<?php
/**
 * Script to update Telegram bot configuration files
 * Updates bot token and chat ID in specified configuration files
 */

// Enable error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

/**
 * Validates and sanitizes input parameters
 * @param string|null $token Bot token
 * @param string|null $chat Chat ID
 * @return array Validated and sanitized parameters or throws exception
 * @throws Exception If parameters are invalid
 */
function validateInputs($token, $chat): array {
    if (!$token || !$chat) {
        throw new Exception("Error: Missing required parameters.");
    }

    // Validate token format (basic check)
    if (!preg_match('/^[\w\-:]+$/', $token)) {
        throw new Exception("Error: Invalid token format.");
    }

    // Validate chat ID format (should be numeric)
    if (!preg_match('/^\d+$/', $chat)) {
        throw new Exception("Error: Invalid chat ID format.");
    }

    return [
        'token' => htmlspecialchars($token, ENT_QUOTES, 'UTF-8'),
        'chat' => htmlspecialchars($chat, ENT_QUOTES, 'UTF-8')
    ];
}

/**
 * Updates configuration files with new values
 * @param array $files List of files to update
 * @param array $patterns Regex patterns to match
 * @param array $replacements Replacement patterns
 * @return array Results of the update operation
 */
function updateConfigFiles(array $files, array $patterns, array $replacements): array {
    $results = [];
    $anyUpdated = false;
    
    foreach ($files as $file) {
        try {
            if (!file_exists($file)) {
                throw new Exception("File not found");
            }

            if (!is_writable($file)) {
                throw new Exception("File not writable");
            }

            // Read file content
            $content = file_get_contents($file);
            if ($content === false) {
                throw new Exception("Unable to read file");
            }

            // Apply regex replacements
            $newContent = preg_replace($patterns, $replacements, $content, -1, $count);
            
            if ($newContent === null) {
                throw new Exception("Regex replacement error");
            }

            if ($count > 0) {
                // Write back the updated content
                if (file_put_contents($file, $newContent) === false) {
                    throw new Exception("Unable to write file");
                }
                $results[$file] = ['status' => 'success', 'message' => "updated"];
                $anyUpdated = true;
            } else {
                $results[$file] = ['status' => 'skip', 'message' => "No changes needed"];
            }
        } catch (Exception $e) {
            $results[$file] = ['status' => 'error', 'message' => $e->getMessage()];
        }
    }
    
    // Simple text response for Python script
    if ($anyUpdated) {
        echo "updated";
        exit;
    }
    
    echo "no changes";
    exit;
}

// Main execution
try {
    // Get and validate inputs
    $token = $_GET['token'] ?? null;
    $chat = $_GET['chat'] ?? null;
    
    $validatedInputs = validateInputs($token, $chat);
    
    // Define configuration
    $files = ["config.php", "telegram.php"];
    
    $patterns = [
        '/\$botToken\s*=\s*["\']([\w:-]+)["\']\s*;/',
        '/\$chat_id\s*=\s*\[\s*["\']?(\d+)["\']?\s*\]\s*;/',
        '/\$id\s*=\s*["\']?(\d+)["\']?\s*;/'
    ];

    $replacements = [
        '$botToken = "' . addslashes($validatedInputs['token']) . '";',
        '$chat_id = ["' . addslashes($validatedInputs['chat']) . '"];',
        '$id = "' . addslashes($validatedInputs['chat']) . '";'
    ];

    // Perform updates
    updateConfigFiles($files, $patterns, $replacements);

} catch (Exception $e) {
    http_response_code(400);
    echo "error: " . $e->getMessage();
    exit;
}
?>