<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

use \Elementor\Widget_Base as WB;
use \Elementor\Controls_Manager as KFCM;
use \Elementor\Icons_Manager as KFIM;

use \Elementor\Group_Control_Image_Size as KFGCIS;
use \Elementor\Group_Control_Background as KFGCBG;
use \Elementor\Group_Control_Typography as KFGCT;
use \Elementor\Group_Control_Border as KFGCBR;
use \Elementor\Group_Control_Box_Shadow as KFGCBS;
use \Elementor\Group_Control_Text_Shadow as KFGCTS;
use \Elementor\Group_Control_Text_Stroke as KFGCTST;
use \Elementor\Group_Control_Css_Filter as KFGCSSF;

class KFW_Ad_Banner extends WB {

	/**
	 * Get element name.
	 *
	 * Retrieve the element name.
	 *
	 * @return string The name.
	 */
	public function get_name() {
        return 'kfw-ad-banner';
    }

	/**
	 * Get element title.
	 *
	 * Retrieve the element title.
	 *
	 * @return string Element title.
	 */
	public function get_title() {
		return esc_html__('Ad Banner', 'onnat-extra');
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the widget categories.
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ '_kfw_core_widgets' ];
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve the widget icon.
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'kfw-badge eicon-wordart kfw-ad-banner';
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the widget keywords.
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'kfw', 'kin', 'ad', 'banner' ];
	}

	/**
	 * Get widget help url.
	 *
	 * Retrieve the widget help url.
	 *
	 * @return string Widget help url.
	 */
	public function get_custom_help_url() {
		return KFE_CONST_THEME_DOC . 'elementor-custom-widgets/ad-banner';
    }

	/**
	 * Get HTML wrapper class.
     *
	 * Retrieve the widget container class.
     *
     * @return string Widget container css classes.
	 */
	public function get_html_wrapper_class() {
        $name = $this->get_name();

		return sprintf( 'kfw-ele-%s elementor-widget-%s', $name, $name );
	}

	/**
	 * Widget base constructor.
	 *
	 * Initializing the widget base class.
	 */
	public function __construct( $data = [], $args = null ) {
        parent::__construct( $data, $args );

		wp_register_style( 'kfw-ad-banner',
			KFE_CONST_URL . 'libraries/elementor/assets/css/ad-banner' . KFE_CONST_DEBUG_SUFFIX . '.css',
			[],
			KFE_CONST_VERSION
		);
    }

	/**
	 * Get style dependencies.
	 *
	 * Retrieve the list of style dependencies the element requires.
	 *
	 * @return array Element styles dependencies.
	 */
	public function get_style_depends() {
		return [ 'kfw-ad-banner' ];
	}

	/**
	 * Register controls.
	 *
	 * Used to add new controls to any element type.
	 */
	protected function register_controls() {
		$this->_register_banner_section();
		$this->_register_content_section();
		$this->_register_button_section();
		$this->_register_banner_style_section();
		$this->_register_content_box_style_section();
		$this->_register_title_style_section();
		$this->_register_sub_title_style_section();
		$this->_register_description_style_section();
		$this->_register_button_style_section();
	}

	public function _register_banner_section() {
		$this->start_controls_section( 'section_banner', [
			'label' => esc_html__( 'Banner Image', 'onnat-extra' ),
		]);
			$this->add_control( 'banner', [
				'label'   => esc_html__( 'Image', 'onnat-extra' ),
				'type'    => KFCM::MEDIA,
				'default' => [ 'url' => KFE_CONST_URL .'libraries/elementor/assets/images/image-placeholder-1.svg' ],
				'dynamic' => [ 'active' => true ],
			]);
			$this->add_group_control( KFGCIS::get_type(), [
				'name'    => 'image_size',
				'default' => 'full',
            ]);
		$this->end_controls_section();
	}

	public function _register_content_section() {
		$this->start_controls_section( 'section_content', [
			'label' => esc_html__( 'Content', 'onnat-extra' )
		]);
			$this->add_control('title',[
				'label'       => esc_html__( 'Title', 'onnat-extra' ),
				'type'        => KFCM::TEXT,
				'dynamic'     => [ 'active' => true ],
				'default'     => esc_html__( 'Sample Title' , 'onnat-extra' ),
				'label_block' => true,
			]);
			$this->add_control( 'show_sub_title', [
				'label'        => esc_html__('Show Sub Title','onnat-extra' ),
				'type'         => KFCM::SWITCHER,
				'label_on'     => esc_html__( 'On', 'onnat-extra' ),
				'label_off'    => esc_html__( 'Off', 'onnat-extra' ),
				'return_value' => 'yes',
				'default'      => 'no'
			]);
			$this->add_control('sub_title',[
				'label'       => esc_html__( 'Sub Title', 'onnat-extra' ),
				'type'        => KFCM::TEXT,
				'dynamic'     => [ 'active' => true, ],
				'placeholder' => esc_html__( 'Enter Sub Title', 'onnat-extra' ),
				'label_block' => true,
				'condition'   => [ 'show_sub_title' => 'yes' ]
			]);
			$this->add_control('sub_title_pos',[
                'label'     => esc_html__( 'Position?', 'onnat-extra' ),
				'type'      => KFCM::CHOOSE,
				'options'   => [
					'top'    => [
						'title' => esc_html__( 'Above Title', 'onnat-extra' ),
						'icon'  => 'eicon-v-align-top',
					],
					'bottom' => [
						'title' => esc_html__( 'Below Title', 'onnat-extra' ),
						'icon'  => 'eicon-v-align-bottom',
					],
				],
				'default'   => 'top',
				'toggle'    => false,
				'condition' => [
					'title!'         => '',
					'show_sub_title' => 'yes',
					'sub_title!'     => ''
				]
			]);
			$this->add_control('description',[
				'label'       => esc_html__( 'Description', 'onnat-extra' ),
				'type'        => KFCM::TEXTAREA,
				'dynamic'     => [ 'active' => true, ],
				'placeholder' => esc_html__( 'Enter Description', 'onnat-extra' ),
				'default'     => esc_html__( 'Lorem ipsum dolor sit amet consectetur adipiscing elit dolor', 'onnat-extra' ),
				'label_block' => true,
			]);
		$this->end_controls_section();
	}

	public function _register_button_section() {
		$this->start_controls_section( 'section_button', [
			'label' => esc_html__( 'Button', 'onnat-extra' )
		]);
            $this->add_control('use_button',[
                'label'        => esc_html__( 'Use Button', 'onnat-extra' ),
                'type'         => KFCM::SWITCHER,
                'return_value' => 'yes',
                'default'      => 'yes'
            ]);
			$this->add_control( 'btn_text', [
				'label'     => esc_html__( 'Text', 'onnat-extra' ),
				'type'      => KFCM::TEXT,
				'default'   => esc_html__( 'Button', 'onnat-extra' ),
				'condition' => [
					'use_button' => 'yes'
				]
			]);
			$this->add_control( 'btn_link', [
				'label'       => esc_html__( 'Link', 'onnat-extra' ),
				'type'        => KFCM::URL,
				'placeholder' => 'https://your-link.com',
				'default'     => [ 'url' => '#' ],
				'conditions'  => [
					'relation' => 'and',
					'terms'    => [
						[
							'name'     => 'use_button',
							'operator' => '===',
							'value'    => 'yes'
						],
						[
							'relation' => 'or',
							'terms'    => [
								[
									'name'     => 'btn_text',
									'operator' => '!=',
									'value'    => ''
								],
								[
									'name'     => 'btn_icon[value]',
									'operator' => '!=',
									'value'    => ''
								],
							],
						]
					]
				],
			]);
			$this->add_control( 'btn_icon', [
				'label'       => esc_html__( 'Icon', 'onnat-extra' ),
				'type'        => KFCM::ICONS,
				'skin'        => 'inline',
				'label_block' => false,
				'condition'   => [
					'use_button' => 'yes'
				],
				'default'     => [
					'value'   => 'fa fa-check',
					'library' => 'fa-solid'
				]
			]);
		$this->end_controls_section();
	}

	public function _register_banner_style_section() {
		$this->start_controls_section( 'section_banner_style', [
			'label'     => esc_html__( 'Banner Image', 'onnat-extra' ),
			'tab'       => KFCM::TAB_STYLE,
		]);
		$this->add_control('animation_type',[
			'label'     => esc_html__( 'Animation', 'onnat-extra' ),
			'type'      => KFCM::SELECT,
			'default'   => '',
			'options'   => [
				''                 => esc_html__('None', 'onnat-extra'),
				'kinfw-zoom-in'    => esc_html__('Zoom In', 'onnat-extra'),
				'kinfw-zoom-out'   => esc_html__('Zoom Out', 'onnat-extra'),
				'kinfw-move-up'    => esc_html__('Move Up', 'onnat-extra'),
				'kinfw-move-down'  => esc_html__('Move Down', 'onnat-extra'),
				'kinfw-move-left'  => esc_html__('Move Left', 'onnat-extra'),
				'kinfw-move-right' => esc_html__('Move Right', 'onnat-extra'),
			],
			'prefix_class' => 'elementor-bg-transform elementor-bg-transform-',
		]);
		$this->add_control('animation_type_duration',[
			'label'     => esc_html__( 'Animation Duration', 'onnat-extra' ),
			'type'      => KFCM::SLIDER,
			'range'     => [
				'px' => [
					'min' => 0,
					'max' => 3000,
				],
			],
			'selectors' => [
				'{{WRAPPER}} .kfw-elementor-ad-banner-image-wrap' => 'transition-duration:{{SIZE}}ms;'
			],
			'condition' => [
				'animation_type!' => ''
			],
		]);
		$this->start_controls_tabs( 'tabs_banner_style' );
			$this->start_controls_tab( 'tabs_banner_normal', [
				'label' => esc_html__( 'Normal', 'onnat-extra' ),
			]);
				$this->add_control( 'normal_bg_overlay', [
					'label'     => esc_html__( 'Background Overlay', 'onnat-extra' ),
					'type'      => KFCM::COLOR,
					'selectors' => [
						'{{WRAPPER}} .kfw-elementor-ad-banner-image-wrap-overlay' => 'background-color: {{VALUE}};'
					],
				]);
				$this->add_group_control( KFGCSSF::get_type(), [
					'name'      => 'normal_bg_css_filters',
					'selector'  => '{{WRAPPER}} .kfw-elementor-ad-banner-image-wrap',
				]);
			$this->end_controls_tab();
			$this->start_controls_tab( 'tabs_banner_hover', [
				'label' => esc_html__( 'Hover', 'onnat-extra' ),
			]);
				$this->add_control( 'hover_bg_overlay', [
					'label'     => esc_html__( 'Background Overlay', 'onnat-extra' ),
					'type'      => KFCM::COLOR,
					'selectors' => [
						'{{WRAPPER}}:hover .kfw-elementor-ad-banner-image-wrap-overlay' => 'background-color: {{VALUE}};'
					],
				]);
				$this->add_group_control( KFGCSSF::get_type(), [
					'name'      => 'hover_bg_css_filters',
					'selector'  => '{{WRAPPER}} .kfw-ele-widget-ad-banner-main:hover .kfw-elementor-ad-banner-image-wrap',
				]);
			$this->end_controls_tab();
		$this->end_controls_tabs();
		$this->end_controls_section();
	}

	public function _register_content_box_style_section() {
		$this->start_controls_section( 'section_content_box_style', [
			'label' => esc_html__( 'Content Box', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
			$this->add_responsive_control('align',[
				'label'        => esc_html__( 'Alignment', 'onnat-extra' ),
				'type'         => KFCM::CHOOSE,
				'options'      => [
					'left'    => [
						'title' => esc_html__( 'Left', 'onnat-extra' ),
						'icon'  => 'eicon-text-align-left',
					],
					'center'  => [
						'title' => esc_html__( 'Center', 'onnat-extra' ),
						'icon'  => 'eicon-text-align-center',
					],
					'right'   => [
						'title' => esc_html__( 'Right', 'onnat-extra' ),
						'icon'  => 'eicon-text-align-right',
					],
				],
				'toggle'       => false,
				'default'      => 'center',
				'prefix_class' => 'kfw-ele-ad-banner%s-align-',
			]);
			$this->add_control( 'horizontal_position', [
				'label'        => esc_html__( 'Horizontal Position', 'onnat-extra' ),
				'type'         => KFCM::CHOOSE,
				'options'      => [
					'left'   => [
						'title' => esc_html__( 'Left', 'onnat-extra' ),
						'icon'  => 'eicon-h-align-left'
					],
					'center' => [
						'title' => esc_html__( 'Center', 'onnat-extra' ),
						'icon'  => 'eicon-h-align-center'
					],
					'right'  => [
						'title' => esc_html__( 'Right', 'onnat-extra' ),
						'icon'  => 'eicon-h-align-right'
					],
				],
				'default'      => 'left',
				'prefix_class' => 'kfw-ele-ad-banner-h-align-',
			]);
			$this->add_control( 'vertical_position', [
				'label'        => esc_html__( 'Vertical Position', 'onnat-extra' ),
				'type'         => KFCM::CHOOSE,
				'options'      => [
					'top'    => [
						'title' => esc_html__( 'Top', 'onnat-extra' ),
						'icon'  => 'eicon-v-align-top'
					],
					'middle' => [
						'title' => esc_html__( 'Middle', 'onnat-extra' ),
						'icon' => 'eicon-v-align-middle'
					],
					'bottom' => [
						'title' => esc_html__( 'Bottom', 'onnat-extra' ),
						'icon'  => 'eicon-v-align-bottom'
					],
				],
				'default'      => 'middle',
				'prefix_class' => 'kfw-ele-ad-banner-v-align-',
			]);
			$this->add_responsive_control('wrap_height',[
				'label'      => esc_html__( 'Height', 'onnat-extra' ),
				'type'       => KFCM::SLIDER,
				'size_units' => [ 'px', '%' ],
				'range'      => [
					'%'  => [
						'min' => 10,
						'max' => 100,
					],
					'px' => [
						'min' => 0,
						'max' => 1000,
					],
				],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-ad-banner-content-wrap' => 'min-height:{{SIZE}}{{UNIT}};'

				],
			]);
			$this->add_responsive_control('c_box_width',[
				'label'      => esc_html__( 'Width', 'onnat-extra' ),
				'type'       => KFCM::SLIDER,
				'size_units' => [ 'px', '%' ],
				'range'      => [
					'%'  => [
						'min' => 10,
                        'max' => 100,
					],
					'px' => [
						'min' => 0,
						'max' => 1000,
					],
				],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-ad-banner-content' => 'width:{{SIZE}}{{UNIT}};'
				],
			]);
		$this->end_controls_section();
	}

	public function _register_title_style_section() {
		$this->start_controls_section( 'section_title_style', [
			'label'     => esc_html__( 'Title', 'onnat-extra' ),
			'tab'       => KFCM::TAB_STYLE,
			'condition' => [ 'title!'=> '' ]
		]);
			$this->add_group_control(KFGCT::get_type(), [
				'name'      => 'title_typo',
				'label'     => esc_html__( 'Typography', 'onnat-extra' ),
				'condition' => [ 'title!'=> '' ],
				'selector'  => '{{WRAPPER}} .kfw-ele-widget-ad-banner-title',
			]);
			$this->add_responsive_control( 'title_margin', [
				'label'      => esc_html__('Margin', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'condition'  => [ 'title!'=> '' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-ad-banner-title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
			$this->add_responsive_control( 'title_padding', [
				'label'      => esc_html__('Padding', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'condition'  => [ 'title!'=> '' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-ad-banner-title' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
			$this->start_controls_tabs( 'title_customize_tabs' );
				$this->start_controls_tab( 'title_normal_state', [
					'label'     => esc_html__( 'Normal', 'onnat-extra' ),
					'condition' => [ 'title!'=> '' ],
				]);
					$this->add_control( 'title_color', [
						'label'     => esc_html__('Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'selectors' => ['{{WRAPPER}} .kfw-ele-widget-ad-banner-title, {{WRAPPER}} .kfw-ele-widget-ad-banner-title a' => 'color: {{VALUE}};' ],
						'condition' => [ 'title!'=> '' ],
					]);
					$this->add_group_control( KFGCTS::get_type(), [
						'name'      => 'title_txt_shadow',
						'condition' => [ 'title!'=> '' ],
						'selector'  => '{{WRAPPER}} .kfw-ele-widget-ad-banner-title, {{WRAPPER}} .kfw-ele-widget-ad-banner-title a',
					]);
					$this->add_group_control( KFGCTST::get_type(), [
						'name'      => 'title_txt_stroke',
						'condition' => [ 'title!'=> '' ],
						'selector'  => '{{WRAPPER}} .kfw-ele-widget-ad-banner-title, {{WRAPPER}} .kfw-ele-widget-ad-banner-title a',
					]);
				$this->end_controls_tab();
				$this->start_controls_tab( 'title_hover_state', [
					'label'     => esc_html__( 'Hover', 'onnat-extra' ),
					'condition' => [ 'title!'=> '' ],
				]);
					$this->add_control( 'title_hover_color', [
						'label'     => esc_html__('Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'selectors' => ['{{WRAPPER}}:hover .kfw-ele-widget-ad-banner-title, {{WRAPPER}}:hover .kfw-ele-widget-ad-banner-title a' => 'color: {{VALUE}};' ],
						'condition' => [ 'title!'=> '' ],
					]);
					$this->add_group_control( KFGCTS::get_type(), [
						'name'      => 'title_hover_txt_shadow',
						'condition' => [ 'title!'=> '' ],
						'selector'  => '{{WRAPPER}}:hover .kfw-ele-widget-ad-banner-title, {{WRAPPER}}:hover .kfw-ele-widget-ad-banner-title a',
					]);
					$this->add_group_control( KFGCTST::get_type(), [
						'name'      => 'title_hover_txt_stroke',
						'condition' => [ 'title!'=> '' ],
						'selector'  => '{{WRAPPER}}:hover .kfw-ele-widget-ad-banner-title, {{WRAPPER}}:hover .kfw-ele-widget-ad-banner-title a',
					]);
				$this->end_controls_tab();
			$this->end_controls_tabs();
		$this->end_controls_section();
	}

	public function _register_sub_title_style_section() {
		$sub_title_condition = [
			'title!'         => '',
			'show_sub_title' => 'yes',
			'sub_title!'     => ''
		];

		$this->start_controls_section( 'section_sub_title_style', [
			'label'     => esc_html__( 'Sub Title', 'onnat-extra' ),
			'tab'       => KFCM::TAB_STYLE,
			'condition' => $sub_title_condition
		]);
			$this->add_group_control(KFGCT::get_type(), [
				'name'      => 'sub_title_typo',
				'label'     => esc_html__( 'Typography', 'onnat-extra' ),
				'condition' => $sub_title_condition,
				'selector'  => '{{WRAPPER}} .kfw-ele-widget-ad-banner-sub-title',
			]);
			$this->add_responsive_control( 'sub_title_margin', [
				'label'      => esc_html__('Margin', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'condition'  => $sub_title_condition,
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-ad-banner-sub-title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
			$this->add_responsive_control( 'sub_title_padding', [
				'label'      => esc_html__('Padding', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'condition'  => $sub_title_condition,
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-ad-banner-sub-title' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
			$this->start_controls_tabs( 'sub_title_customize_tabs' );
				$this->start_controls_tab( 'sub_title_normal_state', [
					'label'     => esc_html__( 'Normal', 'onnat-extra' ),
					'condition' => $sub_title_condition,
				]);
					$this->add_control( 'sub_title_color', [
						'label'     => esc_html__('Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'selectors' => ['{{WRAPPER}} .kfw-ele-widget-ad-banner-sub-title' => 'color: {{VALUE}};' ],
						'condition' => $sub_title_condition,
					]);
					$this->add_group_control( KFGCTS::get_type(), [
						'name'      => 'sub_title_txt_shadow',
						'condition' => $sub_title_condition,
						'selector'  => '{{WRAPPER}} .kfw-ele-widget-ad-banner-sub-title',
					]);
					$this->add_group_control( KFGCTST::get_type(), [
						'name'      => 'sub_title_txt_stroke',
						'condition' => $sub_title_condition,
						'selector'  => '{{WRAPPER}} .kfw-ele-widget-ad-banner-sub-title',
					]);
				$this->end_controls_tab();
				$this->start_controls_tab( 'sub_title_hover_state', [
					'label'     => esc_html__( 'Hover', 'onnat-extra' ),
					'condition' => $sub_title_condition,
				]);
					$this->add_control( 'sub_title_hover_color', [
						'label'     => esc_html__('Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'selectors' => ['{{WRAPPER}}:hover .kfw-ele-widget-ad-banner-sub-title' => 'color: {{VALUE}};' ],
						'condition' => $sub_title_condition,
					]);
					$this->add_group_control( KFGCTS::get_type(), [
						'name'      => 'sub_title_hover_txt_shadow',
						'condition' => $sub_title_condition,
						'selector'  => '{{WRAPPER}}:hover .kfw-ele-widget-ad-banner-sub-title',
					]);
					$this->add_group_control( KFGCTST::get_type(), [
						'name'      => 'sub_title_hover_txt_stroke',
						'condition' => $sub_title_condition,
						'selector'  => '{{WRAPPER}}:hover .kfw-ele-widget-ad-banner-sub-title',
					]);
				$this->end_controls_tab();
			$this->end_controls_tabs();
		$this->end_controls_section();
	}

	public function _register_description_style_section() {
		$this->start_controls_section( 'section_description_style', [
			'label'     => esc_html__( 'Description', 'onnat-extra' ),
			'tab'       => KFCM::TAB_STYLE,
			'condition' => [ 'description!'=> '' ]
		]);
			$this->add_group_control(KFGCT::get_type(), [
				'name'      => 'description_typo',
				'label'     => esc_html__( 'Typography', 'onnat-extra' ),
				'condition' => [ 'description!'=> '' ],
				'selector'  => '{{WRAPPER}} .kfw-ele-widget-ad-banner-description',
			]);
			$this->add_responsive_control( 'description_margin', [
				'label'      => esc_html__('Margin', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'condition'  => [ 'description!'=> '' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-ad-banner-description' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
			$this->add_responsive_control( 'description_padding', [
				'label'      => esc_html__('Padding', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'condition'  => [ 'description!'=> '' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-ad-banner-description' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
			$this->start_controls_tabs( 'description_customize_tabs' );
				$this->start_controls_tab( 'description_normal_state', [
					'label'     => esc_html__( 'Normal', 'onnat-extra' ),
					'condition' => [ 'description!'=> '' ],
				]);
					$this->add_control( 'description_color', [
						'label'     => esc_html__('Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'selectors' => ['{{WRAPPER}} .kfw-ele-widget-ad-banner-description' => 'color: {{VALUE}};' ],
						'condition' => [ 'description!'=> '' ],
					]);
					$this->add_group_control( KFGCTS::get_type(), [
						'name'      => 'description_txt_shadow',
						'condition' => [ 'description!'=> '' ],
						'selector'  => '{{WRAPPER}} .kfw-ele-widget-ad-banner-description',
					]);
					$this->add_group_control( KFGCTST::get_type(), [
						'name'      => 'description_txt_stroke',
						'condition' => [ 'description!'=> '' ],
						'selector'  => '{{WRAPPER}} .kfw-ele-widget-ad-banner-description',
					]);
				$this->end_controls_tab();
				$this->start_controls_tab( 'description_hover_state', [
					'label'     => esc_html__( 'Hover', 'onnat-extra' ),
					'condition' => [ 'description!'=> '' ],
				]);
					$this->add_control( 'description_hover_color', [
						'label'     => esc_html__('Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'selectors' => ['{{WRAPPER}}:hover .kfw-ele-widget-ad-banner-description' => 'color: {{VALUE}};' ],
						'condition' => [ 'description!'=> '' ],
					]);
					$this->add_group_control( KFGCTS::get_type(), [
						'name'      => 'description_hover_txt_shadow',
						'condition' => [ 'description!'=> '' ],
						'selector'  => '{{WRAPPER}}:hover .kfw-ele-widget-ad-banner-description',
					]);
					$this->add_group_control( KFGCTST::get_type(), [
						'name'      => 'description_hover_txt_stroke',
						'condition' => [ 'description!'=> '' ],
						'selector'  => '{{WRAPPER}}:hover .kfw-ele-widget-ad-banner-description',
					]);
				$this->end_controls_tab();
			$this->end_controls_tabs();
		$this->end_controls_section();
	}

	public function _register_button_style_section() {
		$style_condition = [
			'relation' => 'and',
			'terms'    => [
				[
					'name'     => 'use_button',
					'operator' => '===',
					'value'    => 'yes'
				],
				[
					'relation' => 'or',
					'terms'    => [
						[
							'name'     => 'btn_text',
							'operator' => '!=',
							'value'    => ''
						],
						[
							'name'     => 'btn_icon[value]',
							'operator' => '!=',
							'value'    => ''
						],
					],
				]
			]
		];

		$btn_txt_condition = [
			'relation' => 'and',
			'terms'    => [
				[
					'name'     => 'use_button',
					'operator' => '===',
					'value'    => 'yes'
				],
				[
					'name'     => 'btn_text',
					'operator' => '!=',
					'value'    => ''
				],
			]
		];

		$this->start_controls_section( 'section_button_style', [
			'label'      => esc_html__( 'Button', 'onnat-extra' ),
			'tab'        => KFCM::TAB_STYLE,
			'conditions' => $style_condition,
		]);
			$this->add_responsive_control( 'btn_padding', [
				'label'      => esc_html__( 'Padding', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-ad-banner-button' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				],
				'conditions' => $btn_txt_condition
			]);
			$this->add_responsive_control( 'btn_margin', [
				'label'      => esc_html__( 'Margin', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'separator'  => 'after',
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-ad-banner-button-wrap' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				],
				'conditions' => $btn_txt_condition
			]);
			$this->add_control( 'btn_animation_hover', [
				'label'      => esc_html__( 'Hover Animation', 'onnat-extra' ),
				'type'       => KFCM::HOVER_ANIMATION,
				'conditions' => $style_condition,
			]);

			/**
			 * Button Text Style
			 */
				$this->add_control('btn_txt_style_before_hr', [
					'type'       => KFCM::DIVIDER,
					'conditions' => $btn_txt_condition,
				]);
				$this->add_control('btn_txt_style_heading',[
					'label'      => esc_html__( 'BUTTON TEXT', 'onnat-extra' ),
					'type'       => KFCM::HEADING,
					'conditions' => $btn_txt_condition,
				]);
				$this->add_control('btn_txt_style_after_hr', [
					'type'       => KFCM::DIVIDER,
					'conditions' => $btn_txt_condition,
				]);
				$this->add_group_control( KFGCT::get_type(), [
					'name'        => 'btn_typo',
					'description' => esc_html__( 'Typography', 'onnat-extra' ),
					'selector'    => '{{WRAPPER}} .kfw-ele-widget-ad-banner-button',
					'conditions'  => $btn_txt_condition,
				]);

				$this->start_controls_tabs( 'btn_customize_tabs' );
					$this->start_controls_tab( 'btn_normal_state', [
						'label'      => esc_html__( 'Normal', 'onnat-extra' ),
						'conditions' => $btn_txt_condition
					]);
						$this->add_control( 'btn_color', [
							'label'      => esc_html__('Text Color', 'onnat-extra' ),
							'type'       => KFCM::COLOR,
							'selectors'  => ['{{WRAPPER}} .kfw-ele-widget-ad-banner-button' => 'color: {{VALUE}};' ],
							'conditions' => $btn_txt_condition
						]);
						$this->add_group_control( KFGCBG::get_type(), [
							'name'       => 'btn_bg',
							'label'      => esc_html__('Background', 'onnat-extra' ),
							'types'      => [ 'classic', 'gradient' ],
							'exclude'    => [ 'image' ], # To remove background image option in classic mode
							'selector'   => '{{WRAPPER}} .kfw-ele-widget-ad-banner-button',
							'conditions' => $btn_txt_condition
						]);
						$this->add_group_control( KFGCBR::get_type(), [
							'name'           => 'btn_border',
							'fields_options' => [
								'width' => [ 'label' => _x( 'Border Width', 'Border Control', 'onnat-extra' ) ],
								'color' => [ 'label' => _x( 'Border Color', 'Border Control', 'onnat-extra' ) ],
							],
							'selector'       => '{{WRAPPER}} .kfw-ele-widget-ad-banner-button',
							'conditions'     => $btn_txt_condition,
						]);
						$this->add_responsive_control( 'btn_border_radius', [
							'label'      => esc_html__('Border Radius', 'onnat-extra' ),
							'type'       => KFCM::DIMENSIONS,
							'size_units' => [ 'px', '%' ],
							'selectors'  => [ '{{WRAPPER}} .kfw-ele-widget-ad-banner-button' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};' ],
							'conditions' => $btn_txt_condition,
						]);
						$this->add_group_control( KFGCBS::get_type(), [
							'name'       => 'btn_box_shadow',
							'selector'   => '{{WRAPPER}} .kfw-ele-widget-ad-banner-button',
							'conditions' => $btn_txt_condition,
						]);
					$this->end_controls_tab();
					$this->start_controls_tab( 'btn_hover_state', [
						'label'      => esc_html__( 'Hover', 'onnat-extra' ),
						'conditions' => $btn_txt_condition
					]);
						$this->add_control( 'btn_hover_color', [
							'label'      => esc_html__('Text Color', 'onnat-extra' ),
							'type'       => KFCM::COLOR,
							'selectors'  => ['{{WRAPPER}} .kfw-ele-widget-ad-banner-button:hover' => 'color: {{VALUE}};' ],
							'conditions' => $btn_txt_condition
						]);
						$this->add_group_control( KFGCBG::get_type(), [
							'name'       => 'btn_hover_bg',
							'label'      => esc_html__('Background', 'onnat-extra' ),
							'types'      => [ 'classic', 'gradient' ],
							'exclude'    => [ 'image' ], # To remove background image option in classic mode
							'selector'   => '{{WRAPPER}} .kfw-ele-widget-ad-banner-button:hover',
							'conditions' => $btn_txt_condition
						]);
						$this->add_group_control( KFGCBR::get_type(), [
							'name'           => 'btn_hover_border',
							'fields_options' => [
								'width' => [ 'label' => _x( 'Border Width', 'Border Control', 'onnat-extra' ) ],
								'color' => [ 'label' => _x( 'Border Color', 'Border Control', 'onnat-extra' ) ],
							],
							'selector'       => '{{WRAPPER}} .kfw-ele-widget-ad-banner-button:hover',
							'conditions'     => $btn_txt_condition,
						]);
						$this->add_responsive_control( 'btn_hover_border_radius', [
							'label'      => esc_html__('Border Radius', 'onnat-extra' ),
							'type'       => KFCM::DIMENSIONS,
							'size_units' => [ 'px', '%' ],
							'selectors'  => [ '{{WRAPPER}} .kfw-ele-widget-ad-banner-button:hover' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};' ],
							'conditions' => $btn_txt_condition,
						]);
						$this->add_group_control( KFGCBS::get_type(), [
							'name'       => 'btn_hover_box_shadow',
							'selector'   => '{{WRAPPER}} .kfw-ele-widget-ad-banner-button:hover',
							'conditions' => $btn_txt_condition,
						]);
					$this->end_controls_tab();
				$this->end_controls_tabs();

			/**
			 * Button Icon Style
			 */
				$btn_icon_condition = [
					'relation' => 'and',
					'terms'    => [
						[
							'name'     => 'use_button',
							'operator' => '===',
							'value'    => 'yes'
						],
						[
							'name'     => 'btn_icon[value]',
							'operator' => '!=',
							'value'    => ''
						],
					]
				];
				$btn_icon_animation_condition = [
					'relation' => 'and',
					'terms'    => [
						[
							'name'     => 'use_button',
							'operator' => '===',
							'value'    => 'yes'
						],
						[
							'name'     => 'btn_text',
							'operator' => '!=',
							'value'    => ''
						],
						[
							'name'     => 'btn_icon[value]',
							'operator' => '!=',
							'value'    => ''
						],
					]
				];

				$this->add_control('btn_icon_style_before_hr', [
					'type'       => KFCM::DIVIDER,
					'conditions' => $btn_icon_condition,
				]);
				$this->add_control('btn_icon_style_heading',[
					'label'      => esc_html__( 'BUTTON ICON', 'onnat-extra' ),
					'type'       => KFCM::HEADING,
					'conditions' => $btn_icon_condition,
				]);
				$this->add_control('btn_icon_style_after_hr', [
					'type'       => KFCM::DIVIDER,
					'conditions' => $btn_icon_condition,
				]);
				$this->add_responsive_control( 'btn_icon_padding', [
					'label'      => esc_html__( 'Padding', 'onnat-extra' ),
					'type'       => KFCM::DIMENSIONS,
					'size_units' => [ 'px', 'em', '%' ],
					'selectors'  => [
						'{{WRAPPER}} .kfw-ele-widget-ad-banner-button .kfw-elementor-icon-wrap' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
					],
					'conditions' => $btn_icon_condition
				]);
				$this->add_responsive_control( 'btn_icon_margin', [
					'label'      => esc_html__( 'Margin', 'onnat-extra' ),
					'type'       => KFCM::DIMENSIONS,
					'size_units' => [ 'px', 'em', '%' ],
					'separator'  => 'after',
					'selectors'  => [
						'{{WRAPPER}} .kfw-ele-widget-ad-banner-button .kfw-elementor-icon-wrap' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
					],
					'conditions' => $btn_icon_condition
				]);
				$this->add_control( 'btn_icon_pos', [
					'label'      => esc_html__( 'Position', 'onnat-extra' ),
					'type'       => KFCM::CHOOSE,
					'options'    => [
						'left'           => [
							'title' => esc_html__('Left', 'onnat-extra'),
							'icon'  => 'eicon-text-align-left',
						],
						'right'          => [
							'title' => esc_html__('Right', 'onnat-extra'),
							'icon'  => 'eicon-text-align-right',
						],
						'animated-left'  => [
							'title' => esc_html__('Animated Left', 'onnat-extra'),
							'icon'  => 'eicon-animation',
						],
						'animated-right' => [
							'title' => esc_html__('Animated Right', 'onnat-extra'),
							'icon'  => 'eicon-animation kfw-ctrl-rotate-180',
						],
						'toggle'         => [
							'title' => esc_html__('Toogle Button Text', 'onnat-extra'),
							'icon'  => 'eicon-share-arrow',
						]
					],
					'default'    => 'left',
					'toggle'     => false,
					'conditions' => $btn_icon_animation_condition
				]);
				$this->add_control( 'alr_btn_icon_animation', [
					'label'      => esc_html__( 'Animation Direction', 'onnat-extra' ),
					'type'       => KFCM::CHOOSE,
					'options'    => [
						'in'  => [
							'title' => esc_html__('Animate In', 'onnat-extra'),
							'icon'  => 'eicon-chevron-left'
						],
						'out' => [
							'title' => esc_html__('Animate Out', 'onnat-extra'),
							'icon'  => 'eicon-chevron-right'
						],
					],
					'default'    => 'in',
					'toggle'     => false,
					'conditions' => [
						'relation' => 'and',
						'terms'    => [
							[
								'name'     => 'use_button',
								'operator' => '===',
								'value'    => 'yes'
							],
							[
								'name'     => 'btn_text',
								'operator' => '!=',
								'value'    => ''
							],
							[
								'name'     => 'btn_icon[value]',
								'operator' => '!=',
								'value'    => ''
							],
							[
								'name'     => 'btn_icon_pos',
								'operator' => 'in',
								'value'    => ['animated-left', 'animated-right' ]
							]
						]
					]
				]);
				$this->add_control( 'toggle_btn_icon_animation', [
					'label'      => esc_html__( 'Animation Direction', 'onnat-extra' ),
					'type'       => KFCM::CHOOSE,
					'options'    => [
						'top'    => [
							'title' => esc_html__('Top', 'onnat-extra'),
							'icon'  => 'eicon-arrow-up',
						],
						'right'  => [
							'title' => esc_html__('Right', 'onnat-extra'),
							'icon'  => 'eicon-arrow-right',
						],
						'bottom' => [
							'title' => esc_html__('Bottom', 'onnat-extra'),
							'icon'  => 'eicon-arrow-down',
						],
						'left'   => [
							'title' => esc_html__('Left', 'onnat-extra'),
							'icon'  => 'eicon-arrow-left',
						],
					],
					'default'    => 'left',
					'toggle'     => false,
					'conditions' => [
						'relation' => 'and',
						'terms'    => [
							[
								'name'     => 'use_button',
								'operator' => '===',
								'value'    => 'yes'
							],
							[
								'name'     => 'btn_text',
								'operator' => '!=',
								'value'    => ''
							],
							[
								'name'     => 'btn_icon[value]',
								'operator' => '!=',
								'value'    => ''
							],
							[
								'name'     => 'btn_icon_pos',
								'operator' => '===',
								'value'    => 'toggle'
							]
						]
					]
				]);
				$this->add_responsive_control( 'btn_icon_size', [
					'label'      => esc_html__('Size', 'onnat-extra' ),
					'type'       => KFCM::SLIDER,
					'size_units' => [ 'px' ],
					'range'      => [
						'px' => [
							'min' => 0,
							'max' => 200,
						],
					],
					'selectors'  => [
						'{{WRAPPER}} .kfw-ele-widget-ad-banner-button .kfw-elementor-icon-wrap i' => 'font-size: {{SIZE}}{{UNIT}};'
					],
					'conditions' => $btn_icon_condition
				]);
				$this->start_controls_tabs( 'btn_icon_customize_tabs' );
					$this->start_controls_tab( 'btn_icon_normal_state', [
						'label'      => esc_html__( 'Normal', 'onnat-extra' ),
						'conditions' => $btn_icon_condition
					]);
						$this->add_control( 'btn_icon_color', [
							'label'      => esc_html__('Icon Color', 'onnat-extra' ),
							'type'       => KFCM::COLOR,
							'selectors'  => [
								'{{WRAPPER}} .kfw-ele-widget-ad-banner-button .kfw-elementor-icon-wrap i'   => 'color: {{VALUE}};',
								'{{WRAPPER}} .kfw-ele-widget-ad-banner-button .kfw-elementor-icon-wrap svg' => 'fill: {{VALUE}};'
							],
							'conditions' => $btn_icon_condition
						]);
						$this->add_group_control( KFGCBG::get_type(), [
							'name'       => 'btn_icon_bg',
							'label'      => esc_html__('Background', 'onnat-extra' ),
							'types'      => [ 'classic', 'gradient' ],
							'exclude'    => [ 'image' ], # To remove background image option in classic mode
							'selector'   => '{{WRAPPER}} .kfw-ele-widget-ad-banner-button .kfw-elementor-icon-wrap',
							'conditions' => $btn_icon_condition
						]);
						$this->add_group_control( KFGCBR::get_type(), [
							'name'           => 'btn_icon_border',
							'fields_options' => [
								'width' => [ 'label' => _x( 'Border Width', 'Border Control', 'onnat-extra' ) ],
								'color' => [ 'label' => _x( 'Border Color', 'Border Control', 'onnat-extra' ) ],
							],
							'selector'       => '{{WRAPPER}} .kfw-ele-widget-ad-banner-button .kfw-elementor-icon-wrap',
							'conditions'     => $btn_icon_condition,
						]);
						$this->add_responsive_control( 'btn_icon_border_radius', [
							'label'      => esc_html__('Border Radius', 'onnat-extra' ),
							'type'       => KFCM::DIMENSIONS,
							'size_units' => [ 'px', '%' ],
							'selectors'  => [ '{{WRAPPER}} .kfw-ele-widget-ad-banner-button .kfw-elementor-icon-wrap' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};' ],
							'conditions' => $btn_icon_condition,
						]);
						$this->add_group_control( KFGCBS::get_type(), [
							'name'       => 'btn_icon_box_shadow',
							'selector'   => '{{WRAPPER}} .kfw-ele-widget-ad-banner-button .kfw-elementor-icon-wrap',
							'conditions' => $btn_icon_condition,
						]);
					$this->end_controls_tab();
					$this->start_controls_tab( 'btn_icon_hover_state', [
						'label'      => esc_html__( 'Hover', 'onnat-extra' ),
						'conditions' => $btn_icon_condition
					]);
						$this->add_control( 'btn_icon_hover_color', [
							'label'      => esc_html__('Color', 'onnat-extra' ),
							'type'       => KFCM::COLOR,
							'selectors'  => [
								'{{WRAPPER}} .kfw-ele-widget-ad-banner-button:hover .kfw-elementor-icon-wrap i'   => 'color: {{VALUE}};',
								'{{WRAPPER}} .kfw-ele-widget-ad-banner-button:hover .kfw-elementor-icon-wrap svg' => 'color: {{VALUE}};',
							],
							'conditions' => $btn_icon_condition
						]);
						$this->add_group_control( KFGCBG::get_type(), [
							'name'       => 'btn_icon_hover_bg',
							'label'      => esc_html__('Background', 'onnat-extra' ),
							'types'      => [ 'classic', 'gradient' ],
							'exclude'    => [ 'image' ], # To remove background image option in classic mode
							'selector'   => '{{WRAPPER}} .kfw-ele-widget-ad-banner-button:hover .kfw-elementor-icon-wrap',
							'conditions' => $btn_icon_condition
						]);
						$this->add_group_control( KFGCBR::get_type(), [
							'name'           => 'btn_icon_hover_border',
							'fields_options' => [
								'width' => [ 'label' => _x( 'Border Width', 'Border Control', 'onnat-extra' ) ],
								'color' => [ 'label' => _x( 'Border Color', 'Border Control', 'onnat-extra' ) ],
							],
							'selector'       => '{{WRAPPER}} .kfw-ele-widget-ad-banner-button:hover .kfw-elementor-icon-wrap',
							'conditions'     => $btn_icon_condition,
						]);
						$this->add_responsive_control( 'btn_icon_hover_border_radius', [
							'label'      => esc_html__('Border Radius', 'onnat-extra' ),
							'type'       => KFCM::DIMENSIONS,
							'size_units' => [ 'px', '%' ],
							'selectors'  => [ '{{WRAPPER}} .kfw-ele-widget-ad-banner-button:hover .kfw-elementor-icon-wrap' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};' ],
							'conditions' => $btn_icon_condition,
						]);
						$this->add_group_control( KFGCBS::get_type(), [
							'name'       => 'btn_icon_hover_box_shadow',
							'selector'   => '{{WRAPPER}} .kfw-ele-widget-ad-banner-button:hover .kfw-elementor-icon-wrap',
							'conditions' => $btn_icon_condition,
						]);
					$this->end_controls_tab();
				$this->end_controls_tabs();
		$this->end_controls_section();
	}

	/**
	 * Get attachment image src.
	 */
	public static function _get_image_src( $attachment_id, $image_size_key, array $settings ) {
        return KFGCIS::get_attachment_image_src( $attachment_id, $image_size_key, $settings );
	}

	public function _render_img( $settings = [] ) {

		if( !empty( $settings['banner']['id'] ) ) {
			$image_src = $this->_get_image_src( $settings['banner']['id'], 'image_size', $settings );

			$this->add_render_attribute( 'image_attr', 'style', [
				'background-image: url(' . $image_src. ');',
			]);

			echo '<div class="kfw-elementor-ad-banner-image-wrap" '. $this->get_render_attribute_string( 'image_attr' ) .'>';
				echo '<div class="kfw-elementor-ad-banner-image-wrap-overlay"></div>';
			echo '</div>';

		} else if( !empty( $settings['banner']['url'] ) ) {
			$image_src = $settings['banner']['url'];

			$this->add_render_attribute( 'image_attr', 'style', [
				'background-image: url(' . $image_src. ');',
			]);

			echo '<div class="kfw-elementor-ad-banner-image-wrap" '. $this->get_render_attribute_string( 'image_attr' ) .'>';
				echo '<div class="kfw-elementor-ad-banner-image-wrap-overlay"></div>';
			echo '</div>';

		}
	}

	public function _title( $settings = [] ) {
		if( !empty( $settings['title'] ) ) {

			$title = $settings['title'];

			$this->add_render_attribute( 'title', [
				'class' => array_filter([
					'kfw-ele-widget-ad-banner-title'
				])
			]);

			echo '<div '.$this->get_render_attribute_string( 'title' ).'>' . $title . '</div>';
		}
	}

	public function _sub_title( $settings = [] ) {
		if( $settings['show_sub_title'] == 'yes' && !empty( $settings['sub_title'] ) ) {

			$this->add_render_attribute( 'sub-title', [
				'class' => array_filter([
					'kfw-ele-widget-ad-banner-sub-title'
				])
			]);

			echo '<div '.$this->get_render_attribute_string( 'sub-title' ).'>' . $settings['sub_title'] . '</div>';
		}
	}

	public function _description( $settings = [] ) {
		if( !empty( $settings['description'] ) ) {

			$this->add_render_attribute( 'description', [
				'class' => array_filter([
					'kfw-ele-widget-ad-banner-description'
				])
			]);

			echo '<div '.$this->get_render_attribute_string( 'description' ).'>' . esc_html( $settings['description'] ) . '</div>';
		}
	}

	public function _button( $settings = [] ) {
		if( $settings['use_button'] === 'yes' ) {
			$link = array_filter( $settings['btn_link'] );

			$this->add_render_attribute( 'cta-wrapper', [
				'class' => array_filter([
					'kfw-ele-widget-ad-banner-button-wrap',
					!empty( trim( $settings['btn_text'] ) ) ? 'kfw-ele-widget-ad-banner-button-has-txt' : ''
				])
			]);

			echo '<div '.$this->get_render_attribute_string( 'cta-wrapper' ).'>';

				$this->add_render_attribute( 'cta', [
					'class' => array_filter([
						'kfw-ele-widget-ad-banner-button',
						!empty( $settings['btn_animation_hover'] ) ? 'elementor-animation-'.esc_attr( $settings['btn_animation_hover'] ): '',
					])
				]);

				/**
				 * Button Icon related classes
				 */
				if( $settings['btn_icon']['value'] !== '' ) {
					$this->add_render_attribute( 'cta', [
						'class' => array_filter([
							!empty( trim( $settings['btn_text'] ) ) ? 'kfw-ele-widget-ad-banner-button-icon-position-'.$settings['btn_icon_pos'] : '',
							( !empty( trim( $settings['btn_text'] ) ) ) && ( $settings['btn_icon_pos'] === 'animated-left' || $settings['btn_icon_pos'] === 'animated-right' ) ? 'kfw-ele-widget-ad-banner-button-animated-icon-position-' . $settings['alr_btn_icon_animation'] : '',
							( !empty( trim( $settings['btn_text'] ) ) ) && ( $settings['btn_icon_pos'] === 'toggle' ) ? 'kfw-ele-widget-ad-banner-button-toggle-icon-position-' . $settings['toggle_btn_icon_animation'] : '',
						])
					]);
				}

				if ( ! empty( $link['url'] ) ) {
					$this->add_render_attribute( 'cta', 'href', esc_url( $link['url'] ) );

					if ( ! empty( $link['is_external'] ) ) {
						$this->add_render_attribute( 'cta', 'target', '_blank' );
					}

					if ( ! empty( $link['nofollow'] ) ) {
						$this->add_render_attribute( 'cta', 'rel', 'nofollow' );
					}
				}

				echo '<a '.$this->get_render_attribute_string( 'cta' ).'>';

					if( $settings['btn_icon_pos'] === 'left' || $settings['btn_icon_pos'] === 'animated-left' ) {
						$this->_render_icon( $settings['btn_icon'] );
					}

					if( !empty( trim( $settings['btn_text'] ) ) ) {
						printf( '<span class="kfw-elementor-button-text">%s</span>', esc_html( $settings['btn_text'] ) );
					}

					if( $settings['btn_icon_pos'] === 'right'
						|| $settings['btn_icon_pos'] === 'animated-right'
						|| $settings['btn_icon_pos'] === 'toggle'
						|| $settings['btn_icon_pos'] === 'toggle-alt'
					) {
						$this->_render_icon( $settings['btn_icon'] );
					}
				echo '</a>';

			echo '</div>';
		}
	}

	public function _render_icon( $icon , $echo = true ) {
		$icon = array_filter( $icon );

		if( !empty( $icon ) ) {
			echo '<span class="kfw-elementor-icon-wrap">';
				KFIM::render_icon( $icon, [ 'aria-hidden' => 'true' ] );
			echo '</span>';
		}
	}

	/**
	 * Render widget output on the frontend.
	 */
    protected function render() {
		$render = kf_extra_plugin_elementor()->get_template("kfw-ad-banner");
		include $render;
    }

}