<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

use \Elementor\Widget_Base as WB;
use \Elementor\Controls_Manager as KFCM;
use \Elementor\Group_Control_Image_Size as KFGCIS;

use \Elementor\Group_Control_Typography as KFGCT;
use \Elementor\Group_Control_Text_Shadow as KFGCTS;
use \Elementor\Group_Control_Text_Stroke as KFGCTST;
use \Elementor\Group_Control_Background as KFGCBG;
use \Elementor\Group_Control_Border as KFGCBR;
use \Elementor\Group_Control_Box_Shadow as KFGCBS;

class KFW_Blog_List extends WB {

	/**
	 * Get element name.
	 *
	 * Retrieve the element name.
	 *
	 * @return string The name.
	 */
	public function get_name() {
        return 'kfw-blog-list';
    }

	/**
	 * Get element title.
	 *
	 * Retrieve the element title.
	 *
	 * @return string Element title.
	 */
	public function get_title() {
		return esc_html__('Blog List', 'onnat-extra');
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the widget categories.
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ '_kfw_theme_widgets' ];
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve the widget icon.
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'kfw-badge eicon-posts-group kfw-blog-list';
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the widget keywords.
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'kfw', 'kin', 'blog', 'post', 'list' ];
	}

	/**
	 * Get widget help url.
	 *
	 * Retrieve the widget help url.
	 *
	 * @return string Widget help url.
	 */
	public function get_custom_help_url() {
        return KFE_CONST_THEME_DOC . 'elementor-custom-widgets/blog-list';
    }

	/**
	 * Get HTML wrapper class.
     *
	 * Retrieve the widget container class.
     *
     * @return string Widget container css classes.
	 */
	public function get_html_wrapper_class() {
        $name = $this->get_name();

		return sprintf( 'kfw-ele-%s elementor-widget-%s', $name, $name );
	}

	/**
	 * Widget base constructor.
	 *
	 * Initializing the widget base class.
	 */
	public function __construct( $data = [], $args = null ) {
        parent::__construct( $data, $args );
    }

	/**
	 * Get style dependencies.
	 *
	 * Retrieve the list of style dependencies the element requires.
	 *
	 * @return array Element styles dependencies.
	 */
	public function get_style_depends() {
		return [];
	}

	/**
	 * Get script dependencies.
	 *
	 * Retrieve the list of script dependencies the element requires.
	 *
	 * @return array Element scripts dependencies.
	 */
	public function get_script_depends() {
		return [];
	}

	/**
	 * Register controls.
	 *
	 * Used to add new controls to any element type.
	 */
	protected function register_controls() {
		$this->_register_items_section();
		$this->_register_meta_section();
		$this->_register_layout_style_section();
		$this->_register_post_meta_style_section();
        $this->_register_post_title_style_section();
        $this->_register_readmore_btn_style_section();
    }

    public function _register_items_section() {
		$posts = $this->_blog_posts();

		$this->start_controls_section( 'section_items', [
			'label' => esc_html__( 'Blog Posts', 'onnat-extra' ),
			'tab'   => KFCM::TAB_CONTENT,
		]);
			$this->add_control( 'items', [
				'type'        => KFCM::REPEATER,
				'fields'      => [
					[
						'name'        => 'post',
						'label'       => esc_html__( 'Post', 'onnat-extra' ),
						'type'        => KFCM::SELECT2,
						'multiple'    => false,
						'label_block' => true,
						'dynamic'     => [ 'active' => true ],
						'options'     => $posts,
						'default'     => ''
                    ],
                    [
                        'name'         => 'flip',
                        'label'        => esc_html__( 'Flip it?', 'onnat-extra' ),
                        'type'         => KFCM::SWITCHER,
                        'return_value' => 'yes',
                        'default'      => '',
                        'dynamic'      => [ 'active' => true ],
                        'condition'    => [
                            'post!' => ''
                        ]
                    ],
				],
				'title_field' => "<# let labels = ".json_encode( $posts )."; let label = labels[post]; #>{{{ label }}}",
				'default'     => [],
			]);

		$this->end_controls_section();
	}

    public function _register_meta_section() {
		$meta = [
			'author'   => esc_html__( 'Author', 'onnat-extra' ),
			'category' => esc_html__( 'Category', 'onnat-extra' ),
			'comment'  => esc_html__( 'Comment', 'onnat-extra' ),
			'date'     => esc_html__( 'Date', 'onnat-extra' ),
		];

		$this->start_controls_section( 'section_meta', [
			'label' => esc_html__( 'Meta', 'onnat-extra' ),
			'tab'   => KFCM::TAB_CONTENT,
		]);
            $this->add_control( 'meta', [
                'type'        => KFCM::REPEATER,
                'fields'      => [
					[
						'name'        => 'meta',
						'label'       => '',
						'type'        => KFCM::SELECT,
						'classes'     => 'elementor-control-type-hidden',
						'label_block' => true,
						'dynamic'     => [ 'active' => true ],
						'options'     => $meta,
						'default'     => '',
					],
					[
						'name'         => 'status',
						'label'        => esc_html__( 'Enable it?', 'onnat-extra' ),
						'type'         => KFCM::SWITCHER,
						'dynamic'      => [ 'active' => true ],
						'return_value' => 'yes',
						'default'      => 'yes',
					],
                ],
                'default'      => [
					[
                        'meta'   => 'author',
                        'status' => 'yes',
                    ],
					[
                        'meta'   => 'category',
                        'status' => 'no',
                    ],
					[
                        'meta'   => 'comment',
                        'status' => 'no',
                    ],
					[
                        'meta'   => 'date',
                        'status' => 'yes',
                    ],
                ],
                'title_field'  => "<# let labels = ".json_encode( $meta )."; let label = labels[meta]; let s = ( status === 'yes' ) ? 'Enabled' : 'Disabled'; #>".'{{{ label }}} - {{{ s }}}',
                'item_actions' => [
					'add'       => false,
					'duplicate' => false,
					'remove'    => false,
					'sort'      => true,
				],
			]);
		$this->end_controls_section();
    }

    public function _register_layout_style_section() {
		$this->start_controls_section( 'section_layout_style', [
            'label' => esc_html__( 'Layout', 'onnat-extra' ),
            'tab'   => KFCM::TAB_STYLE,
		]);
            $this->add_control('layout_settings_heading',[
                'label' => esc_html__( 'LAYOUT SETTINGS', 'onnat-extra' ),
                'type'  => KFCM::HEADING,
            ]);
            $this->add_control('hr_layout', [
                'type' => KFCM::DIVIDER,
            ]);
            $this->add_group_control( KFGCBG::get_type(), [
                'name'           => 'layout_bg',
                'types'          => ['classic', 'gradient'],
                'exclude'        => [ 'image'],
                'fields_options' => [
                    'background' => [ 'default' => 'classic' ],
                    'color'      => [ 'default' => '' ],
                ],
                'selector'       => '{{WRAPPER}} .kinfw-post-list-item',
            ]);
            $this->add_group_control( KFGCBR::get_type(), [
                'name'      => 'layout_border',
                'selector'  => '{{WRAPPER}} .kinfw-post-list-item',
            ]);
            $this->add_responsive_control( 'layout_border_radius', [
                'label'      => esc_html__('Border Radius', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-post-list-item' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ]
            ]);
            $this->add_group_control( KFGCBS::get_type(), [
                'name'      => 'layout_box_shadow',
                'selector'  => '{{WRAPPER}} .kinfw-post-list-item',
            ]);
            $this->add_responsive_control( 'layout_margin', [
                'label'      => esc_html__('Margin', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-post-list-item' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);            
            $this->add_responsive_control( 'layout_padding', [
                'label'      => esc_html__('Padding', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-post-list-item .kinfw-entry-content-wrap' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->add_control('hr_layout_img_before', [
                'type' => KFCM::DIVIDER,
            ]);
            $this->add_control('layout_image_settings_heading',[
                'label' => esc_html__( 'FEATURED IMAGE SETTINGS', 'onnat-extra' ),
                'type'  => KFCM::HEADING,
            ]);
            $this->add_control('hr_layout_img_after', [
                'type' => KFCM::DIVIDER,
            ]);
            $this->add_group_control( KFGCBG::get_type(), [
                'name'           => 'featured_img_bg',
                'types'          => ['classic', 'gradient'],
                'exclude'        => [ 'image'],
                'fields_options' => [
                    'background' => [ 'default' => 'classic' ],
                    'color'      => [ 'default' => '' ],
                ],
                'selector'       => '{{WRAPPER}} .kinfw-post-list-item .kinfw-entry-media-wrap',
            ]);
            $this->add_group_control( KFGCBR::get_type(), [
                'name'      => 'featured_img_border',
                'selector'  => '{{WRAPPER}} .kinfw-post-list-item .kinfw-entry-media-wrap',
            ]);
            $this->add_responsive_control( 'featured_img_border_radius', [
                'label'      => esc_html__('Border Radius', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-post-list-item .kinfw-entry-media-wrap, {{WRAPPER}} .kinfw-post-list-item .kinfw-entry-media-wrap > img' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ]
            ]);
            $this->add_group_control( KFGCBS::get_type(), [
                'name'      => 'featured_img_box_shadow',
                'selector'  => '{{WRAPPER}} .kinfw-post-list-item .kinfw-entry-media-wrap'
            ]);
            $this->add_responsive_control( 'featured_img_margin', [
                'label'      => esc_html__('Margin', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-post-list-item .kinfw-entry-media-wrap' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->add_responsive_control( 'featured_img_padding', [
                'label'      => esc_html__('Padding', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-post-list-item .kinfw-entry-media-wrap' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);            
		$this->end_controls_section();
    }

    public function _register_post_meta_style_section() {
		$this->start_controls_section( 'section_meta_style', [
			'label' => esc_html__( 'Post Meta', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
            $this->add_control('meta_icon_settings_heading',[
                'label'     => esc_html__( 'ICON SETTINGS', 'onnat-extra' ),
                'type'      => KFCM::HEADING,
            ]);
            $this->add_control('hr_1', [
                'type'      => KFCM::DIVIDER,
            ]);
            $this->add_responsive_control( 'meta_icon_size', [
                'label'      => esc_html__('Icon Size', 'onnat-extra' ),
                'type'       => KFCM::SLIDER,
                'size_units' => [ 'px' ],
                'range'      => [
                    'px' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-entry-meta-wrap span.kinfw-icon' => 'font-size: {{SIZE}}{{UNIT}};'
                ],
            ]);
            $this->add_control( 'meta_icon_color', [
                'label'     => esc_html__('Color', 'onnat-extra' ),
                'type'      => KFCM::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .kinfw-entry-meta-wrap span.kinfw-icon' => 'color:{{VALUE}};',
                ],
            ]);
            $this->add_responsive_control( 'meta_icon_margin', [
                'label'      => esc_html__('Margin', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-entry-meta-wrap span.kinfw-icon' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ],
            ]);
            $this->add_control('meta_typo_settings_heading',[
                'label' => esc_html__( 'TYPO SETTINGS', 'onnat-extra' ),
                'type'  => KFCM::HEADING,
            ]);
            $this->add_control('hr_2', [
                'type' => KFCM::DIVIDER,
            ]);
            $this->add_group_control(KFGCT::get_type(), [
                'name'     => 'meta_info_typo',
                'label'    => esc_html__( 'Typography', 'onnat-extra' ),
                'selector' => '
                    {{WRAPPER}} .kinfw-entry-meta-wrap a,
                    {{WRAPPER}} .kinfw-entry-meta-wrap .kinfw-meta-author a span:last-child,
                    {{WRAPPER}} .kinfw-entry-meta-wrap .kinfw-meta-date time
                    ',
            ]);
            $this->start_controls_tabs( 'meta_info_customize_tabs' );
                $this->start_controls_tab( 'meta_info_normal_state', [
                    'label' => esc_html__( 'Normal', 'onnat-extra' ),
                ]);
                    $this->add_control( 'meta_info_color', [
                        'label'     => esc_html__('Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'selectors' => ['{{WRAPPER}} .kinfw-entry-meta-wrap a, {{WRAPPER}} .kinfw-entry-meta-wrap .kinfw-meta-date time' => 'color: {{VALUE}};' ],
                    ]);
                    $this->add_group_control( KFGCTS::get_type(), [
                        'name'     => 'meta_info_txt_shadow',
                        'selector' => '{{WRAPPER}} .kinfw-entry-meta-wrap a, {{WRAPPER}} .kinfw-entry-meta-wrap .kinfw-meta-date time',
                    ]);
                    $this->add_group_control( KFGCTST::get_type(), [
                        'name'     => 'meta_info_txt_stroke',
                        'selector' => '{{WRAPPER}} .kinfw-entry-meta-wrap a, {{WRAPPER}} .kinfw-entry-meta-wrap .kinfw-meta-date time',
                    ]);
                $this->end_controls_tab();
                $this->start_controls_tab( 'meta_info_hover_state', [
                    'label' => esc_html__( 'Hover', 'onnat-extra' ),
                ]);
                    $this->add_control( 'meta_info_hover_color', [
                        'label'     => esc_html__('Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'selectors' => ['{{WRAPPER}} .kinfw-entry-meta-wrap a:hover' => 'color: {{VALUE}};' ],
                    ]);
                    $this->add_group_control( KFGCTS::get_type(), [
                        'name'     => 'meta_info_hover_txt_shadow',
                        'selector' => '{{WRAPPER}} .kinfw-entry-meta-wrap a:hover',
                    ]);
                    $this->add_group_control( KFGCTST::get_type(), [
                        'name'     => 'meta_info_hover_txt_stroke',
                        'selector' => '{{WRAPPER}} .kinfw-entry-meta-wrap a:hover',
                    ]);
                $this->end_controls_tab();
            $this->end_controls_tabs();
		$this->end_controls_section();
    }    

    public function _register_post_title_style_section() {
		$this->start_controls_section( 'section_p_title_style', [
			'label' => esc_html__( 'Post Title', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
            $this->add_group_control(KFGCT::get_type(), [
                'name'     => 'title_typo',
                'label'    => esc_html__( 'Typography', 'onnat-extra' ),
                'selector' => '{{WRAPPER}} .kinfw-entry-header h4',
            ]);
            $this->add_responsive_control( 'title_margin', [
                'label'      => esc_html__('Margin', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-entry-header h4' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->add_responsive_control( 'title_padding', [
                'label'      => esc_html__('Padding', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-entry-header h4' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->start_controls_tabs( 'title_customize_tabs' );
                $this->start_controls_tab( 'title_normal_state', [
                    'label' => esc_html__( 'Normal', 'onnat-extra' ),
                ]);
                    $this->add_control( 'title_color', [
                        'label'     => esc_html__('Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'selectors' => ['{{WRAPPER}} .kinfw-entry-header h4 a' => 'color: {{VALUE}};' ],
                    ]);
                    $this->add_group_control( KFGCTS::get_type(), [
                        'name'     => 'title_txt_shadow',
                        'selector' => '{{WRAPPER}} .kinfw-entry-header h4',
                    ]);
                    $this->add_group_control( KFGCTST::get_type(), [
                        'name'     => 'title_txt_stroke',
                        'selector' => '{{WRAPPER}} .kinfw-entry-header h4',
                    ]);
                $this->end_controls_tab();
                $this->start_controls_tab( 'title_hover_state', [
                    'label' => esc_html__( 'Hover', 'onnat-extra' ),
                ]);
                    $this->add_control( 'title_hover_color', [
                        'label'     => esc_html__('Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'selectors' => ['{{WRAPPER}} .kinfw-entry-header h4 a:hover' => 'color: {{VALUE}};' ],
                    ]);
                    $this->add_group_control( KFGCTS::get_type(), [
                        'name'     => 'title_hover_txt_shadow',
                        'selector' => '{{WRAPPER}} .kinfw-entry-header h4 a:hover',
                    ]);
                    $this->add_group_control( KFGCTST::get_type(), [
                        'name'     => 'title_hover_txt_stroke',
                        'selector' => '{{WRAPPER}} .kinfw-entry-header h4 a:hover',
                    ]);
                $this->end_controls_tab();
            $this->end_controls_tabs();
		$this->end_controls_section();
    }

    public function _register_readmore_btn_style_section() {
		$this->start_controls_section( 'section_readmore_btn_style', [
            'label' => esc_html__( 'Read More Button', 'onnat-extra' ),
            'tab'   => KFCM::TAB_STYLE,
		]);
            $this->add_group_control(KFGCT::get_type(), [
                'name'      => 'readmore_btn_typo',
                'label'     => esc_html__( 'Text', 'onnat-extra' ),
                'selector'  => '{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a',
            ]);
            $this->add_responsive_control( 'readmore_btn_icon_size', [
                'label'      => esc_html__('Icon Size', 'onnat-extra' ),
                'type'       => KFCM::SLIDER,
                'size_units' => [ 'px' ],
                'range'      => [
                    'px' => [
                        'min' => 0,
                        'max' => 999,
                    ],
                ],
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a .kinfw-icon' => 'font-size: {{SIZE}}{{UNIT}};'
                ],
            ]);
            $this->add_responsive_control( 'readmore_btn_margin', [
                'label'      => esc_html__('Margin', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->add_responsive_control( 'readmore_btn_padding', [
                'label'      => esc_html__('Padding', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->start_controls_tabs( 'readmore_btn_txt_customize_tabs');
                $this->start_controls_tab( 'readmore_btn_txt_normal_state', [
                    'label' => esc_html__( 'Normal', 'onnat-extra' ),
                ]);
                    $this->add_control( 'readmore_btn_txt_color', [
                        'label'     => esc_html__('Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'selectors' => ['{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a' => 'color: {{VALUE}};' ],
                    ]);
                    $this->add_group_control( KFGCTS::get_type(), [
                        'name'     => 'readmore_btn_txt_txt_shadow',
                        'selector' => '{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a',
                    ]);
                    $this->add_group_control( KFGCTST::get_type(), [
                        'name'     => 'readmore_btn_txt_txt_stroke',
                        'selector' => '{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a',
                    ]);
                    $this->add_control( 'readmore_btn_icon_color', [
                        'label'     => esc_html__('Icon Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'selectors' => [
                            '{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a .kinfw-icon' => 'color:{{VALUE}}; fill:{{VALUE}};',
                        ],
                    ]);
					$this->add_group_control( KFGCBG::get_type(), [
						'name'           => 'readmore_btn_bg',
						'types'          => ['classic', 'gradient'],
						'exclude'        => [ 'image'],
						'fields_options' => [
							'background' => [ 'default' => 'classic' ],
							'color'      => [ 'default' => '' ],
						],
						'selector'       => '{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a',
					]);
					$this->add_group_control( KFGCBR::get_type(), [
						'name'      => 'readmore_btn_border',
						'selector'  => '{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a'
					]);
					$this->add_responsive_control( 'readmore_btn_border_radius', [
						'label'      => esc_html__('Border Radius', 'onnat-extra' ),
						'type'       => KFCM::DIMENSIONS,
						'size_units' => [ 'px', '%' ],
						'selectors'  => [
							'{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
						]
					]);
					$this->add_group_control( KFGCBS::get_type(), [
						'name'      => 'readmore_btn_box_shadow',
						'selector'  => '{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a'
					]);
                $this->end_controls_tab();
                $this->start_controls_tab( 'readmore_btn_txt_hover_state', [
                    'label' => esc_html__( 'Hover', 'onnat-extra' ),
                ]);
                    $this->add_control( 'readmore_btn_txt_h_color', [
                        'label'     => esc_html__('Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'selectors' => ['{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a:hover' => 'color: {{VALUE}};' ],
                    ]);
                    $this->add_group_control( KFGCTS::get_type(), [
                        'name'     => 'readmore_btn_txt_h_txt_shadow',
                        'selector' => '{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a:hover',
                    ]);
                    $this->add_group_control( KFGCTST::get_type(), [
                        'name'     => 'readmore_btn_txt_h_txt_stroke',
                        'selector' => '{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a:hover',
                    ]);
                    $this->add_control( 'readmore_btn_icon_h_color', [
                        'label'     => esc_html__('Icon Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'selectors' => [
                            '{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a:hover .kinfw-icon' => 'color:{{VALUE}}; fill:{{VALUE}};',
                        ],
                    ]);
					$this->add_group_control( KFGCBG::get_type(), [
						'name'           => 'readmore_btn_h_bg',
						'types'          => ['classic', 'gradient'],
						'exclude'        => [ 'image'],
						'fields_options' => [
							'background' => [ 'default' => 'classic' ],
							'color'      => [ 'default' => '' ],
						],
						'selector'       => '{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a:hover',
					]);
					$this->add_group_control( KFGCBR::get_type(), [
						'name'      => 'readmore_btn_h_border',
						'selector'  => '{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a:hover'
					]);
					$this->add_responsive_control( 'readmore_btn_h_border_radius', [
						'label'      => esc_html__('Border Radius', 'onnat-extra' ),
						'type'       => KFCM::DIMENSIONS,
						'size_units' => [ 'px', '%' ],
						'selectors'  => [
							'{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a:hover' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
						]
					]);
					$this->add_group_control( KFGCBS::get_type(), [
						'name'      => 'readmore_btn_h_box_shadow',
						'selector'  => '{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a:hover'
					]);
                $this->end_controls_tab();
            $this->end_controls_tabs();

		$this->end_controls_section();
    }

	public function _blog_posts() {
		$result = [ esc_html__( '-- Select --', 'onnat-extra' ) ];

		$posts = get_posts([
			'post_type'      => 'post',
			'post_status'    => 'publish',
			'posts_per_page' => -1
		]);

		if ( ! empty( $posts ) && ! is_wp_error( $posts ) ) {
			foreach ( $posts as $item ) {
				$result[ $item->ID ] = $item->post_title;
			}
		}

		return $result;
	}

    public function _post_ids( $items ) {
        $post_ids = [];
        foreach( $items as $item ) {
            if( $item['post'] !== "0" ) {
                $post_ids[] = $item['post'];
            }
        }

        return array_filter( $post_ids );
    }

    public function _post_data( $items ) {
        $post_data = [];
        foreach( $items as $item ) {
            if( $item['post'] !== "0" ) {
                $post_data[ $item['post'] ] = $item;
            }
        }

        return array_filter( $post_data );
    }

    public function _post_meta( $meta ) {
        $data = [];

        foreach( $meta as $post_meta ) {
            if( $post_meta['status'] == 'yes' ) {
                $data[] = $post_meta['meta'];
            }
        }

        return array_filter( $data );
    }

	/**
	 * Render widget output on the frontend.
	 */
    protected function render() {
		$render = kf_extra_plugin_elementor()->get_template("kfw-blog-list");
		include $render;
    }
}