<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

use \Elementor\Plugin as KFEl;
use \Elementor\Widget_Base as WB;
use \Elementor\Controls_Manager as KFCM;
use \Elementor\Group_Control_Image_Size as KFGCIS;

use \Elementor\Group_Control_Typography as KFGCT;
use \Elementor\Group_Control_Text_Shadow as KFGCTS;
use \Elementor\Group_Control_Text_Stroke as KFGCTST;
use \Elementor\Group_Control_Background as KFGCBG;
use \Elementor\Group_Control_Border as KFGCBR;
use \Elementor\Group_Control_Box_Shadow as KFGCBS;

class KFW_Blog_Single_Post_Excerpt extends WB {

	/**
	 * Get element name.
	 *
	 * Retrieve the element name.
	 *
	 * @return string The name.
	 */
	public function get_name() {
        return 'kfw-blog-single-post-excerpt';
    }

	/**
	 * Get element title.
	 *
	 * Retrieve the element title.
	 *
	 * @return string Element title.
	 */
	public function get_title() {
		return esc_html__('Post Excerpt', 'onnat-extra');
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the widget categories.
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ '_kfw_blog_post_look_widgets' ];
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve the widget icon.
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'kfw-badge eicon-post-excerpt';
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the widget keywords.
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'kfw', 'kin', 'blog', 'post', 'single', 'excerpt' ];
	}

	/**
	 * Get widget help url.
	 *
	 * Retrieve the widget help url.
	 *
	 * @return string Widget help url.
	 */
	public function get_custom_help_url() {
        return KFE_CONST_THEME_DOC . 'elementor-custom-widgets/blog-single-post-excerpt';
    }

	/**
	 * Get HTML wrapper class.
     *
	 * Retrieve the widget container class.
     *
     * @return string Widget container css classes.
	 */
	public function get_html_wrapper_class() {
        $name = $this->get_name();

		return sprintf( 'kfw-ele-%s elementor-widget-%s', $name, $name );
	}

	/**
	 * Widget base constructor.
	 *
	 * Initializing the widget base class.
	 */
	public function __construct( $data = [], $args = null ) {
        parent::__construct( $data, $args );

        wp_register_style( 'kfw-blog-single',
            KFE_CONST_URL . 'libraries/elementor/assets/css/blog-single' . KFE_CONST_DEBUG_SUFFIX . '.css',
            [],
            KFE_CONST_VERSION
        );

    }

	/**
	 * Get style dependencies.
	 *
	 * Retrieve the list of style dependencies the element requires.
	 *
	 * @return array Element styles dependencies.
	 */
	public function get_style_depends() {
		return [ 'kfw-blog-single' ];
	}

	/**
	 * Get script dependencies.
	 *
	 * Retrieve the list of script dependencies the element requires.
	 *
	 * @return array Element scripts dependencies.
	 */
	public function get_script_depends() {
		return [];
	}

	/**
	 * Register controls.
	 *
	 * Used to add new controls to any element type.
	 */
	protected function register_controls() {
		$this->_register_excerpt_section();
		$this->_register_excerpt_style_section();
	}

	public function _register_excerpt_section() {
		$this->start_controls_section( 'section_excerpt', [
			'label' => esc_html__( 'Excerpt', 'onnat-extra' )
		]);
			$this->add_control( 'limit', [
				'label'   => esc_html__( 'Limit', 'onnat-extra' ),
				'type'    => KFCM::NUMBER,
				'min'     => 4,
				'max'     => 900,
				'step'    => 5,
				'default' => 30,
			]);
			$this->add_control( 'type', [
				'label'     => esc_html__( 'Type', 'onnat-extra' ),
				'type'      => KFCM::CHOOSE,
				'options'   => [
					'chars' => [
						'title' => esc_html__('By Characters', 'onnat-extra'),
						'icon'  => 'eicon-t-letter',
					],
					'words' => [
						'title' => esc_html__('By Words', 'onnat-extra'),
						'icon'  => 'eicon-wordpress-light',
					],
				],
				'default'   => 'chars',
				'condition' => [
					'limit!' => 0
				],
				'toggle'    => false,
			]);
		$this->end_controls_section();
	}

	public function _register_excerpt_style_section() {
		$this->start_controls_section( 'section_excerpt_style', [
			'label' => esc_html__( 'Excerpt', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
			$this->add_responsive_control('align',[
				'label'        => esc_html__( 'Alignment', 'onnat-extra' ),
				'type'         => KFCM::CHOOSE,
				'options'      => [
					'left'    => [
						'title' => esc_html__( 'Left', 'onnat-extra' ),
						'icon'  => 'eicon-text-align-left',
					],
					'center'  => [
						'title' => esc_html__( 'Center', 'onnat-extra' ),
						'icon'  => 'eicon-text-align-center',
					],
					'right'   => [
						'title' => esc_html__( 'Right', 'onnat-extra' ),
						'icon'  => 'eicon-text-align-right',
					],
					'justify' => [
						'title' => esc_html__( 'Justified', 'onnat-extra' ),
						'icon'  => 'eicon-text-align-justify',
					],
				],
				'toggle'       => false,
				'default'      => 'center',
				'selectors'    => [
					'{{WRAPPER}} .kfw-ele-widget-post-excerpt' => 'text-align: {{VALUE}};',
				],
			]);
            $this->add_group_control(KFGCT::get_type(), [
                'name'      => 'typo',
                'label'     => esc_html__( 'Typography', 'onnat-extra' ),
                'selector'  => '{{WRAPPER}} .kfw-ele-widget-post-excerpt',
            ]);
			$this->add_control( 'color', [
				'label'     => esc_html__('Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'selectors' => [
					'{{WRAPPER}} .kfw-ele-widget-post-excerpt' => 'color: {{VALUE}};'
				],
			]);
			$this->add_group_control( KFGCTS::get_type(), [
				'name'      => 'txt_shadow',
				'selector'  => '{{WRAPPER}} .kfw-ele-widget-post-excerpt',
			]);
			$this->add_group_control( KFGCTST::get_type(), [
				'name'      => 'txt_stroke',
				'selector'  => '{{WRAPPER}} .kfw-ele-widget-post-excerpt',
			]);
			$this->add_control( 'blend_mode', [
				'label'     => esc_html__('Blend Mode', 'onnat-extra' ),
				'type'      => KFCM::SELECT,
				'options'   => [
					''            => esc_html__( 'Normal', 'onnat-extra' ),
					'multiply'    => esc_html__( 'Multiply', 'onnat-extra' ),
					'screen'      => esc_html__( 'Screen', 'onnat-extra' ),
					'overlay'     => esc_html__( 'Overlay', 'onnat-extra' ),
					'darken'      => esc_html__( 'Darken', 'onnat-extra' ),
					'lighten'     => esc_html__( 'Lighten', 'onnat-extra' ),
					'color-dodge' => esc_html__( 'Color Dodge', 'onnat-extra' ),
					'saturation'  => esc_html__( 'Saturation', 'onnat-extra' ),
					'color'       => esc_html__( 'Color', 'onnat-extra' ),
					'difference'  => esc_html__( 'Difference', 'onnat-extra' ),
					'exclusion'   => esc_html__( 'Exclusion', 'onnat-extra' ),
					'hue'         => esc_html__( 'Hue', 'onnat-extra' ),
					'luminosity'  => esc_html__( 'Luminosity', 'onnat-extra' ),
				],
				'selectors' => [
					'{{WRAPPER}} .kfw-ele-widget-post-excerpt' => 'mix-blend-mode: {{VALUE}};'
				],
			]);
		$this->end_controls_section();
	}

    public function _is_editor_mode() {
        $preview = KFEl::instance()->preview->is_preview_mode();
        $editor  = KFEl::instance()->editor->is_edit_mode();

        return ( $preview || $editor ) ? true : false;
    }

	public function _excerpt() {
		$settings = $this->get_settings();
		$post_id  = get_the_id();
		$mode     = $this->_is_editor_mode();

		if( $mode ) {
			$post_id = kf_extra_plugin_elementor()->get_latest_post_id( 'post' );
		}

		$limit   = $settings['limit'];
		$excerpt = get_the_excerpt($post_id);

		if( $limit > 0 ) {
			if( 'chars' === $settings['type'] ) {
				$excerpt = substr($excerpt, 0, $limit);
			} else if( 'words' === $settings['type'] ) {
				$words = explode(' ', $excerpt, $limit + 1);
				if (count($words) > $limit) {
					array_pop($words);
				}

				$excerpt = implode(' ', $words);
			}
		}

		if( !empty( $excerpt ) ) {
			echo '<div class="kfw-ele-widget-post-excerpt">'. $excerpt .'</div>';
		}

	}

	/**
	 * Render widget output on the frontend.
	 */
    protected function render() {
		$render = kf_extra_plugin_elementor()->get_template("kfw-blog-single-post-excerpt");
		include $render;
    }
}