<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}
use \Elementor\Widget_Base as WB;
use \Elementor\Controls_Manager as KFCM;

use \Elementor\Group_Control_Background as KFGCBG;
use \Elementor\Group_Control_Typography as KFGCT;
use \Elementor\Group_Control_Border as KFGCBR;
use \Elementor\Group_Control_Box_Shadow as KFGCBS;
use \Elementor\Group_Control_Text_Shadow as KFGCTS;
use \Elementor\Group_Control_Text_Stroke as KFGCTST;

class KFW_Blog_Template extends WB {

	/**
	 * Get element name.
	 *
	 * Retrieve the element name.
	 *
	 * @return string The name.
	 */
	public function get_name() {
        return 'kfw-blog-template';
    }

	/**
	 * Get element title.
	 *
	 * Retrieve the element title.
	 *
	 * @return string Element title.
	 */
	public function get_title() {
		return esc_html__('Blog Template', 'onnat-extra');
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the widget categories.
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ '_kfw_theme_widgets' ];
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve the widget icon.
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'kfw-badge eicon-posts-group';
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the widget keywords.
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'kfw', 'blog', 'template' ];
	}

	/**
	 * Get widget help url.
	 *
	 * Retrieve the widget help url.
	 *
	 * @return string Widget help url.
	 */
	public function get_custom_help_url() {
        return KFE_CONST_THEME_DOC . 'elementor-custom-widgets/blog-template';
    }

	/**
	 * Get HTML wrapper class.
     *
	 * Retrieve the widget container class.
     *
     * @return string Widget container css classes.
	 */
	public function get_html_wrapper_class() {
        $name = $this->get_name();

		return sprintf( 'kfw-ele-%s elementor-widget-%s', $name, $name );
	}

	/**
	 * Widget base constructor.
	 *
	 * Initializing the widget base class.
	 */
	public function __construct( $data = [], $args = null ) {
        parent::__construct( $data, $args );

		wp_register_style( 'kfw-blog-template',
			KFE_CONST_URL . 'libraries/elementor/assets/css/blog-template' . KFE_CONST_DEBUG_SUFFIX . '.css',
			[],
			KFE_CONST_VERSION
		);
    }

	/**
	 * Get style dependencies.
	 *
	 * Retrieve the list of style dependencies the element requires.
	 *
	 * @return array Element styles dependencies.
	 */
	public function get_style_depends() {
		return [ 'kfw-blog-template' ];
	}

	/**
	 * Register controls.
	 *
	 * Used to add new controls to any element type.
	 */
	protected function register_controls() {
		$this->_register_settings_section();
		$this->_register_query_settings_section();

        $this->_register_layout_style_section();

        $this->_register_post_date_meta_style_section();
		$this->_register_s4_post_date_meta_style_section();
        $this->_register_post_meta_style_section();

        $this->_register_post_title_style_section();
        $this->_register_post_expert_style_section();

        $this->_register_readmore_btn_style_section();
        $this->_register_readmore_btn_icon_style_section();
        $this->_register_pagination_style_section();
    }

	public function _register_settings_section() {
		$this->start_controls_section( 'section_settings', [
			'label' => esc_html__( 'Settings', 'onnat-extra' ),
			'tab'   => KFCM::TAB_CONTENT,
		]);
            $this->add_control('layout', [
                'label'   => esc_html__( 'Layout', 'onnat-extra' ),
                'type'    => KFCM::CHOOSE,
                'options' => [
                    'standard' => [
                        'title' => esc_html__('Standard', 'onnat-extra' ),
                        'icon'  => 'eicon-form-horizontal'
                    ],
                    'grid-1'   => [
                        'title' => esc_html__('Grid Style 1', 'onnat-extra' ),
                        'icon'  => 'eicon-gallery-grid'
                    ],
                    'grid-2'   => [
                        'title' => esc_html__('Grid Style 2', 'onnat-extra' ),
                        'icon'  => 'eicon-posts-grid'
                    ],
                    'grid-3'   => [
                        'title' => esc_html__('Grid Style 3', 'onnat-extra' ),
                        'icon'  => 'eicon-container-grid',
                    ],
                    'grid-4'   => [
                        'title' => esc_html__('Grid Style 4', 'onnat-extra' ),
						'icon'  => 'eicon-apps',
                    ],
                ],
                'default' => 'grid-3',
                'toggle'  => false
            ]);
			$this->add_responsive_control('cols', [
				'label'          => esc_html__( 'Columns', 'onnat-extra' ),
				'type'           => KFCM::CHOOSE,
				'options'        => [
					'1' => [ 'title' => esc_html__('1 Column', 'onnat-extra'), 'icon' => 'kfw-ctrl-icon kfw-ctrl-numeric-icon' ],
					'2' => [ 'title' => esc_html__('2 Columns', 'onnat-extra'), 'icon' => 'kfw-ctrl-icon kfw-ctrl-numeric-2-icon' ],
					'3' => [ 'title' => esc_html__('3 Columns', 'onnat-extra'), 'icon' => 'kfw-ctrl-icon kfw-ctrl-numeric-3-icon' ],
					'4' => [ 'title' => esc_html__('4 Columns', 'onnat-extra'), 'icon' => 'kfw-ctrl-icon kfw-ctrl-numeric-4-icon' ],
				],
				'devices'        => [ 'widescreen', 'desktop', 'laptop', 'tablet', 'tablet_extra' ],
                'default'        => '3',
                'tablet_default' => '2',
                'mobile_default' => '1',
                'toggle'         => false,
                'selectors'      => [
                    '{{WRAPPER}} .kfw-ele-widget-blog-tpl-main.kfw-ele-widget-blog-tpl-grid-1 .kinfw-blog-posts-wrap' => 'display:grid; grid-template-columns: repeat({{SIZE}}, minmax(0, 1fr));',
                    '{{WRAPPER}} .kfw-ele-widget-blog-tpl-main.kfw-ele-widget-blog-tpl-grid-2 .kinfw-blog-posts-wrap' => 'display:grid; grid-template-columns: repeat({{SIZE}}, minmax(0, 1fr));',
                    '{{WRAPPER}} .kfw-ele-widget-blog-tpl-main.kfw-ele-widget-blog-tpl-grid-3 .kinfw-blog-posts-wrap' => 'display:grid; grid-template-columns: repeat({{SIZE}}, minmax(0, 1fr));',
                    '{{WRAPPER}} .kfw-ele-widget-blog-tpl-main.kfw-ele-widget-blog-tpl-grid-4 .kinfw-blog-posts-wrap' => 'display:grid; grid-template-columns: repeat({{SIZE}}, minmax(0, 1fr));',
                ],
                'condition'      => [
                    'layout' => [ 'grid-1', 'grid-2', 'grid-3', 'grid-4' ]
                ]
            ]);
			$this->add_responsive_control('col_gap',[
                'label'     => esc_html__( 'Column Gap', 'onnat-extra' ),
                'type'      => KFCM::SLIDER,
                'default'   => [ 'size' => 20, ],
                'devices'   => [ 'widescreen', 'desktop', 'laptop', 'tablet', 'tablet_extra' ],
                'selectors' => [
                    '{{WRAPPER}} .kfw-ele-widget-blog-tpl-main.kfw-ele-widget-blog-tpl-grid-1 .kinfw-blog-posts-wrap' => 'grid-column-gap: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .kfw-ele-widget-blog-tpl-main.kfw-ele-widget-blog-tpl-grid-2 .kinfw-blog-posts-wrap' => 'grid-column-gap: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .kfw-ele-widget-blog-tpl-main.kfw-ele-widget-blog-tpl-grid-3 .kinfw-blog-posts-wrap' => 'grid-column-gap: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .kfw-ele-widget-blog-tpl-main.kfw-ele-widget-blog-tpl-grid-4 .kinfw-blog-posts-wrap' => 'grid-column-gap: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                    'layout' => [ 'grid-1', 'grid-2', 'grid-3', 'grid-4' ]
                ]
            ]);
			$this->add_responsive_control('row_gap',[
                'label'     => esc_html__( 'Row Gap', 'onnat-extra' ),
                'type'      => KFCM::SLIDER,
                'default'   => [ 'size' => 20, ],
                'devices'   => [ 'widescreen', 'desktop', 'laptop', 'tablet', 'tablet_extra' ],
                'selectors' => [
                    '{{WRAPPER}} .kfw-ele-widget-blog-tpl-main.kfw-ele-widget-blog-tpl-grid-1 .kinfw-blog-posts-wrap' => 'grid-row-gap: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .kfw-ele-widget-blog-tpl-main.kfw-ele-widget-blog-tpl-grid-2 .kinfw-blog-posts-wrap' => 'grid-row-gap: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                    'layout' => [ 'grid-1', 'grid-2', 'grid-3', 'grid-4' ]
                ]
            ]);
		$this->end_controls_section();
    }

    public function _register_query_settings_section() {
		$this->start_controls_section( 'section_query_settings', [
			'label' => esc_html__( 'Query Settings', 'onnat-extra' ),
			'tab'   => KFCM::TAB_CONTENT,
		]);
            $this->add_control('posts_per_page', [
                'label'   => esc_html__( 'Items Per Page', 'onnat-extra' ),
                'type'    => KFCM::NUMBER,
                'min'     => -1,
                'max'     => 20,
                'step'    => 1,
                'default' => 10,
            ]);
            $this->add_control('ignore_sticky_posts',[
                'label'        => esc_html__( 'Ignore Sticky Posts?', 'onnat-extra' ),
                'type'         => KFCM::SWITCHER,
                'return_value' => 'yes',
                'default'      => 'yes'
            ]);
            $this->add_control('author_param',[
                'label'        => esc_html__( 'Show Posts From Several Authors?', 'onnat-extra' ),
                'type'         => KFCM::SWITCHER,
                'return_value' => 'yes',
                'default'      => 'yes'
            ]);
            $this->add_control('author',[
                'label'       => esc_html__( 'Authors?', 'onnat-extra' ),
                'label_block' => true,
                'type'        => KFCM::SELECT2,
                'multiple'    => true,
                'options'     => $this->get_authors( 'post' ),
                'condition'   => [
                    'author_param!' => ''
                ]
            ]);
            $this->add_control( 'cat_param', [
                'label'        => esc_html__('Show posts from specific categories.', 'onnat-extra' ),
                'type'         => KFCM::SWITCHER,
                'return_value' => 'yes',
                'default'      => 'yes'
            ]);
            $this->add_control('cat',[
                'label'       => esc_html__( 'Categories?', 'onnat-extra' ),
                'label_block' => true,
                'type'        => KFCM::SELECT2,
                'multiple'    => true,
                'options'     => $this->get_terms( 'category' ),
                'condition'   => [
                    'cat_param!' => ''
                ]
            ]);
            $this->add_control( 'tag_param', [
                'label'        => esc_html__('Show posts associated with certain tags.', 'onnat-extra' ),
                'type'         => KFCM::SWITCHER,
                'return_value' => 'yes',
                'default'      => 'yes'
            ]);
            $this->add_control('tag',[
                'label'       => esc_html__( 'Tags?', 'onnat-extra' ),
                'label_block' => true,
                'type'        => KFCM::SELECT2,
                'multiple'    => true,
                'options'     => $this->get_terms( 'post_tag' ),
                'condition'   => [
                    'tag_param!' => ''
                ]
            ]);
            $this->add_control( 'order', [
                'label'        => esc_html__('Post Order.', 'onnat-extra' ),
                'type'         => KFCM::SWITCHER,
                'label_on'     => esc_html__('ASC', 'onnat-extra' ),
                'label_off'    => esc_html__('DESC', 'onnat-extra' ),
                'return_value' => 'yes',
                'default'      => 'yes'
            ]);
            $this->add_control( 'orderby', [
                'label'   => esc_html__('Sort Posts by.', 'onnat-extra' ),
                'type'    => KFCM::SELECT,
                'options' => [
                    'none'       => esc_html__('None', 'onnat-extra'),
                    'ID'         => esc_html__('Post ID', 'onnat-extra'),
                    'author'     => esc_html__('Author', 'onnat-extra'),
                    'title'      => esc_html__('Title', 'onnat-extra'),
                    'name'       => esc_html__('Slug', 'onnat-extra'),
                    'date'       => esc_html__('Date', 'onnat-extra'),
                    'modified'   => esc_html__('Last Modified Date', 'onnat-extra'),
                    'rand'       => esc_html__('Random order', 'onnat-extra'),
                    'menu_order' => esc_html__('Menu order', 'onnat-extra'),
                ],
                'default' => 'date'
            ]);
		$this->end_controls_section();
    }

    public function _register_layout_style_section() {
		$this->start_controls_section( 'section_layout_style', [
            'label' => esc_html__( 'Layout', 'onnat-extra' ),
            'tab'   => KFCM::TAB_STYLE,
		]);
            $this->add_control('layout_settings_heading',[
                'label' => esc_html__( 'LAYOUT SETTINGS', 'onnat-extra' ),
                'type'  => KFCM::HEADING,
            ]);
            $this->add_control('hr_layout', [
                'type' => KFCM::DIVIDER,
            ]);
            $this->add_group_control( KFGCBG::get_type(), [
                'name'           => 'layout_bg',
                'types'          => ['classic', 'gradient'],
                'exclude'        => [ 'image'],
                'fields_options' => [
                    'background' => [ 'default' => 'classic' ],
                    'color'      => [ 'default' => '' ],
                ],
                'selector'       => '
                    {{WRAPPER}} .kinfw-blog-standard-style .kinfw-entry-content-wrap,
                    {{WRAPPER}} .kinfw-blog-grid-style-1,
                    {{WRAPPER}} .kinfw-blog-grid-style-2 .kinfw-entry-content-wrap,
                    {{WRAPPER}} .kinfw-blog-grid-style-3,
                    {{WRAPPER}} .kinfw-blog-grid-style-4',
            ]);
            $this->add_group_control( KFGCBR::get_type(), [
                'name'      => 'layout_border',
                'selector'  => '
                    {{WRAPPER}} .kinfw-blog-standard-style .kinfw-entry-content-wrap,
                    {{WRAPPER}} .kinfw-blog-grid-style-1,
                    {{WRAPPER}} .kinfw-blog-grid-style-2 .kinfw-entry-content-wrap,
                    {{WRAPPER}} .kinfw-blog-grid-style-3,
                    {{WRAPPER}} .kinfw-blog-grid-style-4',
            ]);
            $this->add_responsive_control( 'layout_border_radius', [
                'label'      => esc_html__('Border Radius', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', '%' ],
                'selectors'  => ['
                    {{WRAPPER}} .kinfw-blog-standard-style .kinfw-entry-content-wrap,
                    {{WRAPPER}} .kinfw-blog-grid-style-1,
                    {{WRAPPER}} .kinfw-blog-grid-style-2 .kinfw-entry-content-wrap,
                    {{WRAPPER}} .kinfw-blog-grid-style-3,
                    {{WRAPPER}} .kinfw-blog-grid-style-4' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ]
            ]);
            $this->add_group_control( KFGCBS::get_type(), [
                'name'      => 'layout_box_shadow',
                'selector'  => '
                    {{WRAPPER}} .kinfw-blog-standard-style .kinfw-entry-content-wrap,
                    {{WRAPPER}} .kinfw-blog-grid-style-1,
                    {{WRAPPER}} .kinfw-blog-grid-style-2 .kinfw-entry-content-wrap,
                    {{WRAPPER}} .kinfw-blog-grid-style-3,
                    {{WRAPPER}} .kinfw-blog-grid-style-4'
            ]);
            $this->add_responsive_control( 'layout_padding', [
                'label'      => esc_html__('Padding', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => ['
                    {{WRAPPER}} .kinfw-blog-standard-style .kinfw-entry-content-wrap,
                    {{WRAPPER}} .kinfw-blog-grid-style-1 .kinfw-entry-content-wrap,
                    {{WRAPPER}} .kinfw-blog-grid-style-2 .kinfw-entry-content-wrap,
                    {{WRAPPER}} .kinfw-blog-grid-style-3,
                    {{WRAPPER}} .kinfw-blog-grid-style-4' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->add_control('hr_layout_img_before', [
                'type' => KFCM::DIVIDER,
            ]);
            $this->add_control('layout_image_settings_heading',[
                'label' => esc_html__( 'FEATURED IMAGE SETTINGS', 'onnat-extra' ),
                'type'  => KFCM::HEADING,
            ]);
            $this->add_control('hr_layout_img_after', [
                'type' => KFCM::DIVIDER,
            ]);
            $this->add_group_control( KFGCBG::get_type(), [
                'name'           => 'featured_img_bg',
                'types'          => ['classic', 'gradient'],
                'exclude'        => [ 'image'],
                'fields_options' => [
                    'background' => [ 'default' => 'classic' ],
                    'color'      => [ 'default' => '' ],
                ],
                'selector'       => '{{WRAPPER}} .kinfw-entry-media-wrap .kinfw-entry-media',
            ]);
            $this->add_group_control( KFGCBR::get_type(), [
                'name'      => 'featured_img_border',
                'selector'  => '{{WRAPPER}} .kinfw-entry-media-wrap .kinfw-entry-media',
            ]);
            $this->add_responsive_control( 'featured_img_border_radius', [
                'label'      => esc_html__('Border Radius', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-entry-media-wrap .kinfw-entry-media, {{WRAPPER}} .kinfw-entry-media-wrap .kinfw-entry-media img' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ]
            ]);
            $this->add_group_control( KFGCBS::get_type(), [
                'name'      => 'featured_img_box_shadow',
                'selector'  => '{{WRAPPER}} .kinfw-entry-media-wrap .kinfw-entry-media'
            ]);
            $this->add_responsive_control( 'featured_img_padding', [
                'label'      => esc_html__('Padding', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-entry-media-wrap .kinfw-entry-media' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
		$this->end_controls_section();
    }

    public function _register_post_date_meta_style_section() {
        $condition = [
            'layout' => [ 'standard', 'grid-1' ]
        ];

		$this->start_controls_section( 'section_date_meta_style', [
            'label'     => esc_html__( 'Post Date', 'onnat-extra' ),
            'tab'       => KFCM::TAB_STYLE,
            'condition' => $condition
		]);
            $this->add_group_control(KFGCT::get_type(), [
                'name'      => 'post_date_typo',
                'label'     => esc_html__( 'Typography', 'onnat-extra' ),
                'condition' => $condition,
                'selector'  => '{{WRAPPER}} .kinfw-blog-standard-style .kinfw-entry-media-wrap .kinfw-meta-date time, {{WRAPPER}} .kinfw-blog-grid-style-1 .kinfw-entry-media-wrap .kinfw-meta-date time',
            ]);
            $this->add_control( 'post_date_color', [
                'label'     => esc_html__('Color', 'onnat-extra' ),
                'type'      => KFCM::COLOR,
                'condition' => $condition,
                'selectors' => [
                    '{{WRAPPER}} .kinfw-blog-standard-style .kinfw-entry-media-wrap .kinfw-meta-date time, {{WRAPPER}} .kinfw-blog-grid-style-1 .kinfw-entry-media-wrap .kinfw-meta-date time' => 'color: {{VALUE}};'
                ],
            ]);
            $this->add_group_control( KFGCTS::get_type(), [
                'name'      => 'post_date_txt_shadow',
                'condition' => $condition,
                'selector'  => '{{WRAPPER}} .kinfw-blog-standard-style .kinfw-entry-media-wrap .kinfw-meta-date time, {{WRAPPER}} .kinfw-blog-grid-style-1 .kinfw-entry-media-wrap .kinfw-meta-date time',
            ]);
            $this->add_group_control( KFGCTST::get_type(), [
                'name'      => 'post_date_txt_stroke',
                'condition' => $condition,
                'selector'  => '{{WRAPPER}} .kinfw-blog-standard-style .kinfw-entry-media-wrap .kinfw-meta-date time, {{WRAPPER}} .kinfw-blog-grid-style-1 .kinfw-entry-media-wrap .kinfw-meta-date time',
            ]);
            $this->add_control( 'post_date_wrapper_color', [
				'label'     => esc_html__('Curve Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
                'condition' => $condition,
				'selectors' => [
					'{{WRAPPER}} .kinfw-blog-standard-style .kinfw-entry-media-wrap .kinfw-meta-date, {{WRAPPER}} .kinfw-blog-grid-style-1 .kinfw-entry-media-wrap .kinfw-meta-date' => 'background-color: {{VALUE}}; color: {{VALUE}};'
				],
			]);
            $this->add_group_control( KFGCBG::get_type(), [
                'name'           => 'post_date_bg',
                'types'          => ['classic', 'gradient'],
                'exclude'        => [ 'image'],
                'fields_options' => [
                    'background' => [ 'default' => 'classic' ],
                    'color'      => [ 'default' => '' ],
                ],
                'condition'      => $condition,
                'selector'       => '{{WRAPPER}} .kinfw-blog-standard-style .kinfw-entry-media-wrap .kinfw-meta-date time, {{WRAPPER}} .kinfw-blog-grid-style-1 .kinfw-entry-media-wrap .kinfw-meta-date time',
            ]);
            $this->add_group_control( KFGCBR::get_type(), [
                'name'      => 'post_date_border',
                'condition' => $condition,
                'selector'  => '{{WRAPPER}} .kinfw-blog-standard-style .kinfw-entry-media-wrap .kinfw-meta-date time, {{WRAPPER}} .kinfw-blog-grid-style-1 .kinfw-entry-media-wrap .kinfw-meta-date time'
            ]);
            $this->add_responsive_control( 'post_date_border_radius', [
                'label'      => esc_html__('Border Radius', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', '%' ],
                'condition'  => $condition,
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-blog-standard-style .kinfw-entry-media-wrap .kinfw-meta-date time, {{WRAPPER}} .kinfw-blog-grid-style-1 .kinfw-entry-media-wrap .kinfw-meta-date time' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ]
            ]);
            $this->add_group_control( KFGCBS::get_type(), [
                'name'      => 'post_date_box_shadow',
                'condition' => $condition,
                'selector'  => '{{WRAPPER}} .kinfw-blog-standard-style .kinfw-entry-media-wrap .kinfw-meta-date time, {{WRAPPER}} .kinfw-blog-grid-style-1 .kinfw-entry-media-wrap .kinfw-meta-date time'
            ]);
            $this->add_responsive_control( 'post_date_margin', [
                'label'      => esc_html__('Margin', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'condition'  => $condition,
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-blog-standard-style .kinfw-entry-media-wrap .kinfw-meta-date time, {{WRAPPER}} .kinfw-blog-grid-style-1 .kinfw-entry-media-wrap .kinfw-meta-date time' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->add_responsive_control( 'post_date_padding', [
                'label'      => esc_html__('Padding', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'condition'  => $condition,
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-blog-standard-style .kinfw-entry-media-wrap .kinfw-meta-date time, {{WRAPPER}} .kinfw-blog-grid-style-1 .kinfw-entry-media-wrap .kinfw-meta-date time' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
		$this->end_controls_section();
    }

    public function _register_s4_post_date_meta_style_section() {
        $condition = [
            'layout' => 'grid-4'
        ];

		$this->start_controls_section( 'section_s4_date_meta_style', [
            'label'     => esc_html__( 'Post Date', 'onnat-extra' ),
            'tab'       => KFCM::TAB_STYLE,
            'condition' => $condition
		]);
            $this->add_group_control(KFGCT::get_type(), [
                'name'      => 's4_post_date_typo',
                'label'     => esc_html__( 'Typography', 'onnat-extra' ),
                'exclude'   => [
                    'font_size',
                ],
                'condition' => $condition,
                'selector'  => '{{WRAPPER}} .kinfw-blog-grid-style-4 .kinfw-meta-date span',
            ]);
            $this->start_controls_tabs( 's4_post_date_customize_tabs', [ 
                'condition' => $condition
            ]);
                $this->start_controls_tab( 's4_date_tab', [
                    'label'     => esc_html__( 'Date', 'onnat-extra' ),
                    'condition' => $condition
                ]);
                    $this->add_responsive_control( 's4_date_size', [
                        'label'      => esc_html_x( 'Font Size', 'Typography Control', 'onnat-extra' ),
                        'type'       => KFCM::SLIDER,
                        'condition'  => $condition,
                        'size_units' => [ 'px', 'em', 'rem', 'vw', 'custom' ],
                        'selectors'  => [
                            '{{WRAPPER}} .kinfw-blog-grid-style-4 .kinfw-meta-date .kinfw-meta-day' => 'font-size:{{SIZE}}{{UNIT}};',
                        ],
                        'range'      => [
                            'px' => [
                                'min' => 1,
                                'max' => 200,
                            ],
                            'em' => [
                                'max' => 20,
                            ],
                            'rem' => [
                                'max' => 20,
                            ],
                            'vw' => [
                                'min' => 0.1,
                                'max' => 10,
                                'step' => 0.1,
                            ],
                        ],
                    ]);
                $this->end_controls_tab();
                $this->start_controls_tab( 's4_month_tab', [
                    'label'     => esc_html__( 'Month', 'onnat-extra' ),
                    'condition' => $condition
                ]);
                    $this->add_responsive_control( 's4_month_size', [
                        'label'      => esc_html_x( 'Font Size', 'Typography Control', 'onnat-extra' ),
                        'type'       => KFCM::SLIDER,
                        'condition'  => $condition,
                        'size_units' => [ 'px', 'em', 'rem', 'vw', 'custom' ],
                        'selectors'  => [
                            '{{WRAPPER}} .kinfw-blog-grid-style-4 .kinfw-meta-date .kinfw-meta-month' => 'font-size:{{SIZE}}{{UNIT}};',
                        ],
                        'range'      => [
                            'px' => [
                                'min' => 1,
                                'max' => 200,
                            ],
                            'em' => [
                                'max' => 20,
                            ],
                            'rem' => [
                                'max' => 20,
                            ],
                            'vw' => [
                                'min' => 0.1,
                                'max' => 10,
                                'step' => 0.1,
                            ],
                        ],
                    ]);
                $this->end_controls_tab();                
            $this->end_controls_tabs();
            $this->add_control('s4_post_date_hr1', [
                'type' => KFCM::DIVIDER,
            ]);     
            
            $this->add_control( 's4_post_date_color_settings', [
                'label'        => esc_html__('Color', 'onnat-extra'),
                'type'         => KFCM::POPOVER_TOGGLE,
                'return_value' => 'yes',
            ]);
            $this->start_popover();
                $this->add_control( 's4_post_date_color', [
                    'label'     => esc_html__('Normal Color', 'onnat-extra' ),
                    'type'      => KFCM::COLOR,
                    'selectors' => [
                        '{{WRAPPER}} .kinfw-blog-grid-style-4 .kinfw-entry-meta-wrap .kinfw-meta-date span' => 'color:{{VALUE}};',
                    ],
                ]);
            $this->add_control( 's4_post_date_h_color', [
                'label'     => esc_html__('Hover Color', 'onnat-extra' ),
                'type'      => KFCM::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .kinfw-blog-grid-style-4:hover .kinfw-entry-meta-wrap .kinfw-meta-date span' => 'color:{{VALUE}};',
                ],
            ]);            
            $this->end_popover();

            $this->add_control( 's4_post_date_bg_color_settings', [
                'label'        => esc_html__('Background Colors', 'onnat-extra'),
                'type'         => KFCM::POPOVER_TOGGLE,
                'return_value' => 'yes',
            ]);
            $this->start_popover();
                $this->add_control( 's4_post_date_bg_color', [
                    'label'     => esc_html__('Normal Color', 'onnat-extra' ),
                    'type'      => KFCM::COLOR,
                    'selectors' => [
                        '{{WRAPPER}} .kinfw-blog-grid-style-4 .kinfw-entry-meta-wrap .kinfw-meta-date time' => 'background-color:{{VALUE}};',
                    ],
                ]);
            $this->add_control( 's4_post_date_bg_h_color', [
                'label'     => esc_html__('Hover Color', 'onnat-extra' ),
                'type'      => KFCM::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .kinfw-blog-grid-style-4:hover .kinfw-entry-meta-wrap .kinfw-meta-date time' => 'background-color:{{VALUE}};',
                ],
            ]);            
            $this->end_popover();
            $this->add_control( 's4_post_date_border_color_settings', [
                'label'        => esc_html__('Border Colors', 'onnat-extra'),
                'type'         => KFCM::POPOVER_TOGGLE,
                'return_value' => 'yes',
            ]);
            $this->start_popover();
                $this->add_control( 's4_post_date_border_color', [
                    'label'     => esc_html__('Normal Color', 'onnat-extra' ),
                    'type'      => KFCM::COLOR,
                    'selectors' => [
                        '{{WRAPPER}} .kinfw-blog-grid-style-4 .kinfw-entry-meta-wrap .kinfw-meta-date:before' => 'background-color:{{VALUE}};',
                    ],
                ]);
            $this->add_control( 's4_post_date_border_h_color', [
                'label'     => esc_html__('Hover Color', 'onnat-extra' ),
                'type'      => KFCM::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .kinfw-blog-grid-style-4:hover .kinfw-entry-meta-wrap .kinfw-meta-date:before' => 'background-color:{{VALUE}};',
                ],
            ]);            
            $this->end_popover();
            $this->add_control('s4_post_date_hr2', [
                'type' => KFCM::DIVIDER,
            ]);
            $this->add_responsive_control( 's4_post_date_border_radius', [
                'label'      => esc_html__('Border Radius', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', '%' ],
                'condition'  => $condition,
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-blog-grid-style-4 .kinfw-entry-meta-wrap .kinfw-meta-date time, {{WRAPPER}} .kinfw-blog-grid-style-4 .kinfw-entry-meta-wrap .kinfw-meta-date:before' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ]
            ]);
            $this->add_group_control( KFGCBS::get_type(), [
                'name'      => 's4_post_date_box_shadow',
                'condition' => $condition,
                'selector'  => '{{WRAPPER}} .kinfw-blog-grid-style-4 .kinfw-entry-meta-wrap .kinfw-meta-date:before'
            ]);
            $this->add_responsive_control( 's4_post_date_margin', [
                'label'      => esc_html__('Margin', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'condition'  => $condition,
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-blog-grid-style-4 .kinfw-entry-meta-wrap .kinfw-meta-date' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->add_responsive_control( 's4_post_date_padding', [
                'label'      => esc_html__('Padding', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'condition'  => $condition,
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-blog-grid-style-4 .kinfw-entry-meta-wrap .kinfw-meta-date time' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);        
        $this->end_controls_section();
    }    

    public function _register_post_meta_style_section() {
        $condition = [
            'layout!' => 'grid-3'
        ];

		$this->start_controls_section( 'section_meta_style', [
			'label' => esc_html__( 'Post Meta', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
            $this->add_control('meta_icon_settings_heading',[
                'label'     => esc_html__( 'ICON SETTINGS', 'onnat-extra' ),
                'type'      => KFCM::HEADING,
                'condition' => $condition,
            ]);
            $this->add_control('hr_1', [
                'type'      => KFCM::DIVIDER,
                'condition' => $condition,
            ]);
            $this->add_responsive_control( 'meta_icon_size', [
                'label'      => esc_html__('Icon Size', 'onnat-extra' ),
                'type'       => KFCM::SLIDER,
                'size_units' => [ 'px' ],
                'range'      => [
                    'px' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'condition'  => $condition,
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-entry-meta-wrap span.kinfw-icon' => 'font-size: {{SIZE}}{{UNIT}};'
                ],
            ]);
            $this->add_control( 'meta_icon_color', [
                'label'     => esc_html__('Color', 'onnat-extra' ),
                'type'      => KFCM::COLOR,
                'condition' => $condition,
                'selectors' => [
                    '{{WRAPPER}} .kinfw-entry-meta-wrap span.kinfw-icon' => 'color:{{VALUE}};',
                ],
            ]);
            $this->add_responsive_control( 'meta_icon_margin', [
                'label'      => esc_html__('Margin', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'condition'  => $condition,
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-entry-meta-wrap span.kinfw-icon' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->add_control('meta_typo_settings_heading',[
                'label' => esc_html__( 'TYPO SETTINGS', 'onnat-extra' ),
                'type'  => KFCM::HEADING,
            ]);
            $this->add_control('hr_2', [
                'type' => KFCM::DIVIDER,
            ]);
            $this->add_group_control(KFGCT::get_type(), [
                'name'     => 'meta_info_typo',
                'label'    => esc_html__( 'Typography', 'onnat-extra' ),
                'selector' => '
                    {{WRAPPER}} .kinfw-entry-meta-wrap a,
                    {{WRAPPER}} .kinfw-entry-meta-wrap .kinfw-meta-author a span:last-child,
                    {{WRAPPER}} .kinfw-entry-meta-wrap .kinfw-meta-date time',
            ]);
            $this->start_controls_tabs( 'meta_info_customize_tabs' );
                $this->start_controls_tab( 'meta_info_normal_state', [
                    'label' => esc_html__( 'Normal', 'onnat-extra' ),
                ]);
                    $this->add_control( 'meta_info_color', [
                        'label'     => esc_html__('Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'selectors' => [
                            '{{WRAPPER}} .kinfw-entry-meta-wrap a, {{WRAPPER}} .kinfw-entry-meta-wrap .kinfw-meta-date time' => 'color: {{VALUE}};',
                            '{{WRAPPER}} .kinfw-blog-grid-style-3 .kinfw-entry-meta-wrap > div:before'                       => 'background-color: {{VALUE}};',
                        ],
                    ]);
                    $this->add_group_control( KFGCTS::get_type(), [
                        'name'     => 'meta_info_txt_shadow',
                        'selector' => '{{WRAPPER}} .kinfw-entry-meta-wrap a, {{WRAPPER}} .kinfw-entry-meta-wrap .kinfw-meta-date time',
                    ]);
                    $this->add_group_control( KFGCTST::get_type(), [
                        'name'     => 'meta_info_txt_stroke',
                        'selector' => '{{WRAPPER}} .kinfw-entry-meta-wrap a, {{WRAPPER}} .kinfw-entry-meta-wrap .kinfw-meta-date time',
                    ]);
                $this->end_controls_tab();
                $this->start_controls_tab( 'meta_info_hover_state', [
                    'label' => esc_html__( 'Hover', 'onnat-extra' ),
                ]);
                    $this->add_control( 'meta_info_hover_color', [
                        'label'     => esc_html__('Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'selectors' => ['{{WRAPPER}} .kinfw-entry-meta-wrap a:hover' => 'color: {{VALUE}};' ],
                    ]);
                    $this->add_group_control( KFGCTS::get_type(), [
                        'name'     => 'meta_info_hover_txt_shadow',
                        'selector' => '{{WRAPPER}} .kinfw-entry-meta-wrap a:hover',
                    ]);
                    $this->add_group_control( KFGCTST::get_type(), [
                        'name'     => 'meta_info_hover_txt_stroke',
                        'selector' => '{{WRAPPER}} .kinfw-entry-meta-wrap a:hover',
                    ]);
                $this->end_controls_tab();
            $this->end_controls_tabs();
		$this->end_controls_section();
    }

    public function _register_post_title_style_section() {
		$this->start_controls_section( 'section_p_title_style', [
			'label' => esc_html__( 'Post Title', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
            $this->add_group_control(KFGCT::get_type(), [
                'name'     => 'title_typo',
                'label'    => esc_html__( 'Typography', 'onnat-extra' ),
                'selector' => '{{WRAPPER}} .kinfw-entry-header h4',
            ]);
            $this->add_responsive_control( 'title_margin', [
                'label'      => esc_html__('Margin', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-entry-header h4' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->add_responsive_control( 'title_padding', [
                'label'      => esc_html__('Padding', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-entry-header h4' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->start_controls_tabs( 'title_customize_tabs' );
                $this->start_controls_tab( 'title_normal_state', [
                    'label' => esc_html__( 'Normal', 'onnat-extra' ),
                ]);
                    $this->add_control( 'title_color', [
                        'label'     => esc_html__('Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'selectors' => ['{{WRAPPER}} .kinfw-entry-header h4 a' => 'color: {{VALUE}};' ],
                    ]);
                    $this->add_group_control( KFGCTS::get_type(), [
                        'name'     => 'title_txt_shadow',
                        'selector' => '{{WRAPPER}} .kinfw-entry-header h4',
                    ]);
                    $this->add_group_control( KFGCTST::get_type(), [
                        'name'     => 'title_txt_stroke',
                        'selector' => '{{WRAPPER}} .kinfw-entry-header h4',
                    ]);
                $this->end_controls_tab();
                $this->start_controls_tab( 'title_hover_state', [
                    'label' => esc_html__( 'Hover', 'onnat-extra' ),
                ]);
                    $this->add_control( 'title_hover_color', [
                        'label'     => esc_html__('Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'selectors' => ['{{WRAPPER}} .kinfw-entry-header h4 a:hover' => 'color: {{VALUE}};' ],
                    ]);
                    $this->add_group_control( KFGCTS::get_type(), [
                        'name'     => 'title_hover_txt_shadow',
                        'selector' => '{{WRAPPER}} .kinfw-entry-header h4 a:hover',
                    ]);
                    $this->add_group_control( KFGCTST::get_type(), [
                        'name'     => 'title_hover_txt_stroke',
                        'selector' => '{{WRAPPER}} .kinfw-entry-header h4 a:hover',
                    ]);
                $this->end_controls_tab();
            $this->end_controls_tabs();
		$this->end_controls_section();
    }

    public function _register_post_expert_style_section() {
        $condition = [
            'layout' => 'standard'
        ];

		$this->start_controls_section( 'section_excerpt_style', [
            'label'     => esc_html__( 'Post Excerpt', 'onnat-extra' ),
            'tab'       => KFCM::TAB_STYLE,
            'condition' => $condition
		]);
            $this->add_group_control(KFGCT::get_type(), [
                'name'      => 'excerpt_typo',
                'label'     => esc_html__( 'Typography', 'onnat-extra' ),
                'condition' => $condition,
                'selector'  => '{{WRAPPER}} .kfw-entry-content',
            ]);
            $this->add_control( 'excerpt_color', [
                'label'     => esc_html__('Color', 'onnat-extra' ),
                'type'      => KFCM::COLOR,
                'condition' => $condition,
                'selectors' => ['{{WRAPPER}} .kfw-entry-content' => 'color: {{VALUE}};' ],
            ]);
            $this->add_group_control( KFGCTS::get_type(), [
                'name'      => 'excerpt_txt_shadow',
                'condition' => $condition,
                'selector'  => '{{WRAPPER}} .kfw-entry-content',
            ]);
            $this->add_group_control( KFGCTST::get_type(), [
                'name'      => 'excerpt_txt_stroke',
                'condition' => $condition,
                'selector'  => '{{WRAPPER}} .kfw-entry-content',
            ]);
            $this->add_responsive_control( 'excerpt_margin', [
                'label'      => esc_html__('Margin', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'condition'  => $condition,
                'selectors'  => [
                    '{{WRAPPER}} .kfw-entry-content' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->add_responsive_control( 'excerpt_padding', [
                'label'      => esc_html__('Padding', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'condition'  => $condition,
                'selectors'  => [
                    '{{WRAPPER}} .kfw-entry-content' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
		$this->end_controls_section();
    }

    public function _register_readmore_btn_style_section() {
		$this->start_controls_section( 'section_readmore_btn_style', [
            'label'     => esc_html__( 'Read More Button Text', 'onnat-extra' ),
            'tab'       => KFCM::TAB_STYLE,
            'condition' => [
                'layout' => [ 'standard', 'grid-1' ]
            ]
		]);
            $this->add_group_control(KFGCT::get_type(), [
                'name'      => 'readmore_btn_typo',
                'label'     => esc_html__( 'Text', 'onnat-extra' ),
                'selector'  => '{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a',
                'condition' => [
                    'layout' => [ 'standard', 'grid-1' ]
                ]
            ]);
            $this->start_controls_tabs( 'readmore_btn_txt_customize_tabs', [
                'condition' => [
                    'layout' => [ 'standard', 'grid-1' ]
                ]
            ]);
                $this->start_controls_tab( 'readmore_btn_txt_normal_state', [
                    'label'     => esc_html__( 'Normal', 'onnat-extra' ),
                    'condition' => [
                        'layout' => [ 'standard', 'grid-1' ]
                    ]
                ]);
                    $this->add_control( 'readmore_btn_txt_color', [
                        'label'     => esc_html__('Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'selectors' => ['{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a' => 'color: {{VALUE}};' ],
                    ]);
                    $this->add_group_control( KFGCBG::get_type(), [
                        'name'      => 'readmore_btn_bg',
                        'label'     => esc_html__('Background', 'onnat-extra' ),
                        'types'     => [ 'classic', 'gradient' ],
                        'exclude'   => [ 'image' ],# To remove background image option in classic mode
                        'selector'  => '{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a',
                        'condition' => [
                            'layout' => [ 'standard', 'grid-1' ],
                        ],
                    ]);
                    $this->add_group_control( KFGCBR::get_type(), [
                        'name'           => 'readmore_btn_border',
                        'condition'      => [
                            'layout' => [ 'standard', 'grid-1' ],
                        ],
                        'fields_options' => [
                            'width' => [ 'label' => _x( 'Border Width', 'Border Control', 'onnat-extra' ) ],
                            'color' => [ 'label' => _x( 'Border Color', 'Border Control', 'onnat-extra' ) ],
                        ],
                        'selector'       => '{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a',                        
                    ]);
                    $this->add_responsive_control( 'readmore_btn_border_radius', [
                        'label'      => esc_html__('Border Radius', 'onnat-extra' ),
                        'type'       => KFCM::DIMENSIONS,
                        'size_units' => [ 'px', '%' ],
                        'selectors'  => [
                            '{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                        ],
                        'condition'  => [
                            'layout' => [ 'standard', 'grid-1' ],
                        ],
                    ]);
                    $this->add_group_control( KFGCBS::get_type(), [
                        'name'      => 'readmore_btn_box_shadow',
                        'selector'  => '{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a',
                        'condition' => [
                            'layout' => [ 'standard', 'grid-1' ],
                        ],
                    ]);
                    $this->add_group_control( KFGCTS::get_type(), [
                        'name'     => 'readmore_btn_txt_txt_shadow',
                        'selector' => '{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a',
                    ]);
                    $this->add_group_control( KFGCTST::get_type(), [
                        'name'     => 'readmore_btn_txt_txt_stroke',
                        'selector' => '{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a',
                    ]);
                $this->end_controls_tab();
                $this->start_controls_tab( 'readmore_btn_txt_hover_state', [
                    'label'     => esc_html__( 'Hover', 'onnat-extra' ),
                    'condition' => [
                        'layout' => [ 'standard', 'grid-1' ]
                    ]
                ]);
                    $this->add_control( 'readmore_btn_txt_h_color', [
                        'label'     => esc_html__('Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'selectors' => ['{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a:hover' => 'color: {{VALUE}};' ],
                    ]);
                    $this->add_group_control( KFGCBG::get_type(), [
                        'name'      => 'readmore_btn_h_bg',
                        'label'     => esc_html__('Background', 'onnat-extra' ),
                        'types'     => [ 'classic', 'gradient' ],
                        'exclude'   => [ 'image' ],# To remove background image option in classic mode
                        'selector'  => '{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a:hover',
                        'condition' => [
                            'layout' => [ 'standard', 'grid-1' ],
                        ],
                    ]);
                    $this->add_group_control( KFGCBR::get_type(), [
                        'name'           => 'readmore_btn_h_border',
                        'condition'      => [
                            'layout' => [ 'standard', 'grid-1' ],
                        ],
                        'fields_options' => [
                            'width' => [ 'label' => _x( 'Border Width', 'Border Control', 'onnat-extra' ) ],
                            'color' => [ 'label' => _x( 'Border Color', 'Border Control', 'onnat-extra' ) ],
                        ],
                        'selector'       => '{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a:hover',                        
                    ]);
                    $this->add_responsive_control( 'readmore_btn_h_border_radius', [
                        'label'      => esc_html__('Border Radius', 'onnat-extra' ),
                        'type'       => KFCM::DIMENSIONS,
                        'size_units' => [ 'px', '%' ],
                        'selectors'  => [
                            '{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a:hover' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                        ],
                        'condition'  => [
                            'layout' => [ 'standard', 'grid-1' ],
                        ],
                    ]);
                    $this->add_group_control( KFGCBS::get_type(), [
                        'name'      => 'readmore_btn_h_box_shadow',
                        'selector'  => '{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a:hover',
                        'condition' => [
                            'layout' => [ 'standard', 'grid-1' ],
                        ],
                    ]);
                    $this->add_group_control( KFGCTS::get_type(), [
                        'name'     => 'readmore_btn_txt_h_txt_shadow',
                        'selector' => '{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a:hover',
                    ]);
                    $this->add_group_control( KFGCTST::get_type(), [
                        'name'     => 'readmore_btn_txt_h_txt_stroke',
                        'selector' => '{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a:hover',
                    ]);
                $this->end_controls_tab();
            $this->end_controls_tabs();

		$this->end_controls_section();
    }

    public function _register_readmore_btn_icon_style_section() {
		$condition = [
			'layout' => [ 'standard', 'grid-1', 'grid-2', 'grid-3' ]
		];

		$this->start_controls_section( 'section_readmore_btn_icon_style', [
            'label'     => esc_html__( 'Read More Button Icon', 'onnat-extra' ),
            'tab'       => KFCM::TAB_STYLE,
            'condition' => $condition,
		]);
            $this->add_responsive_control( 'readmore_btn_icon_wrapper_size', [
                'label'      => esc_html__('Wrapper Size', 'onnat-extra' ),
                'type'       => KFCM::SLIDER,
                'condition'  => $condition,
                'size_units' => [ 'px' ],
                'range'      => [
                    'px' => [
                        'min' => 0,
                        'max' => 500,
                    ],
                ],
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a .kinfw-icon' => 'line-height:{{SIZE}}{{UNIT}};height:{{SIZE}}{{UNIT}};width:{{SIZE}}{{UNIT}};',
                ],
            ]);
            $this->add_responsive_control( 'readmore_btn_icon_size', [
                'label'      => esc_html__('Icon Size', 'onnat-extra' ),
                'type'       => KFCM::SLIDER,
                'condition'  => $condition,
                'size_units' => [ 'px' ],
                'range'      => [
                    'px' => [
                        'min' => 0,
                        'max' => 999,
                    ],
                ],
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a .kinfw-icon' => 'font-size: {{SIZE}}{{UNIT}};'
                ],
            ]);
            $this->add_responsive_control( 'readmore_btn_icon_margin', [
                'label'      => esc_html__('Margin', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'condition'  => $condition,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a .kinfw-icon' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->add_responsive_control( 'readmore_btn_icon_padding', [
                'label'      => esc_html__('Padding', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'condition'  => $condition,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a .kinfw-icon' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
			$this->start_controls_tabs( 'readmore_btn_icon_customize_tabs', [
                'condition'  => $condition,
            ] );
                $this->start_controls_tab( 'readmore_btn_icon_normal_state', [
                    'label'     => esc_html__( 'Normal', 'onnat-extra' ),
                    'condition' => $condition,
                ]);
                    $this->add_control( 'readmore_btn_icon_color', [
                        'label'     => esc_html__('Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'condition' => $condition,
                        'selectors' => [
                            '{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a .kinfw-icon' => 'color:{{VALUE}}; fill:{{VALUE}};',
                        ],
                    ]);
					$this->add_group_control( KFGCBG::get_type(), [
						'name'           => 'readmore_btn_icon_bg',
						'types'          => ['classic', 'gradient'],
						'exclude'        => [ 'image'],
						'condition'      => $condition,
						'fields_options' => [
							'background' => [ 'default' => 'classic' ],
							'color'      => [ 'default' => '' ],
						],
						'selector'       => '{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a .kinfw-icon',
					]);
					$this->add_group_control( KFGCBR::get_type(), [
						'name'      => 'readmore_btn_icon_border',
						'condition' => $condition,
						'selector'  => '{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a .kinfw-icon',
					]);
					$this->add_responsive_control( 'readmore_btn_icon_border_radius', [
						'label'      => esc_html__('Border Radius', 'onnat-extra' ),
						'type'       => KFCM::DIMENSIONS,
						'condition'  => $condition,
						'size_units' => [ 'px', '%' ],
						'selectors'  => [
							'{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a .kinfw-icon' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
						]
					]);
					$this->add_group_control( KFGCBS::get_type(), [
						'name'      => 'readmore_btn_icon_box_shadow',
						'condition' => $condition,
						'selector'  => '{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a .kinfw-icon'
					]);
                $this->end_controls_tab();
                $this->start_controls_tab( 'readmore_btn_icon_hover_state', [
                    'label'     => esc_html__( 'Hover', 'onnat-extra' ),
                    'condition' => $condition,
                ]);
                    $this->add_control( 'readmore_btn_icon_h_color', [
                        'label'     => esc_html__('Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'condition' => $condition,
                        'selectors' => [
                            '{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a:hover .kinfw-icon' => 'color:{{VALUE}}; fill:{{VALUE}};',
                        ],
                    ]);
                    $this->add_group_control( KFGCBG::get_type(), [
                        'name'           => 'readmore_btn_icon_h_bg',
                        'types'          => ['classic', 'gradient'],
                        'exclude'        => ['image'],
                        'condition'      => $condition,
                        'fields_options' => [
                            'background' => [ 'default' => 'classic' ],
                            'color'      => [ 'default' => '' ],
                        ],
                        'selector'       => '{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a:hover .kinfw-icon',
                    ]);
                    $this->add_group_control( KFGCBR::get_type(), [
                        'name'      => 'readmore_btn_icon_h_border',
                        'condition' => $condition,
                        'selector'  => '{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a:hover .kinfw-icon'
                    ]);
                    $this->add_responsive_control( 'readmore_btn_icon_h_border_radius', [
                        'label'      => esc_html__('Border Radius', 'onnat-extra' ),
                        'type'       => KFCM::DIMENSIONS,
                        'condition'  => $condition,
                        'size_units' => [ 'px', '%' ],
                        'selectors'  => [
                            '{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a:hover .kinfw-icon' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                        ]
                    ]);
                    $this->add_group_control( KFGCBS::get_type(), [
                        'name'      => 'readmore_btn_icon_h_box_shadow',
                        'condition' => $condition,
                        'selector'  => '{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a:hover .kinfw-icon',
                    ]);
                $this->end_controls_tab();
			$this->end_controls_tabs();
		$this->end_controls_section();
    }

    public function _register_pagination_style_section() {
		$this->start_controls_section( 'section_pagination_style', [
            'label' => esc_html__( 'Pagination', 'onnat-extra' ),
            'tab'   => KFCM::TAB_STYLE,
		]);
            $this->add_responsive_control( 'pagination_size', [
                'label'      => esc_html__('Wrapper Size', 'onnat-extra' ),
                'type'       => KFCM::SLIDER,
                'size_units' => [ 'px' ],
                'range'      => [
                    'px' => [
                        'min' => 0,
                        'max' => 500,
                    ],
                ],
                'selectors'  => [
                    '{{WRAPPER}} #kinfw-blog-pagination .nav-links a, {{WRAPPER}} #kinfw-blog-pagination .nav-links > span' => 'line-height:{{SIZE}}{{UNIT}};height:{{SIZE}}{{UNIT}};width:{{SIZE}}{{UNIT}};',
                ],
            ]);
            $this->add_responsive_control( 'pagination_icon_size', [
                'label'      => esc_html__('Arrow Size', 'onnat-extra' ),
                'type'       => KFCM::SLIDER,
                'size_units' => [ 'px' ],
                'range'      => [
                    'px' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'selectors'  => [
                    '{{WRAPPER}} #kinfw-blog-pagination .nav-links a span' => 'font-size: {{SIZE}}{{UNIT}};'
                ],
            ]);
            $this->add_group_control(KFGCT::get_type(), [
                'name'     => 'pagination_typo',
                'label'    => esc_html__( 'Text', 'onnat-extra' ),
                'selector' => '{{WRAPPER}} #kinfw-blog-pagination .nav-links a, {{WRAPPER}} #kinfw-blog-pagination .nav-links > span',
            ]);
            $this->add_responsive_control( 'pagination_margin', [
                'label'      => esc_html__('Margin', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} #kinfw-blog-pagination .nav-links a, {{WRAPPER}} #kinfw-blog-pagination .nav-links > span' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->add_responsive_control( 'pagination_padding', [
                'label'      => esc_html__('Padding', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} #kinfw-blog-pagination .nav-links a, {{WRAPPER}} #kinfw-blog-pagination .nav-links > span' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
			$this->start_controls_tabs( 'pagination_customize_tabs' );
                $this->start_controls_tab( 'pagination_normal_state', [
                    'label' => esc_html__( 'Normal', 'onnat-extra' ),
                ]);
                    $this->add_control( 'pagination_icon_color', [
                        'label'     => esc_html__('Arrow Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'selectors' => [
                            '{{WRAPPER}} #kinfw-blog-pagination .nav-links a span' => 'color:{{VALUE}};',
                        ],
                    ]);
                    $this->add_control( 'pagination_txt_color', [
                        'label'     => esc_html__('Text Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'selectors' => [
                            '{{WRAPPER}} #kinfw-blog-pagination .nav-links a' => 'color:{{VALUE}};',
                        ],
                    ]);
					$this->add_group_control( KFGCBG::get_type(), [
						'name'           => 'pagination_bg',
						'types'          => ['classic', 'gradient'],
						'exclude'        => [ 'image'],
						'fields_options' => [
							'background' => [ 'default' => 'classic' ],
							'color'      => [ 'default' => '' ],
						],
						'selector'       => '{{WRAPPER}} #kinfw-blog-pagination .nav-links a',
					]);
					$this->add_group_control( KFGCBR::get_type(), [
						'name'      => 'pagination_border',
						'selector'  => '{{WRAPPER}} #kinfw-blog-pagination .nav-links a'
					]);
					$this->add_responsive_control( 'pagination_border_radius', [
						'label'      => esc_html__('Border Radius', 'onnat-extra' ),
						'type'       => KFCM::DIMENSIONS,
						'size_units' => [ 'px', '%' ],
						'selectors'  => [
							'{{WRAPPER}} #kinfw-blog-pagination .nav-links a' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
						]
					]);
					$this->add_group_control( KFGCBS::get_type(), [
						'name'     => 'pagination_box_shadow',
						'selector' => '{{WRAPPER}} #kinfw-blog-pagination .nav-links a'
					]);
                $this->end_controls_tab();
                $this->start_controls_tab( 'pagination_hover_state', [
                    'label' => esc_html__( 'Hover', 'onnat-extra' ),
                ]);
                    $this->add_control( 'pagination_icon_h_color', [
                        'label'     => esc_html__('Arrow Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'selectors' => [
                            '{{WRAPPER}} #kinfw-blog-pagination .nav-links a:hover span' => 'color:{{VALUE}};',
                        ],
                    ]);
                    $this->add_control( 'pagination_txt_h_color', [
                        'label'     => esc_html__('Text Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'selectors' => [
                            '{{WRAPPER}} #kinfw-blog-pagination .nav-links a:hover' => 'color:{{VALUE}};',
                        ],
                    ]);
                    $this->add_group_control( KFGCBG::get_type(), [
                        'name'           => 'pagination_h_bg',
                        'types'          => ['classic', 'gradient'],
                        'exclude'        => [ 'image'],
                        'fields_options' => [
                            'background' => [ 'default' => 'classic' ],
                            'color'      => [ 'default' => '' ],
                        ],
                        'selector'       => '{{WRAPPER}} #kinfw-blog-pagination .nav-links a:hover',
                    ]);
                    $this->add_group_control( KFGCBR::get_type(), [
                        'name'      => 'pagination_h_border',
                        'selector'  => '{{WRAPPER}} #kinfw-blog-pagination .nav-links a:hover'
                    ]);
                    $this->add_responsive_control( 'pagination_h_border_radius', [
                        'label'      => esc_html__('Border Radius', 'onnat-extra' ),
                        'type'       => KFCM::DIMENSIONS,
                        'size_units' => [ 'px', '%' ],
                        'selectors'  => [
                            '{{WRAPPER}} #kinfw-blog-pagination .nav-links a:hover' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                        ]
                    ]);
                    $this->add_group_control( KFGCBS::get_type(), [
                        'name'     => 'pagination_h_box_shadow',
                        'selector' => '{{WRAPPER}} #kinfw-blog-pagination .nav-links a:hover'
                    ]);
                $this->end_controls_tab();
                $this->start_controls_tab( 'pagination_active_state', [
                    'label' => esc_html__( 'Active', 'onnat-extra' ),
                ]);
                    $this->add_control( 'pagination_txt_a_color', [
                        'label'     => esc_html__('Text Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'selectors' => [
                            '{{WRAPPER}} #kinfw-blog-pagination .nav-links > span.current' => 'color:{{VALUE}};',
                        ],
                    ]);
                    $this->add_group_control( KFGCBG::get_type(), [
                        'name'           => 'pagination_a_bg',
                        'types'          => ['classic', 'gradient'],
                        'exclude'        => [ 'image'],
                        'fields_options' => [
                            'background' => [ 'default' => 'classic' ],
                            'color'      => [ 'default' => '' ],
                        ],
                        'selector'       => '{{WRAPPER}} #kinfw-blog-pagination .nav-links > span.current',
                    ]);
                    $this->add_group_control( KFGCBR::get_type(), [
                        'name'      => 'pagination_a_border',
                        'selector'  => '{{WRAPPER}} #kinfw-blog-pagination .nav-links > span.current'
                    ]);
                    $this->add_responsive_control( 'pagination_a_border_radius', [
                        'label'      => esc_html__('Border Radius', 'onnat-extra' ),
                        'type'       => KFCM::DIMENSIONS,
                        'size_units' => [ 'px', '%' ],
                        'selectors'  => [
                            '{{WRAPPER}} #kinfw-blog-pagination .nav-links > span.current' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                        ]
                    ]);
                    $this->add_group_control( KFGCBS::get_type(), [
                        'name'     => 'pagination_a_box_shadow',
                        'selector' => '{{WRAPPER}} #kinfw-blog-pagination .nav-links > span.current'
                    ]);
                $this->end_controls_tab();
			$this->end_controls_tabs();
		$this->end_controls_section();
    }

    public function get_authors( $post_type ) {
        $options = [];

        $post_type_object = get_post_type_object( $post_type );

        $args  = [
            'capability' => [ $post_type_object->cap->edit_posts ],
            'orderby'    => 'display_name',
            'order'      => 'ASC',
        ];

        $users = get_users( $args );

        if( !empty( $users ) && count( $users ) > 1 ) {
            foreach( $users as $user ) {
                $options[ $user->ID ] = sprintf( '%1$s (%2$s)', $user->display_name, $user->user_login );
            }
        }

        return $options;
    }

    public function get_terms( $taxonomy = 'category' ) {
        $options = [];

        if( !empty( $taxonomy ) ) {
            $terms = get_terms([
                'taxonomy'   => $taxonomy,
                'hide_empty' => true,
            ]);

            if ( ! empty( $terms ) && ! is_wp_error( $terms ) ){
                foreach ( $terms as $term ) {
                    $options[ $term->term_id ] = $term->name;
                }
            }
        }

        return $options;
    }

    public function get_args( $settings = [] ) {

        $args = [
            'post_status'         => 'publish',
            'posts_per_page'      => $settings['posts_per_page'],
            'ignore_sticky_posts' => ( $settings['ignore_sticky_posts'] === 'yes' ) ? true : false,
            'order'               => ( $settings['order'] === 'yes' ) ? 'ASC' : 'DESC',
            'orderby'             => $settings['orderby']
        ];

        // Paged
            if ( get_query_var('paged') ) {
                $paged = get_query_var('paged');
            } elseif ( get_query_var('page') ) {
                $paged = get_query_var('page');
            } else {
                $paged = 1;
            }

            $args['paged'] = $paged;

        // Author Param
            if( $settings['author_param'] === 'yes' ) {
                if( is_array( $settings['author'] ) && count( $settings['author'] ) > 0 ) {
                    $args['author__in'] = implode( ",", $settings['author'] );
                }
            }

        // Category Param
            if( $settings['cat_param'] === 'yes' ) {
                if( is_array( $settings['cat'] ) && count( $settings['cat'] ) > 0 ) {
                    $args['category__in'] = implode( ",", $settings['cat'] );
                }
            }

        // Tag Param
            if( $settings['tag_param'] === 'yes' ) {
                if( is_array( $settings['tag'] ) && count( $settings['tag'] ) > 0 ) {
                    $args['tag__in'] = implode( ",", $settings['tag'] );
                }
            }

        return $args;
    }

	/**
	 * Render widget output on the frontend.
	 */
    protected function render() {
		$render = kf_extra_plugin_elementor()->get_template("kfw-blog-template");
		include $render;
    }

}