<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

use \Elementor\Widget_Base as KFWB;
use \Elementor\Controls_Manager as KFCM;
use \Elementor\Icons_Manager as KFIM;

use \Elementor\Group_Control_Typography as KFGCT;
use \Elementor\Group_Control_Border as KFGCBR;
use \Elementor\Group_Control_Box_Shadow as KFGCBS;
use \Elementor\Group_Control_Text_Shadow as KFGCTS;
use \Elementor\Group_Control_Text_Stroke as KFGCTST;
use \Elementor\Group_Control_Background as KFGCBG;
use \Elementor\Group_Control_Css_Filter as KFGCSSF;

class KFW_Counter_Minimal extends KFWB {

	/**
	 * Get element name.
	 *
	 * Retrieve the element name.
	 *
	 * @return string The name.
	 */
	public function get_name() {
        return 'kfw-counter-minimal';
    }

	/**
	 * Get element title.
	 *
	 * Retrieve the element title.
	 *
	 * @return string Element title.
	 */
	public function get_title() {
		return esc_html__('Counter : Minimal', 'onnat-extra');
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the widget categories.
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ '_kfw_core_widgets' ];
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve the widget icon.
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'kfw-badge eicon-counter';
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the widget keywords.
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'kfw', 'kin', 'counter', 'minimal'];
	}

	/**
	 * Get widget help url.
	 *
	 * Retrieve the widget help url.
	 *
	 * @return string Widget help url.
	 */
	public function get_custom_help_url() {
		return KFE_CONST_THEME_DOC . 'elementor-custom-widgets/counter-minimal';
    }

	/**
	 * Get HTML wrapper class.
     *
	 * Retrieve the widget container class.
     *
     * @return string Widget container css classes.
	 */
	public function get_html_wrapper_class() {
        $name = $this->get_name();

		return sprintf( 'kfw-ele-%s elementor-widget-%s', $name, $name );
	}

	/**
	 * Widget base constructor.
	 *
	 * Initializing the widget base class.
	 */
	public function __construct( $data = [], $args = null ) {
        parent::__construct( $data, $args );

		wp_register_script( 'kfw-counter-minimal',
			KFE_CONST_URL . 'libraries/elementor/assets/js/counter-minimal' . KFE_CONST_DEBUG_SUFFIX . '.js',
			[ 'jquery', 'elementor-frontend' ],
			KFE_CONST_VERSION,
			true
		);

		wp_register_style( 'kfw-counter-minimal',
			KFE_CONST_URL . 'libraries/elementor/assets/css/counter-minimal' . KFE_CONST_DEBUG_SUFFIX . '.css',
			[],
			KFE_CONST_VERSION
		);
    }

	/**
	 * Get style dependencies.
	 *
	 * Retrieve the list of style dependencies the element requires.
	 *
	 * @return array Element styles dependencies.
	 */
	public function get_style_depends() {
		return [ 'kfw-counter-minimal' ];
	}

	/**
	 * Get script dependencies.
	 *
	 * Retrieve the list of script dependencies the element requires.
	 *
	 * @return array Element scripts dependencies.
	 */
	public function get_script_depends() {
		return [ 'jquery-numerator', 'kfw-counter-minimal' ];
	}

	/**
	 * Register controls.
	 *
	 * Used to add new controls to any element type.
	 */
	protected function register_controls() {
		$this->_register_title_section();
		$this->_register_icon_section();
		$this->_register_counter_section();

		$this->_register_title_style_section();
		$this->_register_icon_style_section();
		$this->_register_prefix_style_section();
		$this->_register_number_style_section();
		$this->_register_suffix_style_section();
	}

	public function _register_title_section() {
		$this->start_controls_section( 'section_title', [
			'label' => esc_html__( 'Title', 'onnat-extra' )
		]);
			$this->add_control('title',[
				'label'       => esc_html__( 'Title', 'onnat-extra' ),
				'type'        => KFCM::TEXT,
				'dynamic'     => [ 'active' => true, ],
				'placeholder' => esc_html__( 'Enter Title', 'onnat-extra' ),
				'default'     => esc_html__( 'Here is the title', 'onnat-extra' ),
				'label_block' => true,
			]);
			$this->add_control('title_pos',[
				'label'     => esc_html__( 'Title Position?', 'onnat-extra' ),
				'type'      => KFCM::CHOOSE,
				'options'   => [
					'top'    => [
						'title' => esc_html__( 'Top', 'onnat-extra' ),
						'icon'  => 'eicon-v-align-top',
					],
					'bottom' => [
						'title' => esc_html__( 'Bottom', 'onnat-extra' ),
						'icon'  => 'eicon-v-align-bottom',
					],
				],
				'default'   => 'top',
				'toggle'    => false,
				'condition' => [
					'title!' => ''
				]
			]);

		$this->end_controls_section();
	}

	public function _register_icon_section() {
		$this->start_controls_section( 'section_icon', [
			'label' => esc_html__( 'Icon', 'onnat-extra' )
		]);
			$this->add_control('icon_type',[
				'label'   => esc_html__( 'Type', 'onnat-extra' ),
				'type'    => KFCM::CHOOSE,
				'options' => [
					'blank' => [
						'title' => esc_html__('Blank', 'onnat-extra'),
						'icon'  => '',
					],
					'icon'  => [
						'title' => esc_html__('Icon', 'onnat-extra'),
						'icon'  => 'eicon-plus-circle-o',
					],
					'image'  => [
						'title' => esc_html__('Image', 'onnat-extra'),
						'icon'  => 'eicon-image',
					],
				],
				'default' => 'icon',
				'toggle'  => false
			]);
			$this->add_control( 'icon', [
				'label'     => esc_html__( 'Icon', 'onnat-extra' ),
				'type'      => KFCM::ICONS,
				'default'   => [
					'value'   => 'fas fa-heart',
					'library' => 'fa-solid',
				],
				'condition' => [ 'icon_type' => 'icon' ],
			]);
			$this->add_control( 'image', [
				'label'     => esc_html__( 'Image', 'onnat-extra' ),
				'type'      => KFCM::MEDIA,
				'dynamic'   => [ 'active' => true ],
				'default'   => [
					'url' => \Elementor\Utils::get_placeholder_image_src()
				],
				'condition' => [ 'icon_type' => 'image' ],
			]);
			$this->add_control('icon_position', [
				'label'          => esc_html__( 'Position', 'onnat-extra' ),
				'type'           => KFCM::CHOOSE,
				'options'        => [
					'left'  => [
						'title' => esc_html__( 'Left', 'onnat-extra' ),
						'icon'  => 'eicon-h-align-left',
					],
					'top'   => [
						'title' => esc_html__( 'Top', 'onnat-extra' ),
						'icon'  => 'eicon-v-align-top',
					],
					'right' => [
						'title' => esc_html__( 'Right', 'onnat-extra' ),
						'icon'  => 'eicon-h-align-right',
					],
				],
				'default'        => 'top',
				'tablet_default' => 'top',
				'mobile_default' => 'top',
				'toggle'         => false,
				'prefix_class'   => 'kfw-ele-counter-minl-icon-position-',
				'conditions'     => [
					'relation' => 'and',
					'terms'    => [
						[
							'name'     => 'icon_type',
							'operator' => '!==',
							'value'    => 'blank'
						],
						[
							'relation' => 'or',
							'terms'    => [
								[
									'name'     => 'icon[value]',
									'operator' => '!=',
									'value'    => '',
								],
								[
									'name'     => 'image[url]',
									'operator' => '!=',
									'value'    => '',
								],
							]
						]
					]
				]
			]);
			$this->add_control('icon_align',[
				'label'        => esc_html__( 'Alignment', 'onnat-extra' ),
				'type'         => KFCM::CHOOSE,
				'options'      => [
					'top'    => [
						'title' => esc_html__( 'Top', 'onnat-extra' ),
						'icon'  => 'eicon-v-align-top',
					],
					'middle' => [
						'title' => esc_html__( 'Middle', 'onnat-extra' ),
						'icon'  => 'eicon-v-align-middle',
					],
					'bottom' => [
						'title' => esc_html__( 'Bottom', 'onnat-extra' ),
						'icon'  => 'eicon-v-align-bottom',
					],
				],
				'prefix_class' => 'kfw-ele-counter-minl-icon-vertical-align-',
				'default'      => 'top',
				'toggle'       => false,
				'conditions'   => [
					'relation' => 'and',
					'terms'    => [
						[
							'name'     => 'icon_type',
							'operator' => '!==',
							'value'    => 'blank'
						],
						[
							'name'     => 'icon_position',
							'operator' => 'in',
							'value'    => [ 'left', 'right' ]
						],
					]
				]
			]);
		$this->end_controls_section();
	}

	public function _register_counter_section() {
		$this->start_controls_section( 'section_counter', [
			'label' => esc_html__( 'Counter', 'onnat-extra' )
		]);
			$this->add_control('starting_number',[
				'label'              => esc_html__( 'Starting Number', 'onnat-extra' ),
				'type'               => KFCM::NUMBER,
				'default'            => 0,
				'min'                => 0,
				'step'               => 1,
				'dynamic'            => [ 'active' => true, ],
				'frontend_available' => true,
			]);
			$this->add_control('ending_number',[
				'label'              => esc_html__( 'Ending Number', 'onnat-extra' ),
				'type'               => KFCM::NUMBER,
				'default'            => 100,
				'min'                => 0,
				'step'               => 1,
				'dynamic'            => [ 'active' => true, ],
				'frontend_available' => true,
			]);
			$this->add_control('prefix',[
				'label'       => esc_html__( 'Number Prefix', 'onnat-extra' ),
				'type'        => KFCM::TEXT,
				'dynamic'     => [ 'active' => true, ],
				'default'     => '',
				'placeholder' => esc_html__( 'Enter Prefix', 'onnat-extra' ),
			]);
			$this->add_control('suffix',[
				'label'       => esc_html__( 'Number Suffix', 'onnat-extra' ),
				'type'        => KFCM::TEXT,
				'dynamic'     => [ 'active' => true, ],
				'default'     => '+',
				'placeholder' => esc_html__( 'Enter Suffix', 'onnat-extra' ),
			]);
			$this->add_control('thousand_separator',[
				'label'     => esc_html__( 'Thousand Separator', 'onnat-extra' ),
				'type'      => KFCM::SWITCHER,
				'default'   => 'yes',
				'label_on'  => esc_html__( 'Show', 'onnat-extra' ),
				'label_off' => esc_html__( 'Hide', 'onnat-extra' ),
			]);
			$this->add_control('thousand_separator_char',[
				'label'              => esc_html__( 'Separator', 'onnat-extra' ),
				'type'               => KFCM::CHOOSE,
				'condition'          => [ 'thousand_separator' => 'yes', ],
				'options'            => [
					'default'    => [
						'title' => esc_html__('Default', 'onnat-extra'),
						'icon'  => 'kfw-ctrl-icon kfw-ctrl-comma-icon',
					],
					'dot'        => [
						'title' => esc_html__('Dot', 'onnat-extra'),
						'icon'  => 'eicon-dot-circle-o',
					],
					'space'      => [
						'title' => esc_html__('Space', 'onnat-extra'),
						'icon'  => 'eicon-justify-space-between-h',
					],
					'underline'  => [
						'title' => esc_html__('Underline', 'onnat-extra'),
						'icon'  => 'kfw-ctrl-icon kfw-ctrl-underline-icon',
					],
					'apostrophe' => [
						'title' => esc_html__('Apostrophe', 'onnat-extra'),
						'icon'  => 'kfw-ctrl-icon kfw-ctrl-apostrophe-icon',
					],
				],
				'default'            => 'default',
				'toggle'             => false,
				'frontend_available' => true,
			]);
			$this->add_control('rounding',[
				'label'              => esc_html__( 'Decimal Places', 'onnat-extra' ),
				'type'               => KFCM::NUMBER,
				'default'            => 0,
				'min'                => 0,
				'step'               => 1,
				'max'                => 10,
				'frontend_available' => true,
			]);
			$this->add_control('duration',[
				'label'              => esc_html__( 'Animation Duration', 'onnat-extra' ),
				'type'               => KFCM::NUMBER,
				'default'            => 2000,
				'min'                => 500,
				'step'               => 100,
				'frontend_available' => true,
			]);
			$this->add_responsive_control('counter_align', [
				'label'          => esc_html__( 'Alignment', 'onnat-extra' ),
				'type'           => KFCM::CHOOSE,
				'options'        => [
					'left'   => [
						'title' => esc_html__( 'Left', 'onnat-extra' ),
						'icon'  => 'eicon-text-align-left',
					],
					'center' => [
						'title' => esc_html__( 'Center', 'onnat-extra' ),
						'icon'  => 'eicon-text-align-center',
					],
					'right'  => [
						'title' => esc_html__( 'Right', 'onnat-extra' ),
						'icon'  => 'eicon-text-align-right',
					],
				],
				'default'        => 'center',
				'tablet_default' => 'center',
				'mobile_default' => 'center',
				'toggle'         => false,
				'default'        => 'center',
				'tablet_default' => 'center',
				'mobile_default' => 'center',
				'toggle'         => false,
				'selectors_dictionary' => [
					'left'   => 'justify-content:start; text-align:left;',
					'center' => 'justify-content:center; text-align:center;',
					'right'  => 'justify-content:end; text-align:right;',
				],
				'selectors'            => [
					'{{WRAPPER}}.kfw-ele-kfw-counter-minimal, {{WRAPPER}}.kfw-ele-kfw-counter-minimal .kfw-ele-widget-counter-minl-content' => '{{VALUE}};'
				],
				'condition'      => [ 'icon_type' => 'blank' ]
			]);
			$this->add_responsive_control('counter_align_alt', [
				'label'          => esc_html__( 'Alignment', 'onnat-extra' ),
				'type'           => KFCM::CHOOSE,
				'options'        => [
					'left'   => [
						'title' => esc_html__( 'Left', 'onnat-extra' ),
						'icon'  => 'eicon-text-align-left',
					],
					'center' => [
						'title' => esc_html__( 'Center', 'onnat-extra' ),
						'icon'  => 'eicon-text-align-center',
					],
					'right'  => [
						'title' => esc_html__( 'Right', 'onnat-extra' ),
						'icon'  => 'eicon-text-align-right',
					],
				],
				'toggle'               => false,
				'default'              => 'center',
				'tablet_default'       => 'center',
				'mobile_default'       => 'center',
				'selectors_dictionary' => [
					'left'   => 'justify-content:start; text-align:left;',
					'center' => 'justify-content:center; text-align:center;',
					'right'  => 'justify-content:end; text-align:right;',
				],
				'selectors'            => [
					'{{WRAPPER}}.kfw-ele-kfw-counter-minimal, {{WRAPPER}}.kfw-ele-kfw-counter-minimal .kfw-ele-widget-counter-minl-content' => '{{VALUE}};'
				],
				'conditions'     => [
					'relation' => 'and',
					'terms'    => [
						[
							'name'     => 'icon_type',
							'operator' => '!==',
							'value'    => 'blank'
						],
						[
							'name'     => 'icon_position',
							'operator' => '===',
							'value'    => 'top'
						],
					]
				]
			]);
		$this->end_controls_section();
	}

	public function _register_title_style_section() {
		$this->start_controls_section( 'section_title_style',[
			'label'     => esc_html__( 'Title', 'onnat-extra' ),
			'tab'       => KFCM::TAB_STYLE,
			'condition' => [
				'title!' => ''
			]
		]);
			$this->add_group_control(KFGCT::get_type(), [
				'name'      => 'title_typo',
				'label'     => esc_html__( 'Typography', 'onnat-extra' ),
				'selector'  => '{{WRAPPER}} .kfw-ele-widget-counter-minl-title',
				'condition' => [
					'title!' => ''
				]
			]);
			$this->add_control( 'title_color', [
				'label'     => esc_html__('Typo Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'selectors' => [
					'{{WRAPPER}} .kfw-ele-widget-counter-minl-title' => 'color:{{VALUE}};'
				],
				'condition' => [
					'title!' => ''
				],
			]);
			$this->add_group_control( KFGCTS::get_type(), [
				'name'     => 'title_text_shadow',
				'selector' => '{{WRAPPER}} .kfw-ele-widget-counter-minl-title'
			]);
			$this->add_group_control( KFGCTST::get_type(), [
				'name'     => 'title_text_stroke',
				'selector' => '{{WRAPPER}} .kfw-ele-widget-counter-minl-title'
			]);
			$this->add_responsive_control( 'title_padding',[
				'label'      => esc_html__('Padding', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-counter-minl-title' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				],
				'condition'  => [
					'title!' => ''
				],
			]);
			$this->add_responsive_control( 'title_margin',[
				'label'      => esc_html__('Margin', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-counter-minl-title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				],
				'condition'  => [
					'title!' => ''
				],
			]);
		$this->end_controls_section();
	}

	public function _register_icon_style_section() {
		$this->start_controls_section( 'section_icon_style',[
			'label'      => esc_html__( 'Icon', 'onnat-extra' ),
			'tab'        => KFCM::TAB_STYLE,
			'conditions' => [
				'relation' => 'and',
				'terms'    => [
					[
						'name'     => 'icon_type',
						'operator' => '!==',
						'value'    => 'blank'
					],
					[
						'relation' => 'or',
						'terms'    => [
							[
								'name'     => 'icon[value]',
								'operator' => '!=',
								'value'    => '',
							],
							[
								'name'     => 'image[url]',
								'operator' => '!=',
								'value'    => '',
							],
						]
					]
				]
			]
		]);
			$this->add_control( 'icon_color', [
				'label'     => esc_html__('Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'selectors' => [
					'{{WRAPPER}} .kfw-ele-widget-counter-minl-icon'     => 'color:{{VALUE}};',
					'{{WRAPPER}} .kfw-ele-widget-counter-minl-icon svg' => 'fill:{{VALUE}};',
				],
				'condition' => [
					'icon_type' => 'icon'
				],
			]);
			$this->add_group_control( KFGCBG::get_type(), [
				'name'     => 'icon_bg',
				'types'    => ['classic', 'gradient'],
				'exclude'  => [ 'image'],
				'selector' => '{{WRAPPER}} .kfw-ele-widget-counter-minl-icon-type .kfw-ele-widget-counter-minl-icon, {{WRAPPER}} .kfw-ele-widget-counter-minl-image-type img',
			]);
			$this->add_responsive_control( 'icon_padding',[
				'label'      => esc_html__('Padding', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-counter-minl-icon-type .kfw-ele-widget-counter-minl-icon, {{WRAPPER}} .kfw-ele-widget-counter-minl-image-type img' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				],
			]);
			$this->add_responsive_control( 'icon_margin',[
				'label'      => esc_html__('Margin', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-counter-minl-icon-wrap' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				],
			]);
			$this->add_group_control( KFGCBR::get_type(), [
				'name'     => 'icon_border',
				'selector' => '{{WRAPPER}} .kfw-ele-widget-counter-minl-icon-type .kfw-ele-widget-counter-minl-icon, {{WRAPPER}} .kfw-ele-widget-counter-minl-image-type img'
			]);
			$this->add_responsive_control( 'icon_border_radius', [
				'label'      => esc_html__('Border Radius', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-counter-minl-icon-type .kfw-ele-widget-counter-minl-icon, {{WRAPPER}} .kfw-ele-widget-counter-minl-image-type img' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
			$this->add_group_control( KFGCBS::get_type(), [
				'name'     => 'icon_box_shadow',
				'selector' => '{{WRAPPER}} .kfw-ele-widget-counter-minl-icon-type .kfw-ele-widget-counter-minl-icon, {{WRAPPER}} .kfw-ele-widget-counter-minl-image-type img'
			]);
			$this->add_responsive_control( 'icon_size', [
				'label'      => esc_html__('Size', 'onnat-extra' ),
				'type'       => KFCM::SLIDER,
				'size_units' => [ 'px' ],
				'range'      => [
					'px' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-counter-minl-icon' => 'font-size: {{SIZE}}{{UNIT}};',
				],
				'condition'  => [
					'icon_type'    => 'icon',
					'icon[value]!' => ''
				],
			]);
			$this->add_responsive_control( 'img_size', [
				'label'      => esc_html__('Size', 'onnat-extra' ),
				'type'       => KFCM::SLIDER,
				'size_units' => ['%'],
				'range'      => [
					'%' => [
						'min' => 5,
						'max' => 100,
					],
				],
				'condition'  => [
					'icon_type'   => 'image',
					'image[url]!' => ''
				],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-counter-minl-icon img' => 'max-width:{{SIZE}}%;'
				],
			]);
			$this->add_control('rotate',[
				'label'     => esc_html__( 'Rotate', 'onnat-extra' ),
				'type'      => KFCM::SLIDER,
				'default'   => [
					'size' => 0,
					'unit' => 'deg',
				],
				'range'     => [
					'deg' => [
						'max' => 360,
						'min' => - 360,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .kfw-ele-widget-counter-minl-icon i'   => 'transform: rotate({{SIZE}}{{UNIT}});',
					'{{WRAPPER}} .kfw-ele-widget-counter-minl-icon img' => 'transform: rotate({{SIZE}}{{UNIT}});',
				],
			]);
			$this->add_control('icon_background_rotate',[
				'label'     => esc_html__( 'Background Rotate', 'onnat-extra' ),
				'type'      => KFCM::SLIDER,
				'default'   => [
					'size' => 0,
					'unit' => 'deg',
				],
				'range'     => [
					'deg' => [
						'max' => 360,
						'min' => - 360,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .kfw-ele-widget-counter-minl-icon'   => 'transform: rotate({{SIZE}}{{UNIT}});',
				],
				'condition'  => [
					'icon_type'    => 'icon',
					'icon[value]!' => ''
				],
			]);
			$this->add_control('image_settings_heading',[
				'label'     => esc_html__( 'Image Effects', 'onnat-extra' ),
				'type'      => KFCM::HEADING,
				'separator' => 'before',
				'condition' => [
					'icon_type' => 'image'
				],
			]);
			$this->add_group_control( KFGCSSF::get_type(), [
				'name'      => 'img_css_filters',
				'selector'  => '{{WRAPPER}} .kfw-ele-widget-counter-minl-icon img',
				'condition' => [
					'icon_type' => 'image'
				],
			]);
			$this->add_control('img_opacity',[
				'label'     => esc_html__( 'Opacity', 'onnat-extra' ),
				'type'      => KFCM::SLIDER,
				'range'     => [
					'px' => [
						'max'  => 1,
						'min'  => 0.10,
						'step' => 0.01,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .kfw-ele-widget-counter-minl-icon img' => 'opacity: {{SIZE}};'
				],
				'condition' => [
					'icon_type' => 'image'
				],
			]);
		$this->end_controls_section();
	}

	public function _register_prefix_style_section() {
		$this->start_controls_section( 'section_prefix_style',[
			'label'     => esc_html__( 'Prefix', 'onnat-extra' ),
			'tab'       => KFCM::TAB_STYLE,
			'condition' => [
				'prefix!' => ''
			]
		]);
			$this->add_group_control(KFGCT::get_type(), [
				'name'      => 'prefix_typo',
				'label'     => esc_html__( 'Typography', 'onnat-extra' ),
				'selector'  => '{{WRAPPER}} .kfw-ele-widget-counter-minl-prefix',
			]);
			$this->add_control( 'prefix_color', [
				'label'     => esc_html__('Typo Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'selectors' => [
					'{{WRAPPER}} .kfw-ele-widget-counter-minl-prefix' => 'color:{{VALUE}};'
				],
			]);
			$this->add_group_control( KFGCTS::get_type(), [
				'name'     => 'prefix_text_shadow',
				'selector' => '{{WRAPPER}} .kfw-ele-widget-counter-minl-prefix'
			]);
			$this->add_group_control( KFGCTST::get_type(), [
				'name'     => 'prefix_text_stroke',
				'selector' => '{{WRAPPER}} .kfw-ele-widget-counter-minl-prefix'
			]);
			$this->add_responsive_control( 'prefix_padding',[
				'label'      => esc_html__('Padding', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-counter-minl-prefix' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				],
			]);
			$this->add_responsive_control( 'prefix_margin',[
				'label'      => esc_html__('Margin', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-counter-minl-prefix' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				],
			]);
		$this->end_controls_section();
	}

	public function _register_number_style_section() {
		$this->start_controls_section( 'section_number_style',[
			'label' => esc_html__( 'Number', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
			$this->add_group_control(KFGCT::get_type(), [
				'name'      => 'number_typo',
				'label'     => esc_html__( 'Typography', 'onnat-extra' ),
				'selector'  => '{{WRAPPER}} .kfw-ele-widget-counter-minl-number',
			]);
			$this->add_control( 'number_color', [
				'label'     => esc_html__('Typo Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'selectors' => [
					'{{WRAPPER}} .kfw-ele-widget-counter-minl-number' => 'color:{{VALUE}};'
				],
			]);
			$this->add_group_control( KFGCTS::get_type(), [
				'name'     => 'number_text_shadow',
				'selector' => '{{WRAPPER}} .kfw-ele-widget-counter-minl-number'
			]);
			$this->add_group_control( KFGCTST::get_type(), [
				'name'     => 'number_text_stroke',
				'selector' => '{{WRAPPER}} .kfw-ele-widget-counter-minl-number'
			]);
			$this->add_responsive_control( 'number_padding',[
				'label'      => esc_html__('Padding', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-counter-minl-number' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				],
			]);
			$this->add_responsive_control( 'number_margin',[
				'label'      => esc_html__('Margin', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-counter-minl-number' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				],
			]);
		$this->end_controls_section();
	}

	public function _register_suffix_style_section() {
		$this->start_controls_section( 'section_suffix_style',[
			'label'     => esc_html__( 'Suffix', 'onnat-extra' ),
			'tab'       => KFCM::TAB_STYLE,
			'condition' => [
				'suffix!' => ''
			]
		]);
			$this->add_group_control(KFGCT::get_type(), [
				'name'      => 'suffix_typo',
				'label'     => esc_html__( 'Typography', 'onnat-extra' ),
				'selector'  => '{{WRAPPER}} .kfw-ele-widget-counter-minl-suffix',
			]);
			$this->add_control( 'suffix_color', [
				'label'     => esc_html__('Typo Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'selectors' => [
					'{{WRAPPER}} .kfw-ele-widget-counter-minl-suffix' => 'color:{{VALUE}};'
				],
			]);
			$this->add_group_control( KFGCTS::get_type(), [
				'name'     => 'suffix_text_shadow',
				'selector' => '{{WRAPPER}} .kfw-ele-widget-counter-minl-suffix'
			]);
			$this->add_group_control( KFGCTST::get_type(), [
				'name'     => 'suffix_text_stroke',
				'selector' => '{{WRAPPER}} .kfw-ele-widget-counter-minl-suffix'
			]);
			$this->add_responsive_control( 'suffix_padding',[
				'label'      => esc_html__('Padding', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-counter-minl-suffix' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				],
			]);
			$this->add_responsive_control( 'suffix_margin',[
				'label'      => esc_html__('Margin', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-counter-minl-suffix' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				],
			]);
		$this->end_controls_section();
	}

	public function _counter_title() {
		$settings = $this->get_settings();
		extract( $settings );

		if( !empty( $title ) ) {
			$this->add_render_attribute( 'title', [
				'class' => array_filter([
					'kfw-ele-widget-counter-minl-title'
				])
			]);

			echo '<div '.$this->get_render_attribute_string( 'title' ).'>' . $title . '</div>';
		}
	}

	/**
	 * Render icon from icon manger
	 */
    public function _render_icon( $icon , $echo = true ) {
		KFIM::render_icon( $icon, [ 'aria-hidden' => 'true' ] );
	}

	public function _get_image_title( $attachment ) {
		if ( empty( $attachment['id'] ) ) {
			return '';
		}

		return get_the_title( $attachment['id'] );
	}

	public function _get_image_alt( $instance ) {
		if ( empty( $instance['id'] ) ) {
			// For `Insert From URL` images.
			return isset( $instance['alt'] ) ? trim( strip_tags( $instance['alt'] ) ) : '';
		}

		$attachment_id = $instance['id'];
		if ( ! $attachment_id ) {
			return '';
		}

		$attachment = get_post( $attachment_id );
		if ( ! $attachment ) {
			return '';
		}

		$alt = get_post_meta( $attachment_id, '_wp_attachment_image_alt', true );
		if ( ! $alt ) {
			$alt = $attachment->post_excerpt;
			if ( ! $alt ) {
				$alt = $attachment->post_title;
			}
		}
		return trim( strip_tags( $alt ) );
	}

	/**
	 * Render image
	 */
    public function _render_img() {
		$settings = $this->get_settings();

		if( !empty( $settings['image']['url'] ) ) {

			$image_src = $settings['image']['url'];

			$image_title = $this->_get_image_title( $settings['image'] );
			$image_title = empty( $image_title ) ? $this->get_title() : $image_title;

			$image_alt = $this->_get_image_alt( $settings['image'] );
			$image_alt = empty( $image_alt ) ? $image_title : $image_title;

			$this->add_render_attribute( 'image_attr', [
				'alt'   => esc_attr( $image_alt ),
				'title' => esc_attr( $image_title ),
				'src'   => esc_url( $image_src ),
			] );

			echo '<img '.$this->get_render_attribute_string( 'image_attr' ).'/>';
		}
	}

	/**
	 * Render widget output on the frontend.
	 */
    protected function render() {
		$render = kf_extra_plugin_elementor()->get_template("kfw-counter-minimal");
		include $render;
    }
}