<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

use \Elementor\Plugin as KFEl;
use \Elementor\Widget_Base as KFWB;
use \Elementor\Controls_Manager as KFCM;
use \Elementor\Repeater as KFRTR;
use \Elementor\Icons_Manager as KFIM;
use \Elementor\Group_Control_Border as KFGCBR;

class KFW_Header_Action_Icons extends KFWB {

	/**
	 * Get element name.
	 *
	 * Retrieve the element name.
	 *
	 * @return string The name.
	 */
	public function get_name() {
        return 'kfw-header-action-icons';
    }

	/**
	 * Get element title.
	 *
	 * Retrieve the element title.
	 *
	 * @return string Element title.
	 */
	public function get_title() {
		return esc_html__('Action Icons', 'onnat-extra');
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the widget categories.
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ '_kfw_header_footer_widgets' ];
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve the widget icon.
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'kfw-badge eicon-cogs-check kfw-header-action-icons';
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the widget keywords.
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'kfw', 'kin', 'header', 'action' ];
	}

	/**
	 * Get widget help url.
	 *
	 * Retrieve the widget help url.
	 *
	 * @return string Widget help url.
	 */
	public function get_custom_help_url() {
		return KFE_CONST_THEME_DOC . 'elementor-custom-widgets/header-action-icons';
    }

	/**
	 * Get HTML wrapper class.
     *
	 * Retrieve the widget container class.
     *
     * @return string Widget container css classes.
	 */
	public function get_html_wrapper_class() {
        $name = $this->get_name();

		return sprintf( 'kfw-ele-%s elementor-widget-%s', $name, $name );
	}

	/**
	 * Widget base constructor.
	 *
	 * Initializing the widget base class.
	 */
	public function __construct( $data = [], $args = null ) {
        parent::__construct( $data, $args );
    }

	/**
	 * Get style dependencies.
	 *
	 * Retrieve the list of style dependencies the element requires.
	 *
	 * @return array Element styles dependencies.
	 */
	public function get_style_depends() {
		return [];
	}

	/**
	 * Get script dependencies.
	 *
	 * Retrieve the list of script dependencies the element requires.
	 *
	 * @return array Element scripts dependencies.
	 */
	public function get_script_depends() {
		return [];
	}

	/**
	 * Register controls.
	 *
	 * Used to add new controls to any element type.
	 */
	protected function register_controls() {
		$this->_register_action_items_section();
    }

	public function _register_action_items_section() {
		$repeater = new KFRTR();

		$this->start_controls_section( 'section_action_items', [
			'label' => esc_html__( 'Action Items', 'onnat-extra' )
		]);
			$repeater->start_controls_tabs('items_tabs');
				$repeater->start_controls_tab('item_content',[
					'label' => esc_html__( 'Content', 'onnat-extra' ),
				]);
					$repeater->add_control( 'action', [
						'label'   => esc_html__( 'Source', 'onnat-extra' ),
						'type'    => KFCM::CHOOSE,
						'options' => [
							'cart'         => [
								'title' => esc_html__('Cart', 'onnat-extra'),
								'icon'  => 'eicon-cart',
							],
							'hamburger'    => [
								'title' => esc_html__('Hamburger Menu', 'onnat-extra'),
								'icon'  => 'eicon-menu-toggle',
							],
							'search'       => [
								'title' => esc_html__('Search', 'onnat-extra'),
								'icon'  => 'eicon-search',
							],
							'user_account' => [
								'title' => esc_html__('User', 'onnat-extra'),
								'icon'  => 'eicon-person',
							],
							'wishlist'     => [
								'title' => esc_html__('Wishlist', 'onnat-extra'),
								'icon'  => 'eicon-product-meta',
							],
						],
						'default' => 'search',
						'toggle'  => false,
					]);
					$repeater->add_control('cart_note', [
						'type'            => KFCM::RAW_HTML,
						'content_classes' => 'kfw-elementor-editor-notice-ctrl',
						'condition'       => [
							'action' => 'cart'
						],
						'raw'             => sprintf(
							esc_html__('Kindly install and activate %1$s plugin.', 'onnat-extra'),
							sprintf('<a href="%1$s" target="_blank">%2$s</a>', esc_url( 'https://wordpress.org/plugins/woocommerce/' ), esc_html_x( 'WooCommerce', 'link text for lordicon.com', 'onnat-extra' ) )
						)
					]);
					$repeater->add_control('wishlist_note', [
						'type'            => KFCM::RAW_HTML,
						'content_classes' => 'kfw-elementor-editor-notice-ctrl',
						'condition'       => [
							'action' => 'wishlist'
						],
						'raw'             => sprintf(
							esc_html__('Kindly install and activate %1$s plugin.', 'onnat-extra'),
							sprintf('<a href="%1$s" target="_blank">%2$s</a>', esc_url( 'https://wordpress.org/plugins/yith-woocommerce-wishlist/' ), esc_html_x( 'YITH WooCommerce Wishlist', 'link text for lordicon.com', 'onnat-extra' ) )
						)
					]);
					$repeater->add_control( 'hamburger_source', [
						'label'     => esc_html__('Template', 'onnat-extra'),
						'type'      => KFCM::SELECT,
						'default'   => 0,
						'options'   => kf_extra_plugin_elementor()->get_elementor_library(),
						'condition' => [
							'action'  => 'hamburger',
						],
					]);
					$repeater->add_control('icon-type',[
						'label'   => esc_html__( 'Icon Type', 'onnat-extra' ),
						'type'    => KFCM::CHOOSE,
						'options' => [
							'default' => [
								'title' => esc_html__('Default', 'onnat-extra'),
								'icon'  => 'eicon-theme-style',
							],
							'icon'    => [
								'title' => esc_html__('Icon', 'onnat-extra'),
								'icon'  => 'eicon-plus-circle-o',
							],
						],
						'default' => 'default',
						'toggle'  => false,
					]);
					$repeater->add_control( 'trigger-icon', [
						'label'     => esc_html__( 'Icon', 'onnat-extra' ),
						'type'      => KFCM::ICONS,
						'default'   => [
							'value'   => 'fas fa-search',
							'library' => 'fa-solid',
						],
						'condition' => [
							'icon-type' => 'icon'
						],
					]);
				$repeater->end_controls_tab();
				$repeater->start_controls_tab('item_icon',[
					'label' => esc_html__( 'Icon', 'onnat-extra' ),
				]);
					$repeater->add_responsive_control( 'icon_size', [
						'label'      => esc_html__('Size', 'onnat-extra' ),
						'type'       => KFCM::SLIDER,
						'size_units' => [ 'px' ],
						'range'      => [
							'px' => [
								'min' => 0,
								'max' => 200,
							],
						],
						'selectors'  => [
							'{{WRAPPER}} .kfw-ele-widget-header-group-action-main {{CURRENT_ITEM}} span' => 'font-size: {{SIZE}}{{UNIT}};'
						],
					]);
					$repeater->add_control( 'icon_color_settings', [
						'label'        => esc_html__('Colors', 'onnat-extra'),
						'type'         => KFCM::POPOVER_TOGGLE,
						'return_value' => 'yes',
					]);
					$repeater->start_popover();
						$repeater->add_control( 'icon_color', [
							'label'     => esc_html__('Normal Color', 'onnat-extra' ),
							'type'      => KFCM::COLOR,
							'selectors' => [
								'{{WRAPPER}} .kfw-ele-widget-header-group-action-main {{CURRENT_ITEM}} span'     => 'color:{{VALUE}};',
								'{{WRAPPER}} .kfw-ele-widget-header-group-action-main {{CURRENT_ITEM}} span svg' => 'fill:{{VALUE}};',
							],
						]);
						$repeater->add_control( 'icon_h_color', [
							'label'     => esc_html__('Hover Color', 'onnat-extra' ),
							'type'      => KFCM::COLOR,
							'selectors' => [
								'{{WRAPPER}} .kfw-ele-widget-header-group-action-main {{CURRENT_ITEM}}:hover span'     => 'color:{{VALUE}};',
								'{{WRAPPER}} .kfw-ele-widget-header-group-action-main {{CURRENT_ITEM}}:hover span svg' => 'fill:{{VALUE}};',
							],
						]);
					$repeater->end_popover();
				$repeater->end_controls_tab();
				$repeater->start_controls_tab('item_wrap',[
					'label' => esc_html__( 'Wrap', 'onnat-extra' ),
				]);
					$repeater->add_responsive_control( 'wrap_size', [
						'label'      => esc_html__('Size', 'onnat-extra' ),
						'type'       => KFCM::SLIDER,
						'size_units' => [ 'px' ],
						'range'      => [
							'px' => [
								'min' => 0,
								'max' => 500,
							],
						],
						'selectors'  => [
							'{{WRAPPER}} .kfw-ele-widget-header-group-action-main {{CURRENT_ITEM}}' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};'
						],
					]);
					$repeater->add_control( 'wrap_color_settings', [
						'label'        => esc_html__('Background', 'onnat-extra'),
						'type'         => KFCM::POPOVER_TOGGLE,
						'return_value' => 'yes',
					]);
						$repeater->start_popover();
							$repeater->add_control( 'wrap_color', [
								'label'     => esc_html__('Normal Color', 'onnat-extra' ),
								'type'      => KFCM::COLOR,
								'selectors' => [
									'{{WRAPPER}} .kfw-ele-widget-header-group-action-main {{CURRENT_ITEM}}' => 'background-color:{{VALUE}};',
								],
							]);
							$repeater->add_control( 'wrap_h_color', [
								'label'     => esc_html__('Hover Color', 'onnat-extra' ),
								'type'      => KFCM::COLOR,
								'selectors' => [
									'{{WRAPPER}} .kfw-ele-widget-header-group-action-main {{CURRENT_ITEM}}:hover' => 'background-color:{{VALUE}};',
								],
							]);
						$repeater->end_popover();
					$repeater->add_control('wrap_border_before_hr', [
						'type' => KFCM::DIVIDER,
					]);
					$repeater->add_group_control( KFGCBR::get_type(), [
						'name'      => 'wrap_border',
						'selector'  => '{{WRAPPER}} .kfw-ele-widget-header-group-action-main {{CURRENT_ITEM}}',
						'exclude'   => [ 'color' ],
					]);
					$repeater->add_control( 'wrap_border_color_settings', [
						'label'        => esc_html__('Border Color', 'onnat-extra'),
						'type'         => KFCM::POPOVER_TOGGLE,
						'return_value' => 'yes',
						'condition'    => [
							'wrap_border_border!' => [ '', 'none' ],
						],
					]);
						$repeater->start_popover();
							$repeater->add_control( 'wrap_border_color', [
								'label'     => esc_html__('Normal Color', 'onnat-extra' ),
								'type'      => KFCM::COLOR,
								'selectors' => [
									'{{WRAPPER}} .kfw-ele-widget-header-group-action-main {{CURRENT_ITEM}}' => 'border-color:{{VALUE}};',
								],
							]);
							$repeater->add_control( 'wrap_h_border_color', [
								'label'     => esc_html__('Hover Color', 'onnat-extra' ),
								'type'      => KFCM::COLOR,
								'selectors' => [
									'{{WRAPPER}} .kfw-ele-widget-header-group-action-main {{CURRENT_ITEM}}:hover' => 'border-color:{{VALUE}};',
								],
							]);
						$repeater->end_popover();
					$repeater->add_control('wrap_border_radius_before_hr', [
						'type' => KFCM::DIVIDER,
					]);		
					$repeater->add_control('wrap_border_radius_heading',[
						'label' => esc_html__( 'BORDER RADIUS', 'onnat-extra' ),
						'type'  => KFCM::HEADING,
					]);
					$repeater->add_responsive_control( 'wrap_border_radius', [
						'label'      => esc_html__('Normal', 'onnat-extra' ),
                        'type'       => KFCM::DIMENSIONS,
                        'size_units' => [ 'px', '%' ],
                        'selectors'  => [
                            '{{WRAPPER}} .kfw-ele-widget-header-group-action-main {{CURRENT_ITEM}}' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                        ]
                    ]);
					$repeater->add_responsive_control( 'wraph_h_border_radius', [
						'label'      => esc_html__('Hover', 'onnat-extra' ),
                        'type'       => KFCM::DIMENSIONS,
                        'size_units' => [ 'px', '%' ],
                        'selectors'  => [
                            '{{WRAPPER}} .kfw-ele-widget-header-group-action-main {{CURRENT_ITEM}}:hover' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                        ]
                    ]);	
				$repeater->end_controls_tab();
			$repeater->end_controls_tabs();

            $this->add_control('items', [
				'type'    => KFCM::REPEATER,
				'fields'  => $repeater->get_controls(),
				'default' => [],
            ]);
			$this->add_control('align', [
				'label'          => esc_html__( 'Alignment', 'onnat-extra' ),
				'type'           => KFCM::CHOOSE,
				'options'        => [
					'left'   => [
						'title' => esc_html__( 'Left', 'onnat-extra' ),
						'icon'  => 'eicon-text-align-left',
					],
					'center' => [
						'title' => esc_html__( 'Center', 'onnat-extra' ),
						'icon'  => 'eicon-text-align-center',
					],
					'right'  => [
						'title' => esc_html__( 'Right', 'onnat-extra' ),
						'icon'  => 'eicon-text-align-right',
					],
				],
				'default'        => 'center',
				'toggle'         => false,
				'prefix_class'   => 'kfw-ele-widget-header-group-action-icons-align-',
			]);
			$this->add_responsive_control( 'gap', [
				'label'      => esc_html__('Gap', 'onnat-extra' ),
				'type'       => KFCM::SLIDER,
				'size_units' => [ 'px' ],
				'range'      => [
					'px' => [
						'min' => 0,
						'max' => 500,
					],
				],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-header-group-action-main > div' => 'margin: 0 {{SIZE}}{{UNIT}};',
				],
			]);			
		$this->end_controls_section();
	}

	/**
	 * Get template content
	 */
	public function _generate_template_content( $template_id ) {
		return KFEl::instance()->frontend->get_builder_content_for_display( $template_id );
	}

	public function _render_icon( $icon, $class = '', $echo = false ) {
		$icon = array_filter( $icon );

		if( !empty( $icon ) ) {
			if( $echo ) {
				echo '<span class="kinfw-header-element kfw-elementor-icon-wrap '. esc_attr( $class ).'">';
					KFIM::render_icon( $icon, [ 'aria-hidden' => 'true' ] );
				echo '</span>';
			} else {
				return '<span class="kinfw-header-element kfw-elementor-icon-wrap '. esc_attr( $class ).'">'. KFIM::try_get_icon_html( $icon, [ 'aria-hidden' => 'true' ] ) . '</span>';
			}
		}
    }

	public function render_cart_action_icon( $item ) {
		$icon            = kinfw_icon( 'shopping-bag', 'kinfw-header-element' );
		$cart_link       = 'javascript:void(0);';
		$cart_link_class = 'kinfw-header-mini-cart-a-trigger';
		$count_html      = '<span class="kinfw-header-cart-count">0</span>';

		if( class_exists( 'WooCommerce' ) ) {
            $cart  = WC()->cart;
            $count = !is_null( $cart ) ? esc_html( $cart->get_cart_contents_count() ) : '';

			if( apply_filters('woocommerce_widget_cart_is_hidden', is_cart() || is_checkout()) ) {
				$cart_link       = esc_url( wc_get_cart_url() );
				$cart_link_class = '';
			}

			$count_html = sprintf( '<span class="kinfw-header-cart-count %1$s">%2$s</span>',
				$count ? 'kinfw-show': 'kinfw-hidden',
				$count
			);
		}

		if( 'icon' == $item['icon-type'] ) {
			$icon = $this->_render_icon( $item['trigger-icon'] );
		}

		printf('<!-- .kinfw-header-mini-cart-trigger -->
			<div class="kinfw-header-mini-cart-trigger elementor-repeater-item elementor-repeater-item-%5$s">
				<a href="%1$s" class="%2$s"> %3$s %4$s </a>
			</div><!-- /.kinfw-header-mini-cart-trigger -->',
			$cart_link,
			$cart_link_class,
			$icon,
			$count_html,
			esc_attr( $item['_id'] ),
		);

		if( class_exists( 'WooCommerce' ) && function_exists('kinfw_action_header_mini_cart') ) {
			add_action( 'wp_footer', 'kinfw_action_header_mini_cart', -1 );
		}
	}

	public function render_hamburger_action_icon( $item ) {
		if( !empty( $item['hamburger_source'] ) ) {
			$icon = kinfw_icon( 'misc-menu-dots', 'kinfw-header-element' );
			if( 'icon' == $item['icon-type'] ) {
				$icon = $this->_render_icon( $item['trigger-icon'] );
			}

			printf('<!-- .kinfw-header-hamburger-btn-trigger -->
				<div class="kinfw-header-hamburger-btn-trigger elementor-repeater-item elementor-repeater-item-%1$s">
					%2$s
				</div><!-- /.kinfw-header-hamburger-btn-trigger -->',
				esc_attr( $item['_id'] ),
				$icon,
			);

			$content = $this->_generate_template_content( $item['hamburger_source'] );
			/**
			 * use keyword: Allows you to import variables from the parent scope into an anonymous function introduced in PHP 5.3.
			 */
			add_action( 'wp_footer', function() use( $content ) {
				get_template_part( 'header-templates/hamburger/hamburger', '', [ 'content' => $content ] );
			}, -1 );
		}
	}

	public function render_search_action_icon( $item ) {
		$icon = kinfw_icon( 'misc-search', 'kinfw-header-element' );
		if( 'icon' == $item['icon-type'] ) {
			$icon = $this->_render_icon( $item['trigger-icon'] );
		}

		printf('<!-- .kinfw-header-search-trigger -->
            <div class="kinfw-header-search-trigger elementor-repeater-item elementor-repeater-item-%1$s">
                %2$s
            </div><!-- /.kinfw-header-search-trigger -->',
			esc_attr( $item['_id'] ),
            $icon,
        );

        if( function_exists('kinfw_action_search_form') ) {
            add_action( 'wp_footer', 'kinfw_action_search_form', -1 );
        }
	}

	public function render_user_account_action_icon( $item ) {
        global $wp;
		if ( is_user_logged_in() ) {
			$current_user    = wp_get_current_user();
			$current_user_id = $current_user->ID;
			$user_image      = get_avatar( $current_user_id, 28 );

			printf('<!-- .kinfw-header-login-trigger -->
				<div class="kinfw-header-login-trigger elementor-repeater-item elementor-repeater-item-%1$s">', esc_attr( $item['_id'] ) );
				if( $user_image ) {
					printf( '<span class="kinfw-header-element kinfw-user-avatar">%1$s</span>', $user_image );
				} else {
					$icon = kinfw_icon( 'user-single', 'kinfw-header-element' );
					if( 'icon' == $item['icon-type'] ) {
						$icon = $this->_render_icon( $item['trigger-icon'] );
					}

					echo $icon;
				}

                /**
                 * User Nav Menu
                 */
				$nav_menu_items = apply_filters( 'kinfw-filter/theme/header/action/user-nav-items',[
					'logout' => [
						'url'  => wp_logout_url( home_url( add_query_arg( [], $wp->request ) ) ),
						'text' => esc_html__( 'Log Out', 'onnat' )
					]
				] );

				echo '<ul class="kinfw-header-user-nav">';
					foreach( $nav_menu_items as $menu_items ) {
						printf( '<li><a href="%1$s">%2$s</a></li>',
							esc_url( $menu_items['url'] ),
                            esc_html( $menu_items['text'] )
                        );
					}
				echo '</ul>';
			echo '</div> <!-- /.kinfw-header-login-trigger -->';
		} else {

			$icon = kinfw_icon( 'user-single', 'kinfw-header-element kinfw-header-login-user-trigger' );
			if( 'icon' == $item['icon-type'] ) {
				$icon = $this->_render_icon( $item['trigger-icon'], 'kinfw-header-login-user-trigger' );
			}
			printf('<!-- .kinfw-header-login-trigger -->
				<div class="kinfw-header-login-trigger elementor-repeater-item elementor-repeater-item-%1$s">
					%2$s
				</div><!-- /.kinfw-header-login-trigger -->',
				esc_attr( $item['_id'] ),
				$icon,
			);
		}

        if( function_exists('kinfw_action_user_login_form') ) {
			add_action( 'wp_footer', 'kinfw_action_user_login_form', -1 );
		}
	}

	public function render_wishlist_action_icon( $item ) {
		$icon       = kinfw_icon( 'heart-regular', 'kinfw-header-element' );
		$count      = function_exists( 'yith_wcwl_count_all_products' ) ? yith_wcwl_count_all_products() : 1;
		$count_html = sprintf( '<span class="kinfw-header-wishlist-count %1$s">%2$s</span>',
			$count ? 'kinfw-show': 'kinfw-hidden',
			$count
		);

		if( 'icon' == $item['icon-type'] ) {
			$icon = $this->_render_icon( $item['trigger-icon'] );
		}

		printf( '<!-- .kinfw-header-wishlist-trigger -->
			<div class="kinfw-header-wishlist-trigger elementor-repeater-item elementor-repeater-item-%4$s">
				<a href="%1$s" class="kinfw-header-wishlist-a-trigger"> %2$s %3$s </a>
			</div> <!-- / .kinfw-header-wishlist-trigger -->',
			esc_url( get_permalink( get_option( 'yith_wcwl_wishlist_page_id' ) ) ),
			$icon,
			$count_html,
			esc_attr( $item['_id'] )
		);
	}

	/**
	 * Render widget output on the frontend.
	 */
    protected function render() {
		$render = kf_extra_plugin_elementor()->get_template("kfw-header-action-icons");
		include $render;
    }
}