<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

use \Elementor\Widget_Base as KFWB;
use \Elementor\Controls_Manager as KFCM;
use \Elementor\Group_Control_Image_Size as KFGCIS;
use \Elementor\Group_Control_Typography as KFGCT;
use \Elementor\Group_Control_Border as KFGCBR;
use \Elementor\Group_Control_Background as KFGCBG;
use \Elementor\Group_Control_Box_Shadow as KFGCBS;
use \Elementor\Group_Control_Text_Shadow as KFGCTS;
use \Elementor\Group_Control_Text_Stroke as KFGCTST;
use \Elementor\Repeater as KFRTR;
use \Elementor\Utils as KFUtils;

class KFW_Image_Accordion extends KFWB {

	/**
	 * Get element name.
	 *
	 * Retrieve the element name.
	 *
	 * @return string The name.
	 */
	public function get_name() {
        return 'kfw-image-accordion';
    }

	/**
	 * Get element title.
	 *
	 * Retrieve the element title.
	 *
	 * @return string Element title.
	 */
	public function get_title() {
		return esc_html__('Image Accordion', 'onnat-extra');
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the widget categories.
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ '_kfw_core_widgets' ];
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve the widget icon.
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'kfw-badge eicon-accordion';
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the widget keywords.
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'images', 'accordion', 'image' ];
	}

	/**
	 * Get widget help url.
	 *
	 * Retrieve the widget help url.
	 *
	 * @return string Widget help url.
	 */
	public function get_custom_help_url() {
		return KFE_CONST_THEME_DOC . 'elementor-custom-widgets/image-accordion';
    }

	/**
	 * Get HTML wrapper class.
     *
	 * Retrieve the widget container class.
     *
     * @return string Widget container css classes.
	 */
	public function get_html_wrapper_class() {
        $name = $this->get_name();

		return sprintf( 'kfw-ele-%s elementor-widget-%s', $name, $name );
	}

	/**
	 * Widget base constructor.
	 *
	 * Initializing the widget base class.
	 */
	public function __construct( $data = [], $args = null ) {
        parent::__construct( $data, $args );

		wp_register_script( 'kfw-image-accordion',
			KFE_CONST_URL . 'libraries/elementor/assets/js/image-accordion' . KFE_CONST_DEBUG_SUFFIX . '.js',
			[ 'jquery', 'elementor-frontend' ],
			KFE_CONST_VERSION,
			true
		);

		wp_register_style( 'kfw-image-accordion',
			KFE_CONST_URL . 'libraries/elementor/assets/css/image-accordion' . KFE_CONST_DEBUG_SUFFIX . '.css',
			[],
			KFE_CONST_VERSION
		);
    }

	/**
	 * Get style dependencies.
	 *
	 * Retrieve the list of style dependencies the element requires.
	 *
	 * @return array Element styles dependencies.
	 */
	public function get_style_depends() {
		return [ 'kfw-image-accordion' ];
	}

	/**
	 * Get script dependencies.
	 *
	 * Retrieve the list of script dependencies the element requires.
	 *
	 * @return array Element scripts dependencies.
	 */
	public function get_script_depends() {
		return [ 'imagesloaded', 'kfw-image-accordion' ];
	}

	/**
	 * Register controls.
	 *
	 * Used to add new controls to any element type.
	 */
	protected function register_controls() {
		$this->_register_items_section();
		$this->_register_settings_section();
		$this->_register_container_style_section();
		$this->_register_item_style_section();
		$this->_register_item_index_style_section();
		$this->_register_item_title_style_section();
		$this->_register_item_sub_title_style_section();
		$this->_register_item_content_style_section();
		$this->_register_item_content_boxed_style_section();
		$this->_register_item_action_btn_style_section();
	}

	public function _register_items_section() {
		$this->start_controls_section( 'section_items', [
			'label' => esc_html__( 'Items', 'onnat-extra' )
        ]);

			$repeater = new KFRTR();

			$repeater->start_controls_tabs('tabs');

				/**
				 * Tab: Image
				 */
				$repeater->start_controls_tab('tab_image',[
					'label' => esc_html__( 'Image', 'onnat-extra' )
				]);
					$repeater->add_control('image',[
						'label'   => esc_html__( 'Image', 'onnat-extra' ),
						'type'    => KFCM::MEDIA,
						'default' => [ 'url' => KFE_CONST_URL .'libraries/elementor/assets/images/image-placeholder-'.rand(1,5).'.svg' ],
						'dynamic' => [ 'active' => true ],
					]);
				$repeater->end_controls_tab();

				/**
				 * Tab: Content
				 */
				$repeater->start_controls_tab('tab_content',[
					'label' => esc_html__( 'Content', 'onnat-extra' )
				]);
					$repeater->add_control('title',[
						'label'       => esc_html__( 'Title', 'onnat-extra' ),
						'type'        => KFCM::TEXT,
						'dynamic'     => [ 'active' => true ],
						'default'     => esc_html__( 'Title' , 'onnat-extra' ),
						'label_block' => true,
					]);
					$repeater->add_control('sub_title',[
						'label'       => esc_html__( 'Sub Title', 'onnat-extra' ),
						'type'        => KFCM::TEXT,
						'dynamic'     => [ 'active' => true ],
						'label_block' => true,
					]);
					$repeater->add_control('button',[
						'label' => esc_html__('Button', 'onnat-extra'),
						'type'  => KFCM::POPOVER_TOGGLE,
					]);
						$repeater->start_popover();
							$repeater->add_control('button_txt',[
								'label'   => esc_html__( 'Button Text', 'onnat-extra' ),
								'type'    => KFCM::TEXT,
								'dynamic' => [ 'active' => true ],
								'default' => esc_html__( 'Read More' , 'onnat-extra' ),
							]);
							$repeater->add_control('button_link',[
								'label'     => esc_html__( 'Button Link', 'onnat-extra' ),
								'type'      => KFCM::URL,
								'dynamic'   => [
									'active' => true
								],
								'options'   => [
									'url',
									'is_external',
									'nofollow'
								],
								'default'   => [
									'url'         => '#',
									'is_external' => true,
									'nofollow'    => true,
								],
								'condition' => [
									'button_txt!' => ''
								]
							]);
						$repeater->end_popover();
					$repeater->add_control('content',[
						'label'   => esc_html__( 'Content', 'onnat-extra' ),
						'type'    => KFCM::WYSIWYG,
						'default' => 'Neque porro quisquam est qui dolorem ipsum quia dolor sit amet, consectetur, adipisci velit.',
						'dynamic' => [ 'active' => true ],
					]);
				$repeater->end_controls_tab();

			$repeater->end_controls_tabs();

			$this->add_control( 'items', [
				'type'    => KFCM::REPEATER,
				'fields'  => $repeater->get_controls(),
				'default' => [
					[
						'image' => [
							'url' => KFE_CONST_URL .'libraries/elementor/assets/images/image-placeholder-1.svg'
						]
					],
					[
						'image' => [
							'url' => KFE_CONST_URL .'libraries/elementor/assets/images/image-placeholder-2.svg'
						]
					],
					[
						'image' => [
							'url' => KFE_CONST_URL .'libraries/elementor/assets/images/image-placeholder-3.svg'
						]
					],
					[
						'image' => [
							'url' => KFE_CONST_URL .'libraries/elementor/assets/images/image-placeholder-4.svg'
						]
					],
					[
						'image' => [
							'url' => KFE_CONST_URL .'libraries/elementor/assets/images/image-placeholder-5.svg'
						]
					],
				]
			]);

		$this->end_controls_section();
	}

	public function _register_settings_section() {
		$this->start_controls_section( 'section_settings', [
			'label' => esc_html__( 'Settings', 'onnat-extra' )
        ]);
			$this->add_group_control( KFGCIS::get_type(), [
				'name'    => 'image_size',
				'default' => 'full',
			]);
			$this->add_control( 'trigger', [
				'label'   => esc_html__( 'Selection On', 'onnat-extra' ),
				'type'    => KFCM::CHOOSE,
				'options' => [
					'click'     => [
						'title' => esc_html__('Mouse Click', 'onnat-extra'),
						'icon'  => 'eicon-click',
					],
					'mouseover'  => [
						'title' => esc_html__('Mouse Hover', 'onnat-extra'),
						'icon'  => 'eicon-cursor-move',
					],
				],
				'default' => 'mouseover',
				'toggle'  => false,
			]);
			$this->add_responsive_control('orientation',[
				'label'                => esc_html__( 'Orientation', 'onnat-extra' ),
				'type'                 => KFCM::CHOOSE,
				'options'              => [
					'vertical'   => [
						'title' => esc_html__('Vertical','onnat-extra'),
						'icon'  => 'eicon-navigation-vertical'
					],
					'horizontal' => [
						'title' => esc_html__('Horizontal','onnat-extra'),
						'icon'  => 'eicon-navigation-horizontal'
					],
				],
				'default'              => 'horizontal',
				'tablet_default'       => 'horizontal',
				'mobile_default'       => 'horizontal',
				'toggle'               => false,
				'prefix_class'         => 'kfw-elementor-image-accordion%s-orientation-',
				'selectors_dictionary' => [
                    'horizontal' => 'flex-direction: unset;',
                    'vertical'   => 'flex-direction: column;',
				],
				'selectors'            => [
					'{{WRAPPER}} .kfw-elementor-widget-image-accordion-wrapper' => '{{VALUE}};',
				]
			]);
			$this->add_control('active_item',[
				'label' => esc_html__( 'Active Item?', 'onnat-extra' ),
				'type'  => KFCM::SWITCHER,
			]);
			$this->add_control('active_item_number',[
				'label'       => esc_html__( 'Item Number', 'onnat-extra' ),
				'description' => esc_html__( 'Type your item number', 'onnat-extra' ),
				'type'        => KFCM::NUMBER,
				'default'     => 1,
				'min'         => 1,
				'setp'        => 1,
				'condition'   => [
					'active_item' => 'yes'
				]
			]);
			$this->add_control('active_size',[
				'label'     => esc_html__( 'Active Size', 'onnat-extra' ),
				'type'      => KFCM::SLIDER,
				'range'     => [ '%' => ['min' => 50, 'max' => 100  ] ],
				'default'   => [
					'unit' => '%',
					'size' => 50,
				],
			]);
			$this->add_control('show_index',[
				'label' => esc_html__( 'Show Item Index?', 'onnat-extra' ),
				'type'  => KFCM::SWITCHER,
			]);
			$this->add_control('anim_heading',[
				'label'     => esc_html__( 'Animation Settings', 'onnat-extra' ),
				'type'      => KFCM::HEADING,
				'separator' => 'before',
			]);
			$this->add_control('use_anim',[
				'label' => esc_html__( 'Use Animation?', 'onnat-extra' ),
				'type'  => KFCM::SWITCHER,
			]);
			$this->add_control('anim_duration',[
				'label'     => esc_html__( 'Duration', 'onnat-extra' ),
				'type'      => KFCM::SLIDER,
				'range'     => [ '%' => ['min' => 100, 'max' => 3000, 'step' => 100  ] ],
				'condition' => [
					'use_anim' => 'yes'
				],
				'default'   => [
					'unit' => '%',
					'size' => 500,
				],
			]);
			$this->add_control('anim_type',[
				'label'     => esc_html__( 'Type', 'onnat-extra' ),
				'type'      => KFCM::SELECT,
				'default'   => 'kfw-elementor-image-accordion-cubic-anim',
				'options'   => [
					'kfw-elementor-image-accordion-back-anim'  => esc_html__( 'Back', 'onnat-extra' ),
					'kfw-elementor-image-accordion-circ-anim'  => esc_html__( 'Circ', 'onnat-extra' ),
					'kfw-elementor-image-accordion-cubic-anim' => esc_html__( 'Cubic', 'onnat-extra' ),
					'kfw-elementor-image-accordion-expo-anim'  => esc_html__( 'Expo', 'onnat-extra' ),
					'kfw-elementor-image-accordion-quad-anim'  => esc_html__( 'Quad', 'onnat-extra' ),
					'kfw-elementor-image-accordion-quart-anim' => esc_html__( 'Quart', 'onnat-extra' ),
					'kfw-elementor-image-accordion-quint-anim' => esc_html__( 'Quint', 'onnat-extra' ),
					'kfw-elementor-image-accordion-sine-anim'  => esc_html__( 'Sine', 'onnat-extra' ),
				],
				'condition' => [
					'use_anim' => 'yes'
				],
			]);
		$this->end_controls_section();
	}

	public function _register_container_style_section() {
		$this->start_controls_section( 'section_container_style',[
			'label' => esc_html__( 'Container', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
			$this->add_responsive_control('container_height',[
				'label'     => esc_html__( 'Height', 'onnat-extra' ),
				'type'      => KFCM::SLIDER,
				'range'     => [ 'px' => ['min' => 100, 'max' => 1000  ] ],
				'default'   => [
					'unit' => 'px',
					'size' => 600,
				],
				'selectors' => [
					'{{WRAPPER}} .kfw-elementor-widget-image-accordion-wrapper' => 'height:{{SIZE}}{{UNIT}};'
				],
			]);
			$this->add_group_control( KFGCBG::get_type(), [
				'name'     => 'container_bg',
				'types'    => ['classic', 'gradient'],
				'exclude'  => [ 'image'],
				'selector' => '{{WRAPPER}} .kfw-elementor-widget-image-accordion-wrapper',
			]);
			$this->add_group_control( KFGCBR::get_type(), [
				'name'     => 'border',
				'selector' => '{{WRAPPER}} .kfw-elementor-widget-image-accordion-wrapper'
			]);
			$this->add_responsive_control( 'container_border_radius', [
				'label'      => esc_html__('Border Radius', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-elementor-widget-image-accordion-wrapper' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
			$this->add_group_control( KFGCBS::get_type(), [
				'name'     => 'container_box_shadow',
				'selector' => '{{WRAPPER}} .kfw-elementor-widget-image-accordion-wrapper'
			]);
			$this->add_responsive_control( 'container_padding',[
				'label'      => esc_html__('Padding', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-elementor-widget-image-accordion-wrapper' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
		$this->end_controls_section();
	}

	public function _register_item_style_section() {
		$this->start_controls_section( 'section_item_style',[
			'label' => esc_html__( 'Gutter', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
			$this->add_responsive_control('guter_width_horizontal',[
				'label'          => esc_html__( 'Width', 'onnat-extra' ),
				'type'           => KFCM::SLIDER,
				'range'          => [
					'px' => [
						'min' => 0,
						'max' => 100
					]
				],
				'default'        => [
					'unit' => 'px',
					'size' => 5,
				],
				'tablet_default' => [
					'size' => 5,
					'unit' => 'px',
				],
				'mobile_default' => [
					'size' => 5,
					'unit' => 'px',
				],
				'condition'      => [
					'orientation' => 'horizontal'
				],
				'selectors'      => [
					'{{WRAPPER}} .kfw-elementor-image-accordion-item:after' => 'width:{{SIZE}}{{UNIT}}; height:100%;'
				],
			]);
			$this->add_responsive_control('guter_width_vertical',[
				'label'          => esc_html__( 'Width', 'onnat-extra' ),
				'type'           => KFCM::SLIDER,
				'range'          => [
					'px' => [
						'min' => 0,
						'max' => 100
					]
				],
				'default'        => [
					'unit' => 'px',
					'size' => 5,
				],
				'tablet_default' => [
					'size' => 5,
					'unit' => 'px',
				],
				'mobile_default' => [
					'size' => 5,
					'unit' => 'px',
				],
				'condition'      => [
					'orientation' => 'vertical'
				],
				'selectors'      => [
					'{{WRAPPER}} .kfw-elementor-image-accordion-item:after' => 'height:{{SIZE}}{{UNIT}}; width:100%;'
				],
			]);
			$this->add_control( 'guter_color', [
				'label'     => esc_html__('Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'selectors' => [
					'{{WRAPPER}} .kfw-elementor-image-accordion-item:after' => 'background:{{VALUE}};'
				]
			]);
		$this->end_controls_section();
	}

	public function _register_item_index_style_section() {
		$condition = [ 'show_index' => 'yes' ];

		$this->start_controls_section( 'section_item_index_style',[
			'label'     => esc_html__( 'Item::Index', 'onnat-extra' ),
			'tab'       => KFCM::TAB_STYLE,
			'condition' => $condition,
		]);
			$this->add_group_control(KFGCT::get_type(), [
				'name'      => 'index_typo',
				'label'     => esc_html__( 'Typography', 'onnat-extra' ),
				'condition' => $condition,
				'selector'  => '{{WRAPPER}} .kfw-elementor-image-accordion-index-wrapper',
			]);
			$this->add_control( 'index_color', [
				'label'     => esc_html__('Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'condition' => $condition,
				'selectors' => [
					'{{WRAPPER}} .kfw-elementor-image-accordion-index-wrapper' => 'color:{{VALUE}};',
				]
			]);
			$this->add_group_control( KFGCTS::get_type(), [
				'name'      => 'index_text_shadow',
				'condition' => $condition,
				'selector'  => '{{WRAPPER}} .kfw-elementor-image-accordion-index-wrapper',
			]);
			$this->add_group_control( KFGCTST::get_type(), [
				'name'      => 'index_text_stroke',
				'condition' => $condition,
				'selector'  => '{{WRAPPER}} .kfw-elementor-image-accordion-index-wrapper',
			]);		
		$this->end_controls_section();
	}

	public function _register_item_title_style_section() {
		$this->start_controls_section( 'section_item_title_style',[
			'label' => esc_html__( 'Item::Title', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
			$this->add_group_control(KFGCT::get_type(), [
				'name'     => 'title_typo',
				'label'    => esc_html__( 'Typography', 'onnat-extra' ),
				'selector' => '{{WRAPPER}} .kfw-elementor-image-accordion-title'
			]);
			$this->add_control( 'title_color', [
				'label'     => esc_html__('Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'selectors' => [
					'{{WRAPPER}} .kfw-elementor-image-accordion-title' => 'color:{{VALUE}};'
				]
			]);
			$this->add_group_control( KFGCTS::get_type(), [
				'name'     => 'title_text_shadow',
				'selector' => '{{WRAPPER}} .kfw-elementor-image-accordion-title'
			]);
			$this->add_group_control( KFGCTST::get_type(), [
				'name'     => 'title_text_stroke',
				'selector' => '{{WRAPPER}} .kfw-elementor-image-accordion-title'
			]);
			$this->add_responsive_control( 'title_padding',[
				'label'      => esc_html__('Padding', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-elementor-image-accordion-title' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
			$this->add_responsive_control( 'title_margin',[
				'label'      => esc_html__('Margin', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-elementor-image-accordion-title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
		$this->end_controls_section();
	}

	public function _register_item_sub_title_style_section() {
		$this->start_controls_section( 'section_item_stitle_style',[
			'label' => esc_html__( 'Item::Sub Title', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
			$this->add_group_control(KFGCT::get_type(), [
				'name'     => 'sub_title_typo',
				'label'    => esc_html__( 'Typography', 'onnat-extra' ),
				'selector' => '{{WRAPPER}} .kfw-elementor-image-accordion-sub-title'
			]);
			$this->add_control( 'sub_title_color', [
				'label'     => esc_html__('Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'selectors' => [
					'{{WRAPPER}} .kfw-elementor-image-accordion-sub-title' => 'color:{{VALUE}};'
				]
			]);
			$this->add_group_control( KFGCTS::get_type(), [
				'name'     => 'sub_title_text_shadow',
				'selector' => '{{WRAPPER}} .kfw-elementor-image-accordion-sub-title'
			]);
			$this->add_group_control( KFGCTST::get_type(), [
				'name'     => 'sub_title_text_stroke',
				'selector' => '{{WRAPPER}} .kfw-elementor-image-accordion-sub-title'
			]);
			$this->add_responsive_control( 'sub_title_padding',[
				'label'      => esc_html__('Padding', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-elementor-image-accordion-sub-title' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
			$this->add_responsive_control( 'sub_title_margin',[
				'label'      => esc_html__('Margin', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-elementor-image-accordion-sub-title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
		$this->end_controls_section();
	}

	public function _register_item_content_style_section() {
		$this->start_controls_section( 'section_item_content_style',[
			'label' => esc_html__( 'Item::Content', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
			$this->add_control( 'content_style', [
				'label'        => esc_html__('Style', 'onnat-extra' ),
				'type'         => KFCM::SWITCHER,
				'label_on'     => esc_html__( 'Fill', 'onnat-extra' ),
				'label_off'    => esc_html__( 'Boxed', 'onnat-extra' ),
				'return_value' => 'yes',
				'default'      => 'yes',
			]);
			$this->add_group_control(KFGCT::get_type(), [
				'name'     => 'content_typo',
				'label'    => esc_html__( 'Typography', 'onnat-extra' ),
				'selector' => '{{WRAPPER}} .kfw-elementor-image-accordion-content'
			]);
			$this->add_control( 'content_color', [
				'label'     => esc_html__('Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'selectors' => [
					'{{WRAPPER}} .kfw-elementor-image-accordion-content' => 'color:{{VALUE}};'
				]
			]);
			$this->add_group_control( KFGCTS::get_type(), [
				'name'     => 'content_text_shadow',
				'selector' => '{{WRAPPER}} .kfw-elementor-image-accordion-content'
			]);
			$this->add_group_control( KFGCTST::get_type(), [
				'name'     => 'content_text_stroke',
				'selector' => '{{WRAPPER}} .kfw-elementor-image-accordion-content'
			]);
			$this->add_responsive_control( 'content_padding',[
				'label'      => esc_html__('Padding', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-elementor-image-accordion-content' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
			$this->add_responsive_control( 'content_margin',[
				'label'      => esc_html__('Margin', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-elementor-image-accordion-content' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
			$this->add_group_control( KFGCBG::get_type(), [
				'name'           => 'content_overlay',
				'types'          => ['classic', 'gradient'],
				'exclude'        => [ 'image'],
				'selector'       => '{{WRAPPER}} .kfw-elementor-image-accordion-content-boxed-style .kfw-elementor-image-accordion-content-wrapper, {{WRAPPER}} .kfw-elementor-image-accordion-content-fill-style .kfw-elementor-image-accordion-item:before',
				'fields_options' => [
					'background' => [
						'label'       => esc_html__( 'Overlay Type', 'onnat-extra' ),
						'type'        => KFCM::CHOOSE,
						'render_type' => 'ui',
					]
				],
			]);
			$this->add_responsive_control('content_width',[
				'label'     => esc_html__( 'Width', 'onnat-extra' ),
				'type'      => KFCM::SLIDER,
				'range'     => [ 'px' => ['min' => 100, 'max' => 1000  ] ],
				'default'   => [
					'unit' => 'px',
					'size' => 350,
				],
				'selectors' => [
					'{{WRAPPER}} .kfw-elementor-image-accordion-content-wrapper' => 'width:{{SIZE}}{{UNIT}};'
				],
			]);
		$this->end_controls_section();
	}

	public function _register_item_content_boxed_style_section() {
		$this->start_controls_section( 'section_item_boxed_content_style',[
			'label'     => esc_html__( 'Item::Content Boxed Style', 'onnat-extra' ),
			'tab'       => KFCM::TAB_STYLE,
			'condition' => [ 'content_style!' => 'yes' ],
		]);
			$this->add_group_control( KFGCBR::get_type(), [
				'name'      => 'boxed_content_border',
				'condition' => [ 'content_style!' => 'yes' ],
				'selector'  => '{{WRAPPER}} .kfw-elementor-image-accordion-content-boxed-style .kfw-elementor-image-accordion-content-wrapper'
			]);
			$this->add_responsive_control( 'boxed_content_border_radius', [
				'label'      => esc_html__('Border Radius', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'condition'  => [ 'content_style!' => 'yes' ],
				'size_units' => [ 'px', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-elementor-image-accordion-content-boxed-style .kfw-elementor-image-accordion-content-wrapper' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
			$this->add_group_control( KFGCBS::get_type(), [
				'name'      => 'boxed_content_box_shadow',
				'condition' => [ 'content_style!' => 'yes' ],
				'selector'  => '{{WRAPPER}} .kfw-elementor-image-accordion-content-boxed-style .kfw-elementor-image-accordion-content-wrapper'
			]);
		$this->end_controls_section();
	}

	public function _register_item_action_btn_style_section() {
		$this->start_controls_section( 'section_item_action_btn_style',[
			'label' => esc_html__( 'Item::Action Button', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
			$this->add_group_control(KFGCT::get_type(), [
				'name'     => 'content_btn_typo',
				'label'    => esc_html__( 'Typography', 'onnat-extra' ),
				'selector' => '{{WRAPPER}} .kfw-elementor-image-accordion-button'
			]);
			$this->add_responsive_control( 'content_btn_padding',[
				'label'      => esc_html__('Padding', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-elementor-image-accordion-button' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
			$this->add_responsive_control( 'content_btn_margin',[
				'label'      => esc_html__('Margin', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-elementor-image-accordion-button' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
			$this->start_controls_tabs('tabs_content_btn_style');
				$this->start_controls_tab('tab_content_btn_normal_style',[
					'label' => esc_html__( 'Normal', 'onnat-extra' )
				]);
					$this->add_control( 'btn_normal_color', [
						'label'     => esc_html__('Text Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'selectors' => [
							'{{WRAPPER}} .kfw-elementor-image-accordion-button' => 'color:{{VALUE}};'
						]
					]);
					$this->add_control( 'btn_normal_bg_color', [
						'label'     => esc_html__('Background Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'selectors' => [
							'{{WRAPPER}} .kfw-elementor-image-accordion-button' => 'background:{{VALUE}};'
						]
					]);
					$this->add_group_control( KFGCTS::get_type(), [
						'name'     => 'btn_normal_text_shadow',
						'selector' => '{{WRAPPER}} .kfw-elementor-image-accordion-button'
					]);
					$this->add_group_control( KFGCTST::get_type(), [
						'name'     => 'btn_normal_text_stroke',
						'selector' => '{{WRAPPER}} .kfw-elementor-image-accordion-button'
					]);
					$this->add_group_control( KFGCBR::get_type(), [
						'name'     => 'btn_normal_border',
						'selector' => '{{WRAPPER}} .kfw-elementor-image-accordion-button'
					]);
					$this->add_responsive_control( 'btn_normal_border_radius', [
						'label'      => esc_html__('Border Radius', 'onnat-extra' ),
						'type'       => KFCM::DIMENSIONS,
						'size_units' => [ 'px', '%' ],
						'selectors'  => [
							'{{WRAPPER}} .kfw-elementor-image-accordion-button' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
						]
					]);
					$this->add_group_control( KFGCBS::get_type(), [
						'name'     => 'btn_normal_box_shadow',
						'selector' => '{{WRAPPER}} .kfw-elementor-image-accordion-button'
					]);
				$this->end_controls_tab();
				$this->start_controls_tab('tab_content_btn_hover_style',[
					'label' => esc_html__( 'Hover', 'onnat-extra' )
				]);
					$this->add_control( 'btn_hover_color', [
						'label'     => esc_html__('Text Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'selectors' => [
							'{{WRAPPER}} .kfw-elementor-image-accordion-button:hover' => 'color:{{VALUE}};'
						]
					]);
					$this->add_control( 'btn_hover_bg_color', [
						'label'     => esc_html__('Background Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'selectors' => [
							'{{WRAPPER}} .kfw-elementor-image-accordion-button:hover' => 'background:{{VALUE}};'
						]
					]);
					$this->add_group_control( KFGCTS::get_type(), [
						'name'     => 'btn_hover_text_shadow',
						'selector' => '{{WRAPPER}} .kfw-elementor-image-accordion-button:hover'
					]);
					$this->add_group_control( KFGCTST::get_type(), [
						'name'     => 'btn_hover_text_stroke',
						'selector' => '{{WRAPPER}} .kfw-elementor-image-accordion-button:hover'
					]);
					$this->add_group_control( KFGCBR::get_type(), [
						'name'     => 'btn_hover_border',
						'selector' => '{{WRAPPER}} .kfw-elementor-image-accordion-button:hover'
					]);
					$this->add_responsive_control( 'btn_hover_border_radius', [
						'label'      => esc_html__('Border Radius', 'onnat-extra' ),
						'type'       => KFCM::DIMENSIONS,
						'size_units' => [ 'px', '%' ],
						'selectors'  => [
							'{{WRAPPER}} .kfw-elementor-image-accordion-button:hover' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
						]
					]);
					$this->add_group_control( KFGCBS::get_type(), [
						'name'     => 'btn_hover_box_shadow',
						'selector' => '{{WRAPPER}} .kfw-elementor-image-accordion-button:hover'
					]);
				$this->end_controls_tab();
			$this->end_controls_tabs();
		$this->end_controls_section();
	}

	/**
	 * Generate setting json
	 */
	protected function _generate_setting_json( array $settings ) {
		extract( $settings );

		$options = [
			'id'             => 'kfw-elementor-widget-image-accordion-'.esc_attr( $this->get_id() ),
			'trigger'        => $trigger,
			'active_item'    => ( $active_item === 'yes' ) ? true : false,
			'active_item_no' => $active_item_number,
			'total_items'    => count( $items ),
			'active_size'    => $active_size
		];

		if( $use_anim === 'yes' ) {
			$options['anim_duration'] = $anim_duration['size'];
		}

		return wp_json_encode( $options );
	}

	/**
	 * Get attachment image src.
	 */
	public static function _get_image_src( $attachment_id, $image_size_key, array $settings ) {
        return KFGCIS::get_attachment_image_src( $attachment_id, $image_size_key, $settings );
	}

	public static function _get_image_html( $settings ) {
		return KFGCIS::get_attachment_image_html( $settings );
	}

	public function _get_image_title( $attachment ) {
		if ( empty( $attachment['id'] ) ) {
			return '';
		}

		return get_the_title( $attachment['id'] );
	}

	public function _get_image_alt( $instance ) {
		if ( empty( $instance['id'] ) ) {
			// For `Insert From URL` images.
			return isset( $instance['alt'] ) ? trim( strip_tags( $instance['alt'] ) ) : '';
		}

		$attachment_id = $instance['id'];
		if ( ! $attachment_id ) {
			return '';
		}

		$attachment = get_post( $attachment_id );
		if ( ! $attachment ) {
			return '';
		}

		$alt = get_post_meta( $attachment_id, '_wp_attachment_image_alt', true );
		if ( ! $alt ) {
			$alt = $attachment->post_excerpt;
			if ( ! $alt ) {
				$alt = $attachment->post_title;
			}
		}
		return trim( strip_tags( $alt ) );
	}

	public function _formatted_number( $index ) {
		return str_pad($index, 2, "0", STR_PAD_LEFT);
	}

	/**
	 * Render widget output on the frontend.
	 */
    protected function render() {
		$render = kf_extra_plugin_elementor()->get_template("kfw-image-accordion");
		include $render;
    }
}