<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

use \Elementor\Widget_Base as KFWB;
use \Elementor\Controls_Manager as KFCM;
use \Elementor\Icons_Manager as KFIM;

use \Elementor\Repeater as KFRTR;
use \Elementor\Group_Control_Typography as KFGCT;
use \Elementor\Group_Control_Border as KFGCBR;
use \Elementor\Group_Control_Background as KFGCBG;
use \Elementor\Group_Control_Box_Shadow as KFGCBS;

class KFW_Info_Box extends KFWB {

	/**
	 * Get element name.
	 *
	 * Retrieve the element name.
	 *
	 * @return string The name.
	 */
	public function get_name() {
        return 'kfw-info-box';
    }

	/**
	 * Get element title.
	 *
	 * Retrieve the element title.
	 *
	 * @return string Element title.
	 */
	public function get_title() {
		return esc_html__('Info Box', 'onnat-extra');
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the widget categories.
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ '_kfw_core_widgets' ];
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve the widget icon.
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'kfw-badge eicon-favorite';
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the widget keywords.
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'kfw', 'kin', 'info', 'box', 'fancy',];
	}

	/**
	 * Get widget help url.
	 *
	 * Retrieve the widget help url.
	 *
	 * @return string Widget help url.
	 */
	public function get_custom_help_url() {
        return KFE_CONST_THEME_DOC . 'elementor-custom-widgets/info-box';
    }

	/**
	 * Get HTML wrapper class.
     *
	 * Retrieve the widget container class.
     *
     * @return string Widget container css classes.
	 */
	public function get_html_wrapper_class() {
        $name = $this->get_name();

		return sprintf( 'kfw-ele-%s elementor-widget-%s', $name, $name );
	}

	/**
	 * Widget base constructor.
	 *
	 * Initializing the widget base class.
	 */
	public function __construct( $data = [], $args = null ) {
        parent::__construct( $data, $args );

		wp_register_style( 'kfw-info-box',
			KFE_CONST_URL . 'libraries/elementor/assets/css/info-box' . KFE_CONST_DEBUG_SUFFIX . '.css',
			[],
			KFE_CONST_VERSION
		);
    }

	/**
	 * Get style dependencies.
	 *
	 * Retrieve the list of style dependencies the element requires.
	 *
	 * @return array Element styles dependencies.
	 */
	public function get_style_depends() {
		return [ 'kfw-info-box' ];
	}

	/**
	 * Get script dependencies.
	 *
	 * Retrieve the list of script dependencies the element requires.
	 *
	 * @return array Element scripts dependencies.
	 */
	public function get_script_depends() {
		return [];
	}

	/**
	 * Register controls.
	 *
	 * Used to add new controls to any element type.
	 */
	protected function register_controls() {
		$this->_register_titles_section();
		$this->_register_features_section();
        $this->_register_button_section();

		$this->_register_box_style_section();
		$this->_register_title_style_section();
        $this->_register_features_style_section();
        $this->_register_button_style_section();
    }

	public function _register_titles_section() {
		$this->start_controls_section( 'section_title', [
			'label' => esc_html__( 'Title & Sub Title', 'onnat-extra' )
		]);
			$this->add_control( 'title', [
				'label'       => esc_html__( 'Title', 'onnat-extra' ),
				'type'        => KFCM::TEXT,
				'label_block' => true,
				'default'     => esc_html__( 'Title', 'onnat-extra' ),
			]);
			$this->add_control( 'show_sub_title', [
				'label'        => esc_html__('Show Sub Title','onnat-extra' ),
				'type'         => KFCM::SWITCHER,
				'label_on'     => esc_html__( 'On', 'onnat-extra' ),
				'label_off'    => esc_html__( 'Off', 'onnat-extra' ),
				'return_value' => 'yes',
				'default'      => 'no'
			]);
			$this->add_control( 'sub_title', [
				'label'       => esc_html__( 'Sub Title', 'onnat-extra' ),
				'type'        => KFCM::TEXT,
				'label_block' => true,
				'default'     => esc_html__( 'Sub Title', 'onnat-extra' ),
				'condition'   => [ 'show_sub_title' => 'yes' ]
			]);
			$this->add_control('sub_title_pos',[
                'label'     => esc_html__( 'Position?', 'onnat-extra' ),
				'type'      => KFCM::CHOOSE,
				'options'   => [
					'top'    => [
						'title' => esc_html__( 'Above Title', 'onnat-extra' ),
						'icon'  => 'eicon-v-align-top',
					],
					'bottom' => [
						'title' => esc_html__( 'Below Title', 'onnat-extra' ),
						'icon'  => 'eicon-v-align-bottom',
					],
				],
				'default'   => 'top',
				'toggle'    => false,
				'condition' => [
					'title!'         => '',
					'show_sub_title' => 'yes',
					'sub_title!'     => ''
				]
			]);

		$this->end_controls_section();
	}

	public function _register_features_section() {
		$repeater = new KFRTR();

		$this->start_controls_section( 'section_features', [
			'label' => esc_html__( 'Features', 'onnat-extra' )
		]);
            $repeater->add_control( 'label', [
                'label'   => esc_html__( 'Label', 'onnat-extra' ),
                'type'    => KFCM::TEXT,
                'default' => esc_html__( 'List Item', 'onnat-extra' ),
            ]);
			$repeater->add_control( 'icon', [
				'label'       => esc_html__( 'Icon', 'onnat-extra' ),
				'type'        => KFCM::ICONS,
				'skin'        => 'inline',
				'label_block' => false,
				'default'     => [
					'value'   => 'fa fa-check',
					'library' => 'fa-solid'
				]
			]);
            $repeater->add_control( 'customize', [
                'label'   => esc_html__( 'Customize It?', 'onnat-extra' ),
                'type'    => KFCM::SWITCHER,
                'default' => 'no',
            ]);
            $repeater->add_control('customize_after_hr',[
                'type'      => KFCM::DIVIDER,
                'condition' => [
                    'customize' => 'yes'
                ]
            ]);
            $repeater->add_control('icon_color',[
                'label'     => esc_html__('Color', 'onnat-extra' ),
                'type'      => KFCM::COLOR,
                'selectors' => [
                    '{{WRAPPER}} {{CURRENT_ITEM}} .kfw-ele-info-box-features-list-item-label i'   => 'color:{{VALUE}};',
                    '{{WRAPPER}} {{CURRENT_ITEM}} .kfw-ele-info-box-features-list-item-label svg' => 'fill:{{VALUE}};',
                ],
                'condition' => [
                    'customize' => 'yes'
                ]
            ]);
			$this->add_control( 'features', [
				'type'        => KFCM::REPEATER,
				'fields'      => $repeater->get_controls(),
				'default'     => [
					[
						'label'  => esc_html__( 'List Item #1', 'onnat-extra' ),
						'icon'   => [ 'value' => 'fa fa-check', 'library' => 'fa-brands' ],
					],
					[
						'label'  => esc_html__( 'List Item #2', 'onnat-extra' ),
						'icon'   => [ 'value' => 'fa fa-check', 'library' => 'fa-brands' ],
					],
					[
						'label'  => esc_html__( 'List Item #3', 'onnat-extra' ),
						'icon'   => [ 'value' => 'fa fa-check', 'library' => 'fa-brands' ],
					],
					[
						'label'  => esc_html__( 'List Item #4', 'onnat-extra' ),
						'icon'   => [ 'value' => 'fa fa-check', 'library' => 'fa-brands' ],
					],
					[
						'label'  => esc_html__( 'List Item #5', 'onnat-extra' ),
						'icon'   => [ 'value' => 'fa fa-check', 'library' => 'fa-brands' ],
					],
				],
				'title_field' => '{{{ elementor.helpers.renderIcon( this, icon, {}, "i", "panel" ) || \'<i class="{{ icon }}" aria-hidden="true"></i>\' }}} {{{ label }}}',
			]);
		$this->end_controls_section();
    }

    public function _register_button_section() {
		$this->start_controls_section( 'section_button', [
			'label' => esc_html__( 'Button', 'onnat-extra' )
		]);
			$this->add_control('use_button',[
				'label'        => esc_html__( 'Use Button', 'onnat-extra' ),
				'type'         => KFCM::SWITCHER,
				'return_value' => 'yes',
				'default'      => 'yes'
			]);
			$this->add_control( 'btn_text', [
				'label'     => esc_html__( 'Text', 'onnat-extra' ),
				'type'      => KFCM::TEXT,
				'default'   => esc_html__( 'Button', 'onnat-extra' ),
				'condition' => [
					'use_button' => 'yes'
				]
			]);
			$this->add_control( 'btn_link', [
				'label'       => esc_html__( 'Link', 'onnat-extra' ),
				'type'        => KFCM::URL,
				'placeholder' => 'https://your-link.com',
				'default'     => [ 'url' => '#' ],
				'conditions'  => [
					'relation' => 'and',
					'terms'    => [
						[
							'name'     => 'use_button',
							'operator' => '===',
							'value'    => 'yes'
						],
						[
							'relation' => 'or',
							'terms'    => [
								[
									'name'     => 'btn_text',
									'operator' => '!=',
									'value'    => ''
								],
								[
									'name'     => 'btn_icon[value]',
									'operator' => '!=',
									'value'    => ''
								],
							],
						]
					]
				],
			]);
			$this->add_control( 'btn_icon', [
				'label'       => esc_html__( 'Icon', 'onnat-extra' ),
				'type'        => KFCM::ICONS,
				'skin'        => 'inline',
				'label_block' => false,
				'condition'   => [
					'use_button' => 'yes'
				],
				'default'     => [
					'value'   => 'fa fa-check',
					'library' => 'fa-solid'
				]
			]);
		$this->end_controls_section();
    }

    public function _register_box_style_section() {
		$this->start_controls_section( 'section_box_style', [
			'label' => esc_html__( 'Box', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
            $this->add_group_control( KFGCBG::get_type(), [
                'name'     => 'box_background',
                'label'    => esc_html__('Background', 'onnat-extra' ),
                'types'    => [ 'classic', 'gradient' ],
                'exclude'  => [ 'image' ], // to remove background image option in classic mode
                'selector' => '{{WRAPPER}} .kfw-ele-info-box-main'
            ]);
            $this->add_responsive_control( 'box_padding', [
                'label'      => esc_html__( 'Padding', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-ele-info-box-main' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ],
            ]);
            $this->add_responsive_control( 'box_margin', [
                'label'      => esc_html__( 'Margin', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-ele-info-box-main' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ],
            ]);
            $this->add_group_control( KFGCBR::get_type(), [
                'name'           => 'box_border',
                'fields_options' => [
                    'width' => [ 'label' => _x( 'Border Width', 'Border Control', 'onnat-extra' ) ],
                    'color' => [ 'label' => _x( 'Border Color', 'Border Control', 'onnat-extra' ) ],
                ],
                'selector'       => '{{WRAPPER}} .kfw-ele-info-box-main'
            ]);
            $this->add_responsive_control( 'box_border_radius', [
                'label'      => esc_html__('Border Radius', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-ele-info-box-main' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ]
            ]);
            $this->add_group_control( KFGCBS::get_type(), [
                'name'     => 'box_box_shadow',
                'selector' => '{{WRAPPER}} .kfw-ele-info-box-main'
            ]);
            $this->add_control( 'add_box_animation_hover', [
                'label'        => esc_html__('Hover Effect','onnat-extra' ),
                'type'         => KFCM::SWITCHER,
                'label_on'     => esc_html__( 'On', 'onnat-extra' ),
                'label_off'    => esc_html__( 'Off', 'onnat-extra' ),
                'return_value' => 'yes',
                'default'      => 'no'
            ]);
            $this->add_control( 'box_animation_hover', [
                'label'     => esc_html__( 'Animation', 'onnat-extra' ),
                'type'      => KFCM::HOVER_ANIMATION,
                'default'   => 'grow',
                'condition' => [ 'add_table_animation_hover' => 'yes', ],
            ]);
		$this->end_controls_section();
    }

    public function _register_title_style_section() {
		$conditions = [
			'relation' => 'or',
			'terms'    => [
				[
					'name'     => 'title',
					'operator' => '!=',
					'value'    => '',
				],
                [
					'relation' => 'and',
					'terms'    => [
                        [
                            'name'     => 'show_sub_title',
                            'operator' => '==',
                            'value'    => 'yes',
                        ],
                        [
                            'name'     => 'sub_title',
                            'operator' => '!=',
                            'value'    => '',
                        ],
                    ]
                ]
			]
		];

        $stitle_condition = [
            'relation' => 'and',
            'terms'    => [
                [
                    'name'     => 'show_sub_title',
                    'operator' => '===',
                    'value'    => 'yes',
                ],
                [
                    'name'     => 'sub_title',
                    'operator' => '!=',
                    'value'    => '',
                ],
            ]
        ];

		$this->start_controls_section( 'section_title_style', [
            'label'      => esc_html__( 'Title & Sub Title', 'onnat-extra' ),
            'tab'        => KFCM::TAB_STYLE,
            'conditions' => $conditions
		]);
            $this->add_responsive_control('align',[
                'label'        => esc_html__( 'Alignment', 'onnat-extra' ),
                'type'         => KFCM::CHOOSE,
                'options'      => [
                    'left'    => [
                        'title' => esc_html__( 'Left', 'onnat-extra' ),
                        'icon'  => 'eicon-text-align-left',
                    ],
                    'center'  => [
                        'title' => esc_html__( 'Center', 'onnat-extra' ),
                        'icon'  => 'eicon-text-align-center',
                    ],
                    'right'   => [
                        'title' => esc_html__( 'Right', 'onnat-extra' ),
                        'icon'  => 'eicon-text-align-right',
                    ],
                ],
                'toggle'       => false,
                'default'      => 'center',
                'prefix_class' => 'kfw-ele-info-box-heading%s-align-',
            ]);
            $this->add_control('title_heading_before_hr', [
                'type' => KFCM::DIVIDER,
                'condition' => [ 'title!' => '', ]
            ]);
            $this->add_control( 'title_heading', [
                'label'     => esc_html__( 'Title', 'onnat-extra' ),
                'type'      => KFCM::HEADING,
                'condition' => [ 'title!' => '', ]
            ]);
			$this->add_control( 'title_color', [
				'label'     => esc_html__('Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'selectors' => [
					'{{WRAPPER}} .kfw-ele-info-box-main .kfw-ele-info-box-title' => 'color: {{VALUE}};'
				]
			]);
			$this->add_group_control( KFGCT::get_type(), [
				'name'        => 'title_typo',
				'description' => esc_html__( 'Typography', 'onnat-extra' ),
				'selector'    => '{{WRAPPER}} .kfw-ele-info-box-main .kfw-ele-info-box-title'
			]);
			$this->add_responsive_control( 'title_margin', [
				'label'      => esc_html__( 'Margin', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-info-box-main .kfw-ele-info-box-title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				],
			]);
            $this->add_control('sub_title_heading_before_hr', [
				'type'       => KFCM::DIVIDER,
				'conditions' => $stitle_condition
            ]);
			$this->add_control( 'sub_title_heading', [
				'label'      => esc_html__( 'Sub Title', 'onnat-extra' ),
				'type'       => KFCM::HEADING,
				'conditions' => $stitle_condition
			]);
			$this->add_control( 'sub_title_color', [
                'label'      => esc_html__('Color', 'onnat-extra' ),
                'type'       => KFCM::COLOR,
                'selectors'  => [
					'{{WRAPPER}} .kfw-ele-info-box-main .kfw-ele-info-box-sub-title' => 'color: {{VALUE}};'
				],
                'conditions' => $stitle_condition,
			]);
			$this->add_group_control( KFGCT::get_type(), [
                'name'        => 'sub_title_typo',
                'description' => esc_html__( 'Typography', 'onnat-extra' ),
                'selector'    => '{{WRAPPER}} .kfw-ele-info-box-main .kfw-ele-info-box-sub-title',
                'conditions'  => $stitle_condition,
			]);
			$this->add_responsive_control( 'sub_title_margin', [
				'label'      => esc_html__( 'Margin', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-info-box-main .kfw-ele-info-box-sub-title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				],
                'conditions' => $stitle_condition,
			]);
		$this->end_controls_section();
    }

    public function _register_features_style_section() {
		$this->start_controls_section( 'section_features_style', [
			'label' => esc_html__( 'Features', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
			$this->add_responsive_control('item_alignment',[
				'label'        => esc_html__( 'Alignment', 'onnat-extra' ),
				'type'         => KFCM::CHOOSE,
				'options'      => [
					'left'    => [
						'title' => esc_html__( 'Left', 'onnat-extra' ),
						'icon'  => 'eicon-text-align-left',
					],
					'center'  => [
						'title' => esc_html__( 'Center', 'onnat-extra' ),
						'icon'  => 'eicon-text-align-center',
					],
					'right'   => [
						'title' => esc_html__( 'Right', 'onnat-extra' ),
						'icon'  => 'eicon-text-align-right',
					],
				],
				'toggle'       => false,
				'default'      => 'center',
				'prefix_class' => 'kfw-ele-info-box-features-list%s-align-',
			]);
            $this->add_group_control( KFGCBG::get_type(), [
                'name'     => 'features_background',
                'label'    => esc_html__('Background', 'onnat-extra' ),
                'types'    => [ 'classic', 'gradient' ],
                'exclude'  => [ 'image' ],
                'selector' => '{{WRAPPER}} .kfw-ele-info-box-main .kfw-ele-info-box-features-list'
            ]);
			$this->add_responsive_control( 'features_padding', [
				'label'      => esc_html__( 'Padding', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-info-box-main .kfw-ele-info-box-features-list' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				],
			]);
            $this->add_control('icon_heading_before_hr', [
                'type' => KFCM::DIVIDER,
            ]);
			$this->add_control( 'icon_heading', [
				'label' => esc_html__( 'Icon', 'onnat-extra' ),
				'type'  => KFCM::HEADING,
			]);
			$this->add_responsive_control( 'item_icon_size', [
				'label'     => esc_html__( 'Size', 'onnat-extra' ),
				'type'      => KFCM::SLIDER,
				'range'     => [
					'px' => [ 'min' => 10, 'max' => 100 ],
				],
				'selectors' => [
					'{{WRAPPER}} .kfw-ele-info-box-main ul.kfw-ele-info-box-features-list .kfw-elementor-icon-wrapper' => 'font-size:{{SIZE}}{{UNIT}};',
				]
			]);
            $this->add_control( 'item_icon_color', [
				'label'     => esc_html__( 'Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'default'   => '',
				'selectors' => [
					'{{WRAPPER}} .kfw-ele-info-box-features-list-item-label i'   => 'color: {{VALUE}};',
					'{{WRAPPER}} .kfw-ele-info-box-features-list-item-label svg' => 'fill: {{VALUE}};',
                ]
			]);
            $this->add_control('text_heading_before_hr', [
                'type' => KFCM::DIVIDER,
            ]);
			$this->add_control( 'text_heading', [
				'label' => esc_html__( 'Text', 'onnat-extra' ),
				'type'  => KFCM::HEADING,
			]);
			$this->add_group_control( KFGCT::get_type(), [
				'name'        => 'item_text_typo',
				'description' => esc_html__( 'Typography', 'onnat-extra' ),
				'selector'    => '{{WRAPPER}} .kfw-ele-info-box-main .kfw-ele-info-box-features-list-item'
			]);
            $this->add_control( 'item_text_color_settings', [
                'label'        => esc_html__('Text Color', 'onnat-extra'),
                'type'         => KFCM::POPOVER_TOGGLE,
                'return_value' => 'yes',
            ]);
            $this->start_popover();
                $this->add_control( 'item_text_active_color', [
                    'label'     => esc_html__('Active', 'onnat-extra' ),
                    'type'      => KFCM::COLOR,
                    'selectors' => ['{{WRAPPER}} .kfw-ele-info-box-main .kfw-ele-info-box-features-list-item.kfw-ele-info-box-features-list-active-item' => 'color: {{VALUE}};' ],
                ]);
                $this->add_control( 'item_text_inactive_color', [
                    'label'     => esc_html__('Inactive', 'onnat-extra' ),
                    'type'      => KFCM::COLOR,
                    'selectors' => ['{{WRAPPER}} .kfw-ele-info-box-main .kfw-ele-info-box-features-list-item.kfw-ele-info-box-features-list-inactive-item' => 'color: {{VALUE}};' ],
                ]);
            $this->end_popover();
		$this->end_controls_section();
    }

    public function _register_button_style_section() {
		$style_condition = [
			'relation' => 'and',
			'terms'    => [
				[
					'name'     => 'use_button',
					'operator' => '===',
					'value'    => 'yes'
				],
				[
					'relation' => 'or',
					'terms'    => [
						[
							'name'     => 'btn_text',
							'operator' => '!=',
							'value'    => ''
						],
						[
							'name'     => 'btn_icon[value]',
							'operator' => '!=',
							'value'    => ''
						],
					],
				]
			]
		];

		$btn_txt_condition = [
			'relation' => 'and',
			'terms'    => [
				[
					'name'     => 'use_button',
					'operator' => '===',
					'value'    => 'yes'
				],
				[
					'name'     => 'btn_text',
					'operator' => '!=',
					'value'    => ''
				],
			]
		];

		$this->start_controls_section( 'section_button_style', [
			'label'      => esc_html__( 'Button', 'onnat-extra' ),
			'tab'        => KFCM::TAB_STYLE,
			'conditions' => $style_condition,
		]);
			$this->add_responsive_control('btn_alignment',[
				'label'        => esc_html__( 'Alignment', 'onnat-extra' ),
				'type'         => KFCM::CHOOSE,
				'options'      => [
					'left'    => [
						'title' => esc_html__( 'Left', 'onnat-extra' ),
						'icon'  => 'eicon-text-align-left',
					],
					'center'  => [
						'title' => esc_html__( 'Center', 'onnat-extra' ),
						'icon'  => 'eicon-text-align-center',
					],
					'right'   => [
						'title' => esc_html__( 'Right', 'onnat-extra' ),
						'icon'  => 'eicon-text-align-right',
					],
				],
				'toggle'       => false,
				'default'      => 'center',
				'prefix_class' => 'kfw-ele-info-box-button%s-align-',
				'conditions'   => $style_condition,
			]);
			$this->add_responsive_control('btn_padding', [
				'label'      => esc_html__( 'Padding', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-info-box-button' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				],
				'conditions' => $btn_txt_condition
			]);
			$this->add_responsive_control('btn_margin', [
				'label'      => esc_html__( 'Margin', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'separator'  => 'after',
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-info-box-button' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				],
				'conditions' => $btn_txt_condition
			]);

			/**
			 * Button Text Style
			 */
				$this->add_control('btn_txt_style_before_hr', [
					'type'       => KFCM::DIVIDER,
					'conditions' => $btn_txt_condition,
				]);
				$this->add_control('btn_txt_style_heading',[
					'label'      => esc_html__( 'BUTTON TEXT', 'onnat-extra' ),
					'type'       => KFCM::HEADING,
					'conditions' => $btn_txt_condition,
				]);
				$this->add_group_control( KFGCT::get_type(), [
					'name'        => 'btn_typo',
					'description' => esc_html__( 'Typography', 'onnat-extra' ),
					'selector'    => '{{WRAPPER}} .kfw-ele-info-box-button-text',
					'conditions'  => $btn_txt_condition,
				]);
				$this->add_control( 'btn_text_color_settings', [
					'label'        => esc_html__('Text Color', 'onnat-extra'),
					'type'         => KFCM::POPOVER_TOGGLE,
					'return_value' => 'yes',
					'conditions'   => $btn_txt_condition,
				]);
				$this->start_popover();
					$this->add_control( 'btn_text_color', [
						'label'      => esc_html__('Normal', 'onnat-extra' ),
						'type'       => KFCM::COLOR,
						'selectors'  => ['{{WRAPPER}} a .kfw-ele-info-box-button-text' => 'color: {{VALUE}};' ],
						'conditions' => $btn_txt_condition,
					]);
					$this->add_control( 'btn_text_hover_color', [
						'label'      => esc_html__('Hover', 'onnat-extra' ),
						'type'       => KFCM::COLOR,
						'selectors'  => ['{{WRAPPER}} a:hover .kfw-ele-info-box-button-text' => 'color: {{VALUE}};' ],
						'conditions' => $btn_txt_condition,
					]);
				$this->end_popover();

			/**
			 * Button Icon Style
			 */
				$btn_icon_condition = [
					'relation' => 'and',
					'terms'    => [
						[
							'name'     => 'use_button',
							'operator' => '===',
							'value'    => 'yes'
						],
						[
							'name'     => 'btn_icon[value]',
							'operator' => '!=',
							'value'    => ''
						],
					]
				];
				$this->add_control('btn_icon_style_before_hr', [
					'type'       => KFCM::DIVIDER,
					'conditions' => $btn_icon_condition,
				]);
				$this->add_control('btn_icon_style_heading',[
					'label'      => esc_html__( 'BUTTON ICON', 'onnat-extra' ),
					'type'       => KFCM::HEADING,
					'conditions' => $btn_icon_condition,
				]);
				$this->add_responsive_control( 'btn_icon_margin', [
					'label'      => esc_html__( 'Margin', 'onnat-extra' ),
					'type'       => KFCM::DIMENSIONS,
					'size_units' => [ 'px', 'em', '%' ],
					'separator'  => 'after',
					'selectors'  => [
						'{{WRAPPER}} .kfw-ele-widget-info-box-button .kfw-elementor-icon-wrapper' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
					],
					'conditions' => $btn_icon_condition
				]);
				$this->add_control( 'btn_icon_pos', [
					'label'      => esc_html__( 'Position', 'onnat-extra' ),
					'type'       => KFCM::CHOOSE,
					'options'    => [
						'left'           => [
							'title' => esc_html__('Left', 'onnat-extra'),
							'icon'  => 'eicon-text-align-left',
						],
						'right'          => [
							'title' => esc_html__('Right', 'onnat-extra'),
							'icon'  => 'eicon-text-align-right',
						],
					],
					'default'    => 'left',
					'toggle'     => false,
					'conditions' => [
						'relation' => 'and',
						'terms'    => [
							[
								'name'     => 'btn_text',
								'operator' => '!=',
								'value'    => ''
							],
							[
								'name'     => 'use_button',
								'operator' => '===',
								'value'    => 'yes'
							],
							[
								'name'     => 'btn_icon[value]',
								'operator' => '!=',
								'value'    => ''
							],
						]
					]
				]);
				$this->add_responsive_control( 'btn_icon_size', [
					'label'      => esc_html__('Icon Size', 'onnat-extra' ),
					'type'       => KFCM::SLIDER,
					'size_units' => [ 'px' ],
					'range'      => [
						'px' => [
							'min' => 0,
							'max' => 100,
						],
					],
					'selectors'  => [
						'{{WRAPPER}} .kfw-ele-widget-info-box-button .kfw-elementor-icon-wrapper i' => 'font-size: {{SIZE}}{{UNIT}};'
					],
					'conditions' => $btn_icon_condition
				]);
				$this->add_responsive_control( 'btn_icon_wrapper_size', [
					'label'      => esc_html__('Size', 'onnat-extra' ),
					'type'       => KFCM::SLIDER,
					'size_units' => [ 'px' ],
					'range'      => [
						'px' => [
							'min' => 0,
							'max' => 500,
						],
					],
					'selectors'  => [
						'{{WRAPPER}} .kfw-ele-widget-info-box-button .kfw-elementor-icon-wrapper' => 'line-height:{{SIZE}}{{UNIT}};height:{{SIZE}}{{UNIT}};width:{{SIZE}}{{UNIT}};',
					],
					'conditions' => $btn_icon_condition
				]);
				$this->start_controls_tabs( 'btn_icon_customize_tabs' );
					$this->start_controls_tab( 'btn_icon_normal_state', [
						'label'      => esc_html__( 'Normal', 'onnat-extra' ),
						'conditions' => $btn_icon_condition
					]);
						$this->add_control( 'btn_icon_color', [
							'label'      => esc_html__('Icon Color', 'onnat-extra' ),
							'type'       => KFCM::COLOR,
							'selectors'  => [
								'{{WRAPPER}} .kfw-ele-widget-info-box-button .kfw-elementor-icon-wrapper i'   => 'color: {{VALUE}};',
								'{{WRAPPER}} .kfw-ele-widget-info-box-button .kfw-elementor-icon-wrapper svg' => 'fill: {{VALUE}};'
							],
							'conditions' => $btn_icon_condition
						]);
						$this->add_group_control( KFGCBG::get_type(), [
							'name'       => 'btn_icon_bg',
							'label'      => esc_html__('Background', 'onnat-extra' ),
							'types'      => [ 'classic', 'gradient' ],
							'exclude'    => [ 'image' ],
							'selector'   => '{{WRAPPER}} .kfw-ele-widget-info-box-button .kfw-elementor-icon-wrapper',
							'conditions' => $btn_icon_condition
						]);
						$this->add_group_control( KFGCBR::get_type(), [
							'name'           => 'btn_icon_border',
							'fields_options' => [
								'width' => [ 'label' => _x( 'Border Width', 'Border Control', 'onnat-extra' ) ],
								'color' => [ 'label' => _x( 'Border Color', 'Border Control', 'onnat-extra' ) ],
							],
							'selector'       => '{{WRAPPER}} .kfw-ele-widget-info-box-button .kfw-elementor-icon-wrapper',
							'conditions'     => $btn_icon_condition,
						]);
						$this->add_responsive_control( 'btn_icon_border_radius', [
							'label'      => esc_html__('Border Radius', 'onnat-extra' ),
							'type'       => KFCM::DIMENSIONS,
							'size_units' => [ 'px', '%' ],
							'selectors'  => [ '{{WRAPPER}} .kfw-ele-widget-info-box-button .kfw-elementor-icon-wrapper' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};' ],
							'conditions' => $btn_icon_condition,
						]);
					$this->end_controls_tab();
					$this->start_controls_tab( 'btn_icon_hover_state', [
						'label'      => esc_html__( 'Hover', 'onnat-extra' ),
						'conditions' => $btn_icon_condition
					]);
						$this->add_control( 'btn_icon_hover_color', [
							'label'      => esc_html__('Icon Color', 'onnat-extra' ),
							'type'       => KFCM::COLOR,
							'selectors'  => [
								'{{WRAPPER}} .kfw-ele-widget-info-box-button:hover .kfw-elementor-icon-wrapper i'   => 'color: {{VALUE}};',
								'{{WRAPPER}} .kfw-ele-widget-info-box-button:hover .kfw-elementor-icon-wrapper svg' => 'fill: {{VALUE}};'
							],
							'conditions' => $btn_icon_condition
						]);
						$this->add_group_control( KFGCBG::get_type(), [
							'name'       => 'btn_icon_hover_bg',
							'label'      => esc_html__('Background', 'onnat-extra' ),
							'types'      => [ 'classic', 'gradient' ],
							'exclude'    => [ 'image' ],
							'selector'   => '{{WRAPPER}} .kfw-ele-widget-info-box-button:hover .kfw-elementor-icon-wrapper',
							'conditions' => $btn_icon_condition
						]);
						$this->add_group_control( KFGCBR::get_type(), [
							'name'           => 'btn_icon_hover_border',
							'fields_options' => [
								'width' => [ 'label' => _x( 'Border Width', 'Border Control', 'onnat-extra' ) ],
								'color' => [ 'label' => _x( 'Border Color', 'Border Control', 'onnat-extra' ) ],
							],
							'selector'       => '{{WRAPPER}} .kfw-ele-widget-info-box-button:hover .kfw-elementor-icon-wrapper',
							'conditions'     => $btn_icon_condition,
						]);
						$this->add_responsive_control( 'btn_icon_hover_border_radius', [
							'label'      => esc_html__('Border Radius', 'onnat-extra' ),
							'type'       => KFCM::DIMENSIONS,
							'size_units' => [ 'px', '%' ],
							'selectors'  => [ '{{WRAPPER}} .kfw-ele-widget-info-box-button:hover .kfw-elementor-icon-wrapper' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};' ],
							'conditions' => $btn_icon_condition,
						]);
					$this->end_controls_tab();
				$this->end_controls_tabs();
		$this->end_controls_section();
    }

    public function _render_icons( $icon , $echo = true ) {
		$icon = array_filter( $icon );

		if( !empty( $icon ) ) {
			echo '<span class="kfw-elementor-icon-wrapper">';
				KFIM::render_icon( $icon, [ 'aria-hidden' => 'true' ] );
			echo '</span>';
		}
	}

	public function _titles( $settings ) {
		$title         = '';
		$sub_title     = '';
		$sub_title_pos = $settings['sub_title_pos'];

		if( !empty( $settings['title'] ) ) {
			$title = sprintf( '<span class="kfw-ele-info-box-title">%s</span>', esc_html( $settings['title'] ) );
		}

		if( $settings['show_sub_title'] == 'yes' && !empty( $settings['sub_title'] ) ) {
			$sub_title = sprintf( '<span class="kfw-ele-info-box-sub-title">%s</span>', esc_html( $settings['sub_title'] ) );
		}

		if( !empty( $title ) || !empty( $sub_title ) ) {
			echo '<div class="kfw-ele-info-box-header">';

				if( 'top' === $sub_title_pos ) {
					echo $sub_title;
				}

				echo $title;

				if( 'bottom' === $sub_title_pos ) {
					echo $sub_title;
				}

			echo '</div>';
		}
	}

	public function _button( $settings ) {
		if( $settings['use_button'] === 'yes' ) {
			$link = array_filter( $settings['btn_link'] );

			$this->add_render_attribute( 'cta-wrapper', [
				'class' => array_filter([
					'kfw-ele-info-box-button-wrap',
					!empty( trim( $settings['btn_text'] ) ) ? 'kfw-ele-info-box-button-has-txt' : ''
				])
			]);

			echo '<div '.$this->get_render_attribute_string( 'cta-wrapper' ).'>';

				$this->add_render_attribute( 'cta', [
					'class' => array_filter([
						'kfw-ele-widget-info-box-button',
						!empty( trim( $settings['btn_text'] ) ) ? 'kfw-ele-widget-icon-box-button-icon-position-'.$settings['btn_icon_pos'] : '',
					])
				]);

				if ( ! empty( $link['url'] ) ) {
					$this->add_render_attribute( 'cta', 'href', esc_url( $link['url'] ) );

					if ( ! empty( $link['is_external'] ) ) {
						$this->add_render_attribute( 'cta', 'target', '_blank' );
					}

					if ( ! empty( $link['nofollow'] ) ) {
						$this->add_render_attribute( 'cta', 'rel', 'nofollow' );
					}
				}

				echo '<a '.$this->get_render_attribute_string( 'cta' ).'>';
					if( $settings['btn_icon_pos'] === 'left' ) {
						$this->_render_icons( $settings['btn_icon'] );
					}

					if( !empty( trim( $settings['btn_text'] ) ) ) {
						printf( '<span class="kfw-ele-info-box-button-text">%s</span>', esc_html( $settings['btn_text'] ) );
					}

					if( $settings['btn_icon_pos'] === 'right' ) {
						$this->_render_icons( $settings['btn_icon'] );
					}
				echo '</a>';

			echo '</div>';

		}
	}

	/**
	 * Render widget output on the frontend.
	 */
    protected function render() {
		$render = kf_extra_plugin_elementor()->get_template("kfw-info-box");
		include $render;
    }
}