<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

use \Elementor\Widget_Base as WB;
use \Elementor\Controls_Manager as KFCM;
use \Elementor\Icons_Manager as KFIM;

use \Elementor\Repeater as KFRTR;
use \Elementor\Group_Control_Background as KFGCBG;
use \Elementor\Group_Control_Border as KFGCBR;
use \Elementor\Group_Control_Typography as KFGCT;
use \Elementor\Group_Control_Text_Shadow as KFGCTS;
use \Elementor\Group_Control_Text_Stroke as KFGCTST;

class KFW_Marquee_Text extends WB {

	/**
	 * Get element name.
	 *
	 * Retrieve the element name.
	 *
	 * @return string The name.
	 */
	public function get_name() {
        return 'kfw-marquee-text';
    }

	/**
	 * Get element title.
	 *
	 * Retrieve the element title.
	 *
	 * @return string Element title.
	 */
	public function get_title() {
		return esc_html__('Marquee Text', 'onnat-extra');
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the widget categories.
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ '_kfw_core_widgets' ];
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve the widget icon.
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'kfw-badge eicon-user-preferences kfw-marquee-txt';
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the widget keywords.
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'kfw', 'kin', 'marquee', 'text', 'animate' ];
	}

	/**
	 * Get widget help url.
	 *
	 * Retrieve the widget help url.
	 *
	 * @return string Widget help url.
	 */
	public function get_custom_help_url() {
        return KFE_CONST_THEME_DOC . 'elementor-custom-widgets/marquee-text';
    }

	/**
	 * Get HTML wrapper class.
     *
	 * Retrieve the widget container class.
     *
     * @return string Widget container css classes.
	 */
	public function get_html_wrapper_class() {
        $name = $this->get_name();

		return sprintf( 'kfw-ele-%s elementor-widget-%s', $name, $name );
	}

	/**
	 * Widget base constructor.
	 *
	 * Initializing the widget base class.
	 */
	public function __construct( $data = [], $args = null ) {
        parent::__construct( $data, $args );

		wp_register_style( 'kfw-marquee-text',
			KFE_CONST_URL . 'libraries/elementor/assets/css/marquee-text' . KFE_CONST_DEBUG_SUFFIX . '.css',
			[],
			KFE_CONST_VERSION
		);
    }

	/**
	 * Get style dependencies.
	 *
	 * Retrieve the list of style dependencies the element requires.
	 *
	 * @return array Element styles dependencies.
	 */
	public function get_style_depends() {
		return [ 'kfw-marquee-text' ];
	}

	/**
	 * Get script dependencies.
	 *
	 * Retrieve the list of script dependencies the element requires.
	 *
	 * @return array Element scripts dependencies.
	 */
	public function get_script_depends() {
		return [];
	}

	/**
	 * Register controls.
	 *
	 * Used to add new controls to any element type.
	 */
	protected function register_controls() {

		$this->_register_items_section();
		$this->_register_settings_section();

		$this->_register_main_style_section();
		$this->_register_title_style_section();

		$this->_register_icon_type_icon_style_section();
		$this->_register_icon_type_text_style_section();
		$this->_register_icon_type_image_style_section();
    }

	public function _register_items_section() {
		$repeater = new KFRTR();

		$this->start_controls_section( 'section_items', [
			'label' => esc_html__( 'Items', 'onnat-extra' )
		]);
			$repeater->start_controls_tabs('tabs_marquee');
				$repeater->start_controls_tab('tab_marquee_data',[
					'label' => esc_html__( 'Data', 'onnat-extra' )
				]);
					$repeater->add_control( 'title', [
						'label'       => esc_html__( 'Title', 'onnat-extra' ),
						'type'        => KFCM::TEXT,
						'label_block' => true,
						'default'     => 'Contrary to popular belief random'
					]);
					$repeater->add_control( 'link', [
						'label'       => esc_html__( 'Link', 'onnat-extra' ),
						'type'        => KFCM::URL,
						'placeholder' => 'https://your-link.com',
						'default'     => [
							'url' => ''
						]
					]);
				$repeater->end_controls_tab();
				$repeater->start_controls_tab('tab_marquee_icon',[
					'label' => esc_html__( 'Icon', 'onnat-extra' )
				]);
					$repeater->add_control('icon_type',[
						'label'   => esc_html__( 'Icon Type', 'onnat-extra' ),
						'type'    => KFCM::CHOOSE,
						'default' => 'global',
						'toggle'  => false,
						'options' => [
							'global' => [
								'title' => esc_html__('Global', 'onnat-extra'),
								'icon'  => 'eicon-globe',
							],
							'blank'  => [
								'title' => esc_html__('Blank', 'onnat-extra'),
								'icon'  => '',
							],
							'icon'  => [
								'title' => esc_html__('Icon', 'onnat-extra'),
								'icon'  => 'eicon-plus-circle-o',
							],
							'image' => [
								'title' => esc_html__('Image', 'onnat-extra'),
								'icon'  => 'eicon-image',
							],
							'input' => [
								'title' => esc_html__('Input', 'onnat-extra'),
								'icon'  => 'eicon-text-field',
							],
						],
					]);
					$repeater->add_control( 'icon', [
						'label'     => esc_html__( 'Icon', 'onnat-extra' ),
						'type'      => KFCM::ICONS,
						'default'   => [
							'value'   => 'fas fa-heart',
							'library' => 'fa-solid',
						],
						'condition' => [
							'icon_type' => 'icon'
						],
					]);
					$repeater->add_control( 'image', [
						'label'     => esc_html__( 'Image', 'onnat-extra' ),
						'type'      => KFCM::MEDIA,
						'dynamic'   => [ 'active' => true ],
						'default'   => [
							'url' => \Elementor\Utils::get_placeholder_image_src()
						],
						'condition' => [
							'icon_type' => 'image'
						],
					]);
					$repeater->add_control( 'input', [
						'label'       => esc_html__( 'Custom Text', 'onnat-extra' ),
						'type'        => KFCM::TEXT,
						'label_block' => true,
						'default'     => '+',
						'condition'   => [
							'icon_type' => 'input'
						],
					]);
				$repeater->end_controls_tab();
			$repeater->end_controls_tabs();

            $this->add_control( 'items', [
                'type'    => KFCM::REPEATER,
                'fields'  => $repeater->get_controls(),
                'default' => [
					[ 'title' => 'Aenean ornare neque' ],
					[ 'title' => 'Mauris sit amet purus' ],
					[ 'title' => 'Praesent convallis augue ' ],
					[ 'title' => 'laoreet scelerisque' ],
					[ 'title' => 'Eget libero ante' ],
					[ 'title' => 'Accumsan sodales' ],
				]
            ]);
			$this->add_control('icon_type',[
				'label'     => esc_html__( 'Icon Type', 'onnat-extra' ),
				'type'      => KFCM::CHOOSE,
				'separator' => 'before',
				'options'   => [
					'blank'  => [
                        'title' => esc_html__('Blank', 'onnat-extra'),
                        'icon'  => '',
                    ],
                    'icon'  => [
                        'title' => esc_html__('Icon', 'onnat-extra'),
                        'icon'  => 'eicon-plus-circle-o',
                    ],
                    'image' => [
                        'title' => esc_html__('Image', 'onnat-extra'),
                        'icon'  => 'eicon-image',
                    ],
                    'input' => [
                        'title' => esc_html__('Input', 'onnat-extra'),
                        'icon'  => 'eicon-text-field',
                    ],
                ],
                'default' => 'input',
                'toggle'  => false
            ]);
            $this->add_control( 'icon', [
                'label'     => esc_html__( 'Icon', 'onnat-extra' ),
                'type'      => KFCM::ICONS,
                'default'   => [
                    'value'   => 'fas fa-heart',
                    'library' => 'fa-solid',
                ],
                'condition' => [
                    'icon_type' => 'icon'
                ],
            ]);
            $this->add_control( 'image', [
                'label'     => esc_html__( 'Image', 'onnat-extra' ),
                'type'      => KFCM::MEDIA,
                'dynamic'   => [ 'active' => true ],
                'default'   => [
                    'url' => \Elementor\Utils::get_placeholder_image_src()
                ],
                'condition' => [
                    'icon_type' => 'image'
                ],
            ]);
            $this->add_control( 'input', [
                'label'       => esc_html__( 'Custom Text', 'onnat-extra' ),
                'type'        => KFCM::TEXT,
                'label_block' => true,
                'default'     => '+',
                'condition'   => [
                    'icon_type' => 'input'
                ],
            ]);
			$this->add_control('link_to',[
				'label'     => esc_html__( 'Apply Link To', 'onnat-extra' ),
				'type'      => KFCM::CHOOSE,
				'separator' => 'before',
				'options'   => [
					'title'   => [
						'title' => esc_html__('Title Only', 'onnat-extra'),
						'icon'  => 'eicon-site-title',
					],
					'icon'    => [
						'title' => esc_html__('Icon Only', 'onnat-extra'),
						'icon'  => ' eicon-cursor-move',
					],
					'all'     => [
						'title' => esc_html__('All', 'onnat-extra'),
						'icon'  => ' eicon-font',
					],
				],
				'default'   => 'title',
				'toggle'    => false
			]);

		$this->end_controls_section();
    }

    public function _register_settings_section() {
		$this->start_controls_section( 'section_settings', [
			'label' => esc_html__( 'Settings', 'onnat-extra' )
		]);
			$this->add_control('marquee_type',[
				'label'   => esc_html__( 'Type', 'onnat-extra' ),
				'type'    => KFCM::CHOOSE,
				'toggle'  => false,
				'default' => 'type-1',
				'options' => [
					'type-1' => [
						'title' => esc_html__('Type 1', 'onnat-extra'),
						'icon'  => 'kfw-ctrl-icon kfw-ctrl-type-1-icon'
					],
					'type-2' => [
						'title' => esc_html__('Type 2', 'onnat-extra'),
						'icon'  => 'kfw-ctrl-icon kfw-ctrl-type-2-icon'
					],
				],
				'prefix_class' => 'kfw-marquee-type kfw-marquee-',
			]);
            $this->add_control('speed',[
				'label'       => esc_html__('Speed (s)', 'onnat-extra' ),
				'type'        => KFCM::SLIDER,
				'size_units'  => [],
				'range'       => [
					'' => [
						'min'  => 1,
						'max'  => 200,
						'step' => 1
					]
                ],
				'selectors' => [
					'{{WRAPPER}}' => '--kfw-ele-widget-marquee-speed:{{SIZE}}s;',
				],
				'default'     => [
                    'unit' => '',
                    'size' => 30
                ],
            ]);
            $this->add_control( 'pause', [
				'label'        => esc_html__( 'Pause on Hover?', 'onnat-extra' ),
				'type'         => KFCM::SWITCHER,
				'return_value' => 'yes',
				'default'      => 'yes',
				'prefix_class' => 'kfw-ele-widget-marquee-pause-',
            ]);
            $this->add_control('direction',[
				'label'        => esc_html__( 'Marquee Direction', 'onnat-extra' ),
				'type'         => KFCM::CHOOSE,
				'options'      => [
                    'left'  => [
                        'title' => esc_html__('Left Side', 'onnat-extra'),
                        'icon'  => 'eicon-chevron-left',
                    ],
                    'right' => [
                        'title' => esc_html__('Right Side', 'onnat-extra'),
                        'icon'  => 'eicon-chevron-right',
                    ],
                ],
				'prefix_class' => 'kfw-ele-widget-marquee-dir-',
				'default'      => 'left',
				'toggle'       => false
            ]);
		$this->end_controls_section();
    }

	public function _register_main_style_section() {
		$this->start_controls_section( 'section_main_style', [
			'label' => esc_html__( 'Main Wrap', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
			$this->add_group_control( KFGCBG::get_type(), [
				'name'     => 'main_bg',
				'label'    => esc_html__('Background', 'onnat-extra' ),
				'types'    => [ 'classic', 'gradient', 'image', ],
				'exclude'  => [ 'video' ],
				'selector' => '{{WRAPPER}} .kfw-ele-widget-marquee-txt-main',
			]);
            $this->add_responsive_control( 'main_margin', [
                'label'      => esc_html__('Margin', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-ele-widget-marquee-txt-main' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->add_responsive_control( 'main_padding', [
                'label'      => esc_html__('Padding', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-ele-widget-marquee-txt-main' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->add_control( 'rotation', [
				'label'        => esc_html__( 'Enable Rotation?', 'onnat-extra' ),
				'type'         => KFCM::SWITCHER,
				'return_value' => 'yes',
				'default'      => '',
				'prefix_class' => 'kfw-ele-widget-marquee-rotation-',
            ]);
			$this->add_control('main_rotate',[
				'label'    => esc_html__( 'Rotate', 'onnat-extra' ),
				'type'     => KFCM::SLIDER,
				'default'  => [
					'size' => -7,
					'unit' => 'deg',
				],
				'range'     => [
					'deg' => [
						'max' => 360,
						'min' => -360,
					],
				],
                'condition'   => [
                    'rotation!' => ''
                ],
				'selectors' => [
					'{{WRAPPER}}.kfw-ele-widget-marquee-rotation-yes .kfw-ele-widget-marquee-txt-main' => 'transform: rotate({{SIZE}}{{UNIT}});',
				],
			]);
		$this->end_controls_section();
	}

	public function _register_title_style_section() {
		$this->start_controls_section( 'section_title_style', [
			'label' => esc_html__( 'Marquee : Title', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
			$this->add_group_control(KFGCT::get_type(), [
				'name'      => 'title_typo',
				'label'     => esc_html__( 'Typography', 'onnat-extra' ),
				'selector'  => '{{WRAPPER}} .kfw-ele-marquee-list .kfw-ele-marquee-title',
			]);
			$this->add_responsive_control( 'title_margin', [
				'label'      => esc_html__('Margin', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-marquee-list .kfw-ele-marquee-title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
			$this->add_responsive_control( 'title_padding', [
				'label'      => esc_html__('Padding', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-marquee-list .kfw-ele-marquee-title' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
			$this->start_controls_tabs( 'title_customize_tabs' );
				$this->start_controls_tab( 'title_normal_state', [
					'label'	=> esc_html__( 'Normal', 'onnat-extra' ),
				]);
					$this->add_group_control( KFGCBG::get_type(), [
						'name'     => 'title_bg',
						'label'    => esc_html__('Background', 'onnat-extra' ),
						'types'    => [ 'classic', 'gradient', 'image', ],
						'exclude'  => [ 'video' ],
						'selector' => '{{WRAPPER}} .kfw-ele-marquee-list .kfw-ele-marquee-title',
					]);	
					$this->add_group_control( KFGCBR::get_type(), [
						'name'       => 'title_border',
						'selector' => '{{WRAPPER}} .kfw-ele-marquee-list .kfw-ele-marquee-title',
					]);
					$this->add_control( 'title_color', [
						'label'     => esc_html__('Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'selectors' => ['{{WRAPPER}} .kfw-ele-marquee-list .kfw-ele-marquee-title' => 'color: {{VALUE}};' ],
					]);
					$this->add_group_control( KFGCTS::get_type(), [
						'name'      => 'title_txt_shadow',
						'selector'  => '{{WRAPPER}} .kfw-ele-marquee-list .kfw-ele-marquee-title',
					]);
					$this->add_group_control( KFGCTST::get_type(), [
						'name'      => 'title_txt_stroke',
						'selector'  => '{{WRAPPER}} .kfw-ele-marquee-list .kfw-ele-marquee-title',
					]);
					$this->add_responsive_control( 'title_border_radius', [
						'label'      => esc_html__('Border Radius', 'onnat-extra' ),
						'type'       => KFCM::DIMENSIONS,
						'size_units' => [ 'px', '%' ],
						'selectors'  => [
							'{{WRAPPER}} .kfw-ele-marquee-list .kfw-ele-marquee-title' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
						]
					]);
				$this->end_controls_tab();
				$this->start_controls_tab( 'title_hover_state', [
					'label'	=> esc_html__( 'Hover', 'onnat-extra' ),
				]);
					$this->add_group_control( KFGCBG::get_type(), [
						'name'     => 'title_hover_bg',
						'label'    => esc_html__('Background', 'onnat-extra' ),
						'types'    => [ 'classic', 'gradient', 'image', ],
						'exclude'  => [ 'video' ],
						'selector' => '{{WRAPPER}} .kfw-ele-marquee-list .kfw-ele-marquee-title:hover',
					]);	
					$this->add_group_control( KFGCBR::get_type(), [
						'name'       => 'title_hover_border',
						'selector' => '{{WRAPPER}} .kfw-ele-marquee-list .kfw-ele-marquee-title:hover  ',
					]);				
					$this->add_control( 'title_hover_color', [
						'label'     => esc_html__('Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'selectors' => [
							'{{WRAPPER}}.kfw-marquee-type-1 .kfw-ele-marquee-list .kfw-ele-marquee-title:hover'  => 'color: {{VALUE}};',
							'{{WRAPPER}}.kfw-marquee-type-2 .kfw-ele-marquee-list .kfw-ele-marquee-title:before' => '-webkit-text-fill-color:{{VALUE}};'
						],
					]);
					$this->add_group_control( KFGCTS::get_type(), [
						'name'      => 'title_hover_txt_shadow',
						'selector'  => '{{WRAPPER}} .kfw-ele-marquee-list .kfw-ele-marquee-title:hover',
					]);
					$this->add_group_control( KFGCTST::get_type(), [
						'name'      => 'title_hover_txt_stroke',
						'selector'  => '{{WRAPPER}} .kfw-ele-marquee-list .kfw-ele-marquee-title:hover',
					]);
					$this->add_responsive_control( 'title_hover_border_radius', [
						'label'      => esc_html__('Border Radius', 'onnat-extra' ),
						'type'       => KFCM::DIMENSIONS,
						'size_units' => [ 'px', '%' ],
						'selectors'  => [
							'{{WRAPPER}} .kfw-ele-marquee-list .kfw-ele-marquee-title:hover' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
						]
					]);					
				$this->end_controls_tab();
			$this->end_controls_tabs();
		$this->end_controls_section();
	}

	public function _register_icon_type_icon_style_section() {
		$this->start_controls_section( 'section_icon_type_icon_style', [
			'label' => esc_html__( 'Marquee : Icon Type : Icon', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
			$this->add_responsive_control( 'icon_size', [
				'label'      => esc_html__('Icon Size', 'onnat-extra' ),
				'type'       => KFCM::SLIDER,
				'size_units' => [ 'px' ],
				'range'      => [
					'px' => [
						'min' => 0,
						'max' => 999,
					],
				],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-marquee-list .kfw-ele-marquee-icon-wrap .kfw-elementor-icon-wrap' => 'font-size:{{SIZE}}{{UNIT}};'
				],
			]);
			$this->add_responsive_control( 'icon_margin', [
				'label'      => esc_html__('Margin', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-marquee-list .kfw-ele-marquee-icon-wrap' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
			$this->add_responsive_control( 'icon_padding', [
				'label'      => esc_html__('Padding', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-marquee-list .kfw-ele-marquee-icon-wrap' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
			$this->start_controls_tabs( 'icon_customize_tabs' );
				$this->start_controls_tab( 'icon_normal_state', [
					'label'	=> esc_html__( 'Normal', 'onnat-extra' ),
				]);
					$this->add_control( 'icon_color', [
						'label'     => esc_html__('Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'selectors' => [
							'{{WRAPPER}} .kfw-ele-marquee-list .kfw-ele-marquee-icon-wrap .kfw-elementor-icon-wrap i'   => 'color:{{VALUE}};',
							'{{WRAPPER}} .kfw-ele-marquee-list .kfw-ele-marquee-icon-wrap .kfw-elementor-icon-wrap svg' => 'fill:{{VALUE}};',
						],
					]);
					$this->add_control('icon_rotate',[
						'label'     => esc_html__( 'Rotate', 'onnat-extra' ),
						'type'      => KFCM::SLIDER,
						'default'   => [
							'size' => 0,
							'unit' => 'deg',
						],
						'range'     => [
							'deg' => [
								'max' => 360,
								'min' => - 360,
							],
						],
						'selectors' => [
							'{{WRAPPER}} .kfw-ele-marquee-list .kfw-ele-marquee-icon-wrap i'   => 'transform: rotate({{SIZE}}{{UNIT}});',
							'{{WRAPPER}} .kfw-ele-marquee-list .kfw-ele-marquee-icon-wrap svg' => 'transform: rotate({{SIZE}}{{UNIT}});',
						],
					]);
				$this->end_controls_tab();
				$this->start_controls_tab( 'icon_hover_state', [
					'label'	=> esc_html__( 'Hover', 'onnat-extra' ),
				]);
					$this->add_control( 'icon_h_color', [
						'label'     => esc_html__('Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'selectors' => [
							'{{WRAPPER}} .kfw-ele-marquee-list .kfw-ele-marquee-icon-wrap .kfw-elementor-icon-wrap:hover i'   => 'color: {{VALUE}};',
							'{{WRAPPER}} .kfw-ele-marquee-list .kfw-ele-marquee-icon-wrap .kfw-elementor-icon-wrap:hover svg' => 'fill: {{VALUE}};',
						],
					]);
					$this->add_control('icon_h_rotate',[
						'label'     => esc_html__( 'Rotate', 'onnat-extra' ),
						'type'      => KFCM::SLIDER,
						'default'   => [
							'size' => 0,
							'unit' => 'deg',
						],
						'range'     => [
							'deg' => [
								'max' => 360,
								'min' => - 360,
							],
						],
						'selectors' => [
							'{{WRAPPER}} .kfw-ele-marquee-list .kfw-ele-marquee-icon-wrap:hover i'   => 'transform: rotate({{SIZE}}{{UNIT}});',
							'{{WRAPPER}} .kfw-ele-marquee-list .kfw-ele-marquee-icon-wrap:hover svg' => 'transform: rotate({{SIZE}}{{UNIT}});',
						],
					]);
				$this->end_controls_tab();
			$this->end_controls_tabs();
		$this->end_controls_section();
	}

	public function _register_icon_type_text_style_section() {
		$this->start_controls_section( 'section_icon_type_text_style', [
			'label' => esc_html__( 'Marquee : Icon Type : Text', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
			$this->add_group_control(KFGCT::get_type(), [
				'name'      => 'icon_text_typo',
				'label'     => esc_html__( 'Typography', 'onnat-extra' ),
				'selector'  => '{{WRAPPER}} .kfw-ele-marquee-list .kfw-ele-marquee-icon-wrap span',
			]);
			$this->add_responsive_control( 'icon_text_margin', [
				'label'      => esc_html__('Margin', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-marquee-list .kfw-ele-marquee-icon-wrap span' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
			$this->add_responsive_control( 'icon_text_padding', [
				'label'      => esc_html__('Padding', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-marquee-list .kfw-ele-marquee-icon-wrap span' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
			$this->start_controls_tabs( 'icon_text_customize_tabs' );
				$this->start_controls_tab( 'icon_text_normal_state', [
					'label'	=> esc_html__( 'Normal', 'onnat-extra' ),
				]);
					$this->add_control( 'icon_text_color', [
						'label'     => esc_html__('Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'selectors' => ['{{WRAPPER}} .kfw-ele-marquee-list .kfw-ele-marquee-icon-wrap span' => 'color: {{VALUE}};' ],
					]);
					$this->add_group_control( KFGCTS::get_type(), [
						'name'      => 'icon_text_txt_shadow',
						'selector'  => '{{WRAPPER}} .kfw-ele-marquee-list .kfw-ele-marquee-icon-wrap span',
					]);
					$this->add_group_control( KFGCTST::get_type(), [
						'name'      => 'icon_text_txt_stroke',
						'selector'  => '{{WRAPPER}} .kfw-ele-marquee-list .kfw-ele-marquee-icon-wrap span',
					]);
					$this->add_control('icon_text_rotate',[
						'label'     => esc_html__( 'Rotate', 'onnat-extra' ),
						'type'      => KFCM::SLIDER,
						'default'   => [
							'size' => 0,
							'unit' => 'deg',
						],
						'range'     => [
							'deg' => [
								'max' => 360,
								'min' => - 360,
							],
						],
						'selectors' => [
							'{{WRAPPER}} .kfw-ele-marquee-list .kfw-ele-marquee-icon-wrap .kfw-elementor-custom-data-wrap span'   => 'display:inline-block; transform: rotate({{SIZE}}{{UNIT}});',
						],
					]);
				$this->end_controls_tab();
				$this->start_controls_tab( 'icon_text_hover_state', [
					'label'	=> esc_html__( 'Hover', 'onnat-extra' ),
				]);
					$this->add_control( 'icon_text_hover_color', [
						'label'     => esc_html__('Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'selectors' => ['{{WRAPPER}} .kfw-ele-marquee-list .kfw-ele-marquee-icon-wrap:hover span' => 'color: {{VALUE}};' ],
					]);
					$this->add_group_control( KFGCTS::get_type(), [
						'name'      => 'icon_text_hover_txt_shadow',
						'selector'  => '{{WRAPPER}} .kfw-ele-marquee-list .kfw-ele-marquee-icon-wrap:hover span',
					]);
					$this->add_group_control( KFGCTST::get_type(), [
						'name'      => 'icon_text_hover_txt_stroke',
						'selector'  => '{{WRAPPER}} .kfw-ele-marquee-list .kfw-ele-marquee-icon-wrap:hover span',
					]);
					$this->add_control('icon_text_hover_rotate',[
						'label'     => esc_html__( 'Rotate', 'onnat-extra' ),
						'type'      => KFCM::SLIDER,
						'default'   => [
							'size' => 0,
							'unit' => 'deg',
						],
						'range'     => [
							'deg' => [
								'max' => 360,
								'min' => - 360,
							],
						],
						'selectors' => [
							'{{WRAPPER}} .kfw-ele-marquee-list .kfw-ele-marquee-icon-wrap .kfw-elementor-custom-data-wrap:hover span'   => 'display:inline-block; transform: rotate({{SIZE}}{{UNIT}});',
						],
					]);
				$this->end_controls_tab();
			$this->end_controls_tabs();

		$this->end_controls_section();
	}

	public function _register_icon_type_image_style_section() {
		$this->start_controls_section( 'section_icon_type_image_style', [
			'label' => esc_html__( 'Marquee : Icon Type : Image', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
			$this->add_responsive_control( 'icon_type_img_size', [
				'label'      => esc_html__('Image Size', 'onnat-extra' ),
				'type'       => KFCM::SLIDER,
				'size_units' => ['px'],
				'range'      => [
					'px' => [
						'min' => 5,
						'max' => 300,
					],
				],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-marquee-list .kfw-ele-marquee-icon-wrap img' => 'max-width: {{SIZE}}px;'
				],
			]);
			$this->add_responsive_control( 'icon_type_img_margin', [
				'label'      => esc_html__('Margin', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-marquee-list .kfw-ele-marquee-icon-wrap' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
			$this->add_responsive_control( 'icon_type_img_padding', [
				'label'      => esc_html__('Padding', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-marquee-list .kfw-ele-marquee-icon-wrap' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
			$this->start_controls_tabs( 'icon_type_img_customize_tabs' );
				$this->start_controls_tab( 'icon_type_img_normal_state', [
					'label' => esc_html__( 'Normal', 'onnat-extra' ),
				]);
					$this->add_control('icon_type_img_normal_rotate',[
						'label'     => esc_html__( 'Rotate', 'onnat-extra' ),
						'type'      => KFCM::SLIDER,
						'default'   => [
							'size' => 0,
							'unit' => 'deg',
						],
						'range'     => [
							'deg' => [
								'max' => 360,
								'min' => - 360,
							],
						],
						'selectors' => [
							'{{WRAPPER}} .kfw-ele-marquee-list .kfw-ele-marquee-icon-wrap img' => 'transform: rotate({{SIZE}}{{UNIT}});',
						],
					]);
				$this->end_controls_tab();
				$this->start_controls_tab( 'icon_type_img_hover_state', [
					'label' => esc_html__( 'Hover', 'onnat-extra' ),
				]);
					$this->add_control('icon_type_img_hover_rotate',[
						'label'     => esc_html__( 'Rotate', 'onnat-extra' ),
						'type'      => KFCM::SLIDER,
						'default'   => [
							'size' => 0,
							'unit' => 'deg',
						],
						'range'     => [
							'deg' => [
								'max' => 360,
								'min' => - 360,
							],
						],
						'selectors' => [
							'{{WRAPPER}} .kfw-ele-marquee-list .kfw-ele-marquee-icon-wrap:hover img' => 'transform: rotate({{SIZE}}{{UNIT}});',
						],
					]);
				$this->end_controls_tab();
			$this->end_controls_tabs();



		$this->end_controls_section();
	}

	public function _render_icon( $icon , $echo = true ) {
		$icon = array_filter( $icon );

		if( !empty( $icon ) ) {
			echo '<span class="kfw-elementor-icon-wrap">';
				KFIM::render_icon( $icon, [ 'aria-hidden' => 'true' ] );
			echo '</span>';
		}
	}

	public function _render_img( $settings = [], $id = '' ) {
		if( !empty( $settings['image']['url'] ) ) {
			$image_src = $settings['image']['url'];

			$image_title = $this->_get_image_title( $settings['image'] );
			$image_title = empty( $image_title ) ? $this->get_title() : $image_title;

			$image_alt = $this->_get_image_alt( $settings['image'] );
			$image_alt = empty( $image_alt ) ? $image_title : $image_title;

			$this->add_render_attribute( 'image_attr_'.$id , [
				'alt'   => esc_attr( $image_alt ),
				'title' => esc_attr( $image_title ),
				'src'   => esc_url( $image_src ),
			] );

			echo '<span class="kfw-elementor-icon-wrap">';
				echo '<img '.$this->get_render_attribute_string( 'image_attr_'.$id ).'>';
			echo '</span>';
		}
	}

    public function _render_input( $input, $echo = true ) {

		if( !empty( $input ) ) {

            printf( '<span class="kfw-elementor-icon-wrap kfw-elementor-custom-data-wrap"> <span> %1$s </span> </span>', $input );
        }
    }

	public function _get_image_title( $attachment ) {
		if ( empty( $attachment['id'] ) ) {
			return '';
		}

		return get_the_title( $attachment['id'] );
	}

	public function _get_image_alt( $instance ) {
		if ( empty( $instance['id'] ) ) {
			// For `Insert From URL` images.
			return isset( $instance['alt'] ) ? trim( strip_tags( $instance['alt'] ) ) : '';
		}

		$attachment_id = $instance['id'];
		if ( ! $attachment_id ) {
			return '';
		}

		$attachment = get_post( $attachment_id );
		if ( ! $attachment ) {
			return '';
		}

		$alt = get_post_meta( $attachment_id, '_wp_attachment_image_alt', true );
		if ( ! $alt ) {
			$alt = $attachment->post_excerpt;
			if ( ! $alt ) {
				$alt = $attachment->post_title;
			}
		}
		return trim( strip_tags( $alt ) );
	}

	/**
	 * Render widget output on the frontend.
	 */
    protected function render() {
		$render = kf_extra_plugin_elementor()->get_template("kfw-marquee-text");
		include $render;
    }

}