<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

use Elementor\Widget_Base as WB;
use Elementor\Controls_Manager as KFCM;
use Elementor\Icons_Manager as KFIM;

use Elementor\Group_Control_Typography as KFGCT;
use Elementor\Group_Control_Text_Shadow as KFGCTS;
use Elementor\Group_Control_Text_Stroke as KFGCTST;
use \Elementor\Group_Control_Background as KFGCBG;
use \Elementor\Group_Control_Border as KFGCBR;
use \Elementor\Group_Control_Box_Shadow as KFGCBS;

class KFW_Rounded_Text extends WB {

	/**
	 * Get element name.
	 *
	 * Retrieve the element name.
	 *
	 * @return string The name.
	 */
	public function get_name() {
        return 'kfw-rounded-text';
    }

	/**
	 * Get element title.
	 *
	 * Retrieve the element title.
	 *
	 * @return string Element title.
	 */
	public function get_title() {
		return esc_html__('Rounded Text', 'onnat-extra');
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the widget categories.
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ '_kfw_core_widgets' ];
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve the widget icon.
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'kfw-badge eicon-settings kfw-rounded-txt';
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the widget keywords.
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'kfw', 'kin', 'rounded', 'round', 'gsap', 'text', 'animate' ];
	}

	/**
	 * Get widget help url.
	 *
	 * Retrieve the widget help url.
	 *
	 * @return string Widget help url.
	 */
	public function get_custom_help_url() {
        return KFE_CONST_THEME_DOC . 'elementor-custom-widgets/rounded-text';
    }

	/**
	 * Get HTML wrapper class.
     *
	 * Retrieve the widget container class.
     *
     * @return string Widget container css classes.
	 */
	public function get_html_wrapper_class() {
        $name = $this->get_name();

		return sprintf( 'kfw-ele-%s elementor-widget-%s', $name, $name );
	}

	/**
	 * Widget base constructor.
	 *
	 * Initializing the widget base class.
	 */
	public function __construct( $data = [], $args = null ) {
        parent::__construct( $data, $args );

		wp_register_style( 'kfw-rounded-text',
			KFE_CONST_URL . 'libraries/elementor/assets/css/rounded-text' . KFE_CONST_DEBUG_SUFFIX . '.css',
			[],
			KFE_CONST_VERSION
		);
    }

	/**
	 * Get style dependencies.
	 *
	 * Retrieve the list of style dependencies the element requires.
	 *
	 * @return array Element styles dependencies.
	 */
	public function get_style_depends() {
		return [ 'kfw-rounded-text' ];
	}

	/**
	 * Get script dependencies.
	 *
	 * Retrieve the list of script dependencies the element requires.
	 *
	 * @return array Element scripts dependencies.
	 */
	public function get_script_depends() {
		return [];
	}

    public function _get_settings() {
        return $this->get_default_data();
    }

	/**
	 * Register controls.
	 *
	 * Used to add new controls to any element type.
	 */
	protected function register_controls() {
        $this->_register_title_section();
		$this->_register_middle_section();
		$this->_register_settings_section();

        $this->_register_title_style_section();
        $this->_register_middle_style_section();
    }

	public function _register_title_section() {
		$this->start_controls_section( 'section_title', [
			'label' => esc_html__( 'Title', 'onnat-extra' )
		]);
            $this->add_control('title',[
                'label'       => esc_html__( 'Title', 'onnat-extra' ),
                'type'        => KFCM::TEXT,
                'dynamic'     => [ 'active' => true, ],
                'placeholder' => esc_html__( 'Enter Title', 'onnat-extra' ),
                'default'     => 'Phasellus blandit justo quis dui malesuada gravida',
                'label_block' => true,
            ]);
		$this->end_controls_section();
    }

	public function _register_middle_section() {
		$this->start_controls_section( 'section_middle', [
			'label' => esc_html__( 'Middle Section', 'onnat-extra' )
		]);
            $this->add_control('icon_type',[
                'label'   => esc_html__( 'Icon Type', 'onnat-extra' ),
                'type'    => KFCM::CHOOSE,
                'options' => [
                    'blank' => [
                        'title' => esc_html__('Blank', 'onnat-extra'),
                        'icon'  => '',
                    ],
                    'icon'  => [
                        'title' => esc_html__('Icon', 'onnat-extra'),
                        'icon'  => 'eicon-plus-circle-o',
                    ],
                    'image' => [
                        'title' => esc_html__('Image', 'onnat-extra'),
                        'icon'  => 'eicon-image',
                    ],
                    'input' => [
                        'title' => esc_html__('Input', 'onnat-extra'),
                        'icon'  => 'eicon-text-field',
                    ],
                ],
                'default' => 'blank',
                'toggle'  => false
            ]);
            $this->add_control( 'icon', [
                'label'     => esc_html__( 'Icon', 'onnat-extra' ),
                'type'      => KFCM::ICONS,
                'default'   => [
                    'value'   => 'fas fa-heart',
                    'library' => 'fa-solid',
                ],
                'condition' => [
                    'icon_type' => 'icon'
                ],
            ]);
            $this->add_control( 'image', [
                'label'     => esc_html__( 'Image', 'onnat-extra' ),
                'type'      => KFCM::MEDIA,
                'dynamic'   => [ 'active' => true ],
                'default'   => [
                    'url' => \Elementor\Utils::get_placeholder_image_src()
                ],
                'condition' => [
                    'icon_type' => 'image'
                ],
            ]);
            $this->add_control( 'input', [
                'label'       => esc_html__( 'Custom Text', 'onnat-extra' ),
                'type'        => KFCM::TEXT,
                'label_block' => true,
                'default'     => 'A',
                'condition'   => [
                    'icon_type' => 'input'
                ],
            ]);
		$this->end_controls_section();
    }

    public function _register_settings_section() {
		$this->start_controls_section( 'section_settings', [
			'label' => esc_html__( 'Settings', 'onnat-extra' )
		]);
            $this->add_responsive_control('align',[
                'label'        => esc_html__( 'Alignment', 'onnat-extra' ),
                'type'         => KFCM::CHOOSE,
                'options'      => [
                    'left'    => [
                        'title' => esc_html__( 'Left', 'onnat-extra' ),
                        'icon'  => 'eicon-text-align-left',
                    ],
                    'center'  => [
                        'title' => esc_html__( 'Center', 'onnat-extra' ),
                        'icon'  => 'eicon-text-align-center',
                    ],
                    'right'   => [
                        'title' => esc_html__( 'Right', 'onnat-extra' ),
                        'icon'  => 'eicon-text-align-right',
                    ],
                ],
				'toggle'               => false,
				'default'              => 'center',
				'tablet_default'       => 'center',
				'mobile_default'       => 'center',
				'selectors_dictionary' => [
					'left'   => 'justify-content:start;',
					'center' => 'justify-content:center;align-items: center;',
					'right'  => 'justify-content:end;',
				],
				'selectors'            => [
                    '{{WRAPPER}}.kfw-ele-kfw-rounded-text' => '{{VALUE}};',
                ],
            ]);
            $this->add_responsive_control( 'svg_size',[
                'label'      => esc_html__('Size', 'onnat-extra' ),
                'type'       => KFCM::SLIDER,
                'size_units' => [ 'px' ],
                'range'      => [
                    'px' => [
                        'min' => 50,
                        'max' => 800,
                    ],
                ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-ele-widget-rounded-txt-main, {{WRAPPER}} .kfw-ele-widget-rounded-txt-main .kfw-ele-widget-rounded-txt-svg-wrapper svg' => 'width:{{SIZE}}{{UNIT}};height:{{SIZE}}{{UNIT}};'
                ]
            ]);
            $this->add_control( 'spin', [
                'label'        => esc_html__( 'Spin it?', 'onnat-extra' ),
                'type'         => KFCM::SWITCHER,
                'return_value' => 'yes',
                'default'      => 'yes',
            ]);
            $this->add_control( 'pause', [
                'label'        => esc_html__( 'Pause on Hover?', 'onnat-extra' ),
                'type'         => KFCM::SWITCHER,
                'return_value' => 'yes',
                'default'      => 'yes',
                'condition'    => [
                    'spin' => 'yes'
                ],
            ]);
            $this->add_control('spin_dir',[
                'label'     => esc_html__( 'Spin Direction', 'onnat-extra' ),
                'type'      => KFCM::CHOOSE,
                'options'   => [
                    'clock'      => [
                        'title' => esc_html__('Clock Wise', 'onnat-extra'),
                        'icon'  => 'eicon-clock',
                    ],
                    'anti-clock' => [
                        'title' => esc_html__('Anti Clock Wise', 'onnat-extra'),
                        'icon'  => 'eicon-clock-o',
                    ],
                ],
                'default'   => 'clock',
                'condition' => [
                    'spin' => 'yes'
                ],
                'toggle'    => false
            ]);
            $this->add_control('duration',[
                'label'      => esc_html__('Duration (s)', 'onnat-extra' ),
                'type'       => KFCM::SLIDER,
                'size_units' => [],
                'range'      => [
                    'min'  => 1,
                    'max'  => 200,
                    'step' => 1
                ],
                'condition'  => [
                    'spin' => 'yes'
                ],
                'default'    => [
                    'unit' => '',
                    'size' => 25
                ],
				'selectors' => [
					'{{WRAPPER}}' => '--kfw-ele-widget-rounded-txt-anim-speed:{{SIZE}}s;',
				],
            ]);
		$this->end_controls_section();
    }

	public function _register_title_style_section() {
		$this->start_controls_section( 'section_title_style',[
			'label'     => esc_html__( 'Title', 'onnat-extra' ),
			'tab'       => KFCM::TAB_STYLE,
			'condition' => [
				'title!' => ''
			]
		]);
            $this->add_group_control(KFGCT::get_type(), [
                'name'      => 'title_typo',
                'label'     => esc_html__( 'Typography', 'onnat-extra' ),
                'selector'  => '{{WRAPPER}} .kfw-ele-widget-rounded-txt-main .kfw-ele-widget-rounded-txt-path-wrap',
                'condition' => [
                    'title!' => ''
                ]
            ]);
            $this->add_control( 'title_color', [
                'label'     => esc_html__('Typo Color', 'onnat-extra' ),
                'type'      => KFCM::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .kfw-ele-widget-rounded-txt-main .kfw-ele-widget-rounded-txt-path-wrap' => 'fill:{{VALUE}};'
                ],
                'condition' => [
                    'title!' => ''
                ],
            ]);
		$this->end_controls_section();
    }

    public function _register_middle_style_section() {
		$this->start_controls_section( 'section_middle_style',[
			'label'     => esc_html__( 'Middle', 'onnat-extra' ),
			'tab'       => KFCM::TAB_STYLE,
			'condition' => [
                'icon_type!' => 'blank'
			]
		]);
            $this->add_control('icon_type_icon_style_heading',[
                'label'     => esc_html__( 'ICON SETTINGS', 'onnat-extra' ),
                'type'      => KFCM::HEADING,
                'condition' => [
                    'icon_type'    => 'icon',
                    'icon[value]!' => ''
                ],
            ]);
            $this->add_responsive_control( 'icon_size', [
                'label'      => esc_html__('Size', 'onnat-extra' ),
                'type'       => KFCM::SLIDER,
                'size_units' => [ 'px' ],
                'range'      => [
                    'px' => [
                        'min' => 0,
                        'max' => 300,
                    ],
                ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-ele-widget-rounded-txt-icon-wrap.kfw-ele-widget-rounded-txt-icon-type' => 'font-size: {{SIZE}}{{UNIT}};'
                ],
                'condition'  => [
                    'icon_type'    => 'icon',
                    'icon[value]!' => ''
                ],
            ]);
            $this->add_control( 'icon_color', [
                'label'     => esc_html__('Color', 'onnat-extra' ),
                'type'      => KFCM::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .kfw-ele-widget-rounded-txt-icon-wrap.kfw-ele-widget-rounded-txt-icon-type .kfw-elementor-icon-wrap'     => 'color: {{VALUE}};',
                    '{{WRAPPER}} .kfw-ele-widget-rounded-txt-icon-wrap.kfw-ele-widget-rounded-txt-icon-type .kfw-elementor-icon-wrap svg' => 'fill: {{VALUE}};',
                ],
                'condition' => [
                    'icon_type'    => 'icon',
                    'icon[value]!' => ''
                ],
            ]);
            $this->add_control('icon_rotate',[
                'label'     => esc_html__( 'Rotate', 'onnat-extra' ),
                'type'      => KFCM::SLIDER,
                'default'   => [
                    'size' => 0,
                    'unit' => 'deg',
                ],
                'range'     => [
                    'deg' => [
                        'max' => 360,
                        'min' => - 360,
                    ],
                ],
                'condition' => [
                    'icon_type'    => 'icon',
                    'icon[value]!' => ''
                ],
                'selectors' => [
                    '{{WRAPPER}} .kfw-ele-widget-rounded-txt-icon-wrap.kfw-ele-widget-rounded-txt-icon-type .kfw-elementor-icon-wrap i'   => 'transform: rotate({{SIZE}}{{UNIT}});',
                    '{{WRAPPER}} .kfw-ele-widget-rounded-txt-icon-wrap.kfw-ele-widget-rounded-txt-icon-type .kfw-elementor-icon-wrap svg' => 'transform: rotate({{SIZE}}{{UNIT}});',
                ],
            ]);
            $this->add_group_control( KFGCBG::get_type(), [
                'name'           => 'icon_bg',
                'types'          => ['classic', 'gradient'],
                'exclude'        => [ 'image'],
                'condition'      => [
                    'icon_type'    => 'icon',
                    'icon[value]!' => ''
                ],
                'fields_options' => [
                    'background' => [ 'default' => 'classic' ],
                    'color'      => [ 'default' => '' ],
                ],
                'selector'       => '{{WRAPPER}} .kfw-ele-widget-rounded-txt-icon-wrap.kfw-ele-widget-rounded-txt-icon-type .kfw-elementor-icon-wrap',
            ]);
            $this->add_control('icon_bg_rotate',[
                'label'     => esc_html__( 'Background Rotate', 'onnat-extra' ),
                'type'      => KFCM::SLIDER,
                'default'   => [
                    'size' => 0,
                    'unit' => 'deg',
                ],
                'range'     => [
                    'deg' => [
                        'max' => 360,
                        'min' => -360,
                    ],
                ],
                'condition' => [
                    'icon_type'          => 'icon',
                    'icon[value]!'       => '',
                    'icon_bg_background' => [ 'classic', 'gradient' ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .kfw-ele-widget-rounded-txt-icon-wrap.kfw-ele-widget-rounded-txt-icon-type .kfw-elementor-icon-wrap' => 'transform: rotate({{SIZE}}{{UNIT}});',
                ],
            ]);
            $this->add_group_control( KFGCBR::get_type(), [
                'name'      => 'icon_border',
                'condition' => [
                    'icon_type'    => 'icon',
                    'icon[value]!' => '',
                ],
                'selector'  => '{{WRAPPER}} .kfw-ele-widget-rounded-txt-icon-wrap.kfw-ele-widget-rounded-txt-icon-type .kfw-elementor-icon-wrap'
            ]);
            $this->add_responsive_control( 'icon_border_radius', [
                'label'      => esc_html__('Border Radius', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', '%' ],
                'condition'  => [
                    'icon_type'    => 'icon',
                    'icon[value]!' => '',
                ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-ele-widget-rounded-txt-icon-wrap.kfw-ele-widget-rounded-txt-icon-type .kfw-elementor-icon-wrap' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ]
            ]);
            $this->add_group_control( KFGCBS::get_type(), [
                'name'      => 'icon_box_shadow',
                'condition'  => [
                    'icon_type'    => 'icon',
                    'icon[value]!' => '',
                ],
                'selector'  => '{{WRAPPER}} .kfw-ele-widget-rounded-txt-icon-wrap.kfw-ele-widget-rounded-txt-icon-type .kfw-elementor-icon-wrap'
            ]);

            /**
             *  Image Tyoe
             */
                $this->add_control('icon_type_img_style_heading',[
                    'label'     => esc_html__( 'IMAGE SETTINGS', 'onnat-extra' ),
                    'type'      => KFCM::HEADING,
                    'condition' => [
                        'icon_type'   => 'image',
                        'image[url]!' => ''
                    ],
                ]);
                $this->add_responsive_control( 'img_wrap_size', [
                    'label'      => esc_html__('Wrap Size', 'onnat-extra' ),
                    'type'       => KFCM::SLIDER,
                    'size_units' => [ 'px' ],
                    'range'      => [
                        'px' => [
                            'min' => 0,
                            'max' => 300,
                        ],
                    ],
                    'selectors'  => [
                        '{{WRAPPER}} .kfw-ele-widget-rounded-txt-icon-wrap.kfw-ele-widget-rounded-txt-image-type .kfw-elementor-icon-wrap' => 'width:{{SIZE}}{{UNIT}}; height:{{SIZE}}{{UNIT}};'
                    ],
                    'condition'  => [
                        'icon_type'   => 'image',
                        'image[url]!' => ''
                    ],
                ]);
                $this->add_responsive_control( 'img_size', [
                    'label'      => esc_html__('Size', 'onnat-extra' ),
                    'type'       => KFCM::SLIDER,
                    'size_units' => ['%'],
                    'range'      => [
                        '%' => [
                            'min' => 5,
                            'max' => 100,
                        ],
                    ],
                    'condition'  => [
                        'icon_type'   => 'image',
                        'image[url]!' => ''
                    ],
                    'selectors'  => [
                        '{{WRAPPER}} .kfw-ele-widget-rounded-txt-icon-wrap.kfw-ele-widget-rounded-txt-image-type .kfw-elementor-icon-wrap img' => 'max-width: {{SIZE}}%;'
                    ],
                ]);
                $this->add_control('img_rotate',[
                    'label'     => esc_html__( 'Rotate', 'onnat-extra' ),
                    'type'      => KFCM::SLIDER,
                    'default'   => [
                        'size' => 0,
                        'unit' => 'deg',
                    ],
                    'range'     => [
                        'deg' => [
                            'max' => 360,
                            'min' => - 360,
                        ],
                    ],
                    'condition' => [
                        'icon_type'   => 'image',
                        'image[url]!' => ''
                    ],
                    'selectors' => [
                        '{{WRAPPER}} .kfw-ele-widget-rounded-txt-icon-wrap.kfw-ele-widget-rounded-txt-image-type .kfw-elementor-icon-wrap img' => 'transform: rotate({{SIZE}}{{UNIT}});',
                    ],
                ]);
                $this->add_group_control( KFGCBG::get_type(), [
                    'name'           => 'img_bg',
                    'types'          => ['classic', 'gradient'],
                    'exclude'        => [ 'image'],
                    'fields_options' => [
                        'background' => [ 'default' => 'classic' ],
                        'color'      => [ 'default' => '' ],
                    ],
                    'condition'      => [
                        'icon_type'   => 'image',
                        'image[url]!' => ''
                    ],
                    'selector'       => '{{WRAPPER}} .kfw-ele-widget-rounded-txt-icon-wrap.kfw-ele-widget-rounded-txt-image-type .kfw-elementor-icon-wrap',
                ]);
                $this->add_control('img_bg_rotate',[
                    'label'     => esc_html__( 'Background Rotate', 'onnat-extra' ),
                    'type'      => KFCM::SLIDER,
                    'default'   => [
                        'size' => 0,
                        'unit' => 'deg',
                    ],
                    'range'     => [
                        'deg' => [
                            'max' => 360,
                            'min' => -360,
                        ],
                    ],
                    'condition' => [
                        'icon_type'         => 'image',
                        'image[url]!'       => '',
                        'img_bg_background' => [ 'classic', 'gradient' ],
                    ],
                    'selectors' => [
                        '{{WRAPPER}} .kfw-ele-widget-rounded-txt-icon-wrap.kfw-ele-widget-rounded-txt-image-type .kfw-elementor-icon-wrap' => 'transform: rotate({{SIZE}}{{UNIT}});',
                    ],
                ]);
                $this->add_group_control( KFGCBR::get_type(), [
                    'name'      => 'img_border',
                    'condition' => [
                        'icon_type'   => 'image',
                        'image[url]!' => '',
                    ],
                    'selector'  => '{{WRAPPER}} .kfw-ele-widget-rounded-txt-icon-wrap.kfw-ele-widget-rounded-txt-image-type .kfw-elementor-icon-wrap'
                ]);
                $this->add_responsive_control( 'img_border_radius', [
                    'label'      => esc_html__('Border Radius', 'onnat-extra' ),
                    'type'       => KFCM::DIMENSIONS,
                    'size_units' => [ 'px', '%' ],
                    'condition'  => [
                        'icon_type'   => 'image',
                        'image[url]!' => '',
                    ],
                    'selectors'  => [
                        '{{WRAPPER}} .kfw-ele-widget-rounded-txt-icon-wrap.kfw-ele-widget-rounded-txt-image-type .kfw-elementor-icon-wrap' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                    ]
                ]);
                $this->add_group_control( KFGCBS::get_type(), [
                    'name'      => 'img_box_shadow',
                    'condition' => [
                        'icon_type'   => 'image',
                        'image[url]!' => '',
                    ],
                    'selector'  => '{{WRAPPER}} .kfw-ele-widget-rounded-txt-icon-wrap.kfw-ele-widget-rounded-txt-image-type .kfw-elementor-icon-wrap'
                ]);

            /**
             * Text Type
             */
                $this->add_responsive_control( 't_wrap_size', [
                    'label'      => esc_html__('Wrap Size', 'onnat-extra' ),
                    'type'       => KFCM::SLIDER,
                    'size_units' => [ 'px' ],
                    'range'      => [
                        'px' => [
                            'min' => 0,
                            'max' => 300,
                        ],
                    ],
                    'selectors'  => [
                        '{{WRAPPER}} .kfw-ele-widget-rounded-txt-icon-wrap.kfw-ele-widget-rounded-txt-input-type .kfw-elementor-icon-wrap' => 'width:{{SIZE}}{{UNIT}}; height:{{SIZE}}{{UNIT}}; line-height:{{SIZE}}{{UNIT}};'
                    ],
                    'condition'  => [
                        'icon_type' => 'input',
                        'input!'    => ''
                    ],
                ]);
                $this->add_group_control(KFGCT::get_type(), [
                    'name'      => 't_typo',
                    'label'     => esc_html__( 'Typography', 'onnat-extra' ),
                    'exclude'   => [
                        'line_height',
                        'letter_spacing',
                        'word_spacing'
                    ],
                    'selector'  => '{{WRAPPER}} .kfw-ele-widget-rounded-txt-icon-wrap.kfw-ele-widget-rounded-txt-input-type .kfw-elementor-icon-wrap span',
                    'condition' => [
                        'icon_type' => 'input',
                        'input!'    => ''
                    ]
                ]);
                $this->add_control( 't_color', [
                    'label'     => esc_html__('Color', 'onnat-extra' ),
                    'type'      => KFCM::COLOR,
                    'selectors' => ['{{WRAPPER}} .kfw-ele-widget-rounded-txt-icon-wrap.kfw-ele-widget-rounded-txt-input-type .kfw-elementor-icon-wrap span' => 'color: {{VALUE}};' ],
                    'condition' => [
                        'icon_type' => 'input',
                        'input!'    => ''
                    ]
                ]);
                $this->add_control('t_rotate',[
                    'label'     => esc_html__( 'Rotate', 'onnat-extra' ),
                    'type'      => KFCM::SLIDER,
                    'default'   => [
                        'size' => 0,
                        'unit' => 'deg',
                    ],
                    'range'     => [
                        'deg' => [
                            'max' => 360,
                            'min' => - 360,
                        ],
                    ],
                    'condition' => [
                        'icon_type' => 'input',
                        'input!'    => ''
                    ],
                    'selectors' => [
                        '{{WRAPPER}} .kfw-ele-widget-rounded-txt-icon-wrap.kfw-ele-widget-rounded-txt-input-type .kfw-elementor-icon-wrap span' => 'transform: rotate({{SIZE}}{{UNIT}});',
                    ],
                ]);
                $this->add_group_control( KFGCBG::get_type(), [
                    'name'           => 't_bg',
                    'types'          => ['classic', 'gradient'],
                    'exclude'        => [ 'image'],
                    'fields_options' => [
                        'background' => [ 'default' => 'classic' ],
                        'color'      => [ 'default' => '' ],
                    ],
                    'condition'      => [
                        'icon_type' => 'input',
                        'input!'    => ''
                    ],
                    'selector'       => '{{WRAPPER}} .kfw-ele-widget-rounded-txt-icon-wrap.kfw-ele-widget-rounded-txt-input-type .kfw-elementor-icon-wrap',
                ]);
                $this->add_control('t_bg_rotate',[
                    'label'     => esc_html__( 'Background Rotate', 'onnat-extra' ),
                    'type'      => KFCM::SLIDER,
                    'default'   => [
                        'size' => 0,
                        'unit' => 'deg',
                    ],
                    'range'     => [
                        'deg' => [
                            'max' => 360,
                            'min' => -360,
                        ],
                    ],
                    'condition' => [
                        'icon_type'       => 'input',
                        'input!'          => '',
                        't_bg_background' => [ 'classic', 'gradient' ],
                    ],
                    'selectors' => [
                        '{{WRAPPER}} .kfw-ele-widget-rounded-txt-icon-wrap.kfw-ele-widget-rounded-txt-input-type .kfw-elementor-icon-wrap' => 'transform: rotate({{SIZE}}{{UNIT}});',
                    ],
                ]);
                $this->add_group_control( KFGCBR::get_type(), [
                    'name'      => 't_border',
                    'condition' => [
                        'icon_type' => 'input',
                        'input!'    => '',
                    ],
                    'selector'  => '{{WRAPPER}} .kfw-ele-widget-rounded-txt-icon-wrap.kfw-ele-widget-rounded-txt-input-type .kfw-elementor-icon-wrap'
                ]);
                $this->add_responsive_control( 't_border_radius', [
                    'label'      => esc_html__('Border Radius', 'onnat-extra' ),
                    'type'       => KFCM::DIMENSIONS,
                    'size_units' => [ 'px', '%' ],
                    'condition'  => [
                        'icon_type' => 'input',
                        'input!'    => '',
                    ],
                    'selectors'  => [
                        '{{WRAPPER}} .kfw-ele-widget-rounded-txt-icon-wrap.kfw-ele-widget-rounded-txt-input-type .kfw-elementor-icon-wrap' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                    ]
                ]);
                $this->add_group_control( KFGCBS::get_type(), [
                    'name'      => 't_box_shadow',
                    'condition' => [
                        'icon_type' => 'input',
                        'input!'    => '',
                    ],
                    'selector'  => '{{WRAPPER}} .kfw-ele-widget-rounded-txt-icon-wrap.kfw-ele-widget-rounded-txt-input-type .kfw-elementor-icon-wrap'
                ]);
                $this->add_group_control( KFGCTS::get_type(), [
                    'name'     => 't_txt_shadow',
                    'selector' => '{{WRAPPER}} .kfw-ele-widget-rounded-txt-icon-wrap.kfw-ele-widget-rounded-txt-input-type .kfw-elementor-icon-wrap span',
                    'condition'   => [
                        'icon_type' => 'input',
                        'input!'    => ''
                    ]
                ]);
                $this->add_group_control( KFGCTST::get_type(), [
                    'name'     => 't_txt_stroke',
                    'selector' => '{{WRAPPER}} .kfw-ele-widget-rounded-txt-icon-wrap.kfw-ele-widget-rounded-txt-input-type .kfw-elementor-icon-wrap span',
                    'condition'   => [
                        'icon_type' => 'input',
                        'input!'    => ''
                    ]
                ]);

            $this->add_control('icon_type_style_after_hr', [
                'type'      => KFCM::DIVIDER,
                'condition' => [
                    'icon_type!' => 'blank',
                ],
            ]);
            $this->add_responsive_control( 'icon_margin', [
                'label'      => esc_html__('Margin', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'condition'  => [
                    'icon_type!' => 'blank',
                ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-ele-widget-rounded-txt-icon-wrap' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->add_responsive_control( 'icon_padding', [
                'label'      => esc_html__('Padding', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'condition'  => [
                    'icon_type!' => 'blank',
                ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-ele-widget-rounded-txt-icon-wrap .kfw-elementor-icon-wrap' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
		$this->end_controls_section();
    }

	public function _render_icon( $icon , $echo = true ) {
		$icon = array_filter( $icon );

		if( !empty( $icon ) ) {
			echo '<span class="kfw-elementor-icon-wrap">';
				KFIM::render_icon( $icon, [ 'aria-hidden' => 'true' ] );
			echo '</span>';
		}
	}

	public function _render_img( $settings = [] ) {
		if( !empty( $settings['image']['url'] ) ) {
			$image_src = $settings['image']['url'];

			$image_title = $this->_get_image_title( $settings['image'] );
			$image_title = empty( $image_title ) ? $this->get_title() : $image_title;

			$image_alt = $this->_get_image_alt( $settings['image'] );
			$image_alt = empty( $image_alt ) ? $image_title : $image_title;

			$this->add_render_attribute( 'image_attr', [
				'alt'   => esc_attr( $image_alt ),
				'title' => esc_attr( $image_title ),
				'src'   => esc_url( $image_src ),
			] );

			echo '<span class="kfw-elementor-icon-wrap">';
				echo '<img '.$this->get_render_attribute_string( 'image_attr' ).'/>';
			echo '</span>';
		}
	}

    public function _render_input( $input, $echo = true ) {

		if( !empty( $input ) ) {

            printf( '<span class="kfw-elementor-icon-wrap kfw-elementor-custom-data-wrap"> <span> %1$s </span> </span>', $input );
        }
    }

	public function _get_image_title( $attachment ) {
		if ( empty( $attachment['id'] ) ) {
			return '';
		}

		return get_the_title( $attachment['id'] );
	}

	public function _get_image_alt( $instance ) {
		if ( empty( $instance['id'] ) ) {
			// For `Insert From URL` images.
			return isset( $instance['alt'] ) ? trim( strip_tags( $instance['alt'] ) ) : '';
		}

		$attachment_id = $instance['id'];
		if ( ! $attachment_id ) {
			return '';
		}

		$attachment = get_post( $attachment_id );
		if ( ! $attachment ) {
			return '';
		}

		$alt = get_post_meta( $attachment_id, '_wp_attachment_image_alt', true );
		if ( ! $alt ) {
			$alt = $attachment->post_excerpt;
			if ( ! $alt ) {
				$alt = $attachment->post_title;
			}
		}
		return trim( strip_tags( $alt ) );
	}

	/**
	 * Render widget output on the frontend.
	 */
    protected function render() {
		$render = kf_extra_plugin_elementor()->get_template("kfw-rounded-text");
		include $render;
    }

}