<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

use \Elementor\Widget_Base as WB;
use \Elementor\Controls_Manager as KFCM;
use \Elementor\Group_Control_Image_Size as KFGCIS;

use \Elementor\Group_Control_Typography as KFGCT;
use \Elementor\Group_Control_Text_Shadow as KFGCTS;
use \Elementor\Group_Control_Text_Stroke as KFGCTST;
use \Elementor\Group_Control_Background as KFGCBG;
use \Elementor\Group_Control_Border as KFGCBR;
use \Elementor\Group_Control_Box_Shadow as KFGCBS;

class KFW_Blog_Grid extends WB {

	/**
	 * Get element name.
	 *
	 * Retrieve the element name.
	 *
	 * @return string The name.
	 */
	public function get_name() {
        return 'kfw-blog-grid';
    }

	/**
	 * Get element title.
	 *
	 * Retrieve the element title.
	 *
	 * @return string Element title.
	 */
	public function get_title() {
		return esc_html__('Blog Grid', 'onnat-extra');
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the widget categories.
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ '_kfw_theme_widgets' ];
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve the widget icon.
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'kfw-badge eicon-posts-grid kfw-blog-grid';
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the widget keywords.
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'kfw', 'kin', 'blog', 'post', 'grid' ];
	}

	/**
	 * Get widget help url.
	 *
	 * Retrieve the widget help url.
	 *
	 * @return string Widget help url.
	 */
	public function get_custom_help_url() {
        return KFE_CONST_THEME_DOC . 'elementor-custom-widgets/blog-grid';
    }

	/**
	 * Get HTML wrapper class.
     *
	 * Retrieve the widget container class.
     *
     * @return string Widget container css classes.
	 */
	public function get_html_wrapper_class() {
        $name = $this->get_name();

		return sprintf( 'kfw-ele-%s elementor-widget-%s', $name, $name );
	}

	/**
	 * Widget base constructor.
	 *
	 * Initializing the widget base class.
	 */
	public function __construct( $data = [], $args = null ) {
        parent::__construct( $data, $args );

		wp_register_style( 'kfw-blog-grid',
			KFE_CONST_URL . 'libraries/elementor/assets/css/blog-grid' . KFE_CONST_DEBUG_SUFFIX . '.css',
			[],
			KFE_CONST_VERSION
		);
    }

	/**
	 * Get style dependencies.
	 *
	 * Retrieve the list of style dependencies the element requires.
	 *
	 * @return array Element styles dependencies.
	 */
	public function get_style_depends() {
		return [ 'kfw-blog-grid' ];
	}

	/**
	 * Register controls.
	 *
	 * Used to add new controls to any element type.
	 */
	protected function register_controls() {
		$this->_register_items_section();
		$this->_register_settings_section();

		$this->_register_layout_style_section();

		$this->_register_post_date_meta_style_section();
		$this->_register_s4_post_date_meta_style_section();
		$this->_register_post_meta_style_section();

		$this->_register_post_title_style_section();
		$this->_register_readmore_btn_style_section();
		$this->_register_readmore_btn_icon_style_section();
	}

    public function _register_items_section() {
		$posts = $this->_blog_posts();

		$this->start_controls_section( 'section_items', [
			'label' => esc_html__( 'Blog Posts', 'onnat-extra' ),
			'tab'   => KFCM::TAB_CONTENT,
		]);
			$this->add_control( 'items', [
				'type'        => KFCM::REPEATER,
				'fields'      => [
					[
						'name'        => 'post',
						'label'       => esc_html__( 'Post', 'onnat-extra' ),
						'type'        => KFCM::SELECT2,
						'multiple'    => false,
						'label_block' => true,
						'dynamic'     => [ 'active' => true ],
						'options'     => $posts,
						'default'     => ''
					]
				],
				'title_field' => "<# let labels = ".json_encode( $posts )."; let label = labels[post]; #>{{{ label }}}",
				'default'     => [],
			]);
		$this->end_controls_section();
	}

	public function _register_settings_section() {
		$this->start_controls_section( 'section_settings', [
			'label' => esc_html__( 'Settings', 'onnat-extra' ),
			'tab'   => KFCM::TAB_CONTENT,
		]);
			$this->add_control('layout', [
				'label'   => esc_html__( 'Layout', 'onnat-extra' ),
				'type'    => KFCM::CHOOSE,
				'options' => [
					'kinfw-blog-grid-style-1' => [
						'title' => esc_html__('Grid Style 1', 'onnat-extra' ),
						'icon'  => 'eicon-gallery-grid'
					],
					'kinfw-blog-grid-style-2' => [
						'title' => esc_html__('Grid Style 2', 'onnat-extra' ),
						'icon'  => 'eicon-posts-grid'
					],
					'kinfw-blog-grid-style-3' => [
						'title' => esc_html__('Grid Style 3', 'onnat-extra' ),
						'icon'  => 'eicon-container-grid'
					],
					'kinfw-blog-grid-style-4' => [
						'title' => esc_html__('Grid Style 4', 'onnat-extra' ),
						'icon'  => 'eicon-apps'
					],
				],
				'default' => 'kinfw-blog-grid-style-3',
				'toggle'  => false
			]);
			$this->add_group_control( KFGCIS::get_type(), [
                'name'    => 'image_size',
                'default' => 'full',
			]);
			$this->add_responsive_control('cols', [
				'label'          => esc_html__( 'Columns', 'onnat-extra' ),
				'type'           => KFCM::CHOOSE,
				'options'        => [
					'1' => [ 'title' => esc_html__('1 Column', 'onnat-extra'), 'icon' => 'kfw-ctrl-icon kfw-ctrl-numeric-icon' ],
					'2' => [ 'title' => esc_html__('2 Columns', 'onnat-extra'), 'icon' => 'kfw-ctrl-icon kfw-ctrl-numeric-2-icon' ],
					'3' => [ 'title' => esc_html__('3 Columns', 'onnat-extra'), 'icon' => 'kfw-ctrl-icon kfw-ctrl-numeric-3-icon' ],
					'4' => [ 'title' => esc_html__('4 Columns', 'onnat-extra'), 'icon' => 'kfw-ctrl-icon kfw-ctrl-numeric-4-icon' ],
				],
				'devices'        => [ 'widescreen', 'desktop', 'laptop', 'tablet', 'tablet_extra' ],
				'default'        => '3',
				'tablet_default' => '2',
				'mobile_default' => '1',
				'selectors'      => [
					'{{WRAPPER}} .kfw-ele-widget-blog-post-grid-main' => 'display:grid; grid-template-columns: repeat({{SIZE}}, minmax(0, 1fr));',
				],
				'toggle'         => false
			]);
			$this->add_responsive_control('col_gap',[
				'label'     => esc_html__( 'Column Gap', 'onnat-extra' ),
				'type'      => KFCM::SLIDER,
				'devices'   => [ 'widescreen', 'desktop', 'laptop', 'tablet', 'tablet_extra' ],
				'default'   => [
					'size' => 20,
				],
				'selectors' => [
					'{{WRAPPER}} .kfw-ele-widget-blog-post-grid-main' => 'grid-column-gap: {{SIZE}}{{UNIT}};',
				],
			]);
			$this->add_responsive_control('row_gap',[
				'label'     => esc_html__( 'Row Gap', 'onnat-extra' ),
				'type'      => KFCM::SLIDER,
				'devices'   => [ 'widescreen', 'desktop', 'laptop', 'tablet', 'tablet_extra' ],
				'default'   => [ 'size' => 20,],
				'selectors' => [
					'{{WRAPPER}} .kfw-ele-widget-blog-post-grid-main' => 'grid-row-gap: {{SIZE}}{{UNIT}};',
				],
			]);
		$this->end_controls_section();
	}

    public function _register_layout_style_section() {
		$this->start_controls_section( 'section_layout_style', [
            'label' => esc_html__( 'Layout', 'onnat-extra' ),
            'tab'   => KFCM::TAB_STYLE,
		]);
            $this->add_control('layout_settings_heading',[
                'label' => esc_html__( 'LAYOUT SETTINGS', 'onnat-extra' ),
                'type'  => KFCM::HEADING,
            ]);
            $this->add_control('hr_layout', [
                'type' => KFCM::DIVIDER,
            ]);
            $this->add_group_control( KFGCBG::get_type(), [
                'name'           => 'layout_bg',
                'types'          => ['classic', 'gradient'],
                'exclude'        => [ 'image'],
                'fields_options' => [
                    'background' => [ 'default' => 'classic' ],
                    'color'      => [ 'default' => '' ],
                ],
                'selector'       => '
                    {{WRAPPER}} .kinfw-blog-grid-style-1,
                    {{WRAPPER}} .kinfw-blog-grid-style-2 .kinfw-entry-content-wrap,
                    {{WRAPPER}} .kinfw-blog-grid-style-3,
                    {{WRAPPER}} .kinfw-blog-grid-style-4',
                    
            ]);
            $this->add_group_control( KFGCBR::get_type(), [
                'name'      => 'layout_border',
                'selector'  => '
                    {{WRAPPER}} .kinfw-blog-grid-style-1,
                    {{WRAPPER}} .kinfw-blog-grid-style-2 .kinfw-entry-content-wrap,
                    {{WRAPPER}} .kinfw-blog-grid-style-3,
                    {{WRAPPER}} .kinfw-blog-grid-style-4',
            ]);
            $this->add_responsive_control( 'layout_border_radius', [
                'label'      => esc_html__('Border Radius', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-blog-grid-style-1,
                    {{WRAPPER}} .kinfw-blog-grid-style-2 .kinfw-entry-content-wrap,
                    {{WRAPPER}} .kinfw-blog-grid-style-3,
                    {{WRAPPER}} .kinfw-blog-grid-style-4' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ]
            ]);
            $this->add_group_control( KFGCBS::get_type(), [
                'name'      => 'layout_box_shadow',
                'selector'  => '
                    {{WRAPPER}} .kinfw-blog-grid-style-1,
                    {{WRAPPER}} .kinfw-blog-grid-style-2 .kinfw-entry-content-wrap,
                    {{WRAPPER}} .kinfw-blog-grid-style-3,
                    {{WRAPPER}} .kinfw-blog-grid-style-4'
            ]);
            $this->add_responsive_control( 'layout_padding', [
                'label'      => esc_html__('Padding', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-blog-grid-style-1 .kinfw-entry-content-wrap,
                    {{WRAPPER}} .kinfw-blog-grid-style-2 .kinfw-entry-content-wrap,
                    {{WRAPPER}} .kinfw-blog-grid-style-3,
                    {{WRAPPER}} .kinfw-blog-grid-style-4' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->add_control('hr_layout_img_before', [
                'type' => KFCM::DIVIDER,
            ]);
            $this->add_control('layout_image_settings_heading',[
                'label' => esc_html__( 'FEATURED IMAGE SETTINGS', 'onnat-extra' ),
                'type'  => KFCM::HEADING,
            ]);
            $this->add_control('hr_layout_img_after', [
                'type' => KFCM::DIVIDER,
            ]);
            $this->add_group_control( KFGCBG::get_type(), [
                'name'           => 'featured_img_bg',
                'types'          => ['classic', 'gradient'],
                'exclude'        => [ 'image'],
                'fields_options' => [
                    'background' => [ 'default' => 'classic' ],
                    'color'      => [ 'default' => '' ],
                ],
                'selector'       => '{{WRAPPER}} .kinfw-entry-media-wrap .kinfw-entry-media',
            ]);
            $this->add_group_control( KFGCBR::get_type(), [
                'name'      => 'featured_img_border',
                'selector'  => '{{WRAPPER}} .kinfw-entry-media-wrap .kinfw-entry-media',
            ]);
            $this->add_responsive_control( 'featured_img_border_radius', [
                'label'      => esc_html__('Border Radius', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-entry-media-wrap .kinfw-entry-media, {{WRAPPER}} .kinfw-entry-media-wrap .kinfw-entry-media img' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ]
            ]);
            $this->add_group_control( KFGCBS::get_type(), [
                'name'      => 'featured_img_box_shadow',
                'selector'  => '{{WRAPPER}} .kinfw-entry-media-wrap .kinfw-entry-media'
            ]);
            $this->add_responsive_control( 'featured_img_padding', [
                'label'      => esc_html__('Padding', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-entry-media-wrap .kinfw-entry-media' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
		$this->end_controls_section();
    }

    public function _register_s4_post_date_meta_style_section() {
        $condition = [
            'layout' => 'kinfw-blog-grid-style-4'
        ];

		$this->start_controls_section( 'section_s4_date_meta_style', [
            'label'     => esc_html__( 'Post Date', 'onnat-extra' ),
            'tab'       => KFCM::TAB_STYLE,
            'condition' => $condition
		]);
            $this->add_group_control(KFGCT::get_type(), [
                'name'      => 's4_post_date_typo',
                'label'     => esc_html__( 'Typography', 'onnat-extra' ),
                'exclude'   => [
                    'font_size',
                ],
                'condition' => $condition,
                'selector'  => '{{WRAPPER}} .kinfw-blog-grid-style-4 .kinfw-meta-date span',
            ]);
            $this->start_controls_tabs( 's4_post_date_customize_tabs', [ 
                'condition' => $condition
            ]);
                $this->start_controls_tab( 's4_date_tab', [
                    'label'     => esc_html__( 'Date', 'onnat-extra' ),
                    'condition' => $condition
                ]);
                    $this->add_responsive_control( 's4_date_size', [
                        'label'      => esc_html_x( 'Font Size', 'Typography Control', 'onnat-extra' ),
                        'type'       => KFCM::SLIDER,
                        'condition'  => $condition,
                        'size_units' => [ 'px', 'em', 'rem', 'vw', 'custom' ],
                        'selectors'  => [
                            '{{WRAPPER}} .kinfw-blog-grid-style-4 .kinfw-meta-date .kinfw-meta-day' => 'font-size:{{SIZE}}{{UNIT}};',
                        ],
                        'range'      => [
                            'px' => [
                                'min' => 1,
                                'max' => 200,
                            ],
                            'em' => [
                                'max' => 20,
                            ],
                            'rem' => [
                                'max' => 20,
                            ],
                            'vw' => [
                                'min' => 0.1,
                                'max' => 10,
                                'step' => 0.1,
                            ],
                        ],
                    ]);
                $this->end_controls_tab();
                $this->start_controls_tab( 's4_month_tab', [
                    'label'     => esc_html__( 'Month', 'onnat-extra' ),
                    'condition' => $condition
                ]);
                    $this->add_responsive_control( 's4_month_size', [
                        'label'      => esc_html_x( 'Font Size', 'Typography Control', 'onnat-extra' ),
                        'type'       => KFCM::SLIDER,
                        'condition'  => $condition,
                        'size_units' => [ 'px', 'em', 'rem', 'vw', 'custom' ],
                        'selectors'  => [
                            '{{WRAPPER}} .kinfw-blog-grid-style-4 .kinfw-meta-date .kinfw-meta-month' => 'font-size:{{SIZE}}{{UNIT}};',
                        ],
                        'range'      => [
                            'px' => [
                                'min' => 1,
                                'max' => 200,
                            ],
                            'em' => [
                                'max' => 20,
                            ],
                            'rem' => [
                                'max' => 20,
                            ],
                            'vw' => [
                                'min' => 0.1,
                                'max' => 10,
                                'step' => 0.1,
                            ],
                        ],
                    ]);
                $this->end_controls_tab();                
            $this->end_controls_tabs();
            $this->add_control('s4_post_date_hr1', [
                'type' => KFCM::DIVIDER,
            ]);     
            
            $this->add_control( 's4_post_date_color_settings', [
                'label'        => esc_html__('Color', 'onnat-extra'),
                'type'         => KFCM::POPOVER_TOGGLE,
                'return_value' => 'yes',
            ]);
            $this->start_popover();
                $this->add_control( 's4_post_date_color', [
                    'label'     => esc_html__('Normal Color', 'onnat-extra' ),
                    'type'      => KFCM::COLOR,
                    'selectors' => [
                        '{{WRAPPER}} .kinfw-blog-grid-style-4 .kinfw-entry-meta-wrap .kinfw-meta-date span' => 'color:{{VALUE}};',
                    ],
                ]);
            $this->add_control( 's4_post_date_h_color', [
                'label'     => esc_html__('Hover Color', 'onnat-extra' ),
                'type'      => KFCM::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .kinfw-blog-grid-style-4:hover .kinfw-entry-meta-wrap .kinfw-meta-date span' => 'color:{{VALUE}};',
                ],
            ]);            
            $this->end_popover();

            $this->add_control( 's4_post_date_bg_color_settings', [
                'label'        => esc_html__('Background Colors', 'onnat-extra'),
                'type'         => KFCM::POPOVER_TOGGLE,
                'return_value' => 'yes',
            ]);
            $this->start_popover();
                $this->add_control( 's4_post_date_bg_color', [
                    'label'     => esc_html__('Normal Color', 'onnat-extra' ),
                    'type'      => KFCM::COLOR,
                    'selectors' => [
                        '{{WRAPPER}} .kinfw-blog-grid-style-4 .kinfw-entry-meta-wrap .kinfw-meta-date time' => 'background-color:{{VALUE}};',
                    ],
                ]);
            $this->add_control( 's4_post_date_bg_h_color', [
                'label'     => esc_html__('Hover Color', 'onnat-extra' ),
                'type'      => KFCM::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .kinfw-blog-grid-style-4:hover .kinfw-entry-meta-wrap .kinfw-meta-date time' => 'background-color:{{VALUE}};',
                ],
            ]);            
            $this->end_popover();
            $this->add_control( 's4_post_date_border_color_settings', [
                'label'        => esc_html__('Border Colors', 'onnat-extra'),
                'type'         => KFCM::POPOVER_TOGGLE,
                'return_value' => 'yes',
            ]);
            $this->start_popover();
                $this->add_control( 's4_post_date_border_color', [
                    'label'     => esc_html__('Normal Color', 'onnat-extra' ),
                    'type'      => KFCM::COLOR,
                    'selectors' => [
                        '{{WRAPPER}} .kinfw-blog-grid-style-4 .kinfw-entry-meta-wrap .kinfw-meta-date:before' => 'background-color:{{VALUE}};',
                    ],
                ]);
            $this->add_control( 's4_post_date_border_h_color', [
                'label'     => esc_html__('Hover Color', 'onnat-extra' ),
                'type'      => KFCM::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .kinfw-blog-grid-style-4:hover .kinfw-entry-meta-wrap .kinfw-meta-date:before' => 'background-color:{{VALUE}};',
                ],
            ]);            
            $this->end_popover();
            $this->add_control('s4_post_date_hr2', [
                'type' => KFCM::DIVIDER,
            ]);
            $this->add_responsive_control( 's4_post_date_border_radius', [
                'label'      => esc_html__('Border Radius', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', '%' ],
                'condition'  => $condition,
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-blog-grid-style-4 .kinfw-entry-meta-wrap .kinfw-meta-date time, {{WRAPPER}} .kinfw-blog-grid-style-4 .kinfw-entry-meta-wrap .kinfw-meta-date:before' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ]
            ]);
            $this->add_group_control( KFGCBS::get_type(), [
                'name'      => 's4_post_date_box_shadow',
                'condition' => $condition,
                'selector'  => '{{WRAPPER}} .kinfw-blog-grid-style-4 .kinfw-entry-meta-wrap .kinfw-meta-date:before'
            ]);
            $this->add_responsive_control( 's4_post_date_margin', [
                'label'      => esc_html__('Margin', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'condition'  => $condition,
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-blog-grid-style-4 .kinfw-entry-meta-wrap .kinfw-meta-date' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->add_responsive_control( 's4_post_date_padding', [
                'label'      => esc_html__('Padding', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'condition'  => $condition,
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-blog-grid-style-4 .kinfw-entry-meta-wrap .kinfw-meta-date time' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);        
        $this->end_controls_section();
    }

    public function _register_post_date_meta_style_section() {
        $condition = [
            'layout' => 'kinfw-blog-grid-style-1'
        ];

		$this->start_controls_section( 'section_date_meta_style', [
            'label'     => esc_html__( 'Post Date', 'onnat-extra' ),
            'tab'       => KFCM::TAB_STYLE,
            'condition' => $condition
		]);
            $this->add_group_control(KFGCT::get_type(), [
                'name'      => 'post_date_typo',
                'label'     => esc_html__( 'Typography', 'onnat-extra' ),
                'condition' => $condition,
                'selector'  => '{{WRAPPER}} .kinfw-blog-grid-style-1 .kinfw-entry-media-wrap .kinfw-meta-date time',
            ]);
            $this->add_control( 'post_date_color', [
                'label'     => esc_html__('Color', 'onnat-extra' ),
                'type'      => KFCM::COLOR,
                'condition' => $condition,
                'selectors' => ['{{WRAPPER}} .kinfw-blog-grid-style-1 .kinfw-entry-media-wrap .kinfw-meta-date time' => 'color: {{VALUE}};' ],
            ]);
            $this->add_group_control( KFGCTS::get_type(), [
                'name'      => 'post_date_txt_shadow',
                'condition' => $condition,
                'selector'  => '{{WRAPPER}} .kinfw-blog-grid-style-1 .kinfw-entry-media-wrap .kinfw-meta-date time',
            ]);
            $this->add_group_control( KFGCTST::get_type(), [
                'name'      => 'post_date_txt_stroke',
                'condition' => $condition,
                'selector'  => '{{WRAPPER}} .kinfw-blog-grid-style-1 .kinfw-entry-media-wrap .kinfw-meta-date time',
            ]);
			$this->add_control( 'post_date_wrapper_color', [
				'label'     => esc_html__('Curve Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
                'condition' => $condition,
				'selectors' => [
					'{{WRAPPER}} .kinfw-blog-grid-style-1 .kinfw-entry-media-wrap .kinfw-meta-date' => 'background-color: {{VALUE}}; color: {{VALUE}};'
				],
			]);
            $this->add_group_control( KFGCBG::get_type(), [
                'name'           => 'post_date_bg',
                'types'          => ['classic', 'gradient'],
                'exclude'        => [ 'image'],
                'fields_options' => [
                    'background' => [ 'default' => 'classic' ],
                    'color'      => [ 'default' => '' ],
                ],
                'condition'      => $condition,
                'selector'       => '{{WRAPPER}} .kinfw-blog-grid-style-1 .kinfw-entry-media-wrap .kinfw-meta-date time',
            ]);
            $this->add_group_control( KFGCBR::get_type(), [
                'name'      => 'post_date_border',
                'condition' => $condition,
                'selector'  => '{{WRAPPER}} .kinfw-blog-grid-style-1 .kinfw-entry-media-wrap .kinfw-meta-date time'
            ]);
            $this->add_responsive_control( 'post_date_border_radius', [
                'label'      => esc_html__('Border Radius', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', '%' ],
                'condition'  => $condition,
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-blog-grid-style-1 .kinfw-entry-media-wrap .kinfw-meta-date time' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ]
            ]);
            $this->add_group_control( KFGCBS::get_type(), [
                'name'      => 'post_date_box_shadow',
                'condition' => $condition,
                'selector'  => '{{WRAPPER}} .kinfw-blog-grid-style-1 .kinfw-entry-media-wrap .kinfw-meta-date time'
            ]);
            $this->add_responsive_control( 'post_date_margin', [
                'label'      => esc_html__('Margin', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'condition'  => $condition,
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-blog-grid-style-1 .kinfw-entry-media-wrap .kinfw-meta-date time' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->add_responsive_control( 'post_date_padding', [
                'label'      => esc_html__('Padding', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'condition'  => $condition,
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-blog-grid-style-1 .kinfw-entry-media-wrap .kinfw-meta-date time' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
		$this->end_controls_section();
    }

    public function _register_post_meta_style_section() {
        $condition = [
            'layout!' => 'kinfw-blog-grid-style-3'
        ];

		$this->start_controls_section( 'section_meta_style', [
			'label' => esc_html__( 'Post Meta', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
            $this->add_control('meta_icon_settings_heading',[
                'label'     => esc_html__( 'ICON SETTINGS', 'onnat-extra' ),
                'type'      => KFCM::HEADING,
                'condition' => $condition,
            ]);
            $this->add_control('hr_1', [
                'type'      => KFCM::DIVIDER,
                'condition' => $condition,
            ]);
            $this->add_responsive_control( 'meta_icon_size', [
                'label'      => esc_html__('Icon Size', 'onnat-extra' ),
                'type'       => KFCM::SLIDER,
                'size_units' => [ 'px' ],
                'range'      => [
                    'px' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-entry-meta-wrap span.kinfw-icon' => 'font-size: {{SIZE}}{{UNIT}};'
                ],
                'condition'  => $condition,
            ]);
            $this->add_control( 'meta_icon_color', [
                'label'     => esc_html__('Color', 'onnat-extra' ),
                'type'      => KFCM::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .kinfw-entry-meta-wrap span.kinfw-icon' => 'color:{{VALUE}};',
                ],
                'condition' => $condition,
            ]);
            $this->add_responsive_control( 'meta_icon_margin', [
                'label'      => esc_html__('Margin', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-entry-meta-wrap span.kinfw-icon' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ],
                'condition'  => $condition,
            ]);
            $this->add_control('meta_typo_settings_heading',[
                'label' => esc_html__( 'TYPO SETTINGS', 'onnat-extra' ),
                'type'  => KFCM::HEADING,
            ]);
            $this->add_control('hr_2', [
                'type' => KFCM::DIVIDER,
            ]);
            $this->add_group_control(KFGCT::get_type(), [
                'name'     => 'meta_info_typo',
                'label'    => esc_html__( 'Typography', 'onnat-extra' ),
                'selector' => '
                    {{WRAPPER}} .kinfw-entry-meta-wrap a,
                    {{WRAPPER}} .kinfw-entry-meta-wrap .kinfw-meta-author a span:last-child,
                    {{WRAPPER}} .kinfw-entry-meta-wrap .kinfw-meta-date time
                    ',
            ]);
            $this->start_controls_tabs( 'meta_info_customize_tabs' );
                $this->start_controls_tab( 'meta_info_normal_state', [
                    'label' => esc_html__( 'Normal', 'onnat-extra' ),
                ]);
                    $this->add_control( 'meta_info_color', [
                        'label'     => esc_html__('Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'selectors' => [
                            '{{WRAPPER}} .kinfw-entry-meta-wrap a, {{WRAPPER}} .kinfw-entry-meta-wrap .kinfw-meta-date time' => 'color: {{VALUE}};',
                            '{{WRAPPER}} .kinfw-blog-grid-style-3 .kinfw-entry-meta-wrap > div:before' => 'background-color: {{VALUE}};',
                        ],
                    ]);
                    $this->add_group_control( KFGCTS::get_type(), [
                        'name'     => 'meta_info_txt_shadow',
                        'selector' => '{{WRAPPER}} .kinfw-entry-meta-wrap a, {{WRAPPER}} .kinfw-entry-meta-wrap .kinfw-meta-date time',
                    ]);
                    $this->add_group_control( KFGCTST::get_type(), [
                        'name'     => 'meta_info_txt_stroke',
                        'selector' => '{{WRAPPER}} .kinfw-entry-meta-wrap a, {{WRAPPER}} .kinfw-entry-meta-wrap .kinfw-meta-date time',
                    ]);
                $this->end_controls_tab();
                $this->start_controls_tab( 'meta_info_hover_state', [
                    'label' => esc_html__( 'Hover', 'onnat-extra' ),
                ]);
                    $this->add_control( 'meta_info_hover_color', [
                        'label'     => esc_html__('Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'selectors' => ['{{WRAPPER}} .kinfw-entry-meta-wrap a:hover' => 'color: {{VALUE}};' ],
                    ]);
                    $this->add_group_control( KFGCTS::get_type(), [
                        'name'     => 'meta_info_hover_txt_shadow',
                        'selector' => '{{WRAPPER}} .kinfw-entry-meta-wrap a:hover',
                    ]);
                    $this->add_group_control( KFGCTST::get_type(), [
                        'name'     => 'meta_info_hover_txt_stroke',
                        'selector' => '{{WRAPPER}} .kinfw-entry-meta-wrap a:hover',
                    ]);
                $this->end_controls_tab();
            $this->end_controls_tabs();
		$this->end_controls_section();
    }

    public function _register_post_title_style_section() {
		$this->start_controls_section( 'section_p_title_style', [
			'label' => esc_html__( 'Post Title', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
            $this->add_group_control(KFGCT::get_type(), [
                'name'     => 'title_typo',
                'label'    => esc_html__( 'Typography', 'onnat-extra' ),
                'selector' => '{{WRAPPER}} .kinfw-entry-header h4',
            ]);
            $this->add_responsive_control( 'title_margin', [
                'label'      => esc_html__('Margin', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-entry-header h4' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->add_responsive_control( 'title_padding', [
                'label'      => esc_html__('Padding', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-entry-header h4' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->start_controls_tabs( 'title_customize_tabs' );
                $this->start_controls_tab( 'title_normal_state', [
                    'label' => esc_html__( 'Normal', 'onnat-extra' ),
                ]);
                    $this->add_control( 'title_color', [
                        'label'     => esc_html__('Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'selectors' => ['{{WRAPPER}} .kinfw-entry-header h4 a' => 'color: {{VALUE}};' ],
                    ]);
                    $this->add_group_control( KFGCTS::get_type(), [
                        'name'     => 'title_txt_shadow',
                        'selector' => '{{WRAPPER}} .kinfw-entry-header h4',
                    ]);
                    $this->add_group_control( KFGCTST::get_type(), [
                        'name'     => 'title_txt_stroke',
                        'selector' => '{{WRAPPER}} .kinfw-entry-header h4',
                    ]);
                $this->end_controls_tab();
                $this->start_controls_tab( 'title_hover_state', [
                    'label' => esc_html__( 'Hover', 'onnat-extra' ),
                ]);
                    $this->add_control( 'title_hover_color', [
                        'label'     => esc_html__('Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'selectors' => ['{{WRAPPER}} .kinfw-entry-header h4 a:hover' => 'color: {{VALUE}};' ],
                    ]);
                    $this->add_group_control( KFGCTS::get_type(), [
                        'name'     => 'title_hover_txt_shadow',
                        'selector' => '{{WRAPPER}} .kinfw-entry-header h4 a:hover',
                    ]);
                    $this->add_group_control( KFGCTST::get_type(), [
                        'name'     => 'title_hover_txt_stroke',
                        'selector' => '{{WRAPPER}} .kinfw-entry-header h4 a:hover',
                    ]);
                $this->end_controls_tab();
            $this->end_controls_tabs();
		$this->end_controls_section();
    }

    public function _register_readmore_btn_style_section() {
		$condition = [
			'layout' => [ 'kinfw-blog-grid-style-1' ]
		];
		$this->start_controls_section( 'section_readmore_btn_style', [
            'label'     => esc_html__( 'Read More Button Text', 'onnat-extra' ),
            'tab'       => KFCM::TAB_STYLE,
            'condition' => $condition
		]);
            $this->add_group_control(KFGCT::get_type(), [
                'name'      => 'readmore_btn_typo',
                'label'     => esc_html__( 'Text', 'onnat-extra' ),
                'selector'  => '{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a',
                'condition' => $condition
            ]);
            $this->start_controls_tabs( 'readmore_btn_txt_customize_tabs', [
                'condition' => $condition
            ]);
                $this->start_controls_tab( 'readmore_btn_txt_normal_state', [
                    'label'     => esc_html__( 'Normal', 'onnat-extra' ),
					'condition' => $condition,
                ]);
                    $this->add_control( 'readmore_btn_txt_color', [
                        'label'     => esc_html__('Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'selectors' => ['{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a' => 'color: {{VALUE}};' ],
                    ]);
                    $this->add_group_control( KFGCBG::get_type(), [
                        'name'      => 'btn_bg',
                        'label'     => esc_html__('Background', 'onnat-extra' ),
                        'types'     => [ 'classic', 'gradient' ],
                        'exclude'   => [ 'image' ],# To remove background image option in classic mode
                        'selector'  => '{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a',
                        'condition' => [
                            'layout' => 'kinfw-blog-grid-style-1'
                        ],
                    ]);
                    $this->add_group_control( KFGCBR::get_type(), [
                        'name'           => 'btn_border',
                        'condition'      => [ 'layout' => 'kinfw-blog-grid-style-1'],
                        'fields_options' => [
                            'width' => [ 'label' => _x( 'Border Width', 'Border Control', 'onnat-extra' ) ],
                            'color' => [ 'label' => _x( 'Border Color', 'Border Control', 'onnat-extra' ) ],
                        ],
                        'selector'       => '{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a',                        
                    ]);
                    $this->add_responsive_control( 'btn_border_radius', [
                        'label'      => esc_html__('Border Radius', 'onnat-extra' ),
                        'type'       => KFCM::DIMENSIONS,
                        'size_units' => [ 'px', '%' ],
                        'selectors'  => [
                            '{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                        ],
                        'condition'  => [
                            'layout' => 'kinfw-blog-grid-style-1'
                        ],
                    ]);
                    $this->add_group_control( KFGCBS::get_type(), [
                        'name'      => 'btn_box_shadow',
                        'selector'  => '{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a',
                        'condition' => [
                            'layout' => 'kinfw-blog-grid-style-1'
                        ],
                    ]);
                    $this->add_group_control( KFGCTS::get_type(), [
                        'name'     => 'readmore_btn_txt_txt_shadow',
                        'selector' => '{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a',
                    ]);
                    $this->add_group_control( KFGCTST::get_type(), [
                        'name'     => 'readmore_btn_txt_txt_stroke',
                        'selector' => '{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a',
                    ]);
                $this->end_controls_tab();
                $this->start_controls_tab( 'readmore_btn_txt_hover_state', [
                    'label'     => esc_html__( 'Hover', 'onnat-extra' ),
					'condition' => $condition,
                ]);
                    $this->add_control( 'readmore_btn_txt_h_color', [
                        'label'     => esc_html__('Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'selectors' => ['{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a:hover' => 'color: {{VALUE}};' ],
                    ]);
                    $this->add_group_control( KFGCBG::get_type(), [
                        'name'      => 'btn_h_bg',
                        'label'     => esc_html__('Background', 'onnat-extra' ),
                        'types'     => [ 'classic', 'gradient' ],
                        'exclude'   => [ 'image' ],# To remove background image option in classic mode
                        'selector'  => '{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a:hover',
                        'condition' => [
                            'layout' => 'kinfw-blog-grid-style-1'
                        ],
                    ]);
                    $this->add_group_control( KFGCBR::get_type(), [
                        'name'           => 'btn_h_border',
                        'condition'      => [ 'layout' => 'kinfw-blog-grid-style-1'],
                        'fields_options' => [
                            'width' => [ 'label' => _x( 'Border Width', 'Border Control', 'onnat-extra' ) ],
                            'color' => [ 'label' => _x( 'Border Color', 'Border Control', 'onnat-extra' ) ],
                        ],
                        'selector'       => '{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a:hover',                        
                    ]);
                    $this->add_responsive_control( 'btn_h_border_radius', [
                        'label'      => esc_html__('Border Radius', 'onnat-extra' ),
                        'type'       => KFCM::DIMENSIONS,
                        'size_units' => [ 'px', '%' ],
                        'selectors'  => [
                            '{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a:hover' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                        ],
                        'condition'  => [
                            'layout' => 'kinfw-blog-grid-style-1'
                        ],
                    ]);
                    $this->add_group_control( KFGCBS::get_type(), [
                        'name'      => 'btn_h_box_shadow',
                        'selector'  => '{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a:hover',
                        'condition' => [
                            'layout' => 'kinfw-blog-grid-style-1'
                        ],
                    ]);                    
                    $this->add_group_control( KFGCTS::get_type(), [
                        'name'     => 'readmore_btn_txt_h_txt_shadow',
                        'selector' => '{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a:hover',
                    ]);
                    $this->add_group_control( KFGCTST::get_type(), [
                        'name'     => 'readmore_btn_txt_h_txt_stroke',
                        'selector' => '{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a:hover',
                    ]);
                $this->end_controls_tab();
            $this->end_controls_tabs();

		$this->end_controls_section();
    }

    public function _register_readmore_btn_icon_style_section() {
		$condition = [
			'layout' => [ 'kinfw-blog-grid-style-1', 'kinfw-blog-grid-style-2', 'kinfw-blog-grid-style-3' ]
		];

		$this->start_controls_section( 'section_readmore_btn_icon_style', [
            'label'     => esc_html__( 'Read More Button Icon', 'onnat-extra' ),
            'tab'       => KFCM::TAB_STYLE,
            'condition' => $condition,
		]);
            $this->add_responsive_control( 'readmore_btn_icon_wrapper_size', [
                'label'      => esc_html__('Wrapper Size', 'onnat-extra' ),
                'type'       => KFCM::SLIDER,
                'condition'  => $condition,
                'size_units' => [ 'px' ],
                'range'      => [
                    'px' => [
                        'min' => 0,
                        'max' => 500,
                    ],
                ],
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a .kinfw-icon' => 'line-height:{{SIZE}}{{UNIT}};height:{{SIZE}}{{UNIT}};width:{{SIZE}}{{UNIT}};',
                ],
            ]);
            $this->add_responsive_control( 'readmore_btn_icon_size', [
                'label'      => esc_html__('Icon Size', 'onnat-extra' ),
                'type'       => KFCM::SLIDER,
                'condition'  => $condition,
                'size_units' => [ 'px' ],
                'range'      => [
                    'px' => [
                        'min' => 0,
                        'max' => 999,
                    ],
                ],
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a .kinfw-icon' => 'font-size: {{SIZE}}{{UNIT}};'
                ],
            ]);
            $this->add_responsive_control( 'readmore_btn_icon_margin', [
                'label'      => esc_html__('Margin', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'condition'  => $condition,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a .kinfw-icon' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->add_responsive_control( 'readmore_btn_icon_padding', [
                'label'      => esc_html__('Padding', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'condition'  => $condition,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a .kinfw-icon' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
			$this->start_controls_tabs( 'readmore_btn_icon_customize_tabs', [
                'condition'  => $condition,
            ] );
                $this->start_controls_tab( 'readmore_btn_icon_normal_state', [
                    'label'     => esc_html__( 'Normal', 'onnat-extra' ),
                    'condition' => $condition,
                ]);
                    $this->add_control( 'readmore_btn_icon_color', [
                        'label'     => esc_html__('Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'condition' => $condition,
                        'selectors' => [
                            '{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a .kinfw-icon' => 'color:{{VALUE}}; fill:{{VALUE}};',
                        ],
                    ]);
					$this->add_group_control( KFGCBG::get_type(), [
						'name'           => 'readmore_btn_icon_bg',
						'types'          => ['classic', 'gradient'],
						'exclude'        => [ 'image'],
						'condition'      => $condition,
						'fields_options' => [
							'background' => [ 'default' => 'classic' ],
							'color'      => [ 'default' => '' ],
						],
						'selector'       => '{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a .kinfw-icon',
					]);
					$this->add_group_control( KFGCBR::get_type(), [
						'name'      => 'readmore_btn_icon_border',
						'condition' => $condition,
						'selector'  => '{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a .kinfw-icon',
					]);
					$this->add_responsive_control( 'readmore_btn_icon_border_radius', [
						'label'      => esc_html__('Border Radius', 'onnat-extra' ),
						'type'       => KFCM::DIMENSIONS,
						'condition' => $condition,
						'size_units' => [ 'px', '%' ],
						'selectors'  => [
							'{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a .kinfw-icon' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
						]
					]);
					$this->add_group_control( KFGCBS::get_type(), [
						'name'      => 'readmore_btn_icon_box_shadow',
						'condition' => $condition,
						'selector'  => '{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a .kinfw-icon'
					]);
                $this->end_controls_tab();
                $this->start_controls_tab( 'readmore_btn_icon_hover_state', [
                    'label'     => esc_html__( 'Hover', 'onnat-extra' ),
                    'condition' => $condition,
                ]);
                    $this->add_control( 'readmore_btn_icon_h_color', [
                        'label'     => esc_html__('Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'condition' => $condition,
                        'selectors' => [
                            '{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a:hover .kinfw-icon' => 'color:{{VALUE}}; fill:{{VALUE}};',
                        ],
                    ]);
                    $this->add_group_control( KFGCBG::get_type(), [
                        'name'           => 'readmore_btn_icon_h_bg',
                        'types'          => ['classic', 'gradient'],
                        'exclude'        => [ 'image'],
                        'condition'      => $condition,
                        'fields_options' => [
                            'background' => [ 'default' => 'classic' ],
                            'color'      => [ 'default' => '' ],
                        ],
                        'selector'       => '{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a:hover .kinfw-icon',
                    ]);
                    $this->add_group_control( KFGCBR::get_type(), [
                        'name'      => 'readmore_btn_icon_h_border',
                        'condition' => $condition,
                        'selector'  => '{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a:hover .kinfw-icon',
                    ]);
                    $this->add_responsive_control( 'readmore_btn_icon_h_border_radius', [
                        'label'      => esc_html__('Border Radius', 'onnat-extra' ),
                        'type'       => KFCM::DIMENSIONS,
                        'condition'  => $condition,  
                        'size_units' => [ 'px', '%' ],
                        'selectors'  => [
                            '{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a:hover .kinfw-icon' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                        ]
                    ]);
                    $this->add_group_control( KFGCBS::get_type(), [
                        'name'      => 'readmore_btn_icon_h_box_shadow',
                        'condition' => $condition,
                        'selector'  => '{{WRAPPER}} .kinfw-entry-footer .kinfw-entry-readmore a:hover .kinfw-icon',
                    ]);
                $this->end_controls_tab();
			$this->end_controls_tabs();
		$this->end_controls_section();
    }

	public function _blog_posts() {
		$result = [ esc_html__( '-- Select --', 'onnat-extra' ) ];

		$posts = get_posts([
			'post_type'      => 'post',
			'post_status'    => 'publish',
			'posts_per_page' => -1
		]);

		if ( ! empty( $posts ) && ! is_wp_error( $posts ) ) {
			foreach ( $posts as $item ) {
				$result[ $item->ID ] = $item->post_title;
			}
		}

		return $result;
	}

	public function _get_post_comments_popup_link( $post_id ) {
		$output = '';

		$post_link       = get_permalink( $post_id );
		$comments_number = get_comments_number( $post_id );

		if ( ! post_password_required( $post_id ) && ( comments_open( $post_id ) || $comments_number ) ) {

			$zero      = sprintf( '%1$s 0 %2$s', kinfw_icon( 'comment-single' ), esc_html__( 'Comment', 'onnat-extra' ) );
			$one       = sprintf( '%1$s 1 %2$s', kinfw_icon( 'comment-single' ), esc_html__( 'Comment', 'onnat-extra' ) );
			$more      = sprintf( '%1$s %2$s', kinfw_icon( 'comment-multiple' ), esc_html__( '% Comments', 'onnat-extra' ) );
			$css_class = '';
			$none      = sprintf( '%1$s %2$s', kinfw_icon( 'comment-off' ), esc_html__( 'Comments Off', 'onnat-extra' ) );

			$output .= '<div class="kinfw-meta-comments">';

				if ( 0 == $comments_number && !comments_open( $post_id ) && ! pings_open( $post_id ) ) {
					$output .= sprintf(
						'<span%1$s>%2$s</span>',
						! empty( $css_class ) ? ' class="' . esc_attr( $css_class ) . '"' : '',
						$none
					);
				} else {

					$comments_link   = get_comments_link();
					$link_attributes = '';

					if ( 0 == $comments_number ) {
						$respond_link  = $post_link . '#respond';
						$comments_link = apply_filters( 'respond_link', $respond_link, $post_id );
					}

					$link_attributes = apply_filters( 'comments_popup_link_attributes', $link_attributes );
					$output .= sprintf(
						'<a href="%1$s"%2$s%3$s>%4$s</a>',
						esc_url( $comments_link ),
						! empty( $css_class ) ? ' class="' . $css_class . '" ' : '',
						$link_attributes,
						get_comments_number_text( $zero, $one, $more, $post_id )
					);
				}

			$output .= '</div>';
		}

		return $output;
	}

	/**
	 * Get attachment image src.
	 */
	public static function _get_image_src( $attachment_id, $image_size_key, array $settings ) {
        return KFGCIS::get_attachment_image_src( $attachment_id, $image_size_key, $settings );
	}

	public function _get_image_title( $id ) {
		return get_the_title( $id );
	}

	public function _get_image_alt( $id ) {

		$attachment_id = $id;
		if ( ! $attachment_id ) {
			return '';
		}

		$attachment = get_post( $attachment_id );
		if ( ! $attachment ) {
			return '';
		}

		$alt = get_post_meta( $attachment_id, '_wp_attachment_image_alt', true );
		if ( ! $alt ) {
			$alt = $attachment->post_excerpt;
			if ( ! $alt ) {
				$alt = $attachment->post_title;
			}
		}
		return trim( strip_tags( $alt ) );
	}

	public function blog_post_grid_style_1( $post ) {
        $post_id    = $post['ID'];
        $post_title = get_the_title( $post_id );
        $post_link  = get_permalink( $post_id );
        $settings   = $this->get_settings();

		/**
		 * Media
		 */
			$media = '';

			if( has_post_thumbnail( $post_id ) ) {
				// $media = get_the_post_thumbnail( $post_id, 'full' );
                $thumbnail_id = get_post_thumbnail_id( $post_id );
                $thumbnail_url = get_the_post_thumbnail_url( 'full' );

                $image_src = $this->_get_image_src( $thumbnail_id, 'image_size', $settings );
                $image_src = empty($image_src) ? $thumbnail_url : $image_src;

                $image_title = $this->_get_image_title( $thumbnail_id );
                $image_title = empty( $image_title ) ? $post_title : $image_title;

                $image_alt = $this->_get_image_alt( $thumbnail_id );
                $image_alt = empty( $image_alt ) ? $image_title : $image_title;

                $media = sprintf( '
                    <img src="%1$s" alt="%2$s" class="wp-post-image %3$s">',
                    $image_src,
                    $image_alt,
                    $image_title
                );
			} else {
				$media = sprintf( '
					<img src="%1$s" alt="%2$s" class="kinfw-transparent-img">',
					get_theme_file_uri( 'assets/image/public/transparent.jpg' ),
					esc_attr( $post_title )
				);
			}

			printf( '
				<div class="kinfw-entry-media-wrap">
					<div class="kinfw-entry-media">%1$s</div>
					<div class="kinfw-meta-date">
						<time datetime="%2$s"> %3$s </time>
					</div>
				</div>',
				$media,
				esc_attr( get_the_date( 'c', $post_id ) ),
				get_the_date ('M d, Y', $post_id )
			);

		/**
		 * Author
		 */
			$author_id        = get_post_field( 'post_author', $post_id );
			$author_name      = get_the_author_meta( 'display_name', $author_id );
			$author_posts_url = get_author_posts_url( $author_id );

			$author = sprintf('
				<div class="kinfw-meta-author">
					<a href="%1$s" title="%2$s"> %3$s <span> %4$s </span> </a>
				</div>',
				esc_url( $author_posts_url ),
				sprintf( esc_html__( 'Posted by %1$s', 'onnat-extra' ), $author_name ),
				kinfw_icon( 'user-single' ),
				sprintf( esc_html__( 'by %1$s', 'onnat-extra' ), $author_name ),
			);

		/**
		 * Comment
		 */
			$comment = $this->_get_post_comments_popup_link( $post_id );

		/**
		 * Header
		 */
			$header = sprintf(
				'<header class="kinfw-entry-header">
					<h4 class="kinfw-entry-title">
						<a href="%1$s">%2$s</a>
					</h4>
				</header>',
				esc_url( $post_link ),
				esc_html( $post_title )
			);

		/**
		 * Footer
		 */
			$footer = sprintf(
				'<footer class="kinfw-entry-footer">
					<div class="kinfw-entry-readmore">
						<a href="%1$s"> %2$s %3$s </a>
					</div>
				</footer>',
				esc_url( $post_link ),
				esc_html__( 'Read More', 'onnat-extra' ),
				kinfw_icon( 'chevron-simple-right' ),
			);

		/**
		 * Content
		 */
			printf(
				'<div class="kinfw-entry-content-wrap">
					<div class="kinfw-entry-meta-wrap">
						%1$s %2$s
					</div>
					%3$s %4$s
				</div>',
				$author,
				$comment,
				$header,
				$footer
			);

	}

	public function blog_post_grid_style_2( $post ) {
        $post_id    = $post['ID'];
        $post_title = get_the_title( $post_id );
        $post_link  = get_permalink( $post_id );
        $settings   = $this->get_settings();

		/**
		 * Media
		 */
			$media = '';

			if( has_post_thumbnail( $post_id ) ) {
				// $media = get_the_post_thumbnail( $post_id, 'full' );
                $thumbnail_id = get_post_thumbnail_id( $post_id );
                $thumbnail_url = get_the_post_thumbnail_url( 'full' );

                $image_src = $this->_get_image_src( $thumbnail_id, 'image_size', $settings );
                $image_src = empty($image_src) ? $thumbnail_url : $image_src;

                $image_title = $this->_get_image_title( $thumbnail_id );
                $image_title = empty( $image_title ) ? $post_title : $image_title;

                $image_alt = $this->_get_image_alt( $thumbnail_id );
                $image_alt = empty( $image_alt ) ? $image_title : $image_title;

                $media = sprintf( '
                    <img src="%1$s" alt="%2$s" class="wp-post-image %3$s">',
                    $image_src,
                    $image_alt,
                    $image_title
                );
			} else {
				$media = sprintf( '
					<img src="%1$s" alt="%2$s" class="kinfw-transparent-img">',
					get_theme_file_uri( 'assets/image/public/transparent.jpg' ),
					esc_attr( $post_title )
				);
			}

			printf( '
				<div class="kinfw-entry-media-wrap">
					<div class="kinfw-entry-media">%1$s</div>
				</div>',
				$media,
			);

		/**
		 * Author
		 */
			$author_id        = get_post_field( 'post_author', $post_id );
			$author_name      = get_the_author_meta( 'display_name', $author_id );
			$author_posts_url = get_author_posts_url( $author_id );

			$author = sprintf(
				'<div class="kinfw-meta-author">
					<a href="%1$s" title="%2$s"> %3$s <span> %4$s </span> </a>
				</div>',
				esc_url( $author_posts_url ),
				sprintf( esc_html__( 'Posted by %1$s', 'onnat-extra' ), $author_name ),
				kinfw_icon( 'user-single' ),
				sprintf( esc_html__( 'by %1$s', 'onnat-extra' ), $author_name ),
			);

		/**
		 * Date
		 */
			$date = sprintf(
				'<div class="kinfw-meta-date">
					%1$s
					<time datetime="%2$s"> %3$s </time>
				</div>',
				kinfw_icon( 'misc-calendar' ),
				esc_attr( get_the_date( 'c', $post_id ) ),
				get_the_date ('M d, Y', $post_id )
			);

		/**
		 * Header
		 */
			$header = sprintf(
				'<header class="kinfw-entry-header">
					<h4 class="kinfw-entry-title">
						<a href="%1$s">%2$s</a>
					</h4>
				</header>',
				esc_url( $post_link ),
				esc_html( $post_title )
			);

		/**
		 * Footer
		 */
			$footer = sprintf(
				'<footer class="kinfw-entry-footer">
					<div class="kinfw-entry-readmore">
						<a href="%1$s"> %2$s </a>
					</div>
				</footer>',
				esc_url( $post_link ),
				kinfw_icon( 'chevron-simple-right' ),
			);

		/**
		 * Content
		 */
		printf(
			'<div class="kinfw-entry-content-wrap">
				<div class="kinfw-entry-meta-wrap">
					%1$s %2$s
				</div>
				%3$s %4$s
			</div>',
			$author,
			$date,
			$header,
			$footer
		);
	}

    public function blog_post_grid_style_3( $post ) {
		$post_id    = $post['ID'];
		$post_title = get_the_title( $post_id );
		$post_link  = get_permalink( $post_id );
        $settings = $this->get_settings();

		/**
		 * Media
		 */
            $media = '';
            if( has_post_thumbnail( $post_id ) ) {
                $thumbnail_id = get_post_thumbnail_id( $post_id );
                $thumbnail_url = get_the_post_thumbnail_url( 'full' );

                $image_src = $this->_get_image_src( $thumbnail_id, 'image_size', $settings );
                $image_src = empty($image_src) ? $thumbnail_url : $image_src;

                $image_title = $this->_get_image_title( $thumbnail_id );
                $image_title = empty( $image_title ) ? $post_title : $image_title;

                $image_alt = $this->_get_image_alt( $thumbnail_id );
                $image_alt = empty( $image_alt ) ? $image_title : $image_title;

                printf('
                    <div class="kinfw-entry-media-wrap">
                        <div class="kinfw-entry-media">
                            <img src="%1$s" alt="%2$s" title="%3$s">
                        </div>
                    </div>',
                    $image_src,
                    $image_alt,
                    $image_title
                );
            }

		/**
		 * Category
		 */
            $category   = '';
            $categories = get_the_category( $post_id );

            if( !empty( $categories ) ) {
                $first_category = $categories[0];
                $category = sprintf('
                    <div class="kinfw-meta-cat">
                        <a href="%1$s" title="%2$s">%2$s</a>
                    </div>',
                    esc_url( get_category_link( $first_category->term_id ) ),
                    esc_html( $first_category->name )
                );
            }                


		/**
		 * Date
		 */
            $date = sprintf(
                '<div class="kinfw-meta-date">
                    <time datetime="%1$s"> %2$s </time>
                </div>',
                esc_attr( get_the_date( 'c', $post_id ) ),
                get_the_date ('M d, Y', $post_id )
            );

		/**
		 * Header
		 */
            $header = sprintf(
                '<header class="kinfw-entry-header">
                    <h4 class="kinfw-entry-title">
                        <a href="%1$s">%2$s</a>
                    </h4>
                </header>',
                esc_url( $post_link ),
                esc_html( $post_title )
            );

        /**
         * Footer
         */
            $footer = sprintf(
                '<footer class="kinfw-entry-footer">
                    <div class="kinfw-entry-readmore">
                        <a href="%1$s"> %2$s </a>
                    </div>
                </footer>',
                esc_url( $post_link ),
                kinfw_icon( 'onnat-line-arrow-long-right-cross' ),
            );


		/**
		 * Content
		 */
            printf('
                %4$s
                <div class="kinfw-entry-content-wrap">
                    <div class="kinfw-entry-meta-wrap">
                        %1$s %2$s
                    </div>
                    %3$s
                </div>',
                $category,
                $date,
                $header,
                $footer
            );

    }

    public function blog_post_grid_style_4( $post ) {
        $post_id    = $post['ID'];
        $post_title = get_the_title( $post_id );
        $post_link  = get_permalink( $post_id );
        $settings   = $this->get_settings();

		/**
		 * Media
		 */
            $media = '';

            if( has_post_thumbnail( $post_id ) ) {
                // $media = get_the_post_thumbnail( $post_id, 'full' );
                $thumbnail_id = get_post_thumbnail_id( $post_id );
                $thumbnail_url = get_the_post_thumbnail_url( 'full' );

                $image_src = $this->_get_image_src( $thumbnail_id, 'image_size', $settings );
                $image_src = empty($image_src) ? $thumbnail_url : $image_src;

                $image_title = $this->_get_image_title( $thumbnail_id );
                $image_title = empty( $image_title ) ? $post_title : $image_title;

                $image_alt = $this->_get_image_alt( $thumbnail_id );
                $image_alt = empty( $image_alt ) ? $image_title : $image_title;

                $media = sprintf( '
                    <img src="%1$s" alt="%2$s" class="wp-post-image %3$s">',
                    $image_src,
                    $image_alt,
                    $image_title
                );
            } else {
                $media = sprintf( '
                    <img src="%1$s" alt="%2$s" class="kinfw-transparent-img">',
                    get_theme_file_uri( 'assets/image/public/transparent.jpg' ),
                    esc_attr( $post_title )
                );
            }

            printf( '
                <div class="kinfw-entry-media-wrap">
                    <div class="kinfw-entry-media">%1$s</div>
                </div>',
                $media,
            );

		/**
		 * Date
		 */
            $date = sprintf(
                '<div class="kinfw-meta-date">
                    <time datetime="%1$s">
                        <span class="kinfw-meta-day">%2$s</span><span class="kinfw-meta-month">%3$s</span>
                    </time>
                </div>',
                esc_attr( get_the_date( 'c', $post_id ) ),
                get_the_date ('d', $post_id ),
                get_the_date ('/M', $post_id )
            );
            
		/**
		 * Comment
		 */
            $comment = $this->_get_post_comments_popup_link( $post_id );            

		/**
		 * Header
		 */
            $header = sprintf(
                '<header class="kinfw-entry-header">
                    <h4 class="kinfw-entry-title">
                        <a href="%1$s">%2$s</a>
                    </h4>
                </header>',
                esc_url( $post_link ),
                esc_html( $post_title )
            );

		/**
		 * Content
		 */
            printf(
                '<div class="kinfw-entry-content-wrap">
                    <div class="kinfw-entry-meta-wrap">
                        %1$s %2$s
                    </div>
                    %3$s
                </div>',
            $date,
            $comment,
            $header
        );
    }

	/**
	 * Render widget output on the frontend.
	 */
    protected function render() {
		$render = kf_extra_plugin_elementor()->get_template("kfw-blog-grid");
		include $render;
    }
}