<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

use \Elementor\Plugin as KFEl;
use \Elementor\Widget_Base as WB;
use \Elementor\Controls_Manager as KFCM;
use \Elementor\Repeater as KFRTR;
use \Elementor\Icons_Manager as KFIM;
use \Elementor\Group_Control_Typography as KFGCT;
use \Elementor\Group_Control_Background as KFGCBG;
use \Elementor\Group_Control_Border as KFGCBR;

class KFW_Blog_Single_Post_Meta extends WB {

	/**
	 * Get element name.
	 *
	 * Retrieve the element name.
	 *
	 * @return string The name.
	 */
	public function get_name() {
        return 'kfw-blog-single-post-meta';
    }

	/**
	 * Get element title.
	 *
	 * Retrieve the element title.
	 *
	 * @return string Element title.
	 */
	public function get_title() {
		return esc_html__('Post Meta', 'onnat-extra');
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the widget categories.
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ '_kfw_blog_post_look_widgets' ];
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve the widget icon.
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'kfw-badge eicon-meta-data';
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the widget keywords.
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'kfw', 'kin', 'blog', 'post', 'single', 'meta' ];
	}

	/**
	 * Get widget help url.
	 *
	 * Retrieve the widget help url.
	 *
	 * @return string Widget help url.
	 */
	public function get_custom_help_url() {
        return KFE_CONST_THEME_DOC . 'elementor-custom-widgets/blog-single-post-meta';
    }

	/**
	 * Get HTML wrapper class.
     *
	 * Retrieve the widget container class.
     *
     * @return string Widget container css classes.
	 */
	public function get_html_wrapper_class() {
        $name = $this->get_name();

		return sprintf( 'kfw-ele-%s elementor-widget-%s', $name, $name );
	}

	/**
	 * Widget base constructor.
	 *
	 * Initializing the widget base class.
	 */
	public function __construct( $data = [], $args = null ) {
        parent::__construct( $data, $args );

        wp_register_style( 'kfw-blog-single',
            KFE_CONST_URL . 'libraries/elementor/assets/css/blog-single' . KFE_CONST_DEBUG_SUFFIX . '.css',
            [],
            KFE_CONST_VERSION
        );

    }

	/**
	 * Get style dependencies.
	 *
	 * Retrieve the list of style dependencies the element requires.
	 *
	 * @return array Element styles dependencies.
	 */
	public function get_style_depends() {
		return [ 'kfw-blog-single' ];
	}

	/**
	 * Get script dependencies.
	 *
	 * Retrieve the list of script dependencies the element requires.
	 *
	 * @return array Element scripts dependencies.
	 */
	public function get_script_depends() {
		return [];
	}

	/**
	 * Register controls.
	 *
	 * Used to add new controls to any element type.
	 */
	protected function register_controls() {
		$this->_register_meta_section();
	}

	public function _register_meta_section() {
		$repeater = new KFRTR();

		$this->start_controls_section( 'section_meta', [
			'label' => esc_html__( 'Available Meta', 'onnat-extra' )
		]);
			$repeater->start_controls_tabs('meta_tabs');
				$repeater->start_controls_tab('meta_content',[
					'label' => esc_html__( 'Content', 'onnat-extra' ),
				]);
					$repeater->add_control( 'meta', [
						'label'   => esc_html__( 'Meta Source', 'onnat-extra' ),
						'type'    => KFCM::CHOOSE,
						'options' => [
							'author'   => [
								'title' => esc_html__('Author', 'onnat-extra'),
								'icon'  => 'eicon-person',
							],
							'category' => [
								'title' => esc_html__('Category', 'onnat-extra'),
								'icon'  => 'eicon-sitemap',
							],
							'date'     => [
								'title' => esc_html__('Date', 'onnat-extra'),
								'icon'  => 'eicon-calendar',
							],
							'comment'  => [
								'title' => esc_html__('Comments', 'onnat-extra'),
								'icon'  => 'eicon-comments',
							],
							'reading'  => [
								'title' => esc_html__('Reading Time', 'onnat-extra'),
								'icon'  => 'eicon-countdown',
							],
							'tag'      => [
								'title' => esc_html__('Tag', 'onnat-extra'),
								'icon'  => 'eicon-tags',
							],
						],
						'default' => 'author',
						'toggle'  => false,
					]);
					$repeater->add_control('icon_type',[
						'label'   => esc_html__( 'Icon Type', 'onnat-extra' ),
						'type'    => KFCM::CHOOSE,
						'options' => [
							'blank'   => [
								'title' => esc_html__('Blank', 'onnat-extra'),
								'icon'  => '',
							],
							'default' => [
								'title' => esc_html__('Default', 'onnat-extra'),
								'icon'  => 'eicon-theme-style',
							],
							'icon'    => [
								'title' => esc_html__('Icon', 'onnat-extra'),
								'icon'  => 'eicon-plus-circle-o',
							],
						],
						'default' => 'default',
						'toggle'  => false,
					]);
					$repeater->add_control( 'icon', [
						'label'     => esc_html__( 'Icon', 'onnat-extra' ),
						'type'      => KFCM::ICONS,
						'default'   => [
							'value'   => 'far fa-window-minimize',
							'library' => 'fa-regular',
						],
						'condition' => [
							'icon_type' => 'icon'
						],
					]);
					$repeater->add_control( 'comments-icon', [
						'label'     => esc_html__( 'Comments Icon', 'onnat-extra' ),
						'type'      => KFCM::ICONS,
						'default'   => [
							'value'   => 'far fa-comments',
							'library' => 'fa-regular',
						],
						'condition' => [
							'icon_type' => 'icon',
							'meta'      => 'comment',
						],
					]);
					$repeater->add_control( 'comment-off-icon', [
						'label'     => esc_html__( 'Comment Off', 'onnat-extra' ),
						'type'      => KFCM::ICONS,
						'default'   => [
							'value'   => 'fas fa-comment-slash',
							'library' => 'fa-solid',
						],
						'condition' => [
							'icon_type' => 'icon',
							'meta'      => 'comment',
						],
					]);
					$repeater->add_control( 'author', [
						'label'     => esc_html__( 'Prefix', 'onnat-extra' ),
						'type'      => KFCM::TEXT,
						'default'   => esc_html__('By', 'onnat-extra'),
						'condition' => [
							'meta' => 'author'
						],
					]);
					$repeater->add_control( 'tag', [
						'label'     => esc_html__( 'Prefix', 'onnat-extra' ),
						'type'      => KFCM::TEXT,
						'default'   => esc_html__('Posted In:', 'onnat-extra'),
						'condition' => [
							'meta' => 'tag'
						],
					]);
					$repeater->add_control( 'tag-style', [
						'label'     => esc_html__( 'Style', 'onnat-extra' ),
						'type'      => KFCM::CHOOSE,
						'options'   => [
							'meta-tag-type-1' => [
								'title' => esc_html__('Type 1', 'onnat-extra'),
								'icon'  => 'kfw-ctrl-icon kfw-ctrl-type-1-icon'
							],
							'meta-tag-type-2' => [
								'title' => esc_html__('Type 2', 'onnat-extra'),
								'icon'  => 'kfw-ctrl-icon kfw-ctrl-type-2-icon'
							],
						],
						'toggle'    => false,
						'default'   => 'meta-tag-type-1',
						'condition' => [
							'meta' => 'tag'
						],
					]);
					$repeater->add_control( 'reading', [
						'label'     => esc_html__( 'Postfix', 'onnat-extra' ),
						'type'      => KFCM::TEXT,
						'default'   => esc_html__('read', 'onnat-extra'),
						'condition' => [
							'meta' => 'reading'
						],
					]);
				$repeater->end_controls_tab();

				$repeater->start_controls_tab('meta_typo',[
					'label' => esc_html__( 'Typo', 'onnat-extra' ),
				]);
					$repeater->add_group_control(KFGCT::get_type(), [
						'name'     => 'meta_typo',
						'label'    => esc_html__( 'Typography', 'onnat-extra' ),
						'selector' => '{{WRAPPER}} .kfw-ele-widget-post-meta-wrap {{CURRENT_ITEM}}, {{WRAPPER}} .kfw-ele-widget-post-meta-wrap {{CURRENT_ITEM}}.kinfw-entry-post-tag-list.meta-tag-type-2 p',
					]);
					$repeater->add_control( 'typo_color_settings', [
						'label'        => esc_html__('Colors', 'onnat-extra'),
						'type'         => KFCM::POPOVER_TOGGLE,
						'return_value' => 'yes',
					]);
					$repeater->start_popover();
						$repeater->add_control( 'typo_color', [
							'label'     => esc_html__('Normal Color', 'onnat-extra' ),
							'type'      => KFCM::COLOR,
							'selectors' => [
								'{{WRAPPER}} .kfw-ele-widget-post-meta-wrap {{CURRENT_ITEM}}, {{WRAPPER}} .kfw-ele-widget-post-meta-wrap {{CURRENT_ITEM}} a, {{WRAPPER}} .kfw-ele-widget-post-meta-wrap {{CURRENT_ITEM}}.kinfw-entry-post-tag-list.meta-tag-type-2 p' => 'color:{{VALUE}};',
							],
						]);
						$repeater->add_control( 'typo_h_color', [
							'label'     => esc_html__('Hover Color', 'onnat-extra' ),
							'type'      => KFCM::COLOR,
							'selectors' => [
								'{{WRAPPER}} .kfw-ele-widget-post-meta-wrap {{CURRENT_ITEM}} a:hover' => 'color:{{VALUE}};',
							],
						]);
					$repeater->end_popover();
				$repeater->end_controls_tab();

				$tag_wrap_condition = [
					'meta'      => 'tag',
					'tag-style' => 'meta-tag-type-2',
				];
				$repeater->start_controls_tab('tag_meta_wrap',[
					'label'     => esc_html__( 'Tag', 'onnat-extra' ),
					'condition' => $tag_wrap_condition,
				]);
					$repeater->add_group_control(KFGCT::get_type(), [
						'name'      => 'tag_meta_typo',
						'label'     => esc_html__( 'Typography', 'onnat-extra' ),
						'condition' => $tag_wrap_condition,
						'selector'  => '{{WRAPPER}} .kinfw-entry-post-tag-list.meta-tag-type-2 .kinfw-post-tags-list li a',
					]);
					$repeater->add_control('hr_tag_wrap_normal_heading',[
						'label'     => esc_html__( 'NORMAL', 'onnat-extra' ),
						'type'      => KFCM::HEADING,
						'condition' => $tag_wrap_condition,
					]);
					$repeater->add_control('hr_tag_wrap_1', [
						'type'      => KFCM::DIVIDER,
						'condition' => $tag_wrap_condition,
					]);
					$repeater->add_control( 'tag_typo_color', [
						'label'     => esc_html__('Text Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'condition' => $tag_wrap_condition,
						'selectors' => [
							'{{WRAPPER}} .kinfw-entry-post-tag-list.meta-tag-type-2 .kinfw-post-tags-list li a' => 'color:{{VALUE}};',
						],
					]);
					$repeater->add_group_control( KFGCBG::get_type(), [
						'name'           => 'tag_wrap_bg',
						'types'          => ['classic', 'gradient'],
						'exclude'        => ['image'],
						'condition'      => $tag_wrap_condition,
						'fields_options' => [
							'background' => [ 'default' => 'classic' ],
							'color'      => [ 'default' => '' ],
						],
						'selector'       => '{{WRAPPER}} .kinfw-entry-post-tag-list.meta-tag-type-2 .kinfw-post-tags-list li a',
					]);
					$repeater->add_group_control( KFGCBR::get_type(), [
						'name'      => 'tag_wrap_border',
						'condition' => $tag_wrap_condition,
						'selector'  => '{{WRAPPER}} .kinfw-entry-post-tag-list.meta-tag-type-2 .kinfw-post-tags-list li a',
					]);
					$repeater->add_responsive_control( 'item_border_radius', [
						'label'      => esc_html__('Border Radius', 'onnat-extra' ),
						'type'       => KFCM::DIMENSIONS,
						'condition'  => $tag_wrap_condition,
						'size_units' => [ 'px', '%' ],
						'selectors'  => [
							'{{WRAPPER}} .kinfw-entry-post-tag-list.meta-tag-type-2 .kinfw-post-tags-list li a' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
						]
					]);
                    $repeater->add_control('hr_tag_wrap_hover_heading',[
						'label'     => esc_html__( 'HOVER', 'onnat-extra' ),
						'type'      => KFCM::HEADING,
						'condition' => $tag_wrap_condition,
                    ]);
					$repeater->add_control('hr_tag_wrap_2', [
						'type'      => KFCM::DIVIDER,
						'condition' => $tag_wrap_condition,
					]);
					$repeater->add_control( 'tag_typo_h_color', [
						'label'     => esc_html__('Text Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'condition' => $tag_wrap_condition,
						'selectors' => [
							'{{WRAPPER}} .kinfw-entry-post-tag-list.meta-tag-type-2 .kinfw-post-tags-list li a:hover' => 'color:{{VALUE}};',
						],
					]);
					$repeater->add_group_control( KFGCBG::get_type(), [
						'name'           => 'tag_wrap_h_bg',
						'types'          => ['classic', 'gradient'],
						'exclude'        => ['image'],
						'condition'      => $tag_wrap_condition,
						'fields_options' => [
							'background' => [ 'default' => 'classic' ],
							'color'      => [ 'default' => '' ],
						],
						'selector'       => '{{WRAPPER}} .kinfw-entry-post-tag-list.meta-tag-type-2 .kinfw-post-tags-list li a:hover',
					]);
					$repeater->add_group_control( KFGCBR::get_type(), [
						'name'      => 'tag_wrap_h_border',
						'condition' => $tag_wrap_condition,
						'selector'  => '{{WRAPPER}} .kinfw-entry-post-tag-list.meta-tag-type-2 .kinfw-post-tags-list li a:hover',
					]);
					$repeater->add_responsive_control( 'item_h_border_radius', [
						'label'      => esc_html__('Border Radius', 'onnat-extra' ),
						'type'       => KFCM::DIMENSIONS,
						'condition'  => $tag_wrap_condition,
						'size_units' => [ 'px', '%' ],
						'selectors'  => [
							'{{WRAPPER}} .kinfw-entry-post-tag-list.meta-tag-type-2 .kinfw-post-tags-list li a:hover' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
						]
					]);
				$repeater->end_controls_tab();

				$repeater->start_controls_tab('meta_icon',[
					'label'     => esc_html__( 'Icon', 'onnat-extra' ),
					'condition' => [
						'icon_type' => 'icon'
					]
				]);
					$repeater->add_responsive_control( 'icon_size', [
						'label'      => esc_html__('Size', 'onnat-extra' ),
						'type'       => KFCM::SLIDER,
						'size_units' => [ 'px' ],
						'range'      => [
							'px' => [
								'min' => 0,
								'max' => 200,
							],
						],
						'selectors'  => [
							'{{WRAPPER}} .kfw-ele-widget-post-meta-wrap {{CURRENT_ITEM}} span' => 'font-size: {{SIZE}}{{UNIT}};'
						],
					]);
					$repeater->add_responsive_control( 'position', [
                        'label'     => esc_html__('Postion', 'onnat-extra'),
                        'type'      => KFCM::SLIDER,
                        'range'     => [
                            'px' 	=> [
                                'min'  => -50,
                                'max'  => 50,
                                'step' => 1,
                            ],
                        ],
                        'selectors' => [
                            '{{WRAPPER}} .kfw-ele-widget-post-meta-wrap {{CURRENT_ITEM}} span' => 'top:{{SIZE}}{{UNIT}};'
                        ]
                    ]);
					$repeater->add_responsive_control( 'gap', [
                        'label'     => esc_html__('Gap', 'onnat-extra'),
                        'type'      => KFCM::SLIDER,
                        'range'     => [
                            'px' 	=> [
                                'min'  => 1,
                                'max'  => 100,
                                'step' => 1,
                            ],
                        ],
                        'selectors' => [
                            '{{WRAPPER}} .kfw-ele-widget-post-meta-wrap {{CURRENT_ITEM}} span' => 'margin-right:{{SIZE}}{{UNIT}};'
                        ]
                    ]);
					$repeater->add_control( 'icon_color_settings', [
						'label'        => esc_html__('Colors', 'onnat-extra'),
						'type'         => KFCM::POPOVER_TOGGLE,
						'return_value' => 'yes',
					]);
					$repeater->start_popover();
						$repeater->add_control( 'icon_color', [
							'label'     => esc_html__('Normal Color', 'onnat-extra' ),
							'type'      => KFCM::COLOR,
							'selectors' => [
								'{{WRAPPER}} .kfw-ele-widget-post-meta-wrap {{CURRENT_ITEM}} span'     => 'color:{{VALUE}};',
								'{{WRAPPER}} .kfw-ele-widget-post-meta-wrap {{CURRENT_ITEM}} span svg' => 'fill:{{VALUE}};',
							],
						]);
						$repeater->add_control( 'icon_h_color', [
							'label'     => esc_html__('Hover Color', 'onnat-extra' ),
							'type'      => KFCM::COLOR,
							'selectors' => [
								'{{WRAPPER}} .kfw-ele-widget-post-meta-wrap {{CURRENT_ITEM}}:hover span'     => 'color:{{VALUE}};',
								'{{WRAPPER}} .kfw-ele-widget-post-meta-wrap {{CURRENT_ITEM}}:hover span svg' => 'fill:{{VALUE}};',
							],
						]);
					$repeater->end_popover();
				$repeater->end_controls_tab();
			$repeater->end_controls_tabs();


            $this->add_control('items', [
				'type'    => KFCM::REPEATER,
				'fields'  => $repeater->get_controls(),
				'default' => [],
            ]);
		$this->end_controls_section();
	}

    public function _is_editor_mode() {
        $preview = KFEl::instance()->preview->is_preview_mode();
        $editor  = KFEl::instance()->editor->is_edit_mode();

        return ( $preview || $editor ) ? true : false;
    }

	public function render_author_info( $settings, $post_id ){
		$author_id        = get_post_field( 'post_author', $post_id );
		$author_name      = get_the_author_meta( 'display_name', $author_id );
		$author_posts_url = get_author_posts_url( $author_id );

		echo '<div class="kinfw-meta-author elementor-repeater-item-' . esc_attr( $settings['_id'] ). '">';
			echo '<a href="'.esc_url( $author_posts_url ).'" title="'. sprintf( esc_html__( '%1$s %2$s', 'onnat-extra' ), $settings['author'], $author_name ) .'">';
				if( $settings['icon_type'] === 'default' ) {
					echo kinfw_icon( 'user-single' );
				} else if( $settings['icon_type'] === 'icon' ) {
					$this->_render_icon( $settings['icon'] );
				}
				printf( esc_html__( '%1$s %2$s', 'onnat-extra' ), $settings['author'], $author_name );
			echo '</a>';
		echo '</div>';
	}

	public function render_category_info( $settings, $post_id ){
		$categories_list = get_the_category_list(', ', '', $post_id );
		if ( $categories_list ) {
			echo '<div class="kinfw-meta-cat elementor-repeater-item-' . esc_attr( $settings['_id'] ). '">';
				if( $settings['icon_type'] === 'default' ) {
					echo kinfw_icon( 'misc-layers' );
				} else if( $settings['icon_type'] === 'icon' ) {
					$this->_render_icon( $settings['icon'] );
				}
				printf( '<ul class="kinfw-post-cats-list"> <li> %1$s </li> </ul>', str_replace( ', ', ', </li> <li>', $categories_list ) );
			echo '</div>';
		}
	}

	public function render_date_info( $settings, $post_id ){
		echo '<div class="kinfw-meta-date elementor-repeater-item-' . esc_attr( $settings['_id'] ). '">';
			if( $settings['icon_type'] === 'default' ) {
				echo kinfw_icon( 'misc-calendar' );
			} else if( $settings['icon_type'] === 'icon' ) {
				$this->_render_icon( $settings['icon'] );
			}
			printf( '<time datetime="%1$s"> %2$s </time>', esc_attr( get_the_date( 'c', $post_id ) ), get_the_date ( get_option('date_format'), $post_id ) );
		echo '</div>';
	}

	public function render_comment_info( $settings, $post_id ){
		if ( ! post_password_required( $post_id ) && ( comments_open( $post_id ) || get_comments_number( $post_id ) ) ) {

			$css_class       = '';
			$post_link       = get_permalink( $post_id );
			$comments_number = get_comments_number( $post_id );

			$zero = $one = $more = $none = '';

			if( $settings['icon_type'] === 'default' ) {
				$zero = sprintf( '%1$s 0 %2$s', kinfw_icon( 'comment-single' ), esc_html__( 'Comment', 'onnat-extra' ) );
                $one  = sprintf( '%1$s 1 %2$s', kinfw_icon( 'comment-single' ), esc_html__( 'Comment', 'onnat-extra' ) );
                $more = sprintf( '%1$s %2$s', kinfw_icon( 'comment-multiple' ), esc_html__( '% Comments', 'onnat-extra' ) );
                $none = sprintf( '%1$s %2$s', kinfw_icon( 'comment-off' ), esc_html__( 'Comments Off', 'onnat-extra' ) );
			} else {
				$zero = sprintf( '%1$s 0 %2$s', $this->_render_icon( $settings['icon'], false ), esc_html__( 'Comment', 'onnat-extra' ) );
                $one  = sprintf( '%1$s 1 %2$s', $this->_render_icon( $settings['icon'], false ), esc_html__( 'Comment', 'onnat-extra' ) );
                $more = sprintf( '%1$s %2$s', $this->_render_icon( $settings['comments-icon'], false ), esc_html__( '% Comments', 'onnat-extra' ) );
                $none = sprintf( '%1$s %2$s', $this->_render_icon( $settings['comment-off-icon'], false ), esc_html__( 'Comments Off', 'onnat-extra' ) );
			}

			echo '<div class="kinfw-meta-comments elementor-repeater-item-' . esc_attr( $settings['_id'] ). '">';
				if ( 0 == $comments_number && !comments_open( $post_id ) && ! pings_open( $post_id ) ) {
					printf( '<span%1$s>%2$s</span>',
						! empty( $css_class ) ? ' class="' . esc_attr( $css_class ) . '"' : '',
						$none
					);
				} else {
					$comments_link   = get_comments_link();
					$link_attributes = '';

					if ( 0 == $comments_number ) {
						$respond_link  = $post_link . '#respond';
						$comments_link = apply_filters( 'respond_link', $respond_link, $post_id );
					}

					$link_attributes = apply_filters( 'comments_popup_link_attributes', $link_attributes );

					printf( '<a href="%1$s"%2$s%3$s>%4$s</a>',
						esc_url( $comments_link ),
						! empty( $css_class ) ? ' class="' . $css_class . '" ' : '',
						$link_attributes,
						get_comments_number_text( $zero, $one, $more, $post_id )
					);
				}
			echo '</div>';
		}
	}

	public function render_reading_info( $settings, $post_id ){
		$content       = get_the_content( null, false, $post_id );
		$clean_content = esc_html( $content );
		$word_count    = str_word_count( $clean_content );
		$time          = ceil( $word_count / 200 );

		$icon = '';
		if( $settings['icon_type'] === 'default' ) {
			$icon = kinfw_icon( 'misc-eye' );
		} else if( $settings['icon_type'] === 'icon' ) {
			$icon = $this->_render_icon( $settings['icon'], false );
		}

		echo '<div class="kinfw-meta-reading-time elementor-repeater-item-' . esc_attr( $settings['_id'] ). '">';
			printf(
				'%1$s %2$s %3$s',
				$icon,
				$time,
				( $time<=1 ) ? sprintf( esc_html__('minute %1$s', 'onnat-extra' ), $settings['reading'] ) : sprintf( esc_html__('minutes %1$s', 'onnat-extra' ), $settings['reading'] )
			);
		echo '</div>';
	}

	public function render_tag_info( $settings, $post_id ){

		$sep   =  ( $settings['tag-style'] === 'meta-tag-type-1' ) ? ', </li> <li>' : '</li> <li>';
		$after = '</li></ul>';

		if( !empty( $settings['tag'] ) ) {
			$tags_list = get_the_tag_list(
				sprintf( '<p>%s</p> <ul class="kinfw-post-tags-list"><li>', $settings['tag'] ),
				$sep,
				$after,
				$post_id
			);
		} else {
			$tags_list = get_the_tag_list(
				'<ul class="kinfw-post-tags-list"><li>',
				$sep,
				$after,
				$post_id
			);
		}

		if( $tags_list ) {

			$icon = '';
			if( $settings['icon_type'] === 'default' ) {
				$icon = kinfw_icon( 'tag-multiple' );
			} else if( $settings['icon_type'] === 'icon' ) {
				$icon = $this->_render_icon( $settings['icon'], false );
			}

            printf( '
				<div class="kinfw-entry-post-tag-list %1$s %2$s">  %3$s %4$s </div>',
				$settings['tag-style'],
				'elementor-repeater-item-' . esc_attr( $settings['_id'] ),
				$icon,
				$tags_list,
			);

		}
	}

	public function _render_icon( $icon , $echo = true ) {
		$icon = array_filter( $icon );

		if( !empty( $icon ) ) {
			if( $echo ) {
				echo '<span class="kfw-elementor-icon-wrap">';
					KFIM::render_icon( $icon, [ 'aria-hidden' => 'true' ] );
				echo '</span>';
			} else {
				return '<span class="kfw-elementor-icon-wrap">'. KFIM::try_get_icon_html( $icon, [ 'aria-hidden' => 'true' ] ) . '</span>';
			}
		}
    }

	/**
	 * Render widget output on the frontend.
	 */
    protected function render() {
		$render = kf_extra_plugin_elementor()->get_template("kfw-blog-single-post-meta");
		include $render;
    }
}