<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

use \Elementor\Widget_Base as WB;
use \Elementor\Controls_Manager as KFCM;
use \Elementor\Icons_Manager as KFIM;
use \Elementor\Repeater as KFRTR;

use \Elementor\Group_Control_Typography as KFGCT;
use \Elementor\Group_Control_Text_Shadow as KFGCTS;
use \Elementor\Group_Control_Text_Stroke as KFGCTST;
use \Elementor\Group_Control_Background as KFGCBG;
use \Elementor\Group_Control_Border as KFGCBR;
use \Elementor\Group_Control_Box_Shadow as KFGCBS;

class KFW_CPT_Service_Grid extends WB {

	/**
	 * Get element name.
	 *
	 * Retrieve the element name.
	 *
	 * @return string The name.
	 */
	public function get_name() {
        return 'kfw-cpt-service-grid';
    }

	/**
	 * Get element title.
	 *
	 * Retrieve the element title.
	 *
	 * @return string Element title.
	 */
	public function get_title() {
		return esc_html__('Service Grid', 'onnat-extra');
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the widget categories.
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ '_kfw_theme_widgets' ];
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve the widget icon.
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'kfw-badge eicon-gallery-grid kfw-service-grid';
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the widget keywords.
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'kfw', 'kin', 'service', 'post', 'grid' ];
	}

	/**
	 * Get widget help url.
	 *
	 * Retrieve the widget help url.
	 *
	 * @return string Widget help url.
	 */
	public function get_custom_help_url() {
        return KFE_CONST_THEME_DOC . 'elementor-custom-widgets/service-grid';
    }

	/**
	 * Get HTML wrapper class.
     *
	 * Retrieve the widget container class.
     *
     * @return string Widget container css classes.
	 */
	public function get_html_wrapper_class() {
        $name = $this->get_name();

		return sprintf( 'kfw-ele-%s elementor-widget-%s', $name, $name );
	}

	/**
	 * Widget base constructor.
	 *
	 * Initializing the widget base class.
	 */
	public function __construct( $data = [], $args = null ) {
        parent::__construct( $data, $args );

		wp_register_style( 'kfw-cpt-service-grid',
			KFE_CONST_URL . 'libraries/elementor/assets/css/cpt-service-grid' . KFE_CONST_DEBUG_SUFFIX . '.css',
			[],
			KFE_CONST_VERSION
		);
    }

	/**
	 * Get style dependencies.
	 *
	 * Retrieve the list of style dependencies the element requires.
	 *
	 * @return array Element styles dependencies.
	 */
	public function get_style_depends() {
		return [ 'kfw-cpt-service-grid' ];
	}

	/**
	 * Register controls.
	 *
	 * Used to add new controls to any element type.
	 */
	protected function register_controls() {
		$this->_register_items_section();
        $this->_register_settings_section();
        $this->_register_content_wrap_style_section();
        $this->_register_term_style_section();
        $this->_register_title_style_section();
		$this->_register_icon_style_section();
        $this->_register_content_style_section();
        $this->_register_readmore_btn_style_section();
    }

    public function _register_items_section() {
		$repeater = new KFRTR();
		$services = $this->_services();

		$this->start_controls_section( 'section_items', [
			'label' => esc_html__( 'Services', 'onnat-extra' ),
			'tab'   => KFCM::TAB_CONTENT,
		]);
            $repeater->add_control( 'service', [
                'label'       => esc_html__( 'Service', 'onnat-extra' ),
                'type'        => KFCM::SELECT2,
                'multiple'    => false,
                'label_block' => true,
                'options'     => $services,
                'default'     => ''
            ]);
            $repeater->add_control( 'icon-type', [
                'label'   => esc_html__( 'Icon Type', 'onnat-extra' ),
                'type'    => KFCM::CHOOSE,
                'options' => [
                    'blank' => [
                        'title' => esc_html__('Blank', 'onnat-extra'),
                        'icon'  => '',
                    ],
                    'icon'  => [
                        'title' => esc_html__('Icon', 'onnat-extra'),
                        'icon'  => 'eicon-plus-circle-o',
                    ],
                    'image' => [
                        'title' => esc_html__('Image', 'onnat-extra'),
                        'icon'  => 'eicon-image',
                    ],
                ],
                'default' => 'icon',
                'toggle'  => false,
            ]);
            $repeater->add_control( 'icon', [
                'label'     => esc_html__( 'Icon', 'onnat-extra' ),
                'type'      => KFCM::ICONS,
                'default'   => [
                    'value'   => 'fas fa-basketball-ball',
                    'library' => 'fa-solid',
                ],
                'condition' => [
                    'icon-type' => 'icon'
                ],
            ]);
            $repeater->add_control( 'image', [
                'label'     => esc_html__( 'Image', 'onnat-extra' ),
                'type'      => KFCM::MEDIA,
                'default'   => [
                    'url' => \Elementor\Utils::get_placeholder_image_src()
                ],
                'condition' => [
                    'icon-type' => 'image'
                ],
            ]);
            $repeater->add_control('animation', [
                'label'       => esc_html__('Hover Animation', 'onnat-extra'),
                'type'        => KFCM::SELECT,
                'label_block' => true,
                'default'     => '',
                'options'     => [
                    ''                                 => esc_html__('None', 'onnat-extra'),
                    'kfw-service-grid-anim-flipinx'    => esc_html__('Flip In X', 'onnat-extra'),
                    'kfw-service-grid-anim-flipiny'    => esc_html__('Flip In Y', 'onnat-extra'),
                    'kfw-service-grid-anim-float-x'    => esc_html__('Float X', 'onnat-extra'),
                    'kfw-service-grid-anim-float-y'    => esc_html__('Float Y', 'onnat-extra'),
                    'kfw-service-grid-anim-jello'      => esc_html__('Jello', 'onnat-extra'),
                    'kfw-service-grid-anim-pulse'      => esc_html__('Pulse', 'onnat-extra'),
                    'kfw-service-grid-anim-rotate-alt' => esc_html__('Rotate Alt', 'onnat-extra'),
                    'kfw-service-grid-anim-rotate'     => esc_html__('Rotate', 'onnat-extra'),
                    'kfw-service-grid-anim-swing'      => esc_html__('Swing', 'onnat-extra'),
                    'kfw-service-grid-anim-wobble'     => esc_html__('Wobble', 'onnat-extra'),
                ],
            ]);

			$this->add_control('items', [
				'type'        => KFCM::REPEATER,
				'fields'      => $repeater->get_controls(),
				'title_field' => "<# let labels = ".json_encode( $services )."; let label = labels[service]; #>{{{ label }}}",
				'default'     => [],
            ]);
		$this->end_controls_section();
    }

	public function _register_settings_section() {
		$this->start_controls_section( 'section_settings', [
			'label' => esc_html__( 'Settings', 'onnat-extra' ),
			'tab'   => KFCM::TAB_CONTENT,
		]);
			$this->add_responsive_control('cols', [
				'label'          => esc_html__( 'Columns', 'onnat-extra' ),
				'type'           => KFCM::CHOOSE,
				'options'        => [
					'1' => [ 'title' => esc_html__('1 Column', 'onnat-extra'), 'icon' => 'kfw-ctrl-icon kfw-ctrl-numeric-icon' ],
					'2' => [ 'title' => esc_html__('2 Columns', 'onnat-extra'), 'icon' => 'kfw-ctrl-icon kfw-ctrl-numeric-2-icon' ],
					'3' => [ 'title' => esc_html__('3 Columns', 'onnat-extra'), 'icon' => 'kfw-ctrl-icon kfw-ctrl-numeric-3-icon' ],
					'4' => [ 'title' => esc_html__('4 Columns', 'onnat-extra'), 'icon' => 'kfw-ctrl-icon kfw-ctrl-numeric-4-icon' ],
				],
				'devices'        => [ 'widescreen', 'desktop', 'laptop', 'tablet', 'tablet_extra' ],
				'default'        => '3',
				'tablet_default' => '2',
				'mobile_default' => '1',
				'selectors'      => [
					'{{WRAPPER}} .kfw-ele-widget-cpt-service-grid-main' => 'display: grid; grid-template-columns: repeat({{SIZE}}, minmax(0, 1fr));',
				],
				'toggle'         => false
			]);
			$this->add_responsive_control('col-gap',[
				'label'     => esc_html__( 'Column Gap', 'onnat-extra' ),
				'type'      => KFCM::SLIDER,
				'devices'   => [ 'widescreen', 'desktop', 'laptop', 'tablet', 'tablet_extra' ],
				'default'   => [
					'size' => 20,
				],
				'selectors' => [
					'{{WRAPPER}} .kfw-ele-widget-cpt-service-grid-main' => 'grid-column-gap: {{SIZE}}{{UNIT}};',
				],
			]);
			$this->add_responsive_control('row-gap',[
				'label'     => esc_html__( 'Row Gap', 'onnat-extra' ),
				'type'      => KFCM::SLIDER,
				'devices'   => [ 'widescreen', 'desktop', 'laptop', 'tablet', 'tablet_extra' ],
				'default'   => [ 'size' => 20,],
				'selectors' => [
					'{{WRAPPER}} .kfw-ele-widget-cpt-service-grid-main' => 'grid-row-gap: {{SIZE}}{{UNIT}};',
				],
			]);
            $this->add_control('button-icon-heading',[
                'label' => esc_html__( 'READ MORE BUTTON', 'onnat-extra' ),
                'type'  => KFCM::HEADING,
            ]);
            $this->add_control('button-icon-type',[
                'label'   => esc_html__( 'Icon Type', 'onnat-extra' ),
                'type'    => KFCM::CHOOSE,
                'options' => [
                    'icon'  => [
                        'title' => esc_html__('Icon', 'onnat-extra'),
                        'icon'  => 'eicon-plus-circle-o',
                    ],
                    'image' => [
                        'title' => esc_html__('Image', 'onnat-extra'),
                        'icon'  => 'eicon-image',
                    ],
                ],
                'default' => 'icon',
                'toggle'  => false,
            ]);
            $this->add_control('button-icon',[
                'label'     => esc_html__( 'Icon', 'onnat-extra' ),
                'type'      => KFCM::ICONS,
                'default'   => [
                    'value'   => 'fas fa-long-arrow-alt-right',
                    'library' => 'fa-solid',
                ],
                'condition' => [
                    'button-icon-type' => 'icon'
                ],
            ]);
            $this->add_control('button-image',[
                'label'     => esc_html__( 'Image', 'onnat-extra' ),
                'type'      => KFCM::MEDIA,
                'default'   => [
                    'url' => \Elementor\Utils::get_placeholder_image_src()
                ],
                'condition' => [
                    'button-icon-type' => 'image'
                ],
            ]);
		$this->end_controls_section();
	}

    public function _register_content_wrap_style_section() {
		$this->start_controls_section( 'section_wrap_style', [
			'label' => esc_html__( 'Wrapper', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
            $this->start_controls_tabs('tabs');
                $this->start_controls_tab('normal',[
                    'label' => esc_html__( 'Normal', 'onnat-extra' ),
                ]);
                    $this->add_group_control( KFGCBG::get_type(), [
                        'name'     => 'wrap_bg',
                        'types'    => [ 'classic', 'gradient' ],
                        'exclude'  => [ 'image' ],
                        'selector' => '{{WRAPPER}} .kinfw-service-info-wrap',
                    ]);
                    $this->add_control( 'wrap_border_color', [
                        'label'     => esc_html__('Border Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'selectors' => [
                            '{{WRAPPER}} .kinfw-service-item:before' => 'color: {{VALUE}};'
                        ],
                    ]);
                $this->end_controls_tab();
                $this->start_controls_tab('hover',[
                    'label' => esc_html__( 'Hover', 'onnat-extra' ),
                ]);
                    $this->add_group_control( KFGCBG::get_type(), [
                        'name'     => 'wrap_h_bg',
                        'types'    => [ 'classic', 'gradient' ],
                        'exclude'  => [ 'image' ],
                        'selector' => '{{WRAPPER}} .kinfw-service-item:hover .kinfw-service-info-wrap',
                    ]);
                    $this->add_control( 'wrap_h_border_color', [
                        'label'     => esc_html__('Border Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'selectors' => [
                            '{{WRAPPER}} .kinfw-service-item:hover:before' => 'background: {{VALUE}};'
                        ],
                    ]);
                $this->end_controls_tab();
            $this->end_controls_tabs();

            $this->add_responsive_control( 'wrap_margin', [
                'label'      => esc_html__('Margin', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-service-item' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->add_responsive_control( 'wrap_padding', [
                'label'      => esc_html__('Padding', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-service-info-wrap' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
        $this->end_controls_section();
    }

    public function _register_term_style_section() {
		$this->start_controls_section( 'section_term_style', [
			'label' => esc_html__( 'Term', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
			$this->add_group_control(KFGCT::get_type(), [
				'name'      => 'term_typo',
				'label'     => esc_html__( 'Typography', 'onnat-extra' ),
				'selector'  => '{{WRAPPER}} .kinfw-service-terms-list li a',
			]);
            $this->add_responsive_control( 'term_margin', [
                'label'      => esc_html__('Margin', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-service-terms-list' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->add_responsive_control( 'term_padding', [
                'label'      => esc_html__('Padding', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-service-terms-list li a' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
			$this->add_control( 'term_color', [
				'label'     => esc_html__('Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'selectors' => [
					'{{WRAPPER}} .kinfw-service-terms-list li a' => 'color: {{VALUE}};'
				],
			]);
			$this->add_control( 'term_h_color', [
				'label'     => esc_html__('Hover Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'selectors' => [
					'{{WRAPPER}} .kinfw-service-item:hover .kinfw-service-terms-list li a' => 'color: {{VALUE}};'
				],
			]);
			$this->add_group_control( KFGCTS::get_type(), [
				'name'      => 'term_txt_shadow',
				'selector'  => '{{WRAPPER}} .kinfw-service-terms-list li a',
			]);
			$this->add_group_control( KFGCTST::get_type(), [
				'name'      => 'term_txt_stroke',
				'selector'  => '{{WRAPPER}} .kinfw-service-terms-list li a',
			]);
		$this->end_controls_section();
    }

    public function _register_title_style_section() {
		$this->start_controls_section( 'section_title_style', [
			'label' => esc_html__( 'Title', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
			$this->add_group_control(KFGCT::get_type(), [
				'name'      => 'title_typo',
				'label'     => esc_html__( 'Typography', 'onnat-extra' ),
				'selector'  => '{{WRAPPER}} .kinfw-service-title',
			]);
            $this->add_responsive_control( 'title_margin', [
                'label'      => esc_html__('Margin', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-service-title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->add_responsive_control( 'title_padding', [
                'label'      => esc_html__('Padding', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-service-title' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->start_controls_tabs( 'title_customize_tabs' );
                $this->start_controls_tab( 'title_normal_state', [
                    'label' => esc_html__( 'Normal', 'onnat-extra' ),
                ]);
                    $this->add_control( 'title_color', [
                        'label'     => esc_html__('Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'selectors' => ['{{WRAPPER}} .kinfw-service-title' => 'color: {{VALUE}};' ],
                    ]);
                    $this->add_group_control( KFGCTS::get_type(), [
                        'name'     => 'title_txt_shadow',
                        'selector' => '{{WRAPPER}} .kinfw-service-title',
                    ]);
                    $this->add_group_control( KFGCTST::get_type(), [
                        'name'     => 'title_txt_stroke',
                        'selector' => '{{WRAPPER}} .kinfw-service-title',
                    ]);
                $this->end_controls_tab();
                $this->start_controls_tab( 'title_hover_state', [
                    'label' => esc_html__( 'Hover', 'onnat-extra' ),
                ]);
                    $this->add_control( 'title_h_color', [
                        'label'     => esc_html__('Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'selectors' => ['{{WRAPPER}} .kinfw-service-item:hover .kinfw-service-title' => 'color: {{VALUE}};' ],
                    ]);
                    $this->add_group_control( KFGCTS::get_type(), [
                        'name'     => 'title_h_txt_shadow',
                        'selector' => '{{WRAPPER}} .kinfw-service-item:hover .kinfw-service-title',
                    ]);
                    $this->add_group_control( KFGCTST::get_type(), [
                        'name'     => 'title_h_txt_stroke',
                        'selector' => '{{WRAPPER}} .kinfw-service-item:hover .kinfw-service-title',
                    ]);
                $this->end_controls_tab();
            $this->end_controls_tabs();
		$this->end_controls_section();
    }

	public function _register_icon_style_section() {
		$this->start_controls_section( 'section_icon_style', [
			'label' => esc_html__( 'Icon', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
            $this->add_control('item-icon-heading',[
                'label' => esc_html__( 'ICON TYPE', 'onnat-extra' ),
                'type'  => KFCM::HEADING,
            ]);
            $this->add_responsive_control( 'items_icon_size', [
                'label'      => esc_html__('Icon Size', 'onnat-extra' ),
                'type'       => KFCM::SLIDER,
                'size_units' => [ 'px' ],
                'range'      => [
                    'px' => [
                        'min' => 0,
                        'max' => 999,
                    ],
                ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-service-icon-wrap.kfw-service-icon-type .kfw-elementor-icon-wrap' => 'font-size: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .kfw-service-icon-wrap.kfw-service-icon-type' => 'width: {{SIZE}}{{UNIT}};',
                ],
            ]);
            $this->add_control( 'items_icon_colors_settings', [
                'label'        => esc_html__('Colors', 'onnat-extra'),
                'type'         => KFCM::POPOVER_TOGGLE,
                'return_value' => 'yes',
            ]);
            $this->start_popover();
                $this->add_control( 'items_icon_color', [
                    'label'     => esc_html__('Color', 'onnat-extra' ),
                    'type'      => KFCM::COLOR,
                    'selectors' => [
                        '{{WRAPPER}} .kfw-service-icon-wrap.kfw-service-icon-type .kfw-elementor-icon-wrap i' => 'color:{{VALUE}};',
                        '{{WRAPPER}} .kfw-service-icon-wrap.kfw-service-icon-type .kfw-elementor-icon-wrap svg' => 'fill:{{VALUE}};',
                    ],
                ]);
                $this->add_control( 'items_icon_h_color', [
                    'label'     => esc_html__('Hover Color', 'onnat-extra' ),
                    'type'      => KFCM::COLOR,
                    'selectors' => [
                        '{{WRAPPER}} .kinfw-service-item:hover .kfw-service-icon-wrap.kfw-service-icon-type .kfw-elementor-icon-wrap i' => 'color:{{VALUE}};',
                        '{{WRAPPER}} .kinfw-service-item:hover .kfw-service-icon-wrap.kfw-service-icon-type .kfw-elementor-icon-wrap svg' => 'fill:{{VALUE}};',
                    ],
                ]);
            $this->end_popover();
            $this->add_control( 'items_icon_style_1', [
                'type' => KFCM::DIVIDER,
            ]);
            $this->add_control('item-image-heading',[
                'label' => esc_html__( 'IMAGE TYPE', 'onnat-extra' ),
                'type'  => KFCM::HEADING,
            ]);
            $this->add_responsive_control( 'items_img_size', [
                'label'      => esc_html__('Image Size', 'onnat-extra' ),
                'type'       => KFCM::SLIDER,
                'size_units' => [ 'px' ],
                'range'      => [
                    'px' => [
                        'min' => 0,
                        'max' => 500,
                    ],
                ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-service-icon-wrap.kfw-service-image-type .kfw-elementor-icon-wrap img' => 'max-width:{{SIZE}}px;',
                    '{{WRAPPER}} .kfw-service-icon-wrap.kfw-service-image-type' => 'width:{{SIZE}}px;'
                ],
            ]);
            $this->add_control( 'items_icon_style_2', [
                'type' => KFCM::DIVIDER,
            ]);
			$this->add_responsive_control( 'items_icon_margin', [
				'label'      => esc_html__('Margin', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-service-icon-wrap' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
		$this->end_controls_section();
    }

    public function _register_content_style_section() {
		$this->start_controls_section( 'section_content_style', [
			'label' => esc_html__( 'Content', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
            $this->add_group_control(KFGCT::get_type(), [
                'name'     => 'content_typo',
                'label'    => esc_html__( 'Typography', 'onnat-extra' ),
                'selector' => '{{WRAPPER}} .kinfw-service-excerpt',
            ]);
            $this->add_responsive_control( 'content_margin', [
                'label'      => esc_html__('Margin', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-service-excerpt' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->add_responsive_control( 'content_padding', [
                'label'      => esc_html__('Padding', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-service-excerpt' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->start_controls_tabs( 'content_customize_tabs' );
                $this->start_controls_tab( 'content_normal_state', [
                    'label' => esc_html__( 'Normal', 'onnat-extra' ),
                ]);
                    $this->add_control( 'content_color', [
                        'label'     => esc_html__('Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'selectors' => ['{{WRAPPER}} .kinfw-service-excerpt' => 'color: {{VALUE}};' ],
                    ]);
                    $this->add_group_control( KFGCTS::get_type(), [
                        'name'     => 'content_txt_shadow',
                        'selector' => '{{WRAPPER}} .kinfw-service-excerpt',
                    ]);
                    $this->add_group_control( KFGCTST::get_type(), [
                        'name'     => 'content_txt_stroke',
                        'selector' => '{{WRAPPER}} .kinfw-service-excerpt',
                    ]);
                $this->end_controls_tab();
                $this->start_controls_tab( 'content_hover_state', [
                    'label' => esc_html__( 'Hover', 'onnat-extra' ),
                ]);
                    $this->add_control( 'content_h_color', [
                        'label'     => esc_html__('Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'selectors' => ['{{WRAPPER}} .kinfw-service-item:hover .kinfw-service-excerpt' => 'color: {{VALUE}};' ],
                    ]);
                    $this->add_group_control( KFGCTS::get_type(), [
                        'name'     => 'content_h_txt_shadow',
                        'selector' => '{{WRAPPER}} .kinfw-service-item:hover .kinfw-service-excerpt',
                    ]);
                    $this->add_group_control( KFGCTST::get_type(), [
                        'name'     => 'content_h_txt_stroke',
                        'selector' => '{{WRAPPER}} .kinfw-service-item:hover .kinfw-service-excerpt',
                    ]);
                $this->end_controls_tab();
            $this->end_controls_tabs();
		$this->end_controls_section();
    }

    public function _register_readmore_btn_style_section() {
        $icon_condition = [
            'button-icon-type' => 'icon'
        ];

        $img_condition = [
            'button-icon-type' => 'image'
        ];

		$this->start_controls_section( 'section_readmore_btn_icon_style', [
            'label' => esc_html__( 'Read More Button', 'onnat-extra' ),
            'tab'   => KFCM::TAB_STYLE,
		]);
            $this->add_responsive_control( 'readmore_btn_margin', [
                'label'              => esc_html__('Margin', 'onnat-extra' ),
                'type'               => KFCM::DIMENSIONS,
                'size_units'         => [ 'px', '%' ],
                'allowed_dimensions' => [ 'right', 'bottom' ],
                'placeholder'        => [
                    'top'    => 'auto',
                    'right'  => '',
                    'bottom' => '',
                    'left'   => 'auto',
                ],
                'default'            => [
                    'top'      => 'auto',
                    'right'    => '',
                    'bottom'   => '',
                    'left'     => 'auto',
                    'unit'     => 'px',
                    'isLinked' => false,
                ],
                'selectors'          => [
                    '{{WRAPPER}} .kinfw-service-button-wrap' => 'right:{{RIGHT}}{{UNIT}}; bottom:{{BOTTOM}}{{UNIT}};'
                ]
            ]);
            $this->add_responsive_control( 'readmore_btn_icon_wrapper_size', [
                'label'      => esc_html__('Wrapper Size', 'onnat-extra' ),
                'type'       => KFCM::SLIDER,
                'size_units' => [ 'px' ],
                'range'      => [
                    'px' => [
                        'min' => 0,
                        'max' => 500,
                    ],
                ],
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-service-button-wrap a' => 'line-height:{{SIZE}}{{UNIT}};height:{{SIZE}}{{UNIT}};width:{{SIZE}}{{UNIT}};',
                ],
            ]);
            $this->add_responsive_control( 'readmore_btn_icon_size', [
                'label'      => esc_html__('Icon Size', 'onnat-extra' ),
                'type'       => KFCM::SLIDER,
                'size_units' => [ 'px' ],
                'condition'  => $icon_condition,
                'range'      => [
                    'px' => [
                        'min' => 0,
                        'max' => 999,
                    ],
                ],
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-service-button-wrap .kfw-elementor-icon-wrap' => 'font-size: {{SIZE}}{{UNIT}};'
                ],
            ]);
            $this->add_responsive_control( 'readmore_btn_img_size', [
                'label'      => esc_html__('Image Size', 'onnat-extra' ),
                'type'       => KFCM::SLIDER,
                'condition'  => $img_condition,
                'size_units' => [ 'px' ],
                'range'      => [
                    'px' => [
                        'min' => 0,
                        'max' => 500,
                    ],
                ],
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-service-button-wrap .kfw-elementor-icon-wrap img' => 'max-width:{{SIZE}}px;'
                ],
            ]);
            $this->start_controls_tabs( 'readmore_btn_icon_customize_tabs' );
                $this->start_controls_tab( 'readmore_btn_icon_normal_state', [
                    'label' => esc_html__( 'Normal', 'onnat-extra' ),
                ]);
                    $this->add_control( 'readmore_btn_icon_color', [
                        'label'     => esc_html__('Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'condition' => $icon_condition,
                        'selectors' => [
                            '{{WRAPPER}} .kinfw-service-button-wrap .kfw-elementor-icon-wrap i' => 'color:{{VALUE}};',
                            '{{WRAPPER}} .kinfw-service-button-wrap .kfw-elementor-icon-wrap svg' => 'fill:{{VALUE}};',
                        ],
                    ]);
					$this->add_control('icon_rotate',[
						'label'     => esc_html__( 'Rotate', 'onnat-extra' ),
						'type'      => KFCM::SLIDER,
						'default'   => [
							'size' => 0,
							'unit' => 'deg',
						],
						'range'     => [
							'deg' => [
								'max' => 360,
								'min' => - 360,
							],
						],
						'selectors' => [
							'{{WRAPPER}} .kinfw-service-button-wrap .kfw-elementor-icon-wrap'   => 'transform: rotate({{SIZE}}{{UNIT}});',
						],
					]);
                    $this->add_group_control( KFGCBG::get_type(), [
                        'name'           => 'readmore_btn_icon_bg',
                        'types'          => ['classic', 'gradient'],
                        'exclude'        => [ 'image'],
                        'fields_options' => [
                            'background' => [ 'default' => 'classic' ],
                            'color'      => [ 'default' => '' ],
                        ],
                        'selector'       => '{{WRAPPER}} .kinfw-service-button-wrap a',
                    ]);
                    $this->add_group_control( KFGCBR::get_type(), [
                        'name'     => 'readmore_btn_icon_border',
                        'selector' => '{{WRAPPER}} .kinfw-service-button-wrap a',
                    ]);
                    $this->add_responsive_control( 'readmore_btn_icon_border_radius', [
                        'label'      => esc_html__('Border Radius', 'onnat-extra' ),
                        'type'       => KFCM::DIMENSIONS,
                        'size_units' => [ 'px', '%' ],
                        'selectors'  => [
                            '{{WRAPPER}} .kinfw-service-button-wrap a' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                        ]
                    ]);
                    $this->add_group_control( KFGCBS::get_type(), [
                        'name'     => 'readmore_btn_icon_box_shadow',
                        'selector' => '{{WRAPPER}} .kinfw-service-button-wrap a'
                    ]);
                $this->end_controls_tab();
                $this->start_controls_tab( 'readmore_btn_icon_hover_state', [
                    'label' => esc_html__( 'Hover', 'onnat-extra' ),
                ]);
                    $this->add_control( 'readmore_btn_icon_h_color', [
                        'label'     => esc_html__('Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'condition' => $icon_condition,
                        'selectors' => [
                            '{{WRAPPER}} .kinfw-service-item:hover .kinfw-service-button-wrap .kfw-elementor-icon-wrap i' => 'color:{{VALUE}};',
                            '{{WRAPPER}} .kinfw-service-item:hover .kinfw-service-button-wrap .kfw-elementor-icon-wrap svg' => 'fill:{{VALUE}};',
                        ],
                    ]);
                    $this->add_control('icon_h_rotate',[
						'label'     => esc_html__( 'Rotate', 'onnat-extra' ),
						'type'      => KFCM::SLIDER,
						'default'   => [
							'size' => 0,
							'unit' => 'deg',
						],
						'range'     => [
							'deg' => [
								'max' => 360,
								'min' => - 360,
							],
						],
						'selectors' => [
							'{{WRAPPER}} .kinfw-service-item:hover .kinfw-service-button-wrap .kfw-elementor-icon-wrap'   => 'transform: rotate({{SIZE}}{{UNIT}});',
						],
					]);
                    $this->add_control( 'readmore_btn_icon_h_bg_color', [
                        'label'     => esc_html__('Background Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'selectors' => [
                            '{{WRAPPER}}' => '--kinfw-service-button-wrap-hover-color:{{VALUE}};',
                        ],
                    ]);
                    $this->add_group_control( KFGCBR::get_type(), [
                        'name'     => 'readmore_btn_icon_h_border',
                        'selector' => '{{WRAPPER}} .kinfw-service-item:hover .kinfw-service-button-wrap a'
                    ]);
                    $this->add_responsive_control( 'readmore_btn_icon_h_border_radius', [
                        'label'      => esc_html__('Border Radius', 'onnat-extra' ),
                        'type'       => KFCM::DIMENSIONS,
                        'size_units' => [ 'px', '%' ],
                        'selectors'  => [
                            '{{WRAPPER}} .kinfw-service-item:hover .kinfw-service-button-wrap a:before' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                        ]
                    ]);
                    $this->add_group_control( KFGCBS::get_type(), [
                        'name'     => 'readmore_btn_icon_h_box_shadow',
                        'selector' => '{{WRAPPER}} .kinfw-service-item:hover .kinfw-service-button-wrap a'
                    ]);
                $this->end_controls_tab();
            $this->end_controls_tabs();

		$this->end_controls_section();
    }

	public function _services() {
		$result = [ esc_html__( '-- Select --', 'onnat-extra' ) ];

		$posts = get_posts([
			'post_type'      => 'kinfw-service',
			'post_status'    => 'publish',
			'posts_per_page' => -1
		]);

		if ( ! empty( $posts ) && ! is_wp_error( $posts ) ) {
			foreach ( $posts as $item ) {
				$result[ $item->ID ] = $item->post_title;
			}
		}

		return $result;
	}

    public function _render_button_icon( $icon ) {
		$icon = array_filter( $icon );
		if( !empty( $icon ) ) {
            return '<span class="kfw-elementor-icon-wrap">'.
                KFIM::try_get_icon_html( $icon, [ 'aria-hidden' => 'true' ] ) .
            '</span>';
        }
    }

    public function _render_button_img( $image ) {
        if( !empty( $image['url'] ) ) {
            $image_src = $image['url'];

			$image_title = $this->_get_image_title( $image );
			$image_title = empty( $image_title ) ? $this->get_title() : $image_title;

			$image_alt = $this->_get_image_alt( $image );
			$image_alt = empty( $image_alt ) ? $image_title : $image_title;

            return '<span class="kfw-elementor-icon-wrap">'.
                '<img alt="'. esc_attr( $image_alt ) .'" title="'. esc_attr( $image_title ) .'" src="'. esc_url( $image_src ) .'"/>'.
            '</span>';
        }
    }

	public function _render_icon( $icon , $echo = true ) {
		$icon = array_filter( $icon );

		if( !empty( $icon ) ) {
            if( $echo ) {
                echo '<div class="kfw-service-icon-wrap kfw-service-icon-type">';
                    echo '<span class="kfw-elementor-icon-wrap">';
                        KFIM::render_icon( $icon, [ 'aria-hidden' => 'true' ] );
                    echo '</span>';
                echo '</div>';
            } else {
                return '
                    <div class="kfw-service-icon-wrap kfw-service-icon-type">
                        <span class="kfw-elementor-icon-wrap">'.
                        KFIM::try_get_icon_html( $icon, [ 'aria-hidden' => 'true' ] ) .'
                        </span>
                    </div>
                ';
            }
		}
    }

    public function _render_img( $settings = [] ) {
        if( !empty( $settings['url'] ) ) {
            $image_src = $settings['url'];

			$image_title = $this->_get_image_title( $settings );
			$image_title = empty( $image_title ) ? $this->get_title() : $image_title;

			$image_alt = $this->_get_image_alt( $settings );
			$image_alt = empty( $image_alt ) ? $image_title : $image_title;

            echo '<div class="kfw-service-icon-wrap kfw-service-image-type">';
                echo '<span class="kfw-elementor-icon-wrap">';
                    echo '<img alt="'. esc_attr( $image_alt ) .'" title="'. esc_attr( $image_title ) .'" src="'. esc_url( $image_src ) .'"/>';
                echo '</span>';
            echo '</div>';
        }
    }

	public function _get_image_title( $attachment ) {
		if ( empty( $attachment['id'] ) ) {
			return '';
		}

		return get_the_title( $attachment['id'] );
	}

	public function _get_image_alt( $instance ) {
		if ( empty( $instance['id'] ) ) {
			// For `Insert From URL` images.
			return isset( $instance['alt'] ) ? trim( strip_tags( $instance['alt'] ) ) : '';
		}

		$attachment_id = $instance['id'];
		if ( ! $attachment_id ) {
			return '';
		}

		$attachment = get_post( $attachment_id );
		if ( ! $attachment ) {
			return '';
		}

		$alt = get_post_meta( $attachment_id, '_wp_attachment_image_alt', true );
		if ( ! $alt ) {
			$alt = $attachment->post_excerpt;
			if ( ! $alt ) {
				$alt = $attachment->post_title;
			}
		}
		return trim( strip_tags( $alt ) );
	}

	/**
	 * Render widget output on the frontend.
	 */
    protected function render() {
		$render = kf_extra_plugin_elementor()->get_template("kfw-cpt-service-grid");
		include $render;
    }
}