<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

use \Elementor\Widget_Base as WB;
use \Elementor\Controls_Manager as KFCM;
use \Elementor\Icons_Manager as KFIM;
use \Elementor\Repeater as KFRTR;

use \Elementor\Group_Control_Image_Size as KFGCIS;
use \Elementor\Group_Control_Typography as KFGCT;
use \Elementor\Group_Control_Text_Shadow as KFGCTS;
use \Elementor\Group_Control_Text_Stroke as KFGCTST;
use \Elementor\Group_Control_Background as KFGCBG;
use \Elementor\Group_Control_Border as KFGCBR;
use \Elementor\Group_Control_Box_Shadow as KFGCBS;

class KFW_CPT_Service_Vertical extends WB {

	/**
	 * Get element name.
	 *
	 * Retrieve the element name.
	 *
	 * @return string The name.
	 */
	public function get_name() {
        return 'kfw-cpt-service-vertical';
    }

	/**
	 * Get element title.
	 *
	 * Retrieve the element title.
	 *
	 * @return string Element title.
	 */
	public function get_title() {
		return esc_html__('Service Vertical', 'onnat-extra');
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the widget categories.
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ '_kfw_theme_widgets' ];
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve the widget icon.
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'kfw-badge eicon-columns kfw-service-vertical';
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the widget keywords.
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'kfw', 'kin', 'service', 'post', 'vertical' ];
	}

	/**
	 * Get widget help url.
	 *
	 * Retrieve the widget help url.
	 *
	 * @return string Widget help url.
	 */
	public function get_custom_help_url() {
        return KFE_CONST_THEME_DOC . 'elementor-custom-widgets/service-vertical';
    }

	/**
	 * Get HTML wrapper class.
     *
	 * Retrieve the widget container class.
     *
     * @return string Widget container css classes.
	 */
	public function get_html_wrapper_class() {
        $name = $this->get_name();

		return sprintf( 'kfw-ele-%s elementor-widget-%s', $name, $name );
	}

	/**
	 * Widget base constructor.
	 *
	 * Initializing the widget base class.
	 */
	public function __construct( $data = [], $args = null ) {
        parent::__construct( $data, $args );

		wp_register_style( 'kfw-cpt-service-vertical',
			KFE_CONST_URL . 'libraries/elementor/assets/css/cpt-service-vertical' . KFE_CONST_DEBUG_SUFFIX . '.css',
			[],
			KFE_CONST_VERSION
		);

		wp_register_script( 'kfw-cpt-service-vertical',
			KFE_CONST_URL . 'libraries/elementor/assets/js/service-vertical' . KFE_CONST_DEBUG_SUFFIX . '.js',
			[ 'jquery', 'elementor-frontend' ],
			KFE_CONST_VERSION,
			true
		);
    }

	/**
	 * Get style dependencies.
	 *
	 * Retrieve the list of style dependencies the element requires.
	 *
	 * @return array Element styles dependencies.
	 */
	public function get_style_depends() {
		return [ 'kfw-cpt-service-vertical' ];
	}

	/**
	 * Get script dependencies.
	 *
	 * Retrieve the list of script dependencies the element requires.
	 *
	 * @return array Element scripts dependencies.
	 */
	public function get_script_depends() {
		return [ 'kfw-cpt-service-vertical' ];
	}    

	/**
	 * Register controls.
	 *
	 * Used to add new controls to any element type.
	 */
	protected function register_controls() {
		$this->_register_items_section();
        $this->_register_settings_section();
        $this->_register_wrap_style_section();
        $this->_register_image_style_section();
        $this->_register_order_type_style_section();
        $this->_register_title_style_section();
        $this->_register_readmore_btn_style_section();
    }

    public function _register_items_section() {
		$repeater = new KFRTR();
		$services = $this->_services();

		$this->start_controls_section( 'section_items', [
			'label' => esc_html__( 'Services', 'onnat-extra' ),
			'tab'   => KFCM::TAB_CONTENT,
		]);
            $repeater->add_control( 'service', [
                'label'       => esc_html__( 'Service', 'onnat-extra' ),
                'type'        => KFCM::SELECT2,
                'multiple'    => false,
                'label_block' => true,
                'options'     => $services,
                'default'     => ''
            ]);
            $repeater->add_control( 'image-type', [
                'label'   => esc_html__( 'Image Type', 'onnat-extra' ),
                'type'    => KFCM::CHOOSE,
                'options' => [
                    'custom-image'   => [
                        'title' => esc_html__('Custom Image', 'onnat-extra'),
                        'icon'  => 'eicon-plus-circle-o',
                    ],
                    'featured-image' => [
                        'title' => esc_html__('Featured Image', 'onnat-extra'),
                        'icon'  => 'eicon-image',
                    ],
                ],
                'default' => 'featured-image',
                'toggle'  => false,
            ]);
            $repeater->add_control( 'custom-image', [
                'label'     => esc_html__( 'Custom Image', 'onnat-extra' ),
                'type'      => KFCM::MEDIA,
                'default'   => [
                    'url' => \Elementor\Utils::get_placeholder_image_src()
                ],
                'condition' => [
                    'image-type' => 'custom-image'
                ],
            ]);
			$this->add_control( 'items', [
				'type'        => KFCM::REPEATER,
				'fields'      => $repeater->get_controls(),
				'title_field' => "<# let labels = ".json_encode( $services )."; let label = labels[service]; #>{{{ label }}}",
				'default'     => [],
			]);
		$this->end_controls_section();
	}

	public function _register_settings_section() {
		$this->start_controls_section( 'section_settings', [
			'label' => esc_html__( 'Settings', 'onnat-extra' ),
			'tab'   => KFCM::TAB_CONTENT,
        ]);
            $this->add_group_control( KFGCIS::get_type(), [
                'name'    => 'image_size',
                'default' => 'full',
            ]);
            $this->add_control('type',[
                'label'        => esc_html__( 'Order Type', 'onnat-extra' ),
                'type'         => KFCM::CHOOSE,
                'options'      => [
                    'numeric'     => [
                        'title' => esc_html__('Numeric', 'onnat-extra'),
                        'icon'  => 'kfw-ctrl-icon kfw-ctrl-numeric-icon'
                    ],  
                    'decimal-leading-zero'     => [
                        'title' => esc_html__('Decimal Leading Zero', 'onnat-extra'),
                        'icon'  => 'kfw-ctrl-icon kfw-ctrl-decimal-leading-zero-icon'
                    ], 
                    'lower-roman' => [
                        'title' => esc_html__('Lower Roman', 'onnat-extra'),
                        'icon'  => 'kfw-ctrl-icon kfw-ctrl-lower-roman-icon'
                    ],
                    'upper-roman' => [
                        'title' => esc_html__('Upper Roman', 'onnat-extra'),
                        'icon'  => 'kfw-ctrl-icon kfw-ctrl-upper-roman-icon'
                    ],
                    'lower-alpha' => [
                        'title' => esc_html__('Alphabet', 'onnat-extra'),
                        'icon'  => 'kfw-ctrl-icon kfw-ctrl-lower-alphabet-icon'
                    ],
                    'upper-alpha' => [
                        'title' => esc_html__('Alphabet', 'onnat-extra'),
                        'icon'  => 'kfw-ctrl-icon kfw-ctrl-upper-alphabet-icon'
                    ],
                    'lower-greek' => [
                        'title' => esc_html__('Greek', 'onnat-extra'),
                        'icon'  => 'kfw-ctrl-icon kfw-ctrl-alpha-icon'
                    ],
                ],
                'default'      => 'numeric',
                'prefix_class' => 'kfw-ele-service-vertical-type-',
                'toggle'       => false,
                'selectors'    => [
                    '{{WRAPPER}} .kfw-ele-widget-cpt-service-vertical-main'     => 'counter-reset:'.$this->_counter(),
                    '{{WRAPPER}} .kinfw-service-vertical-item:before' => 'counter-increment:'.$this->_counter().';content:counter('.$this->_counter().', {{type}}) ". "',
                ]
            ]);
            $this->add_control('start',[
                'label'     => esc_html__( 'Starts at', 'onnat-extra' ),
                'type'      => KFCM::NUMBER,
                'default'   => 0,
                'min'       => 0,
                'step'      => 1,
                'max'       => 10000,
                'selectors' => [
                    '{{WRAPPER}} .kfw-ele-widget-cpt-service-vertical-main' => 'counter-reset:'.$this->_counter().' {{start}} !important',
                ]
            ]);
            $this->add_control('order',[
                'label'        => esc_html__( 'Order', 'onnat-extra' ),
                'type'         => KFCM::CHOOSE,
                'options'      => [
                    'ascending'  => [
                        'title' => esc_html__( 'Ascending Order', 'onnat-extra' ),
                        'icon'  => 'eicon-arrow-up',
                    ],
                    'descending' => [
                        'title' => esc_html__( 'Descending Order', 'onnat-extra' ),
                        'icon'  => 'eicon-arrow-down',
                    ],
                ],
                'prefix_class' => 'kfw-ele-service-vertical-order-',
                'default'      => 'ascending',
                'selectors_dictionary' => [
                    'ascending'  => '',
                    'descending' => 'counter-increment:'.$this->_counter() . ' -1',
                ],
                'toggle'       => false,
                'selectors'            => [
                    '{{WRAPPER}}.kfw-ele-service-vertical-order-descending .kinfw-service-vertical-item:before' => '{{VALUE}};',
                ]
            ]);
            $this->add_control('active_item', [
                'label'   => esc_html__( 'Active Item Number', 'onnat-extra' ),
                'type'    => KFCM::NUMBER,
                'min'     => 1,
                'max'     => 20,
                'step'    => 1,
                'default' => 1,
            ]);
            $this->add_control('button-icon-heading',[
                'label' => esc_html__( 'READ MORE BUTTON', 'onnat-extra' ),
                'type'  => KFCM::HEADING,
            ]);
            $this->add_control('button-icon-type',[
                'label'   => esc_html__( 'Icon Type', 'onnat-extra' ),
                'type'    => KFCM::CHOOSE,
                'options' => [
                    'icon'  => [
                        'title' => esc_html__('Icon', 'onnat-extra'),
                        'icon'  => 'eicon-plus-circle-o',
                    ],
                    'image' => [
                        'title' => esc_html__('Image', 'onnat-extra'),
                        'icon'  => 'eicon-image',
                    ],
                ],
                'default' => 'icon',
                'toggle'  => false,
            ]);
            $this->add_control('button-icon',[
                'label'     => esc_html__( 'Icon', 'onnat-extra' ),
                'type'      => KFCM::ICONS,
                'default'   => [
                    'value'   => 'fas fa-long-arrow-alt-right',
                    'library' => 'fa-solid',
                ],
                'condition' => [
                    'button-icon-type' => 'icon'
                ],
            ]);
            $this->add_control('button-image',[
                'label'     => esc_html__( 'Image', 'onnat-extra' ),
                'type'      => KFCM::MEDIA,
                'default'   => [
                    'url' => \Elementor\Utils::get_placeholder_image_src()
                ],
                'condition' => [
                    'button-icon-type' => 'image'
                ],
            ]);
		$this->end_controls_section();
    }

    public function _register_wrap_style_section() {
		$this->start_controls_section( 'section_wrap_style', [
			'label' => esc_html__( 'Wrapper', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
            $this->add_group_control( KFGCBG::get_type(), [
                'name'     => 'wrap_bg',
                'types'    => [ 'classic', 'gradient' ],
                'exclude'  => [ 'image' ],
                'selector' => '{{WRAPPER}} .kinfw-service-vertical-item',
            ]);
            $this->add_group_control( KFGCBR::get_type(), [
                'name'           => 'wrap_border',
                'fields_options' => [
                    'width' => [ 'label' => _x( 'Border Width', 'Border Control', 'onnat-extra' ) ],
                    'color' => [ 'label' => _x( 'Border Color', 'Border Control', 'onnat-extra' ) ],
                ],
                'selector'       => '{{WRAPPER}} .kinfw-service-vertical-item',
            ]);
            $this->add_group_control( KFGCBS::get_type(), [
                'name'     => 'wrap_box_shadow',
                'selector' => '{{WRAPPER}} .kinfw-service-vertical-item',
            ]);
            $this->add_responsive_control( 'wrap_margin', [
                'label'      => esc_html__('Margin', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-service-vertical-item' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->add_responsive_control( 'wrap_padding', [
                'label'      => esc_html__('Padding', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-service-vertical-item' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
        $this->end_controls_section();
    }

    public function _register_image_style_section() {
		$this->start_controls_section( 'section_featured_img_style',[
			'label' => esc_html__( 'Featured Image Style', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
            $this->add_responsive_control( 'items_img_size', [
                'label'      => esc_html__('Size', 'onnat-extra' ),
                'type'       => KFCM::SLIDER,
                'size_units' => [ 'px' ],
                'range'      => [
                    'px' => [
                        'min' => 0,
                        'max' => 500,
                    ],
                ],
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-service-vertical-img-wrap img' => 'max-width:{{SIZE}}px;',
                ],
            ]);
            $this->add_responsive_control( 'items_img_margin', [
                'label'              => esc_html__('Margin', 'onnat-extra' ),
                'type'               => KFCM::DIMENSIONS,
                'allowed_dimensions' => [ 'top', 'left' ],
                'placeholder'        => [
                    'top'    => '',
                    'right'  => 'auto',
                    'bottom' => 'auto',
                    'left'   => '',
                ],
                'default'            => [
                    'top'      => '',
                    'right'    => 'auto',
                    'bottom'   => 'auto',
                    'left'     => '',
                    'unit'     => 'px',
                    'isLinked' => false,
                ],
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-service-vertical-img-wrap' => 'top: {{TOP}}{{UNIT}}; left: {{LEFT}}{{UNIT}};'
                ]
            ]);
        $this->end_controls_section();
    }

    public function _register_order_type_style_section() {
		$this->start_controls_section( 'section_order_type_style',[
			'label' => esc_html__( 'Order Type Style', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
            $this->add_responsive_control( 'order_position', [
                'label'      => esc_html__('Position', 'onnat-extra' ),
                'type'       => KFCM::SLIDER,
                'size_units' => [ 'px' ],
                'range'      => [
                    'px' => [
                        'min' => 0,
                        'max' => 250,
                    ],
                ],
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-service-vertical-item:before' => 'left:{{SIZE}}{{UNIT}};',
                ],
            ]);
            $this->add_group_control(KFGCT::get_type(), [
                'name'      => 'order_type_typo',
                'label'     => esc_html__( 'Typography', 'onnat-extra' ),
                'selector'  => '{{WRAPPER}} .kinfw-service-vertical-item:before',
            ]);
            $this->add_control( 'order_type_color', [
                'label'     => esc_html__('Color', 'onnat-extra' ),
                'type'      => KFCM::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .kinfw-service-vertical-item:before' => 'color:{{VALUE}};',
                ],
            ]);
		$this->end_controls_section();
    }

    public function _register_title_style_section() {
		$this->start_controls_section( 'section_title_style', [
			'label' => esc_html__( 'Title', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
			$this->add_group_control(KFGCT::get_type(), [
				'name'      => 'title_typo',
				'label'     => esc_html__( 'Typography', 'onnat-extra' ),
				'selector'  => '{{WRAPPER}} .kinfw-service-vertical-title',
			]);
            $this->add_responsive_control( 'title_margin', [
                'label'      => esc_html__('Margin', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-service-vertical-title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->add_responsive_control( 'title_padding', [
                'label'      => esc_html__('Padding', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-service-vertical-title' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
			$this->add_control( 'title_color', [
				'label'     => esc_html__('Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'selectors' => [
					'{{WRAPPER}} .kinfw-service-vertical-title' => 'color: {{VALUE}};'
				],
			]);
			$this->add_control( 'title_h_color', [
				'label'     => esc_html__('Hover Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'selectors' => [
					'{{WRAPPER}} .kinfw-service-vertical-item.kfw-active-item .kinfw-service-vertical-title' => 'color: {{VALUE}};'
				],
			]);
			$this->add_group_control( KFGCTS::get_type(), [
				'name'      => 'title_txt_shadow',
				'selector'  => '{{WRAPPER}} .kinfw-service-vertical-item .kinfw-service-vertical-title',
			]);
			$this->add_group_control( KFGCTST::get_type(), [
				'name'      => 'title_txt_stroke',
				'selector'  => '{{WRAPPER}} .kinfw-service-vertical-item .kinfw-service-vertical-title',
			]);
		$this->end_controls_section();
    }

    public function _register_readmore_btn_style_section() {
        $icon_condition = [
            'button-icon-type' => 'icon'
        ];

        $img_condition = [
            'button-icon-type' => 'image'
        ];

		$this->start_controls_section( 'section_readmore_btn_icon_style', [
            'label' => esc_html__( 'Read More Button', 'onnat-extra' ),
            'tab'   => KFCM::TAB_STYLE,
		]);
            $this->add_responsive_control( 'readmore_btn_margin', [
                'label'      => esc_html__('Margin', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-service-vertical-button-wrap a' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->add_responsive_control( 'readmore_btn_icon_padding', [
                'label'      => esc_html__('Padding', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-service-vertical-button-wrap a' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->add_responsive_control( 'readmore_btn_icon_wrapper_size', [
                'label'      => esc_html__('Wrapper Size', 'onnat-extra' ),
                'type'       => KFCM::SLIDER,
                'size_units' => [ 'px' ],
                'range'      => [
                    'px' => [
                        'min' => 0,
                        'max' => 500,
                    ],
                ],
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-service-vertical-button-wrap a' => 'line-height:{{SIZE}}{{UNIT}};height:{{SIZE}}{{UNIT}};width:{{SIZE}}{{UNIT}};',
                ],
            ]);
            $this->add_responsive_control( 'readmore_btn_icon_size', [
                'label'      => esc_html__('Icon Size', 'onnat-extra' ),
                'type'       => KFCM::SLIDER,
                'size_units' => [ 'px' ],
                'condition'  => $icon_condition,
                'range'      => [
                    'px' => [
                        'min' => 0,
                        'max' => 999,
                    ],
                ],
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-service-vertical-button-wrap .kfw-elementor-icon-wrap' => 'font-size: {{SIZE}}{{UNIT}};'
                ],
            ]);
            $this->add_responsive_control( 'readmore_btn_img_size', [
                'label'      => esc_html__('Image Size', 'onnat-extra' ),
                'type'       => KFCM::SLIDER,
                'condition'  => $img_condition,
                'size_units' => [ 'px' ],
                'range'      => [
                    'px' => [
                        'min' => 0,
                        'max' => 500,
                    ],
                ],
                'selectors'  => [
                    '{{WRAPPER}} .kinfw-service-vertical-button-wrap .kfw-elementor-icon-wrap img' => 'max-width:{{SIZE}}px;'
                ],
            ]);
            $this->start_controls_tabs( 'readmore_btn_icon_customize_tabs' );
                $this->start_controls_tab( 'readmore_btn_icon_normal_state', [
                    'label' => esc_html__( 'Normal', 'onnat-extra' ),
                ]);
                    $this->add_control( 'readmore_btn_icon_color', [
                        'label'     => esc_html__('Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'condition' => $icon_condition,
                        'selectors' => [
                            '{{WRAPPER}} .kinfw-service-vertical-button-wrap .kfw-elementor-icon-wrap i' => 'color:{{VALUE}};',
                            '{{WRAPPER}} .kinfw-service-vertical-button-wrap .kfw-elementor-icon-wrap svg' => 'fill:{{VALUE}};',
                        ],
                    ]);
					$this->add_control('icon_rotate',[
						'label'     => esc_html__( 'Rotate', 'onnat-extra' ),
						'type'      => KFCM::SLIDER,						
						'range'     => [
							'deg' => [
								'max' => 360,
								'min' => -360,
							],
						],
						'selectors' => [
							#'{{WRAPPER}} .kinfw-service-vertical-button-wrap a .kfw-elementor-icon-wrap'   => 'transform: rotate({{SIZE}}{{UNIT}});',
							'{{WRAPPER}} .kinfw-service-vertical-button-wrap a .kfw-elementor-icon-wrap'   => 'transform: rotate({{SIZE}}deg);',
						],
					]);
                    $this->add_group_control( KFGCBG::get_type(), [
                        'name'           => 'readmore_btn_icon_bg',
                        'types'          => ['classic', 'gradient'],
                        'exclude'        => [ 'image'],
                        'fields_options' => [
                            'background' => [ 'default' => 'classic' ],
                            'color'      => [ 'default' => '' ],
                        ],
                        'selector'       => '{{WRAPPER}} .kinfw-service-vertical-button-wrap a',
                    ]);
                    $this->add_group_control( KFGCBR::get_type(), [
                        'name'     => 'readmore_btn_icon_border',
                        'selector' => '{{WRAPPER}} .kinfw-service-vertical-button-wrap a',
                    ]);
                    $this->add_responsive_control( 'readmore_btn_icon_border_radius', [
                        'label'      => esc_html__('Border Radius', 'onnat-extra' ),
                        'type'       => KFCM::DIMENSIONS,
                        'size_units' => [ 'px', '%' ],
                        'selectors'  => [
                            '{{WRAPPER}} .kinfw-service-vertical-button-wrap a' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                        ]
                    ]);
                    $this->add_group_control( KFGCBS::get_type(), [
                        'name'     => 'readmore_btn_icon_box_shadow',
                        'selector' => '{{WRAPPER}} .kinfw-service-vertical-button-wrap a'
                    ]);
                $this->end_controls_tab();
                $this->start_controls_tab( 'readmore_btn_icon_hover_state', [
                    'label' => esc_html__( 'Hover', 'onnat-extra' ),
                ]);
                    $this->add_control( 'readmore_btn_icon_h_color', [
                        'label'     => esc_html__('Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'condition' => $icon_condition,
                        'selectors' => [
                            '{{WRAPPER}} .kinfw-service-vertical-button-wrap a:hover .kfw-elementor-icon-wrap i' => 'color:{{VALUE}};',
                            '{{WRAPPER}} .kinfw-service-vertical-button-wrap a:hover .kfw-elementor-icon-wrap svg' => 'fill:{{VALUE}};',
                        ],
                    ]);
					$this->add_control('icon_h_rotate',[
						'label'     => esc_html__( 'Rotate', 'onnat-extra' ),
						'type'      => KFCM::SLIDER,
						'range'     => [
							'deg' => [
								'max' => 360,
								'min' => -360,
							],
						],
						'selectors' => [
							#'{{WRAPPER}} .kinfw-service-vertical-button-wrap a:hover .kfw-elementor-icon-wrap'   => 'transform: rotate({{SIZE}}{{UNIT}});',
							'{{WRAPPER}} .kinfw-service-vertical-button-wrap a:hover .kfw-elementor-icon-wrap'   => 'transform: rotate({{SIZE}}deg);',
						],
					]);                    
                    $this->add_group_control( KFGCBG::get_type(), [
                        'name'           => 'readmore_btn_icon_h_bg',
                        'types'          => ['classic', 'gradient'],
                        'exclude'        => [ 'image'],
                        'fields_options' => [
                            'background' => [ 'default' => 'classic' ],
                            'color'      => [ 'default' => '' ],
                        ],
                        'selector'       => '{{WRAPPER}} .kinfw-service-vertical-button-wrap a:hover',
                    ]);
                    $this->add_group_control( KFGCBR::get_type(), [
                        'name'     => 'readmore_btn_icon_h_border',
                        'selector' => '{{WRAPPER}} .kinfw-service-vertical-button-wrap a:hover'
                    ]);
                    $this->add_responsive_control( 'readmore_btn_icon_h_border_radius', [
                        'label'      => esc_html__('Border Radius', 'onnat-extra' ),
                        'type'       => KFCM::DIMENSIONS,
                        'size_units' => [ 'px', '%' ],
                        'selectors'  => [
                            '{{WRAPPER}} .kinfw-service-vertical-button-wrap a:hover' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                        ]
                    ]);
                    $this->add_group_control( KFGCBS::get_type(), [
                        'name'     => 'readmore_btn_icon_h_box_shadow',
                        'selector' => '{{WRAPPER}} .kinfw-service-vertical-button-wrap a:hover'
                    ]);
                $this->end_controls_tab();
            $this->end_controls_tabs();

		$this->end_controls_section();
    }

	public function _services() {
		$result = [ esc_html__( '-- Select --', 'onnat-extra' ) ];

		$posts = get_posts([
			'post_type'      => 'kinfw-service',
			'post_status'    => 'publish',
			'posts_per_page' => -1
		]);

		if ( ! empty( $posts ) && ! is_wp_error( $posts ) ) {
			foreach ( $posts as $item ) {
				$result[ $item->ID ] = $item->post_title;
			}
		}

		return $result;
	}

    public function _render_button_icon( $icon ) {
		$icon = array_filter( $icon );
		if( !empty( $icon ) ) {
            return '<span class="kfw-elementor-icon-wrap">'.
                KFIM::try_get_icon_html( $icon, [ 'aria-hidden' => 'true' ] ) .
            '</span>';
        }
    }

    public function _render_button_img( $image ) {
        if( !empty( $image['url'] ) ) {
            $image_src = $image['url'];

			$image_title = $this->_get_image_title( $image );
			$image_title = empty( $image_title ) ? $this->get_title() : $image_title;

			$image_alt = $this->_get_image_alt( $image );
			$image_alt = empty( $image_alt ) ? $image_title : $image_title;

            return '<span class="kfw-elementor-icon-wrap">'.
                '<img alt="'. esc_attr( $image_alt ) .'" title="'. esc_attr( $image_title ) .'" src="'. esc_url( $image_src ) .'"/>'.
            '</span>';
        }
    }

	public function _get_image_title( $attachment ) {
		if ( empty( $attachment['id'] ) ) {
			return '';
		}

		return get_the_title( $attachment['id'] );
	}

	public function _get_image_alt( $instance ) {
		if ( empty( $instance['id'] ) ) {
			// For `Insert From URL` images.
			return isset( $instance['alt'] ) ? trim( strip_tags( $instance['alt'] ) ) : '';
		}

		$attachment_id = $instance['id'];
		if ( ! $attachment_id ) {
			return '';
		}

		$attachment = get_post( $attachment_id );
		if ( ! $attachment ) {
			return '';
		}

		$alt = get_post_meta( $attachment_id, '_wp_attachment_image_alt', true );
		if ( ! $alt ) {
			$alt = $attachment->post_excerpt;
			if ( ! $alt ) {
				$alt = $attachment->post_title;
			}
		}
		return trim( strip_tags( $alt ) );
	}

	/**
	 * Get attachment image src.
	 */
	public static function _get_image_src( $attachment_id, $image_size_key, array $settings ) {
        return KFGCIS::get_attachment_image_src( $attachment_id, $image_size_key, $settings );
	}

    public function _featured_image( $item, $settings ) {
        $post_id = $item['service'];

        if( 'custom-image' === $item['image-type'] && !empty( $item['custom-image']['url'] ) ) {
            $image_src = $item['custom-image']['url'];

			$image_title = $this->_get_image_title( $item['image-type'] );
			$image_title = empty( $image_title ) ? $this->get_title() : $image_title;

			$image_alt = $this->_get_image_alt( $item['image-type'] );
			$image_alt = empty( $image_alt ) ? $image_title : $image_title;

            return sprintf( '
                <img src="%1$s" alt="%2$s" class="wp-post-image %3$s">',
                $image_src,
                $image_alt,
                $image_title
            );

        } else if( 'featured-image' === $item['image-type'] && has_post_thumbnail( $post_id ) ) {
            $thumbnail_id  = get_post_thumbnail_id( $post_id );
            $thumbnail_url = get_the_post_thumbnail_url( 'full' );

            $image_src = $this->_get_image_src( $thumbnail_id, 'image_size', $settings );
            $image_src = empty($image_src) ? $thumbnail_url : $image_src;

            $image_title = $this->_get_image_title( $thumbnail_id );
            $image_title = empty( $image_title ) ? "" : $image_title;

            $image_alt = $this->_get_image_alt( $thumbnail_id );
            $image_alt = empty( $image_alt ) ? $image_title : $image_title;

            return sprintf( '
                <img src="%1$s" alt="%2$s" class="wp-post-image %3$s">',
                $image_src,
                $image_alt,
                $image_title
            );
        }
    }

	public function _counter() {
		return 'list-counter-'. $this->get_id();
	}

	/**
	 * Generate setting json
	 */
	protected function _generate_setting_json( array $settings ) {
		extract( $settings );

		$options = [
			'id'             => 'kfw-service-vertical'.esc_attr( $this->get_id() ),
			'trigger'        => 'mouseover',
			'active_item'    => true,
			'active_item_no' => !empty( $active_item ) ? $active_item : 1,
			'total_items'    => count( $items ),
		];

		return wp_json_encode( $options );
	}    

	/**
	 * Render widget output on the frontend.
	 */
    protected function render() {
		$render = kf_extra_plugin_elementor()->get_template("kfw-cpt-service-vertical");
		include $render;
    }
}