<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

use \Elementor\Plugin as KFEl;
use \Elementor\Widget_Base as KFWB;
use \Elementor\Controls_Manager as KFCM;
use \Elementor\Icons_Manager as KFIM;

use \Elementor\Group_Control_Typography as KFGCT;
use \Elementor\Group_Control_Text_Shadow as KFGCTS;
use \Elementor\Group_Control_Text_Stroke as KFGCTST;
use \Elementor\Group_Control_Background as KFGCBG;
use \Elementor\Group_Control_Border as KFGCBR;
use \Elementor\Group_Control_Box_Shadow as KFGCBS;

class KFW_Custom_Heading extends KFWB {

	/**
	 * Get element name.
	 *
	 * Retrieve the element name.
	 *
	 * @return string The name.
	 */
	public function get_name() {
        return 'kfw-custom-heading';
    }

	/**
	 * Get element title.
	 *
	 * Retrieve the element title.
	 *
	 * @return string Element title.
	 */
	public function get_title() {
		return esc_html__('Custom Heading', 'onnat-extra');
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the widget categories.
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ '_kfw_core_widgets' ];
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve the widget icon.
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'kfw-badge eicon-heading';
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the widget keywords.
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'kfw', 'kin', 'custom', 'heading'];
	}

	/**
	 * Get widget help url.
	 *
	 * Retrieve the widget help url.
	 *
	 * @return string Widget help url.
	 */
	public function get_custom_help_url() {
        return KFE_CONST_THEME_DOC . 'elementor-custom-widgets/custom-heading';
    }

	/**
	 * Get HTML wrapper class.
     *
	 * Retrieve the widget container class.
     *
     * @return string Widget container css classes.
	 */
	public function get_html_wrapper_class() {
        $name = $this->get_name();

		return sprintf( 'kfw-ele-%s elementor-widget-%s', $name, $name );
	}

	/**
	 * Widget base constructor.
	 *
	 * Initializing the widget base class.
	 */
	public function __construct( $data = [], $args = null ) {
        parent::__construct( $data, $args );

        if ( ! wp_script_is( 'kfw-gsap', 'enqueued' ) ) {
            wp_register_script( 'kfw-gsap', KFE_CONST_URL . 'assets/js/gsap.min.js', [], KFE_CONST_VERSION, true );
        }

        if ( ! wp_script_is( 'kfw-gsap-scroll-trigger', 'enqueued' ) ) {
            wp_register_script( 'kfw-gsap-scroll-trigger', KFE_CONST_URL . 'assets/js/ScrollTrigger.min.js', ['kfw-gsap'], KFE_CONST_VERSION, true );
        }

        if ( ! wp_script_is( 'kfw-gsap-split-text', 'enqueued' ) ) {
            wp_register_script( 'kfw-gsap-split-text', KFE_CONST_URL . 'assets/js/SplitText.min.js', ['kfw-gsap'], KFE_CONST_VERSION, true );
        }

        if ( ! wp_script_is( 'kfw-gsap-css-rule-plugin', 'enqueued' ) ) {
            wp_register_script( 'kfw-gsap-css-rule-plugin', KFE_CONST_URL . 'assets/js/CSSRulePlugin.min.js', ['kfw-gsap'], KFE_CONST_VERSION, true );
        }

		wp_register_script( 'kfw-custom-heading',
			KFE_CONST_URL . 'libraries/elementor/assets/js/custom-heading' . KFE_CONST_DEBUG_SUFFIX . '.js',
			[ 'jquery', 'elementor-frontend' ],
			KFE_CONST_VERSION,
			true
		);

		wp_register_style( 'kfw-custom-heading',
			KFE_CONST_URL . 'libraries/elementor/assets/css/custom-heading' . KFE_CONST_DEBUG_SUFFIX . '.css',
			[],
			KFE_CONST_VERSION
		);

    }

	/**
	 * Get style dependencies.
	 *
	 * Retrieve the list of style dependencies the element requires.
	 *
	 * @return array Element styles dependencies.
	 */
	public function get_style_depends() {
		return [ 'kfw-custom-heading' ];
	}

	/**
	 * Get script dependencies.
	 *
	 * Retrieve the list of script dependencies the element requires.
	 *
	 * @return array Element scripts dependencies.
	 */
	public function get_script_depends() {
        return [ 'kfw-gsap', 'kfw-gsap-scroll-trigger', 'kfw-gsap-split-text', 'kfw-gsap-css-rule-plugin', 'kfw-custom-heading' ];
	}

    public function _get_settings() {
        return $this->get_default_data();
    }

	/**
	 * Register controls.
	 *
	 * Used to add new controls to any element type.
	 */
	protected function register_controls() {
		$this->_register_subtitle_section();
		$this->_register_title_section();
		$this->_register_shadow_text_section();
		$this->_register_settings_section();

		$this->_register_subtitle_style_section();
		$this->_register_subtitle_icon_style_section();
		$this->_register_title_style_section();
		$this->_register_shadow_text_style_section();
    }

	public function _register_subtitle_section() {
		$this->start_controls_section( 'section_subtitle', [
			'label' => esc_html__( 'Sub Title', 'onnat-extra' )
		]);
            $this->add_control('sub_title',[
                'label'       => esc_html__( 'Sub Title', 'onnat-extra' ),
                'type'        => KFCM::TEXT,
                'dynamic'     => [ 'active' => true, ],
                'placeholder' => esc_html__( 'Enter Sub Title', 'onnat-extra' ),
                'default'     => esc_html__( 'Here is the sub title', 'onnat-extra' ),
                'label_block' => true,
            ]);
			$this->add_control('sub_title_pos',[
                'label'     => esc_html__( 'Position?', 'onnat-extra' ),
				'type'      => KFCM::CHOOSE,
				'options'   => [
					'top'    => [
						'title' => esc_html__( 'Above Title', 'onnat-extra' ),
						'icon'  => 'eicon-v-align-top',
					],
					'bottom' => [
						'title' => esc_html__( 'Below Title', 'onnat-extra' ),
						'icon'  => 'eicon-v-align-bottom',
					],
				],
				'default'   => 'top',
				'toggle'    => false,
				'condition' => [
					'title!'     => '',
					'sub_title!' => ''
				]
			]);
			$this->add_control('sub_title_icon_type',[
				'label'     => esc_html__( 'Icon Type', 'onnat-extra' ),
				'type'      => KFCM::CHOOSE,
				'options'   => [
					'blank' => [
						'title' => esc_html__('Blank', 'onnat-extra'),
						'icon'  => '',
					],
					'icon'  => [
						'title' => esc_html__('Icon', 'onnat-extra'),
						'icon'  => 'eicon-plus-circle-o',
					],
					'image'  => [
						'title' => esc_html__('Image', 'onnat-extra'),
						'icon'  => 'eicon-image',
					],
				],
				'default'   => 'blank',
				'toggle'    => false,
				'condition' => [
					'title!'     => '',
					'sub_title!' => '',
                ],
			]);
			$this->add_control( 'sub_title_icon', [
				'label'     => esc_html__( 'Icon', 'onnat-extra' ),
				'type'      => KFCM::ICONS,
				'default'   => [
					'value'   => 'fas fa-heart',
					'library' => 'fa-solid',
				],
				'condition' => [
                    'title!'              => '',
                    'sub_title!'          => '',
                    'sub_title_icon_type' => 'icon',
                ],
			]);
			$this->add_control( 'sub_title_image', [
				'label'     => esc_html__( 'Image', 'onnat-extra' ),
				'type'      => KFCM::MEDIA,
				'dynamic'   => [ 'active' => true ],
				'default'   => [
					'url' => \Elementor\Utils::get_placeholder_image_src()
				],
				'condition' => [ 
                    'title!'              => '',
                    'sub_title!'          => '',
                    'sub_title_icon_type' => 'image',
                ],
			]);
			$this->add_control('sub_title_icon_position', [
				'label'          => esc_html__( 'Position', 'onnat-extra' ),
				'type'           => KFCM::CHOOSE,
				'options'        => [
					'left'   => [
						'title' => esc_html__( 'Left', 'onnat-extra' ),
						'icon'  => 'eicon-h-align-left',
					],
					'top'    => [
						'title' => esc_html__( 'Top', 'onnat-extra' ),
						'icon'  => 'eicon-v-align-top',
					],
					'right'  => [
						'title' => esc_html__( 'Right', 'onnat-extra' ),
						'icon'  => 'eicon-h-align-right',
					],
					'bottom' => [
						'title' => esc_html__( 'Bottom', 'onnat-extra' ),
						'icon'  => 'eicon-v-align-bottom',
					],
				],
				'default'        => 'top',
				'toggle'         => false,
				'prefix_class'   => 'kfw-ele-custom-heading-icon-position-',
				'conditions'     => [
                    'relation' => 'and',
					'terms'    => [
                        [
							'name'     => 'title',
							'operator' => '!==',
							'value'    => ''
                        ],
                        [
							'name'     => 'sub_title',
							'operator' => '!==',
							'value'    => ''
                        ],                        
						[
							'name'     => 'sub_title_icon_type',
							'operator' => '!==',
							'value'    => 'blank'
						],
						[
							'relation' => 'or',
							'terms'    => [
								[
									'name'     => 'sub_title_icon[value]',
									'operator' => '!==',
									'value'    => '',
								],
								[
									'name'     => 'sub_title_image[url]',
									'operator' => '!==',
									'value'    => '',
								],                                
                            ],
						],
                    ],
                ],
			]);
		$this->end_controls_section();
    }

	public function _register_title_section() {
		$this->start_controls_section( 'section_title', [
			'label' => esc_html__( 'Title', 'onnat-extra' )
		]);
			$this->add_control('title',[
				'label'       => esc_html__( 'Title', 'onnat-extra' ),
				'type'        => KFCM::TEXT,
				'dynamic'     => [ 'active' => true, ],
				'placeholder' => esc_html__( 'Enter Title', 'onnat-extra' ),
				'default'     => esc_html__( 'Here is the title', 'onnat-extra' ),
                'description' => esc_html__( '"Secondary Title" Settings will take effect if you use this {{something}} format.', 'onnat-extra' ),
				'label_block' => true,
			]);
            $this->add_control('title_link',[
                'label'     => esc_html__( 'Title Link', 'onnat-extra' ),
                'type'      => KFCM::URL,
                'dynamic'   => [
                    'active' => true
                ],
                'options'   => [
                    'url',
                    'is_external',
                    'nofollow'
                ],
                'default'   => [
                    'url'         => '',
                    'is_external' => true,
                    'nofollow'    => true,
                ],
                'condition' => [
                    'title!' => ''
                ]
            ]);
            $this->add_control('title_tag', [
                'label'     => esc_html__( 'HTML Tag', 'onnat-extra' ),
                'type'      => KFCM::CHOOSE,
                'options'   => [
                    'h1'   => [
                        'title' => esc_html__( 'H1 Tag', 'onnat-extra' ),
                        'icon'  => 'eicon-editor-h1',
                    ],
                    'h2'   => [
                        'title' => esc_html__( 'H2 Tag', 'onnat-extra' ),
                        'icon'  => 'eicon-editor-h2',
                    ],
                    'h3'   => [
                        'title' => esc_html__( 'H3 Tag', 'onnat-extra' ),
                        'icon'  => 'eicon-editor-h3',
                    ],
                    'h4'   => [
                        'title' => esc_html__( 'H4 Tag', 'onnat-extra' ),
                        'icon'  => 'eicon-editor-h4',
                    ],
                    'h5'   => [
                        'title' => esc_html__( 'H5 Tag', 'onnat-extra' ),
                        'icon'  => 'eicon-editor-h5',
                    ],
                    'h6'   => [
                        'title' => esc_html__( 'H6 Tag', 'onnat-extra' ),
                        'icon'  => 'eicon-editor-h6',
                    ],
                    'div'  => [
                        'title' => esc_html__( 'div Tag', 'onnat-extra' ),
                        'icon'  => 'kfw-ctrl-icon  kfw-ctrl-div',
                    ],
                    'span' => [
                        'title' => esc_html__( 'span Tag', 'onnat-extra' ),
                        'icon'  => 'kfw-ctrl-icon kfw-ctrl-span',
                    ],
                    'p'    => [
                        'title' => esc_html__( 'p Tag', 'onnat-extra' ),
                        'icon'  => 'eicon-editor-paragraph',
                    ],
                ],
                'default'   => 'h2',
                'toggle'    => false,
                'condition' => [
                    'title!'      => '',
                ]
            ]);
		$this->end_controls_section();
	}

    public function _register_shadow_text_section() {
		$this->start_controls_section( 'section_shadow_text', [
			'label' => esc_html__( 'Shadow Text', 'onnat-extra' )
		]);
            $this->add_control('use_shadow_text',[
                'label'        => esc_html__( 'Use Shadow Text?', 'onnat-extra' ),
                'type'         => KFCM::SWITCHER,
                'return_value' => 'yes',
                'default'      => 'no'
            ]);
			$this->add_control('shadow_text',[
                'label'       => esc_html__( 'Text', 'onnat-extra' ),
                'type'        => KFCM::TEXT,
                'dynamic'     => [ 'active' => true, ],
                'placeholder' => esc_html__( 'Enter Title', 'onnat-extra' ),
                'default'     => esc_html__( 'Shadow title', 'onnat-extra' ),
                'label_block' => true,
                'condition'   => [
                    'use_shadow_text' => 'yes'
                ],
			]);
            $this->add_control('st_position',[
                'label'     => esc_html__('Position', 'onnat-extra'),
                'type'      => KFCM::POPOVER_TOGGLE,
                'condition' => [
                    'use_shadow_text' => 'yes',
                    'shadow_text!'    => ''
                ],
            ]);
                $this->start_popover();
                    $this->add_responsive_control('st_x_position', [
                        'label'          => esc_html__('X Offset', 'onnat-extra'),
                        'type'           => KFCM::SLIDER,
                        'default'        => [
                            'size' => 0
                        ],
                        'tablet_default' => [ 'size' => 0 ],
                        'mobile_default' => [ 'size' => 0 ],
                        'range'          => [ 'px' => [ 'min' => -1000, 'max' => 1000 ]],
                        'selectors'      => [
                            '{{WRAPPER}}' => '--kfw-custom-heading-pos-x:{{SIZE}}px;'
                        ]
                    ]);
                    $this->add_responsive_control('st_y_position', [
                        'label'          => esc_html__('Y Offset', 'onnat-extra'),
                        'type'           => KFCM::SLIDER,
                        'default'        => [ 'size' => 0 ],
                        'tablet_default' => [ 'size' => 0 ],
                        'mobile_default' => [ 'size' => 0 ],
                        'range'          => [ 'px' => [ 'min' => -1000, 'max' => 1000 ]],
                        'selectors'      => [
                            '{{WRAPPER}}' => '--kfw-custom-heading-pos-y:{{SIZE}}px;'
                        ]
                    ]);
                $this->end_popover();

            $this->add_control('st_rotation',[
                'label'     => esc_html__('Rotation', 'onnat-extra'),
                'type'      => KFCM::POPOVER_TOGGLE,
                'condition' => [
                    'use_shadow_text' => 'yes',
                    'shadow_text!'    => ''
                ],
            ]);
                $this->start_popover();
                    $this->add_responsive_control('st_rotate', [
                        'label'          => esc_html__('Rotate', 'onnat-extra'),
                        'type'           => KFCM::SLIDER,
                        'default'        => [ 'size' => 0 ],
                        'tablet_default' => [ 'size' => 0 ],
                        'mobile_default' => [ 'size' => 0 ],
                        'range'          => [ 'px' => [ 'min' => -180, 'max' => 180 ]],
                        'selectors'      => [
                            '{{WRAPPER}}' => '--kfw-custom-heading-rotate:{{SIZE}}deg;'
                        ]
                    ]);
                    $this->add_control('st_rotation_x_orgin', [
                        'label'   => esc_html__('X Axis', 'onnat-extra'),
                        'type'    => KFCM::CHOOSE,
                        'options' => [
                            'left'   => [
                                'title' => esc_html__('Left', 'onnat-extra'),
                                'icon'  => 'eicon-h-align-left'
                            ],
                            'center' => [
                                'title' => esc_html__('Center', 'onnat-extra'),
                                'icon'  => 'eicon-h-align-center'
                            ],
                            'right'  => [
                                'title' => esc_html__('Right', 'onnat-extra'),
                                'icon'  => 'eicon-h-align-right'
                            ],
                        ],
                        'default'   => 'center',
                        'toggle'    => false,
                        'selectors' => [
                            '{{WRAPPER}}' => '--kfw-custom-heading-transform-origin-x:{{VALUE}};'
                        ]
                    ]);
                    $this->add_control('st_rotation_y_orgin', [
                        'label'   => esc_html__('Y Axis', 'onnat-extra'),
                        'type'    => KFCM::CHOOSE,
                        'options' => [
                            'top'    => [
                                'title' => esc_html__('Top', 'onnat-extra'),
                                'icon'  => 'eicon-v-align-top'
                            ],
                            'center' => [
                                'title' => esc_html__('Center', 'onnat-extra'),
                                'icon'  => 'eicon-h-align-center'
                            ],
                            'bottom' => [
                                'title' => esc_html__('Bottom', 'onnat-extra'),
                                'icon'  => 'eicon-v-align-bottom'
                            ],
                        ],
                        'default'   => 'center',
                        'toggle'    => false,
                        'selectors' => [
                            '{{WRAPPER}}' => '--kfw-custom-heading-transform-origin-y:{{VALUE}};'
                        ]
                    ]);
                $this->end_popover();
		$this->end_controls_section();
    }

	public function _register_settings_section() {
		$this->start_controls_section( 'section_settings', [
			'label' => esc_html__( 'Settings', 'onnat-extra' )
		]);
            $this->add_responsive_control('align',[
				'label'        => esc_html__( 'Alignment', 'onnat-extra' ),
				'type'         => KFCM::CHOOSE,
				'options'      => [
					'left'    => [
                        'title' => esc_html__( 'Left', 'onnat-extra' ),
                        'icon'  => 'eicon-text-align-left',
                    ],
					'center'  => [
                        'title' => esc_html__( 'Center', 'onnat-extra' ),
                        'icon'  => 'eicon-text-align-center',
                    ],
					'right'   => [
                        'title' => esc_html__( 'Right', 'onnat-extra' ),
                        'icon'  => 'eicon-text-align-right',
                    ],
					'justify' => [
                        'title' => esc_html__( 'Justified', 'onnat-extra' ),
                        'icon'  => 'eicon-text-align-justify',
                    ],
				],
                'toggle'       => false,
                'default'      => 'center',
                'prefix_class' => 'kfw-elementor-custom-heading%s-align-',
			]);
            $this->add_control('anim_title',[
				'label' => esc_html__( 'Title Animation?', 'onnat-extra' ),
				'type'  => KFCM::SWITCHER,
			]);
			$this->add_control('anim_type',[
				'label'     => esc_html__( 'Type', 'onnat-extra' ),
				'type'      => KFCM::SELECT,
				'default'   => 'style-1',
				'options'   => [
                    'style-1'  => esc_html__('Style 1', 'onnat-extra'),
                    'style-2'  => esc_html__('Style 2', 'onnat-extra'),
                    'style-3'  => esc_html__('Style 3', 'onnat-extra'),
                    'style-4'  => esc_html__('Style 4', 'onnat-extra'),
                    'style-5'  => esc_html__('Style 5', 'onnat-extra'),
                    'style-6'  => esc_html__('Style 6', 'onnat-extra'),
                    'style-7'  => esc_html__('Style 7', 'onnat-extra'),
                    'style-8'  => esc_html__('Style 8', 'onnat-extra'),
                    'style-9'  => esc_html__('Style 9', 'onnat-extra'),
                    'style-10' => esc_html__('Style 10', 'onnat-extra'),
                ],
                'prefix_class' => 'kfw-custom-heading-anim kfw-custom-heading-anim-',
                'condition' => [
					'anim_title' => 'yes'
				],
			]);
			$this->add_control('animate',[
                'label'     => esc_html__( 'Animate?', 'onnat-extra' ),
                'type'      => KFCM::CHOOSE,
                'options'   => [
                    'chars' => [ 'title' => esc_html__( 'Chars', 'onnat-extra' ), 'icon'  => 'eicon-animation-text',],
                    'words' => [ 'title' => esc_html__( 'Words', 'onnat-extra' ), 'icon'  => 'eicon-wordart',],
                ],
                'toggle'    => false,
                'default'   => 'words',
                'condition' => [
                    'anim_title' => 'yes',
                    'anim_type!' => 'style-6'
                ]
            ]);
		$this->end_controls_section();
    }

    public function _register_subtitle_style_section() {
		$this->start_controls_section( 'section_sub_title_style', [
			'label'     => esc_html__( 'Sub Title', 'onnat-extra' ),
			'tab'       => KFCM::TAB_STYLE,
			'condition' => [ 'sub_title!'=> '' ]
		]);
            $this->add_group_control(KFGCT::get_type(), [
                'name'      => 'sub_title_typo',
                'label'     => esc_html__( 'Typography', 'onnat-extra' ),
                'condition' => [ 'sub_title!'=> '' ],
                'selector'  => '{{WRAPPER}} .kfw-ele-widget-custom-heading-sub-title',
            ]);
            $this->add_responsive_control( 'sub_title_margin', [
                'label'      => esc_html__('Margin', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'condition'  => [ 'sub_title!'=> '' ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-ele-widget-custom-heading-sub-title-wrap' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->add_responsive_control( 'sub_title_padding', [
                'label'      => esc_html__('Padding', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'condition'  => [ 'sub_title!'=> '' ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-ele-widget-custom-heading-sub-title-wrap' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->start_controls_tabs( 'sub_title_customize_tabs' );
                $this->start_controls_tab( 'sub_title_normal_state', [
                    'label'     => esc_html__( 'Normal', 'onnat-extra' ),
                    'condition' => [ 'sub_title!'=> '' ],
                ]);
                    $this->add_control( 'sub_title_color', [
                        'label'     => esc_html__('Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'selectors' => ['{{WRAPPER}} .kfw-ele-widget-custom-heading-sub-title' => 'color: {{VALUE}};' ],
                        'condition' => [ 'sub_title!'=> '' ],
                    ]);
                    $this->add_group_control( KFGCTS::get_type(), [
                        'name'      => 'sub_title_txt_shadow',
                        'condition' => [ 'sub_title!'=> '' ],
                        'selector'  => '{{WRAPPER}} .kfw-ele-widget-custom-heading-sub-title',
                    ]);
                    $this->add_group_control( KFGCTST::get_type(), [
                        'name'      => 'sub_title_txt_stroke',
                        'condition' => [ 'sub_title!'=> '' ],
                        'selector'  => '{{WRAPPER}} .kfw-ele-widget-custom-heading-sub-title',
                    ]);
                    $this->add_group_control( KFGCBG::get_type(), [
                        'name'      => 'sub_title_bg',
                        'label'     => esc_html__('Background', 'onnat-extra' ),
                        'types'     => [ 'classic', 'gradient' ],
                        'exclude'   => [ 'image' ],# To remove background image option in classic mode
                        'selector'  => '{{WRAPPER}} .kfw-ele-widget-custom-heading-sub-title-wrap',
                        'condition' => [ 'sub_title!'=> '' ],
                    ]);
                    $this->add_group_control( KFGCBR::get_type(), [
                        'name'           => 'sub_title_border',
                        'fields_options' => [
                            'width' => [ 'label' => _x( 'Border Width', 'Border Control', 'onnat-extra' ) ],
                            'color' => [ 'label' => _x( 'Border Color', 'Border Control', 'onnat-extra' ) ],
                        ],
                        'selector'       => '{{WRAPPER}} .kfw-ele-widget-custom-heading-sub-title-wrap',
                        'condition'      => [ 'sub_title!'=> '' ],
                    ]);
                    $this->add_responsive_control( 'sub_title_border_radius', [
                        'label'      => esc_html__('Border Radius', 'onnat-extra' ),
                        'type'       => KFCM::DIMENSIONS,
                        'size_units' => [ 'px', '%' ],
                        'selectors'  => [ '{{WRAPPER}} .kfw-ele-widget-custom-heading-sub-title-wrap' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};' ],
                        'condition'  => [ 'sub_title!'=> '' ],
                    ]);
                    $this->add_group_control( KFGCBS::get_type(), [
                        'name'      => 'sub_title_box_shadow',
                        'selector'  => '{{WRAPPER}} .kfw-ele-widget-custom-heading-sub-title-wrap',
                        'condition' => [ 'sub_title!'=> '' ],
                    ]);
                $this->end_controls_tab();
				$this->start_controls_tab( 'sub_title_hover_state', [
					'label'     => esc_html__( 'Hover', 'onnat-extra' ),
					'condition' => [ 'sub_title!'=> '' ],
				]);
                    $this->add_control( 'sub_title_hover_color', [
                        'label'     => esc_html__('Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'selectors' => ['{{WRAPPER}}:hover .kfw-ele-widget-custom-heading-sub-title' => 'color: {{VALUE}};' ],
                        'condition' => [ 'sub_title!'=> '' ],
                    ]);
                    $this->add_group_control( KFGCTS::get_type(), [
                        'name'      => 'sub_title_hover_txt_shadow',
                        'condition' => [ 'sub_title!'=> '' ],
                        'selector'  => '{{WRAPPER}}:hover .kfw-ele-widget-custom-heading-sub-title',
                    ]);
                    $this->add_group_control( KFGCTST::get_type(), [
                        'name'      => 'sub_title_hover_txt_stroke',
                        'condition' => [ 'sub_title!'=> '' ],
                        'selector'  => '{{WRAPPER}}:hover .kfw-ele-widget-custom-heading-sub-title',
                    ]);
                    $this->add_group_control( KFGCBG::get_type(), [
                        'name'      => 'sub_title_hover_bg',
                        'label'     => esc_html__('Background', 'onnat-extra' ),
                        'types'     => [ 'classic', 'gradient' ],
                        'exclude'   => [ 'image' ],# To remove background image option in classic mode
                        'selector'  => '{{WRAPPER}}:hover .kfw-ele-widget-custom-heading-sub-title-wrap',
                        'condition' => [ 'sub_title!'=> '' ],
                    ]);
                    $this->add_group_control( KFGCBR::get_type(), [
                        'name'           => 'sub_title_hover_border',
                        'fields_options' => [
                            'width' => [ 'label' => _x( 'Border Width', 'Border Control', 'onnat-extra' ) ],
                            'color' => [ 'label' => _x( 'Border Color', 'Border Control', 'onnat-extra' ) ],
                        ],
                        'selector'       => '{{WRAPPER}}:hover .kfw-ele-widget-custom-heading-sub-title-wrap',
                        'condition'      => [ 'sub_title!'=> '' ],
                    ]);
                    $this->add_responsive_control( 'sub_title_hover_border_radius', [
                        'label'      => esc_html__('Border Radius', 'onnat-extra' ),
                        'type'       => KFCM::DIMENSIONS,
                        'size_units' => [ 'px', '%' ],
                        'selectors'  => [ '{{WRAPPER}}:hover .kfw-ele-widget-custom-heading-sub-title-wrap' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};' ],
                        'condition'  => [ 'sub_title!'=> '' ],
                    ]);
                    $this->add_group_control( KFGCBS::get_type(), [
                        'name'      => 'sub_title_hover_box_shadow',
                        'selector'  => '{{WRAPPER}}:hover .kfw-ele-widget-custom-heading-sub-title-wrap',
                        'condition' => [ 'sub_title!'=> '' ],
                    ]);
                $this->end_controls_tab();
			$this->end_controls_tabs();
		$this->end_controls_section();
    }

    public function _register_subtitle_icon_style_section() {
        $conditions = [
            'relation' => 'and',
            'terms'    => [
                [
                    'name'     => 'title',
                    'operator' => '!==',
                    'value'    => ''
                ],
                [
                    'name'     => 'sub_title',
                    'operator' => '!==',
                    'value'    => ''
                ],                        
                [
                    'name'     => 'sub_title_icon_type',
                    'operator' => '!==',
                    'value'    => 'blank'
                ],
                [
                    'relation' => 'or',
                    'terms'    => [
                        [
                            'name'     => 'sub_title_icon[value]',
                            'operator' => '!==',
                            'value'    => '',
                        ],
                        [
                            'name'     => 'sub_title_image[url]',
                            'operator' => '!==',
                            'value'    => '',
                        ],                                
                    ],
                ],
            ],
        ];

		$icon_condition = [
			'title!'                 => '',
			'sub_title!'             => '',
			'sub_title_icon_type'    => 'icon',
			'sub_title_icon[value]!' => '',
		];

		$img_condition = [
			'title!'                 => '',
			'sub_title!'             => '',
			'sub_title_icon_type'   => 'image',
			'sub_title_image[url]!' => ''
		];        

		$this->start_controls_section( 'section_sub_title_icon_style', [
			'label'      => esc_html__( 'Sub Title Icon', 'onnat-extra' ),
			'tab'        => KFCM::TAB_STYLE,
			'conditions' => $conditions,
		]);
            $this->add_responsive_control( 'icon_size', [
                'label'      => esc_html__('Icon Size', 'onnat-extra' ),
                'type'       => KFCM::SLIDER,
                'size_units' => [ 'px' ],
                'range'      => [
                    'px' => [
                        'min' => 0,
                        'max' => 999,
                    ],
                ],
                'condition'  => $icon_condition,
                'selectors'  => [
                    '{{WRAPPER}} .kfw-ele-widget-custom-heading-sub-title-wrap .kfw-elementor-icon-wrap' => 'font-size: {{SIZE}}{{UNIT}};'
                ],
            ]); 
            $this->add_responsive_control( 'img_size', [
				'label'      => esc_html__('Image Size', 'onnat-extra' ),
				'type'       => KFCM::SLIDER,
				'size_units' => ['%'],
				'range'      => [
					'%' => [
						'min' => 5,
						'max' => 100,
					],
				],
				'condition'  => $img_condition,
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-custom-heading-sub-title-wrap .kfw-elementor-icon-wrap img' => 'max-width: {{SIZE}}%;'
				],
			]);
			$this->add_responsive_control( 'icon_margin', [
				'label'      => esc_html__('Margin', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'conditions' => $conditions,
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-custom-heading-sub-title-wrap .kfw-elementor-icon-wrap' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
			$this->add_responsive_control( 'icon_padding', [
				'label'      => esc_html__('Padding', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'conditions' => $conditions,
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-custom-heading-sub-title-wrap .kfw-elementor-icon-wrap' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
			$this->start_controls_tabs( 'icon_or_img_customize_tabs' );
				$this->start_controls_tab( 'icon_normal_state', [
					'label'     => esc_html__( 'Normal', 'onnat-extra' ),
					'condition' => $icon_condition
				]);
                    $this->add_control( 'icon_color', [
                        'label'     => esc_html__('Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'condition' => $icon_condition,
                        'selectors' => [
                            '{{WRAPPER}} .kfw-ele-widget-custom-heading-sub-title-wrap .kfw-elementor-icon-wrap i' => 'color:{{VALUE}};',
                            '{{WRAPPER}} .kfw-ele-widget-custom-heading-sub-title-wrap .kfw-elementor-icon-wrap svg' => 'fill:{{VALUE}};',
                        ],
                    ]);
                    $this->add_control('icon_rotate',[
                        'label'     => esc_html__( 'Rotate', 'onnat-extra' ),
                        'type'      => KFCM::SLIDER,
                        'default'   => [
                            'size' => 0,
                            'unit' => 'deg',
                        ],
                        'range'     => [
                            'deg' => [
                                'max' => 360,
                                'min' => - 360,
                            ],
                        ],
                        'condition' => $icon_condition,
                        'selectors' => [
                            '{{WRAPPER}} .kfw-ele-widget-custom-heading-sub-title-wrap .kfw-elementor-icon-wrap i'   => 'transform: rotate({{SIZE}}{{UNIT}});',
                            '{{WRAPPER}} .kfw-ele-widget-custom-heading-sub-title-wrap .kfw-elementor-icon-wrap svg' => 'transform: rotate({{SIZE}}{{UNIT}});',
                        ],
                    ]);
				$this->end_controls_tab();
				$this->start_controls_tab( 'icon_hover_state', [
					'label'     => esc_html__( 'Hover', 'onnat-extra' ),
					'condition' => $icon_condition,
				]);
                    $this->add_control( 'icon_hover_color', [
                        'label'     => esc_html__('Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'condition' => $icon_condition,
                        'selectors' => [
                            '{{WRAPPER}}:hover .kfw-ele-widget-custom-heading-sub-title-wrap .kfw-elementor-icon-wrap i' => 'color:{{VALUE}};',
                            '{{WRAPPER}}:hover .kfw-ele-widget-custom-heading-sub-title-wrap .kfw-elementor-icon-wrap svg' => 'fill:{{VALUE}};',
                        ],
                    ]);
                    $this->add_control('icon_hover_rotate',[
                        'label'     => esc_html__( 'Rotate', 'onnat-extra' ),
                        'type'      => KFCM::SLIDER,
                        'default'   => [
                            'size' => 0,
                            'unit' => 'deg',
                        ],
                        'range'     => [
                            'deg' => [
                                'max' => 360,
                                'min' => - 360,
                            ],
                        ],
                        'condition' => $icon_condition,
                        'selectors' => [
                            '{{WRAPPER}}:hover .kfw-ele-widget-custom-heading-sub-title-wrap .kfw-elementor-icon-wrap i'   => 'transform: rotate({{SIZE}}{{UNIT}});',
                            '{{WRAPPER}}:hover .kfw-ele-widget-custom-heading-sub-title-wrap .kfw-elementor-icon-wrap svg' => 'transform: rotate({{SIZE}}{{UNIT}});',
                        ],
                    ]);                
				$this->end_controls_tab();

				$this->start_controls_tab( 'img_normal_state', [
					'label'     => esc_html__( 'Normal', 'onnat-extra' ),
					'condition' => $img_condition
				]);
                    $this->add_control('img_rotate',[
                        'label'     => esc_html__( 'Rotate', 'onnat-extra' ),
                        'type'      => KFCM::SLIDER,
                        'default'   => [
                            'size' => 0,
                            'unit' => 'deg',
                        ],
                        'range'     => [
                            'deg' => [
                                'max' => 360,
                                'min' => - 360,
                            ],
                        ],
                        'condition' => $img_condition,
                        'selectors' => [
                            '{{WRAPPER}} .kfw-ele-widget-custom-heading-sub-title-wrap .kfw-elementor-icon-wrap img' => 'transform: rotate({{SIZE}}{{UNIT}});',
                        ],
                    ]);                
				$this->end_controls_tab();
				$this->start_controls_tab( 'img_hover_state', [
					'label'     => esc_html__( 'Hover', 'onnat-extra' ),
					'condition' => $img_condition
				]);
                    $this->add_control('img_hover_rotate',[
                        'label'     => esc_html__( 'Rotate', 'onnat-extra' ),
                        'type'      => KFCM::SLIDER,
                        'default'   => [
                            'size' => 0,
                            'unit' => 'deg',
                        ],
                        'range'     => [
                            'deg' => [
                                'max' => 360,
                                'min' => - 360,
                            ],
                        ],
                        'condition' => $img_condition,
                        'selectors' => [
                            '{{WRAPPER}}:hover .kfw-ele-widget-custom-heading-sub-title-wrap .kfw-elementor-icon-wrap img' => 'transform: rotate({{SIZE}}{{UNIT}});',
                        ],
                    ]);                
				$this->end_controls_tab();                
			$this->end_controls_tabs();
		$this->end_controls_section();
    }

    public function _register_title_style_section() {
		$this->start_controls_section( 'section_title_style', [
			'label'     => esc_html__( 'Title', 'onnat-extra' ),
			'tab'       => KFCM::TAB_STYLE,
			'condition' => [ 'title!'=> '' ]
		]);
            $this->add_responsive_control( 'title_margin', [
                'label'      => esc_html__('Margin', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'condition'  => [ 'title!'=> '' ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-ele-widget-custom-heading-title-wrap' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->add_responsive_control( 'title_padding', [
                'label'      => esc_html__('Padding', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'condition'  => [ 'title!'=> '' ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-ele-widget-custom-heading-title-wrap' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);

            $this->add_control('primary_title_style_before_hr', [
                'type'       => KFCM::DIVIDER,
                'condition' => [ 'title!'=> '' ]
            ]);
            $this->add_control('primary_title_style_heading',[
                'label'      => esc_html__( 'PRIMARY TITLE', 'onnat-extra' ),
                'type'       => KFCM::HEADING,
                'condition' => [ 'title!'=> '' ]
            ]);
            $this->add_group_control(KFGCT::get_type(), [
                'name'      => 'title_typo',
                'label'     => esc_html__( 'Typography', 'onnat-extra' ),
                'condition' => [ 'title!'=> '' ],
                'selector'  => '{{WRAPPER}} .kfw-ele-widget-custom-heading-title',
            ]);
            $this->start_controls_tabs( 'title_customize_tabs' );
                $this->start_controls_tab( 'title_normal_state', [
                    'label'     => esc_html__( 'Normal', 'onnat-extra' ),
                    'condition' => [ 'title!'=> '' ],
                ]);
                    $this->add_control( 'title_color', [
                        'label'     => esc_html__('Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'selectors' => ['{{WRAPPER}} .kfw-ele-widget-custom-heading-title' => 'color: {{VALUE}};' ],
                        'condition' => [ 'title!'=> '' ],
                    ]);
                    $this->add_group_control( KFGCTS::get_type(), [
                        'name'      => 'title_txt_shadow',
                        'condition' => [ 'title!'=> '' ],
                        'selector'  => '{{WRAPPER}} .kfw-ele-widget-custom-heading-title',
                    ]);
                    $this->add_group_control( KFGCTST::get_type(), [
                        'name'      => 'title_txt_stroke',
                        'condition' => [ 'title!'=> '' ],
                        'selector'  => '{{WRAPPER}} .kfw-ele-widget-custom-heading-title',
                    ]);
                    $this->add_group_control( KFGCBG::get_type(), [
                        'name'      => 'title_bg',
                        'label'     => esc_html__('Background', 'onnat-extra' ),
                        'types'     => [ 'classic', 'gradient' ],
                        'exclude'   => [ 'image' ],# To remove background image option in classic mode
                        'selector'  => '{{WRAPPER}} .kfw-ele-widget-custom-heading-title-wrap',
                        'condition' => [ 'title!'=> '' ],
                    ]);
                    $this->add_group_control( KFGCBR::get_type(), [
                        'name'           => 'title_border',
                        'fields_options' => [
                            'width' => [ 'label' => _x( 'Border Width', 'Border Control', 'onnat-extra' ) ],
                            'color' => [ 'label' => _x( 'Border Color', 'Border Control', 'onnat-extra' ) ],
                        ],
                        'selector'       => '{{WRAPPER}} .kfw-ele-widget-custom-heading-title-wrap',
                        'condition'      => [ 'title!'=> '' ],
                    ]);
                    $this->add_responsive_control( 'title_border_radius', [
                        'label'      => esc_html__('Border Radius', 'onnat-extra' ),
                        'type'       => KFCM::DIMENSIONS,
                        'size_units' => [ 'px', '%' ],
                        'selectors'  => [ '{{WRAPPER}} .kfw-ele-widget-custom-heading-title-wrap' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};' ],
                        'condition'  => [ 'title!'=> '' ],
                    ]);
                    $this->add_group_control( KFGCBS::get_type(), [
                        'name'      => 'title_box_shadow',
                        'selector'  => '{{WRAPPER}} .kfw-ele-widget-custom-heading-title-wrap',
                        'condition' => [ 'title!'=> '' ],
                    ]);
                $this->end_controls_tab();
				$this->start_controls_tab( 'title_hover_state', [
					'label'     => esc_html__( 'Hover', 'onnat-extra' ),
					'condition' => [ 'title!'=> '' ],
				]);
                    $this->add_control( 'title_hover_color', [
                        'label'     => esc_html__('Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'selectors' => ['{{WRAPPER}}:hover .kfw-ele-widget-custom-heading-title' => 'color: {{VALUE}};' ],
                        'condition' => [ 'title!'=> '' ],
                    ]);
                    $this->add_group_control( KFGCTS::get_type(), [
                        'name'      => 'title_hover_txt_shadow',
                        'condition' => [ 'title!'=> '' ],
                        'selector'  => '{{WRAPPER}}:hover .kfw-ele-widget-custom-heading-title',
                    ]);
                    $this->add_group_control( KFGCTST::get_type(), [
                        'name'      => 'title_hover_txt_stroke',
                        'condition' => [ 'title!'=> '' ],
                        'selector'  => '{{WRAPPER}}:hover .kfw-ele-widget-custom-heading-title',
                    ]);
                    $this->add_group_control( KFGCBG::get_type(), [
                        'name'      => 'title_hover_bg',
                        'label'     => esc_html__('Background', 'onnat-extra' ),
                        'types'     => [ 'classic', 'gradient' ],
                        'exclude'   => [ 'image' ],# To remove background image option in classic mode
                        'selector'  => '{{WRAPPER}}:hover .kfw-ele-widget-custom-heading-title-wrap',
                        'condition' => [ 'title!'=> '' ],
                    ]);
                    $this->add_group_control( KFGCBR::get_type(), [
                        'name'           => 'title_hover_border',
                        'fields_options' => [
                            'width' => [ 'label' => _x( 'Border Width', 'Border Control', 'onnat-extra' ) ],
                            'color' => [ 'label' => _x( 'Border Color', 'Border Control', 'onnat-extra' ) ],
                        ],
                        'selector'       => '{{WRAPPER}}:hover .kfw-ele-widget-custom-heading-title-wrap',
                        'condition'      => [ 'title!'=> '' ],
                    ]);
                    $this->add_responsive_control( 'title_hover_border_radius', [
                        'label'      => esc_html__('Border Radius', 'onnat-extra' ),
                        'type'       => KFCM::DIMENSIONS,
                        'size_units' => [ 'px', '%' ],
                        'selectors'  => [ '{{WRAPPER}}:hover .kfw-ele-widget-custom-heading-title-wrap' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};' ],
                        'condition'  => [ 'title!'=> '' ],
                    ]);
                    $this->add_group_control( KFGCBS::get_type(), [
                        'name'      => 'title_hover_box_shadow',
                        'selector'  => '{{WRAPPER}}:hover .kfw-ele-widget-custom-heading-title-wrap',
                        'condition' => [ 'title!'=> '' ],
                    ]);
                $this->end_controls_tab();
			$this->end_controls_tabs();

            $this->add_control( 'anim_color', [
                'label'     => esc_html__('Animation Color', 'onnat-extra' ),
                'type'      => KFCM::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .kfw-words:before'  => 'background:{{VALUE}};',
                ],
                'condition' => [
                    'title!'    => '',
                    'anim_type' =>[ 'style-7', 'style-8', 'style-9', 'style-10' ]
                ],
            ]);

            $this->add_control('secondary_title_style_before_hr', [
                'type'       => KFCM::DIVIDER,
                'condition' => [ 'title!'=> '' ]
            ]);
            $this->add_control('secondary_title_style_heading',[
                'label'      => esc_html__( 'SECONDARY TITLE', 'onnat-extra' ),
                'type'       => KFCM::HEADING,
                'condition' => [ 'title!'=> '' ]
            ]);
            $this->add_group_control(KFGCT::get_type(), [
                'name'      => 'secondary_title_typo',
                'label'     => esc_html__( 'Typography', 'onnat-extra' ),
                'condition' => [ 'title!'=> '' ],
                'selector'  => '{{WRAPPER}} .kfw-ele-widget-custom-heading-title kfw-ele-widget-custom-heading-secondary-title',
            ]);
            $this->add_responsive_control( 'secondary_title_margin', [
                'label'      => esc_html__('Margin', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'condition'  => [ 'title!'=> '' ],
                'selectors'  => [
                    '{{WRAPPER}} kfw-ele-widget-custom-heading-secondary-title-wrap' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->add_responsive_control( 'secondary_title_padding', [
                'label'      => esc_html__('Padding', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'condition'  => [ 'title!'=> '' ],
                'selectors'  => [
                    '{{WRAPPER}} kfw-ele-widget-custom-heading-secondary-title-wrap' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->start_controls_tabs( 'secondary_title_customize_tabs' );
                $this->start_controls_tab( 'secondary_title_normal_state', [
                    'label'     => esc_html__( 'Normal', 'onnat-extra' ),
                    'condition' => [ 'title!'=> '' ],
                ]);
                    $this->add_control( 'secondary_title_color', [
                        'label'     => esc_html__('Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'selectors' => ['{{WRAPPER}} .kfw-ele-widget-custom-heading-title kfw-ele-widget-custom-heading-secondary-title' => 'color: {{VALUE}};' ],
                        'condition' => [ 'title!'=> '' ],
                    ]);
                    $this->add_group_control( KFGCTS::get_type(), [
                        'name'      => 'secondary_title_txt_shadow',
                        'condition' => [ 'title!'=> '' ],
                        'selector'  => '{{WRAPPER}} .kfw-ele-widget-custom-heading-title kfw-ele-widget-custom-heading-secondary-title',
                    ]);
                    $this->add_group_control( KFGCTST::get_type(), [
                        'name'      => 'secondary_title_txt_stroke',
                        'condition' => [ 'title!'=> '' ],
                        'selector'  => '{{WRAPPER}} .kfw-ele-widget-custom-heading-title kfw-ele-widget-custom-heading-secondary-title',
                    ]);
                    $this->add_group_control( KFGCBG::get_type(), [
                        'name'      => 'secondary_title_bg',
                        'label'     => esc_html__('Background', 'onnat-extra' ),
                        'types'     => [ 'classic', 'gradient' ],
                        'exclude'   => [ 'image' ],
                        'selector'  => '{{WRAPPER}} kfw-ele-widget-custom-heading-secondary-title-wrap',
                        'condition' => [ 'title!'=> '' ],
                    ]);
                    $this->add_group_control( KFGCBR::get_type(), [
                        'name'           => 'secondary_title_border',
                        'fields_options' => [
                            'width' => [ 'label' => _x( 'Border Width', 'Border Control', 'onnat-extra' ) ],
                            'color' => [ 'label' => _x( 'Border Color', 'Border Control', 'onnat-extra' ) ],
                        ],
                        'selector'       => '{{WRAPPER}} kfw-ele-widget-custom-heading-secondary-title-wrap',
                        'condition'      => [ 'title!'=> '' ],
                    ]);
                    $this->add_responsive_control( 'secondary_title_border_radius', [
                        'label'      => esc_html__('Border Radius', 'onnat-extra' ),
                        'type'       => KFCM::DIMENSIONS,
                        'size_units' => [ 'px', '%' ],
                        'selectors'  => [ '{{WRAPPER}} kfw-ele-widget-custom-heading-secondary-title-wrap' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};' ],
                        'condition'  => [ 'title!'=> '' ],
                    ]);
                    $this->add_group_control( KFGCBS::get_type(), [
                        'name'      => 'secondary_title_box_shadow',
                        'selector'  => '{{WRAPPER}} kfw-ele-widget-custom-heading-secondary-title-wrap',
                        'condition' => [ 'title!'=> '' ],
                    ]);
                $this->end_controls_tab();
                $this->start_controls_tab( 'secondary_title_hover_state', [
                    'label'     => esc_html__( 'Hover', 'onnat-extra' ),
                    'condition' => [ 'title!'=> '' ],
                ]);
                    $this->add_control( 'secondary_title_hover_color', [
                        'label'     => esc_html__('Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'selectors' => ['{{WRAPPER}}:hover .kfw-ele-widget-custom-heading-title kfw-ele-widget-custom-heading-secondary-title' => 'color: {{VALUE}};' ],
                        'condition' => [ 'title!'=> '' ],
                    ]);
                    $this->add_group_control( KFGCTS::get_type(), [
                        'name'      => 'secondary_title_hover_txt_shadow',
                        'condition' => [ 'title!'=> '' ],
                        'selector'  => '{{WRAPPER}}:hover .kfw-ele-widget-custom-heading-title kfw-ele-widget-custom-heading-secondary-title',
                    ]);
                    $this->add_group_control( KFGCTST::get_type(), [
                        'name'      => 'secondary_title_hover_txt_stroke',
                        'condition' => [ 'title!'=> '' ],
                        'selector'  => '{{WRAPPER}}:hover .kfw-ele-widget-custom-heading-title kfw-ele-widget-custom-heading-secondary-title',
                    ]);
                    $this->add_group_control( KFGCBG::get_type(), [
                        'name'      => 'secondary_title_hover_bg',
                        'label'     => esc_html__('Background', 'onnat-extra' ),
                        'types'     => [ 'classic', 'gradient' ],
                        'exclude'   => [ 'image' ],
                        'selector'  => '{{WRAPPER}}:hover kfw-ele-widget-custom-heading-secondary-title-wrap',
                        'condition' => [ 'title!'=> '' ],
                    ]);
                    $this->add_group_control( KFGCBR::get_type(), [
                        'name'           => 'secondary_title_hover_border',
                        'fields_options' => [
                            'width' => [ 'label' => _x( 'Border Width', 'Border Control', 'onnat-extra' ) ],
                            'color' => [ 'label' => _x( 'Border Color', 'Border Control', 'onnat-extra' ) ],
                        ],
                        'selector'       => '{{WRAPPER}}:hover kfw-ele-widget-custom-heading-secondary-title-wrap',
                        'condition'      => [ 'title!'=> '' ],
                    ]);
                    $this->add_responsive_control( 'secondary_title_hover_border_radius', [
                        'label'      => esc_html__('Border Radius', 'onnat-extra' ),
                        'type'       => KFCM::DIMENSIONS,
                        'size_units' => [ 'px', '%' ],
                        'selectors'  => [ '{{WRAPPER}}:hover kfw-ele-widget-custom-heading-secondary-title-wrap' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};' ],
                        'condition'  => [ 'title!'=> '' ],
                    ]);
                    $this->add_group_control( KFGCBS::get_type(), [
                        'name'      => 'secondary_title_hover_box_shadow',
                        'selector'  => '{{WRAPPER}}:hover kfw-ele-widget-custom-heading-secondary-title-wrap',
                        'condition' => [ 'title!'=> '' ],
                    ]);
                $this->end_controls_tab();
            $this->end_controls_tabs();
            $this->add_control( 'secondary_title_anim_color', [
                'label'     => esc_html__('Animation Color', 'onnat-extra' ),
                'type'      => KFCM::COLOR,
                'selectors' => [
                    '{{WRAPPER}} kfw-ele-widget-custom-heading-secondary-title .kfw-words:before'  => 'background:{{VALUE}};',
                ],
                'condition' => [
                    'title!'    => '',
                    'anim_type' =>[ 'style-7', 'style-8', 'style-9', 'style-10' ]
                ],
            ]);
		$this->end_controls_section();
    }

    public function _register_shadow_text_style_section() {
		$this->start_controls_section( 'section_shadow_text_style_style', [
            'label'     => esc_html__( 'Shadow Text', 'onnat-extra' ),
            'tab'       => KFCM::TAB_STYLE,
            'condition' => [
                'use_shadow_text' => 'yes',
                'shadow_text!'    => ''
            ],
		]);
            $this->add_group_control(KFGCT::get_type(), [
                'name'      => 'st_typo',
                'label'     => esc_html__( 'Typography', 'onnat-extra' ),
                'condition' => [
                    'use_shadow_text' => 'yes',
                    'shadow_text!'    => ''
                ],
                'selector'  => '{{WRAPPER}} .kfw-ele-widget-custom-heading-shadow-text',
            ]);
            $this->add_control( 'st_color', [
                'label'     => esc_html__('Color', 'onnat-extra' ),
                'type'      => KFCM::COLOR,
                'selectors' => ['{{WRAPPER}} .kfw-ele-widget-custom-heading-shadow-text' => 'color: {{VALUE}};' ],
                'condition' => [
                    'use_shadow_text' => 'yes',
                    'shadow_text!'    => ''
                ],
            ]);
            $this->add_group_control( KFGCTS::get_type(), [
                'name'      => 'st_txt_shadow',
                'condition' => [
                    'use_shadow_text' => 'yes',
                    'shadow_text!'    => ''
                ],
                'selector'  => '{{WRAPPER}} .kfw-ele-widget-custom-heading-shadow-text',
            ]);
            $this->add_group_control( KFGCTST::get_type(), [
                'name'      => 'st_txt_stroke',
                'condition' => [
                    'use_shadow_text' => 'yes',
                    'shadow_text!'    => ''
                ],
                'selector'  => '{{WRAPPER}} .kfw-ele-widget-custom-heading-shadow-text',
            ]);
            $this->add_control( 'st_opacity', [
                'label'     => esc_html__('Opacity', 'onnat-extra' ),
                'type'      => KFCM::SLIDER,
                'range'     => [
                    'px' => [
                        'min'  => 0.05,
                        'max'  => 1,
                        'step' => 0.05,
                    ],
                ],
                'selectors' => ['{{WRAPPER}} .kfw-ele-widget-custom-heading-shadow-text' => 'opacity: {{SIZE}};' ],
                'condition' => [
                    'use_shadow_text' => 'yes',
                    'shadow_text!'    => ''
                ],
            ]);
		$this->end_controls_section();
    }

	/**
	 * Generate setting json
	 */
	protected function _generate_setting_json( array $settings ) {
		extract( $settings );

        $options = [
            'animate'   => false,
            'animation' => 'none',
            'tween'     => ''
        ];

        if( $anim_title === 'yes' ) {
            $options = [
                'animate'   => true,
                'animation' => $anim_type,
                'tween'     => $animate,
                'ease'      => 'back'
            ];
        }

        return wp_json_encode( $options );
    }

    public function _title_tag( $tag ) {
        $tags = [
            'div',
            'span',
            'p',
            'h1',
            'h2',
            'h3',
            'h4',
            'h5',
            'h6',
        ];

        return in_array( strtolower( $tag ), $tags) ? $tag : 'div';
    }

	public function _title( $settings = [] ) {
        $title = esc_html( $settings['title'] );

        $title = str_replace( array( '{{', '}}' ), [
            '<kfw-ele-widget-custom-heading-secondary-title-wrap> <kfw-ele-widget-custom-heading-secondary-title>',
            '</kfw-ele-widget-custom-heading-secondary-title></kfw-ele-widget-custom-heading-secondary-title-wrap>'
        ], $title );

        $link  = array_filter( $settings['title_link'] );

		if( !empty( $title ) ) {
			$this->add_render_attribute( 'title', [
				'class' => array_filter([
					'kfw-ele-widget-custom-heading-title'
				])
			]);

			if( isset( $link['url'] ) ) {

                $mode   = $this->_is_editor_mode();
                $target = ( !empty( $link['is_external'] ) ) ? '_blank' : '_self';

                $this->add_render_attribute( 'title', 'class', 'kfw-ele-widget-custom-heading-linked-title' );

                if( !$mode ) {
                    $this->add_render_attribute( 'title', 'onclick', 'window.open("'. esc_url( $link['url'] ) .'", "'.$target.'")' );
                }
            }

            $tag = $this->_title_tag( $settings['title_tag'] );

            printf(
                '<div class="kfw-ele-widget-custom-heading-title-wrap"><%1$s %2$s>%3$s</%1$s></div>',
                $tag,
                $this->get_render_attribute_string( 'title' ),
                $title,
                $tag
            );
        }
    }

	public function _sub_title( $settings = [] ) {
		if( !empty( $settings['sub_title'] ) ) {
			$this->add_render_attribute( 'sub-title', [
				'class' => array_filter([
					'kfw-ele-widget-custom-heading-sub-title'
				])
			]);

			echo '<div class="kfw-ele-widget-custom-heading-sub-title-wrap">';

                if( $settings['sub_title_icon_type'] === 'icon' ) {
                    $this->_render_icon( $settings['sub_title_icon'] );
                } else if( $settings['sub_title_icon_type'] === 'image' ) {
                    $this->_render_img( $settings );
                }

                echo '<div '.$this->get_render_attribute_string( 'sub-title' ).'>';
                    echo esc_html( $settings['sub_title'] );
                echo '</div>';
            echo '</div>';
        }
    }

	public function _shadow_text( $settings = [] ) {
		if( !empty( $settings['shadow_text']) && 'yes' === $settings['use_shadow_text'] ) {
			$this->add_render_attribute( 'shadow-text', [
				'class' => array_filter([
					'kfw-ele-widget-custom-heading-shadow-text'
				])
			]);

			echo '<div '.$this->get_render_attribute_string( 'shadow-text' ).'>' . esc_html( $settings['shadow_text'] ) . '</div>';
        }
    }

    public function _is_editor_mode() {
        $preview = KFEl::instance()->preview->is_preview_mode();
        $editor  = KFEl::instance()->editor->is_edit_mode();

        return ( $preview || $editor ) ? true : false;
    }

	public function _render_icon( $icon , $echo = true ) {
		$icon = array_filter( $icon );

		if( !empty( $icon ) ) {
			echo '<span class="kfw-elementor-icon-wrap">';
				KFIM::render_icon( $icon, [ 'aria-hidden' => 'true' ] );
			echo '</span>';
		}
	}

	public function _render_img( $settings = [] ) {
		if( !empty( $settings['sub_title_image']['url'] ) ) {
			$image_src = $settings['sub_title_image']['url'];

			$image_title = $this->_get_image_title( $settings['sub_title_image'] );
			$image_title = empty( $image_title ) ? $this->get_title() : $image_title;

			$image_alt = $this->_get_image_alt( $settings['sub_title_image'] );
			$image_alt = empty( $image_alt ) ? $image_title : $image_title;

			$this->add_render_attribute( 'image_attr', [
				'alt'   => esc_attr( $image_alt ),
				'title' => esc_attr( $image_title ),
				'src'   => esc_url( $image_src ),
			] );

			echo '<span class="kfw-elementor-icon-wrap">';
				echo '<img '.$this->get_render_attribute_string( 'image_attr' ).'/>';
			echo '</span>';
		}
	}

	public function _get_image_title( $attachment ) {
		if ( empty( $attachment['id'] ) ) {
			return '';
		}

		return get_the_title( $attachment['id'] );
	}

	public function _get_image_alt( $instance ) {
		if ( empty( $instance['id'] ) ) {
			// For `Insert From URL` images.
			return isset( $instance['alt'] ) ? trim( strip_tags( $instance['alt'] ) ) : '';
		}

		$attachment_id = $instance['id'];
		if ( ! $attachment_id ) {
			return '';
		}

		$attachment = get_post( $attachment_id );
		if ( ! $attachment ) {
			return '';
		}

		$alt = get_post_meta( $attachment_id, '_wp_attachment_image_alt', true );
		if ( ! $alt ) {
			$alt = $attachment->post_excerpt;
			if ( ! $alt ) {
				$alt = $attachment->post_title;
			}
		}
		return trim( strip_tags( $alt ) );
	}

	/**
	 * Render widget output on the frontend.
	 */
    protected function render() {
		$render = kf_extra_plugin_elementor()->get_template("kfw-custom-heading");
		include $render;
    }
}