<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

use \Elementor\Widget_Base as WB;
use \Elementor\Controls_Manager as KFCM;
use \Elementor\Icons_Manager as KFIM;

use \Elementor\Group_Control_Image_Size as KFGCIS;
use \Elementor\Group_Control_Background as KFGCBG;
use \Elementor\Group_Control_Typography as KFGCT;
use \Elementor\Group_Control_Border as KFGCBR;
use \Elementor\Group_Control_Box_Shadow as KFGCBS;
use \Elementor\Group_Control_Css_Filter as KFGCSSF;

class KFW_Flip_Box_Cube extends WB {

	/**
	 * Get element name.
	 *
	 * Retrieve the element name.
	 *
	 * @return string The name.
	 */
	public function get_name() {
        return 'kfw-flip-box-cube';
    }

	/**
	 * Get element title.
	 *
	 * Retrieve the element title.
	 *
	 * @return string Element title.
	 */
	public function get_title() {
		return esc_html__('Flip Box : Cube', 'onnat-extra');
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the widget categories.
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ '_kfw_core_widgets' ];
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve the widget icon.
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'kfw-badge icon-flip-box kfw-flip-box-icon-cube';
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the widget keywords.
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'kfw', 'kin', 'flip', 'box', 'flip box', 'cube' ];
	}

	/**
	 * Get widget help url.
	 *
	 * Retrieve the widget help url.
	 *
	 * @return string Widget help url.
	 */
	public function get_custom_help_url() {
        return KFE_CONST_THEME_DOC . 'elementor-custom-widgets/flip-box-cube';
    }

	/**
	 * Get HTML wrapper class.
     *
	 * Retrieve the widget container class.
     *
     * @return string Widget container css classes.
	 */
	public function get_html_wrapper_class() {
        $name = $this->get_name();

		return sprintf( 'kfw-ele-%s kfw-ele-transition-disabled elementor-widget-%s', $name, $name );
	}

	/**
	 * Widget base constructor.
	 *
	 * Initializing the widget base class.
	 */
	public function __construct( $data = [], $args = null ) {
        parent::__construct( $data, $args );

		wp_register_script( 'kfw-flip-box',
			KFE_CONST_URL . 'libraries/elementor/assets/js/flip-boxes' . KFE_CONST_DEBUG_SUFFIX . '.js',
			[ 'jquery', 'elementor-frontend' ],
			KFE_CONST_VERSION,
			true
		);

		wp_register_style( 'kfw-flip-box-cube',
			KFE_CONST_URL . 'libraries/elementor/assets/css/flip-box-cube' . KFE_CONST_DEBUG_SUFFIX . '.css',
			[],
			KFE_CONST_VERSION
		);
    }

	/**
	 * Get style dependencies.
	 *
	 * Retrieve the list of style dependencies the element requires.
	 *
	 * @return array Element styles dependencies.
	 */
	public function get_style_depends() {
		return [ 'kfw-flip-box-cube' ];
	}

	/**
	 * Get script dependencies.
	 *
	 * Retrieve the list of script dependencies the element requires.
	 *
	 * @return array Element scripts dependencies.
	 */
	public function get_script_depends() {
		return [ 'kfw-flip-box-cube' ];
	}

	/**
	 * Register controls.
	 *
	 * Used to add new controls to any element type.
	 */
	protected function register_controls() {
		$this->_register_front_section();
		$this->_register_back_section();
		$this->_register_other_side_section();

		$this->_register_settings_section();

		$this->_register_front_style_section();
		$this->_register_back_style_section();
	}

	public function _register_front_section() {
		$this->start_controls_section( 'section_front', [
			'label' => esc_html__( 'Front Side', 'onnat-extra' )
		]);
			$this->add_control('front_title',[
				'label'       => esc_html__( 'Title', 'onnat-extra' ),
				'type'        => KFCM::TEXT,
				'dynamic'     => [ 'active' => true, ],
				'placeholder' => esc_html__( 'Enter Title', 'onnat-extra' ),
				'default'     => esc_html__( 'Here is the title', 'onnat-extra' ),
				'label_block' => true,
			]);
			$this->add_control('front_sub_title',[
				'label'       => esc_html__( 'Sub Title', 'onnat-extra' ),
				'type'        => KFCM::TEXT,
				'dynamic'     => [ 'active' => true, ],
				'placeholder' => esc_html__( 'Enter Sub Title', 'onnat-extra' ),
				'label_block' => true,
			]);
			$this->add_control('front_description',[
				'label'       => esc_html__( 'Description', 'onnat-extra' ),
				'type'        => KFCM::TEXTAREA,
				'dynamic'     => [ 'active' => true, ],
				'placeholder' => esc_html__( 'Enter Description', 'onnat-extra' ),
				'default'     => esc_html__( 'Lorem ipsum dolor sit amet consectetur adipiscing elit dolor', 'onnat-extra' ),
				'label_block' => true,
			]);
			$this->add_control('graphic_element_heading',[
				'label'     => esc_html__( 'Graphic Element', 'onnat-extra' ),
				'type'      => KFCM::HEADING,
				'separator' => 'before',
			]);
			$this->add_control('graphic_element', [
				'label'     => esc_html__( 'Type', 'onnat-extra' ),
				'type'      => KFCM::CHOOSE,
				'options'   => [
					'blank' => [
						'title' => esc_html__('Blank', 'onnat-extra'),
						'icon'  => '',
					],
					'icon'  => [
						'title' => esc_html__('Icon', 'onnat-extra'),
						'icon'  => 'eicon-plus-circle-o',
					],
					'image'  => [
						'title' => esc_html__('Image', 'onnat-extra'),
						'icon'  => 'eicon-image',
					],
				],
				'default'   => 'icon',
				'toggle'    => false
			]);
			$this->add_control( 'icon', [
				'label'     => esc_html__( 'Icon', 'onnat-extra' ),
				'type'      => KFCM::ICONS,
				'default'   => [
					'value'   => 'fas fa-heart',
					'library' => 'fa-solid',
				],
				'condition' => [ 'graphic_element' => 'icon' ],
			]);
			$this->add_control( 'image', [
				'label'     => esc_html__( 'Image', 'onnat-extra' ),
				'type'      => KFCM::MEDIA,
				'dynamic'   => [ 'active' => true ],
				'default'   => [
					'url' => \Elementor\Utils::get_placeholder_image_src()
				],
				'condition' => [ 'graphic_element' => 'image' ],
			]);
			$this->add_group_control( KFGCIS::get_type(), [
				'name'      => 'image_size',
				'default'   => 'thumbnail',
				'condition' => [ 'graphic_element' => 'image' ],
			]);
			$this->add_control('front_bg_heading',[
				'label'     => esc_html__( 'Background', 'onnat-extra' ),
				'type'      => KFCM::HEADING,
				'separator' => 'before',
			]);
			$this->add_control( 'front_bg_overlay', [
				'label'     => esc_html__( 'Background Overlay', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'selectors' => [
					'{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-front .kfw-ele-widget-flip-box-layer-overlay' => 'background-color: {{VALUE}};'
				],
				'condition' => [
					'front_bg_background' => [ 'classic', 'gradient' ],
				],
			]);
			$this->add_group_control( KFGCBG::get_type(), [
				'name'     => 'front_bg',
				'types'    => ['classic', 'gradient'],
				'selector' => '{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-front',
			]);
		$this->end_controls_section();
	}

	public function _register_back_section() {
		$this->start_controls_section( 'section_back', [
			'label' => esc_html__( 'Back Side', 'onnat-extra' )
		]);
			$this->add_control('back_title',[
				'label'       => esc_html__( 'Title', 'onnat-extra' ),
				'type'        => KFCM::TEXT,
				'dynamic'     => [ 'active' => true, ],
				'placeholder' => esc_html__( 'Enter Title', 'onnat-extra' ),
				'default'     => esc_html__( 'Here is the title', 'onnat-extra' ),
				'label_block' => true,
			]);
			$this->add_control('back_sub_title',[
				'label'       => esc_html__( 'Sub Title', 'onnat-extra' ),
				'type'        => KFCM::TEXT,
				'dynamic'     => [ 'active' => true, ],
				'placeholder' => esc_html__( 'Enter Sub Title', 'onnat-extra' ),
				'label_block' => true,
			]);
			$this->add_control('back_description',[
				'label'       => esc_html__( 'Description', 'onnat-extra' ),
				'type'        => KFCM::TEXTAREA,
				'dynamic'     => [ 'active' => true, ],
				'placeholder' => esc_html__( 'Enter Description', 'onnat-extra' ),
				'default'     => esc_html__( 'Lorem ipsum dolor sit amet consectetur adipiscing elit dolor', 'onnat-extra' ),
				'label_block' => true,
			]);
			$this->add_control('back_bg_heading',[
				'label'     => esc_html__( 'Background', 'onnat-extra' ),
				'type'      => KFCM::HEADING,
				'separator' => 'before',
			]);
			$this->add_control( 'back_bg_overlay', [
				'label'     => esc_html__( 'Background Overlay', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'selectors' => [
					'{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-back .kfw-ele-widget-flip-box-layer-overlay' => 'background-color: {{VALUE}};'
				],
				'condition' => [
					'back_bg_background' => [ 'classic', 'gradient' ],
				],
			]);
			$this->add_group_control( KFGCBG::get_type(), [
				'name'     => 'back_bg',
				'types'    => ['classic', 'gradient'],
				'selector' => '{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-back',
			]);
			$this->add_control('back_btn_heading',[
				'label'     => esc_html__( 'Button', 'onnat-extra' ),
				'type'      => KFCM::HEADING,
				'separator' => 'before',
			]);
			$this->add_control( 'back_btn_text', [
				'label'   => esc_html__( 'Text', 'onnat-extra' ),
				'type'    => KFCM::TEXT,
				'default' => esc_html__( 'Button', 'onnat-extra' ),
			]);
			$this->add_control( 'back_btn_link', [
				'label'       => esc_html__( 'Link', 'onnat-extra' ),
				'type'        => KFCM::URL,
				'placeholder' => 'https://your-link.com',
				'default'     => [ 'url' => '#' ],
				'conditions'  => [
					'relation' => 'or',
					'terms'    => [
						[
							'name'     => 'back_btn_text',
							'operator' => '!=',
							'value'    => ''
						],
						[
							'name'     => 'back_btn_icon[value]',
							'operator' => '!=',
							'value'    => ''
						],
					],
				],
			]);
			$this->add_control( 'back_btn_icon', [
				'label'       => esc_html__( 'Icon', 'onnat-extra' ),
				'type'        => KFCM::ICONS,
				'skin'        => 'inline',
				'label_block' => false,
				'default'     => [
					'value'   => 'fa fa-check',
					'library' => 'fa-solid'
				]
			]);
			$this->add_control( 'back_btn_icon_pos', [
				'label'     => esc_html__( 'Icon Position', 'onnat-extra' ),
				'type'      => KFCM::CHOOSE,
				'options'   => [
					'left'   => [
						'title' => esc_html__('Left', 'onnat-extra'),
						'icon'  => 'eicon-text-align-left',
					],
					'right'  => [
						'title' => esc_html__('Right', 'onnat-extra'),
						'icon'  => 'eicon-text-align-right',
					],
				],
				'default'   => 'left',
				'condition' => [
					'back_btn_icon[value]!' => ''
				],
			]);
			$this->add_control( 'link_click', [
				'label'     => esc_html__( 'Apply Link On', 'onnat-extra' ),
				'type'      => KFCM::SELECT,
				'options'   => [
					'box'    => esc_html__( 'Whole Box', 'onnat-extra' ),
					'button' => esc_html__( 'Button Only', 'onnat-extra' ),
				],
				'default'   => 'button',
				'condition' => [ 'back_btn_link[url]!' => '', ],
			]);
		$this->end_controls_section();
	}

	public function _register_other_side_section() {
		$this->start_controls_section( 'section_other_side', [
			'label'     => esc_html__( 'Other Side(s)', 'onnat-extra' ),
			'condition' => [
				'effect' => 'flip'
			],
		]);
			$this->add_control('xflank_heading',[
				'label'     => esc_html__( 'XFlank', 'onnat-extra' ),
				'type'      => KFCM::HEADING,
				'condition' => [
					'effect'          => 'flip',
					'flip_direction!' => [ 'w', 'e' ],
				]
			]);
			$this->add_group_control( KFGCBR::get_type(), [
				'name'      => 'xflank_border',
				'selector'  => '{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-cube-inner .kfw-ele-widget-flip-box-xflank',
				'condition' => [
					'effect'          => 'flip',
					'flip_direction!' => [ 'w', 'e' ],
				]
			]);
			$this->add_group_control( KFGCBG::get_type(), [
				'name'           => 'xflank_bg',
				'fields_options' => [
					'background' => [
						'label' => esc_html__( 'Background', 'onnat-extra' )
					]
				],
				'types'          => ['classic', 'gradient'],
				'selector'       => '{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-cube-inner .kfw-ele-widget-flip-box-xflank',
				'condition'      => [
					'flip_direction!' => [ 'w', 'e' ],
				]
			]);
			$this->add_control( 'xflank_bg_overlay', [
				'label'     => esc_html__( 'Background Overlay', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'selectors' => [
					'{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-cube-inner .kfw-ele-widget-flip-box-xflank:before' => 'background-color: {{VALUE}};'
				],
				'condition' => [
					'xflank_bg_background' => [ 'classic', 'gradient' ],
					'flip_direction!'      => [ 'w', 'e' ],
				],
			]);
			$this->add_responsive_control( 'xflank_border_radius', [
				'label'      => esc_html__('Border Radius', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-cube-inner .kfw-ele-widget-flip-box-xflank' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'					
				],
				'condition'  => [
					'effect'          => 'flip',
					'flip_direction!' => [ 'w', 'e' ],
				]
			]);
			$this->add_control('yflank_heading',[
				'label'     => esc_html__( 'YFlank', 'onnat-extra' ),
				'type'      => KFCM::HEADING,
				'separator' => 'before',
				'condition' => [
					'effect'          => 'flip',
					'flip_direction!' => [ 'n', 's' ],
				]
			]);
			$this->add_group_control( KFGCBR::get_type(), [
				'name'      => 'yflank_border',
				'selector'  => '{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-cube-inner .kfw-ele-widget-flip-box-yflank',
				'condition' => [
					'effect'          => 'flip',
					'flip_direction!' => [ 'n', 's' ],
				]
			]);
			$this->add_group_control( KFGCBG::get_type(), [
				'name'           => 'yflank_bg',
				'fields_options' => [
					'background' => [
						'label' => esc_html__( 'Background', 'onnat-extra' )
					]
				],
				'types'          => ['classic', 'gradient'],
				'selector'       => '{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-cube-inner .kfw-ele-widget-flip-box-yflank',
				'condition'      => [
					'flip_direction!' => [ 'n', 's' ],
				]
			]);
			$this->add_control( 'yflank_bg_overlay', [
				'label'     => esc_html__( 'Background Overlay', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'selectors' => [
					'{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-cube-inner .kfw-ele-widget-flip-box-yflank:before' => 'background-color: {{VALUE}};'
				],
				'condition' => [
					'yflank_bg_background' => [ 'classic', 'gradient' ],
					'flip_direction!'      => [ 'n', 's' ],
				],
			]);
			$this->add_responsive_control( 'yflank_border_radius', [
				'label'      => esc_html__('Border Radius', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-cube-inner .kfw-ele-widget-flip-box-yflank' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				],
				'condition'  => [
					'effect'          => 'flip',
					'flip_direction!' => [ 'n', 's' ],
				]
			]);
		$this->end_controls_section();
	}

	public function _register_settings_section() {
		$this->start_controls_section( 'section_settings', [
			'label' => esc_html__( 'Settings', 'onnat-extra' )
		]);
			$this->add_responsive_control( 'height', [
				'label'      => esc_html__( 'Cube Size', 'onnat-extra' ),
				'type'       => KFCM::SLIDER,
				'range'      => [ 'px' => [ 'min' => 260, 'max' => 1500 ],],
				'size_units' => [ 'px' ],
				'selectors'  => [
					'{{WRAPPER}}.kfw-ele-widget-flip-box-effect-tilt .kfw-ele-widget-flip-box-cube-wrap' => 'height:{{SIZE}}{{UNIT}};',
					'{{WRAPPER}}.kfw-ele-widget-flip-box-effect-flip .kfw-ele-widget-flip-box-cube-wrap' => 'height:{{SIZE}}{{UNIT}}; width:{{SIZE}}{{UNIT}};',
				]
			]);
			$this->add_control( 'effect', [
				'label'        => esc_html__( 'Effect', 'onnat-extra' ),
				'type'         => KFCM::CHOOSE,
				'default'      => 'flip',
				'toggle'       => false,
				'options'      => [
					'flip' => [
						'title' => esc_html__( 'Flip', 'onnat-extra' ),
						'icon'  => 'eicon-flip-box'
					],
					'tilt' => [
						'title' => esc_html__( 'Tilt', 'onnat-extra' ),
						'icon'  => 'eicon-instagram-nested-gallery'
					],
				],
				'prefix_class' => 'kfw-ele-widget-flip-box-effect-',
			]);
			$this->add_control( 'flip_direction', [
				'label'        => esc_html__( 'Direction', 'onnat-extra' ),
				'type'         => KFCM::CHOOSE,
				'toggle'       => false,
				'default'      => 'w',
				'options'      => [
					'n'  => [
						'title' => esc_html__( 'North', 'onnat-extra' ),
						'icon'  => 'eicon-arrow-up'
					],
					'ne' => [
						'title' => esc_html__( 'North East', 'onnat-extra' ),
						'icon'  => 'eicon-arrow-right kfw-ctrl-rotate-315'
					],
					'e'  => [
						'title' => esc_html__( 'East', 'onnat-extra' ),
						'icon'  => 'eicon-arrow-right'
					],
					'se' => [
						'title' => esc_html__( 'South East', 'onnat-extra' ),
						'icon'  => 'eicon-arrow-left kfw-ctrl-rotate-225'
					],
					's'  => [
						'title' => esc_html__( 'South', 'onnat-extra' ),
						'icon'  => 'eicon-arrow-down'
					],
					'sw' => [
						'title' => esc_html__( 'South West', 'onnat-extra' ),
						'icon'  => 'eicon-arrow-right kfw-ctrl-rotate-135'
					],
					'w'  => [
						'title' => esc_html__( 'West', 'onnat-extra' ),
						'icon'  => 'eicon-arrow-left'
					],
					'nw' => [
						'title' => esc_html__( 'North West', 'onnat-extra' ),
						'icon'  => 'eicon-arrow-left kfw-ctrl-rotate-45'
					],
				],
				'prefix_class' => 'kfw-ele-widget-flip-box-direction-',
				'condition'    => [
					'effect' => 'flip'
				],
			]);
			$this->add_control( 'tilt_direction', [
				'label'        => esc_html__( 'Direction', 'onnat-extra' ),
				'type'         => KFCM::CHOOSE,
				'toggle'       => false,
				'default'      => 'w',
				'options'      => [
					'n'  => [
						'title' => esc_html__( 'North', 'onnat-extra' ),
						'icon'  => 'eicon-arrow-up'
					],
					'e'  => [
						'title' => esc_html__( 'East', 'onnat-extra' ),
						'icon'  => 'eicon-arrow-right'
					],
					's'  => [
						'title' => esc_html__( 'South', 'onnat-extra' ),
						'icon'  => 'eicon-arrow-down'
					],
					'w'  => [
						'title' => esc_html__( 'West', 'onnat-extra' ),
						'icon'  => 'eicon-arrow-left'
					],
				],
				'prefix_class' => 'kfw-ele-widget-flip-box-direction-',
				'condition'    => [
					'effect' => 'tilt'
				],
			]);
		$this->end_controls_section();
	}

	public function _register_front_style_section() {
		$this->start_controls_section( 'section_front_style', [
			'label' => esc_html__( 'Front Side', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
			$this->add_responsive_control( 'padding_front', [
				'label'      => esc_html__( 'Padding', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-front .kfw-ele-widget-flip-box-layer-overlay' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]);
			$this->add_responsive_control( 'margin_front', [
				'label'      => esc_html__( 'Margin', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-front .kfw-ele-widget-flip-box-layer-overlay' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]);
			$this->add_responsive_control( 'alignment_front', [
				'label'       => esc_html__( 'Alignment', 'onnat-extra' ),
				'type'        => KFCM::CHOOSE,
				'label_block' => false,
				'options'     => [
					'left'   => [
						'title' => esc_html__('Left', 'onnat-extra'),
						'icon'  => 'eicon-text-align-left',
					],
					'center' => [
						'title' => esc_html__('Center', 'onnat-extra'),
						'icon'  => 'eicon-text-align-center',
					],
					'right'  => [
						'title' => esc_html__('Right', 'onnat-extra'),
						'icon'  => 'eicon-text-align-right',
					],
				],
				'default'     => 'center',
				'selectors'   => [
					'{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-front .kfw-ele-widget-flip-box-layer-overlay' => 'text-align: {{VALUE}};',
				],
			]);
			$this->add_responsive_control( 'vertical_position_front', [
				'label'                => esc_html__( 'Vertical Position', 'onnat-extra' ),
				'type'                 => KFCM::CHOOSE,
				'label_block'          => false,
				'options'              => [
					'top'    => [
						'title' => esc_html__('Top', 'onnat-extra'),
						'icon'  => 'eicon-v-align-top',
					],
					'middle' => [
						'title' => esc_html__('Middle', 'onnat-extra'),
						'icon'  => 'eicon-v-align-middle',
					],
					'bottom' => [
						'title' => esc_html__('Bottom', 'onnat-extra'),
						'icon'  => 'eicon-v-align-bottom',
					],
				],
				'selectors_dictionary' => [
					'top'    => 'flex-start',
					'middle' => 'center',
					'bottom' => 'flex-end',
				],
				'selectors'            => [
					'{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-front .kfw-ele-widget-flip-box-layer-overlay' => 'justify-content: {{VALUE}};',
				],
			]);
			$this->add_group_control( KFGCBR::get_type(), [
				'name'      => 'border_front',
				'selector'  => '{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-front',
			]);
			$this->add_responsive_control( 'border_radius_front', [
				'label'      => esc_html__('Border Radius', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-front'                                        => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					'{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-front .kfw-ele-widget-flip-box-layer-overlay' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]);
			$this->add_group_control( KFGCBS::get_type(), [
				'name'     => 'box_shadow_front',
				'selector' => '{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-front',
			]);
			$this->add_control( 'hr', [
				'type' => KFCM::DIVIDER,
			]);
			$this->start_controls_tabs( 'front_style_tabs' );
				$this->start_controls_tab( 'front_title_style_tab', [
					'label'     => esc_html__( 'Title', 'onnat-extra' ),
					'condition' => [ 'front_title!'=> '' ]
				]);
					$this->add_group_control(KFGCT::get_type(), [
						'name'     => 'front_title_typo',
						'label'    => esc_html__( 'Typography', 'onnat-extra' ),
						'selector' => '{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-front .kfw-ele-widget-flip-box-layer-title',
					]);
					$this->add_control( 'font_title_color', [
						'label'     => esc_html__('Typo Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'selectors' => [
							'{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-front .kfw-ele-widget-flip-box-layer-title' => 'color:{{VALUE}};'
						],
					]);
					$this->add_responsive_control( 'front_title_margin', [
						'label'     => esc_html__('Margin', 'onnat-extra' ),
						'type'      => KFCM::DIMENSIONS,
						'size_units' => [ 'px', 'em', '%' ],
						'selectors' => [
							'{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-front .kfw-ele-widget-flip-box-layer-title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
						]
					]);
				$this->end_controls_tab();
				$this->start_controls_tab( 'front_stitle_style_tab', [
					'label'     => esc_html__( 'Sub Title', 'onnat-extra' ),
					'condition' => [ 'front_sub_title!'=> '' ]
				]);
					$this->add_group_control(KFGCT::get_type(), [
						'name'     => 'front_sub_title_typo',
						'label'    => esc_html__( 'Typography', 'onnat-extra' ),
						'selector' => '{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-front .kfw-ele-widget-flip-box-layer-sub-title',
					]);
					$this->add_control( 'font_sub_title_color', [
						'label'     => esc_html__('Typo Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'selectors' => [
							'{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-front .kfw-ele-widget-flip-box-layer-sub-title' => 'color:{{VALUE}};'
						],
					]);
					$this->add_responsive_control( 'front_sub_title_margin', [
						'label'     => esc_html__('Margin', 'onnat-extra' ),
						'type'      => KFCM::DIMENSIONS,
						'size_units' => [ 'px', 'em', '%' ],
						'selectors' => [
							'{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-front .kfw-ele-widget-flip-box-layer-sub-title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
						]
					]);
				$this->end_controls_tab();

				$this->start_controls_tab( 'front_description_style_tab', [
					'label'     => esc_html__( 'Description', 'onnat-extra' ),
					'condition' => [ 'front_description!'=> '' ]
				]);
					$this->add_group_control(KFGCT::get_type(), [
						'name'     => 'front_description_typo',
						'label'    => esc_html__( 'Typography', 'onnat-extra' ),
						'selector' => '{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-front .kfw-ele-widget-flip-box-layer-description',
					]);
					$this->add_control( 'front_description_color', [
						'label'     => esc_html__('Typo Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'selectors' => [
							'{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-front .kfw-ele-widget-flip-box-layer-description' => 'color:{{VALUE}};'
						],
					]);
					$this->add_responsive_control( 'front_description_margin', [
						'label'     => esc_html__('Margin', 'onnat-extra' ),
						'type'      => KFCM::DIMENSIONS,
						'size_units' => [ 'px', 'em', '%' ],
						'selectors' => [
							'{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-front .kfw-ele-widget-flip-box-layer-description' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
						]
					]);
				$this->end_controls_tab();
				$this->start_controls_tab( 'front_icon_style_tab', [
					'label'     => esc_html__( 'Icon', 'onnat-extra' ),
					'condition' => [
						'graphic_element' => 'icon',
						'icon[value]!'    => ''
					]
				]);
					$this->add_control( 'icon_color', [
						'label'     => esc_html__('Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'default'   => '#25ced1',
						'selectors' => [
							'{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-front .kfw-elementor-icon-wrap i' => 'color:{{VALUE}};',
							'{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-front .kfw-elementor-icon-wrap svg' => 'fill:{{VALUE}};',
						],
					]);
					$this->add_group_control( KFGCBG::get_type(), [
						'name'           => 'icon_bg',
						'types'          => ['classic', 'gradient'],
						'exclude'        => [ 'image'],
						'fields_options' => [
							'background' => [ 'default' => 'classic' ],
							'color'      => [ 'default' => '#FFFFFF' ],
						],
						'selector'       => '{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-front .kfw-elementor-icon-wrap',

					]);
					$this->add_responsive_control( 'icon_padding',[
						'label'      => esc_html__('Padding', 'onnat-extra' ),
						'type'       => KFCM::DIMENSIONS,
						'size_units' => [ 'px', 'em', '%' ],
						'default'    => [
							'top'      => '15',
							'right'    => '15',
							'bottom'   => '15',
							'left'     => '15',
							'unit'     => 'px',
							'isLinked' => false
						],
						'selectors'  => [
							'{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-front .kfw-elementor-icon-wrap' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
						],
					]);
					$this->add_responsive_control( 'icon_margin',[
						'label'      => esc_html__('Margin', 'onnat-extra' ),
						'type'       => KFCM::DIMENSIONS,
						'size_units' => [ 'px', 'em', '%' ],
						'selectors'  => [
							'{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-front .kfw-elementor-icon-wrap' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
						],
					]);
					$this->add_group_control( KFGCBR::get_type(), [
						'name'     => 'icon_border',
						'selector' => '{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-front .kfw-elementor-icon-wrap'
					]);
					$this->add_responsive_control( 'icon_border_radius', [
						'label'      => esc_html__('Border Radius', 'onnat-extra' ),
						'type'       => KFCM::DIMENSIONS,
						'size_units' => [ 'px', '%' ],
						'default'    => [
							'top'      => '10',
							'right'    => '10',
							'bottom'   => '10',
							'left'     => '10',
							'unit'     => 'px',
							'isLinked' => false
						],
						'selectors'  => [
							'{{WRAPPER}}  .kfw-ele-widget-flip-box-cube-wrap.kfw-ele-widget-flip-box-cube-graphic-ele-icon .kfw-ele-widget-flip-box-front .kfw-elementor-icon-wrap' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
						]
					]);
					$this->add_group_control( KFGCBS::get_type(), [
						'name'     => 'icon_box_shadow',
						'selector' => '{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-front .kfw-elementor-icon-wrap'
					]);
					$this->add_responsive_control( 'icon_size', [
						'label'      => esc_html__('Size', 'onnat-extra' ),
						'type'       => KFCM::SLIDER,
						'size_units' => [ 'px' ],
						'range'      => [
							'px' => [
								'min' => 0,
								'max' => 100,
							],
						],
						'selectors'  => [
							'{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-front .kfw-elementor-icon-wrap i' => 'font-size: {{SIZE}}{{UNIT}};',
						],
					]);
					$this->add_control('icon_rotate',[
						'label'     => esc_html__( 'Rotate', 'onnat-extra' ),
						'type'      => KFCM::SLIDER,
						'default'   => [
							'size' => 0,
							'unit' => 'deg',
						],
						'range'     => [
							'deg' => [
								'max' => 360,
								'min' => - 360,
							],
						],
						'selectors' => [
							'{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-front .kfw-elementor-icon-wrap i'   => 'transform: rotate({{SIZE}}{{UNIT}});',
							'{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-front .kfw-elementor-icon-wrap svg' => 'transform: rotate({{SIZE}}{{UNIT}});',
						],
					]);
					$this->add_control('icon_background_rotate',[
						'label'     => esc_html__( 'Background Rotate', 'onnat-extra' ),
						'type'      => KFCM::SLIDER,
						'default'   => [
							'size' => 0,
							'unit' => 'deg',
						],
						'range'     => [
							'deg' => [
								'max' => 360,
								'min' => -360,
							],
						],
						'selectors' => [
							'{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap.kfw-ele-widget-flip-box-cube-graphic-ele-icon .kfw-ele-widget-flip-box-front .kfw-elementor-icon-wrap' => 'transform: rotate({{SIZE}}{{UNIT}});',
						],
					]);
				$this->end_controls_tab();
				$this->start_controls_tab( 'front_image_style_tab', [
					'label'     => esc_html__( 'Image', 'onnat-extra' ),
					'condition' => [
						'graphic_element' => 'image',
						'image[url]!'     => ''
					]
				]);
					$this->add_group_control( KFGCBG::get_type(), [
						'name'     => 'img_bg',
						'types'    => ['classic', 'gradient'],
						'exclude'  => [ 'image'],
						'selector' => '{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-front .kfw-elementor-icon-wrap',
					]);
					$this->add_responsive_control( 'img_padding',[
						'label'      => esc_html__('Padding', 'onnat-extra' ),
						'type'       => KFCM::DIMENSIONS,
						'size_units' => [ 'px', 'em', '%' ],
						'selectors'  => [
							'{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-front .kfw-elementor-icon-wrap' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
						],
					]);
					$this->add_responsive_control( 'img_margin',[
						'label'      => esc_html__('Margin', 'onnat-extra' ),
						'type'       => KFCM::DIMENSIONS,
						'size_units' => [ 'px', 'em', '%' ],
						'selectors'  => [
							'{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-front .kfw-elementor-icon-wrap' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
						],
					]);
					$this->add_group_control( KFGCBR::get_type(), [
						'name'     => 'img_border',
						'selector' => '{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-front .kfw-elementor-icon-wrap'
					]);
					$this->add_responsive_control( 'img_border_radius', [
						'label'      => esc_html__('Border Radius', 'onnat-extra' ),
						'type'       => KFCM::DIMENSIONS,
						'size_units' => [ 'px', '%' ],
						'default'    => [
							'top'      => '10',
							'right'    => '10',
							'bottom'   => '10',
							'left'     => '10',
							'unit'     => 'px',
							'isLinked' => false
						],
						'selectors'  => [
							'{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap.kfw-ele-widget-flip-box-cube-graphic-ele-image .kfw-ele-widget-flip-box-front .kfw-elementor-icon-wrap'     => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
							'{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap.kfw-ele-widget-flip-box-cube-graphic-ele-image .kfw-ele-widget-flip-box-front .kfw-elementor-icon-wrap img' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
						]
					]);
					$this->add_group_control( KFGCBS::get_type(), [
						'name'     => 'img_box_shadow',
						'selector' => '{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-front .kfw-elementor-icon-wrap'
					]);
					$this->add_responsive_control( 'img_size', [
						'label'      => esc_html__('Size', 'onnat-extra' ),
						'type'       => KFCM::SLIDER,
						'size_units' => [ '%' ],
						'range'      => [
							'%' => [
								'min' => 5,
								'max' => 100,
							],
						],
						'selectors'  => [
							'{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-front .kfw-elementor-icon-wrap img' => 'width: {{SIZE}}{{UNIT}};',
						],
					]);
					$this->add_control('img_rotate',[
						'label'     => esc_html__( 'Rotate', 'onnat-extra' ),
						'type'      => KFCM::SLIDER,
						'default'   => [
							'size' => 0,
							'unit' => 'deg',
						],
						'range'     => [
							'deg' => [
								'max' => 360,
								'min' => - 360,
							],
						],
						'selectors' => [
							'{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-front .kfw-elementor-icon-wrap img'   => 'transform: rotate({{SIZE}}{{UNIT}});',
						],
					]);
					$this->add_control('img_background_rotate',[
						'label'     => esc_html__( 'Background Rotate', 'onnat-extra' ),
						'type'      => KFCM::SLIDER,
						'default'   => [
							'size' => 0,
							'unit' => 'deg',
						],
						'range'     => [
							'deg' => [
								'max' => 360,
								'min' => - 360,
							],
						],
						'selectors' => [
							'{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap.kfw-ele-widget-flip-box-cube-graphic-ele-image .kfw-ele-widget-flip-box-front .kfw-elementor-icon-wrap' => 'transform: rotate({{SIZE}}{{UNIT}});',
						],
					]);
					$this->add_control( 'image_settings_hr', [
						'type' => KFCM::DIVIDER,
					]);
					$this->add_control('image_settings_heading',[
						'label' => esc_html__( 'Image Effects', 'onnat-extra' ),
						'type'  => KFCM::HEADING,
					]);
					$this->add_group_control( KFGCSSF::get_type(), [
						'name'      => 'img_css_filters',
						'selector'  => '{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-front .kfw-elementor-icon-wrap img',
					]);
					$this->add_control('img_opacity',[
						'label'     => esc_html__( 'Opacity', 'onnat-extra' ),
						'type'      => KFCM::SLIDER,
						'range'     => [
							'px' => [
								'max'  => 1,
								'min'  => 0.10,
								'step' => 0.01,
							],
						],
						'selectors' => [
							'{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-front .kfw-elementor-icon-wrap img' => 'opacity: {{SIZE}};'
						],
					]);
				$this->end_controls_tab();
			$this->end_controls_tabs();
		$this->end_controls_section();
	}

	public function _register_back_style_section() {
		$this->start_controls_section( 'section_back_style', [
			'label' => esc_html__( 'Back Side', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
			$this->add_responsive_control( 'padding_back', [
				'label'      => esc_html__( 'Padding', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-back .kfw-ele-widget-flip-box-layer-overlay' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]);
			$this->add_responsive_control( 'margin_back', [
				'label'      => esc_html__( 'Margin', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-back .kfw-ele-widget-flip-box-layer-overlay' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]);
			$this->add_responsive_control( 'alignment_back', [
				'label'       => esc_html__( 'Alignment', 'onnat-extra' ),
				'type'        => KFCM::CHOOSE,
				'label_block' => false,
				'options'     => [
					'left'   => [
						'title' => esc_html__('Left', 'onnat-extra'),
						'icon'  => 'eicon-text-align-left',
					],
					'center' => [
						'title' => esc_html__('Center', 'onnat-extra'),
						'icon'  => 'eicon-text-align-center',
					],
					'right'  => [
						'title' => esc_html__('Right', 'onnat-extra'),
						'icon'  => 'eicon-text-align-right',
					],
				],
				'default'     => 'center',
				'selectors'   => [
					'{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-back .kfw-ele-widget-flip-box-layer-overlay' => 'text-align: {{VALUE}};',
					'{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-back .kfw-ele-widget-flip-box-button' => 'margin-{{VALUE}}: 0',
				],
			]);
			$this->add_responsive_control( 'vertical_position_back', [
				'label'                => esc_html__( 'Vertical Position', 'onnat-extra' ),
				'type'                 => KFCM::CHOOSE,
				'label_block'          => false,
				'options'              => [
					'top'    => [
						'title' => esc_html__('Top', 'onnat-extra'),
						'icon'  => 'eicon-v-align-top',
					],
					'middle' => [
						'title' => esc_html__('Middle', 'onnat-extra'),
						'icon'  => 'eicon-v-align-middle',
					],
					'bottom' => [
						'title' => esc_html__('Bottom', 'onnat-extra'),
						'icon'  => 'eicon-v-align-bottom',
					],
				],
				'selectors_dictionary' => [
					'top'    => 'flex-start',
					'middle' => 'center',
					'bottom' => 'flex-end',
				],
				'selectors'            => [
					'{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-back .kfw-ele-widget-flip-box-layer-overlay' => 'justify-content: {{VALUE}};',
				],
			]);
			$this->add_group_control( KFGCBR::get_type(), [
				'name'      => 'border_back',
				'selector'  => '{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-back',
			]);
			$this->add_responsive_control( 'border_radius_back', [
				'label'      => esc_html__('Border Radius', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-back'                                        => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					'{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-back .kfw-ele-widget-flip-box-layer-overlay' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]);
			$this->add_group_control( KFGCBS::get_type(), [
				'name'     => 'box_shadow_back',
				'selector' => '{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-back',
			]);
			$this->add_control( 'hr_back', [
				'type' => KFCM::DIVIDER,
			]);
			$this->start_controls_tabs( 'back_style_tabs' );
				$this->start_controls_tab( 'back_title_style_tab', [
					'label'     => esc_html__( 'Title', 'onnat-extra' ),
					'condition' => [ 'back_title!'=> '' ]
				]);
					$this->add_group_control(KFGCT::get_type(), [
						'name'     => 'back_title_typo',
						'label'    => esc_html__( 'Typography', 'onnat-extra' ),
						'selector' => '{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-back .kfw-ele-widget-flip-box-layer-title',
					]);
					$this->add_control( 'back_title_color', [
						'label'     => esc_html__('Typo Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'selectors' => [
							'{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-back .kfw-ele-widget-flip-box-layer-title' => 'color:{{VALUE}};'
						],
					]);
					$this->add_responsive_control( 'back_title_margin', [
						'label'      => esc_html__('Margin', 'onnat-extra' ),
						'type'       => KFCM::DIMENSIONS,
						'size_units' => [ 'px', '%' ],
						'selectors'  => [ '{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-back .kfw-ele-widget-flip-box-layer-title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};' ],
					]);
				$this->end_controls_tab();
				$this->start_controls_tab( 'back_stitle_style_tab', [
					'label'     => esc_html__( 'Sub Title', 'onnat-extra' ),
					'condition' => [ 'back_sub_title!'=> '' ]
				]);
					$this->add_group_control(KFGCT::get_type(), [
						'name'     => 'back_sub_title_typo',
						'label'    => esc_html__( 'Typography', 'onnat-extra' ),
						'selector' => '{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-back .kfw-ele-widget-flip-box-layer-sub-title',
					]);
					$this->add_control( 'back_sub_title_color', [
						'label'     => esc_html__('Typo Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'selectors' => [
							'{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-back .kfw-ele-widget-flip-box-layer-sub-title' => 'color:{{VALUE}};'
						],
					]);
					$this->add_responsive_control( 'back_sub_title_margin', [
						'label'      => esc_html__('Margin', 'onnat-extra' ),
						'type'       => KFCM::DIMENSIONS,
						'size_units' => [ 'px', '%' ],
						'selectors'  => [ '{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-back .kfw-ele-widget-flip-box-layer-sub-title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};' ],
					]);
				$this->end_controls_tab();
				$this->start_controls_tab( 'back_description_style_tab', [
					'label'     => esc_html__( 'Description', 'onnat-extra' ),
					'condition' => [ 'back_description!'=> '' ]
				]);
					$this->add_group_control(KFGCT::get_type(), [
						'name'     => 'back_description_typo',
						'label'    => esc_html__( 'Typography', 'onnat-extra' ),
						'selector' => '{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-back .kfw-ele-widget-flip-box-layer-description',
					]);
					$this->add_control( 'back_description_color', [
						'label'     => esc_html__('Typo Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'selectors' => [
							'{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-back .kfw-ele-widget-flip-box-layer-description' => 'color:{{VALUE}};'
						],
					]);
					$this->add_responsive_control( 'back_description_margin', [
						'label'     => esc_html__('Margin', 'onnat-extra' ),
						'type'      => KFCM::DIMENSIONS,
						'size_units' => [ 'px', 'em', '%' ],
						'selectors' => [
							'{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-back .kfw-ele-widget-flip-box-layer-description' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
						]
					]);
				$this->end_controls_tab();
			$this->end_controls_tabs();

			/**
			 * Common Button Condition
			 */
			$btn_condition = [
				'relation' => 'or',
				'terms'    => [
					[
						'name'     => 'back_btn_text',
						'operator' => '!=',
						'value'    => ''
					],
					[
						'name'     => 'back_btn_icon[value]',
						'operator' => '!=',
						'value'    => ''
					],
				],
			];

			$this->add_control('back_btn_style_heading',[
				'label'      => esc_html__( 'Button', 'onnat-extra' ),
				'type'       => KFCM::HEADING,
				'separator'  => 'before',
				'conditions' => $btn_condition
			]);
			$this->add_group_control( KFGCT::get_type(), [
				'name'        => 'button_typo',
				'description' => esc_html__( 'Typography', 'onnat-extra' ),
				'selector'    => '{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-back .kfw-ele-widget-flip-box-button',
				'conditions'  => $btn_condition,
			]);
			$this->add_responsive_control( 'button_padding', [
				'label'      => esc_html__( 'Padding', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-back .kfw-ele-widget-flip-box-button' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				],
				'conditions' => $btn_condition
			]);
			$this->add_responsive_control( 'button_margin', [
				'label'      => esc_html__( 'Margin', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'separator'  => 'after',
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-back .kfw-ele-widget-flip-box-button' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				],
				'conditions' => $btn_condition
			]);
			$this->start_controls_tabs( 'button_customize_tabs' );
				$this->start_controls_tab( 'button_normal_state', [
					'label' => esc_html__( 'Normal', 'onnat-extra' ),
				]);
					$this->add_control( 'button_color', [
						'label'      => esc_html__('Button Text Color', 'onnat-extra' ),
						'type'       => KFCM::COLOR,
						'selectors'  => ['{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-back .kfw-ele-widget-flip-box-button' => 'color: {{VALUE}};' ],
						'conditions' => $btn_condition
					]);
					$this->add_group_control( KFGCBG::get_type(), [
						'name'       => 'button_background',
						'label'      => esc_html__('Background', 'onnat-extra' ),
						'types'      => [ 'classic', 'gradient' ],
						'exclude'    => [ 'image' ],                                                                                                      // to remove background image option in classic mode
						'selector'   => '{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-back .kfw-ele-widget-flip-box-button',
						'conditions' => $btn_condition
					]);
					$this->add_group_control( KFGCBR::get_type(), [
						'name'           => 'button_border',
						'fields_options' => [
							'width' => [ 'label' => _x( 'Border Width', 'Border Control', 'onnat-extra' ) ],
							'color' => [ 'label' => _x( 'Border Color', 'Border Control', 'onnat-extra' ) ],
						],
						'selector'       => '{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-back .kfw-ele-widget-flip-box-button',
						'conditions'     => $btn_condition,
					]);
					$this->add_responsive_control( 'button_border_radius', [
						'label'      => esc_html__('Border Radius', 'onnat-extra' ),
						'type'       => KFCM::DIMENSIONS,
						'size_units' => [ 'px', '%' ],
						'selectors'  => [ '{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-back .kfw-ele-widget-flip-box-button' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};' ],
						'conditions' => $btn_condition,
					]);
					$this->add_group_control( KFGCBS::get_type(), [
						'name'       => 'button_box_shadow',
						'selector'   => '{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-back .kfw-ele-widget-flip-box-button',
						'conditions' => $btn_condition,
					]);
				$this->end_controls_tab();
				$this->start_controls_tab( 'button_hover_state', [
					'label'     => esc_html__( 'Hover', 'onnat-extra' ),
				]);
					$this->add_control( 'button_color_hover', [
						'label'      => esc_html__('Button Text Color', 'onnat-extra' ),
						'type'       => KFCM::COLOR,
						'selectors'  => [ '{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-back .kfw-ele-widget-flip-box-button:hover' => 'color: {{VALUE}};' ],
						'conditions' => $btn_condition,
					]);
					$this->add_group_control( KFGCBG::get_type(), [
						'name'       => 'button_background_hover',
						'label'      => esc_html__('Background', 'onnat-extra' ),
						'types'      => [ 'classic', 'gradient' ],
						'exclude'    => [ 'image' ], // to remove background image option in classic mode
						'selector'   => '{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-back .kfw-ele-widget-flip-box-button:hover',
						'conditions' => $btn_condition,
					]);
					$this->add_group_control( KFGCBR::get_type(), [
						'name'           => 'button_border_hover',
						'fields_options' => [
							'width' => [ 'label' => _x( 'Border Width', 'Border Control', 'onnat-extra' ) ],
							'color' => [ 'label' => _x( 'Border Color', 'Border Control', 'onnat-extra' ) ],
						],
						'selector'       => '{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-back .kfw-ele-widget-flip-box-button:hover',
						'conditions'     => $btn_condition,
					]);
					$this->add_responsive_control( 'button_border_radius_hover', [
						'label'      => esc_html__('Border Radius', 'onnat-extra' ),
						'type'       => KFCM::DIMENSIONS,
						'size_units' => [ 'px', '%' ],
						'selectors'  => [
							'{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-back .kfw-ele-widget-flip-box-button:hover' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
						],
						'conditions' => $btn_condition,
					]);
					$this->add_group_control( KFGCBS::get_type(), [
						'name'       => 'button_box_shadow_hover',
						'selector'   => '{{WRAPPER}} .kfw-ele-widget-flip-box-cube-wrap .kfw-ele-widget-flip-box-back .kfw-ele-widget-flip-box-button:hover',
						'conditions' => $btn_condition,
					]);
					$this->add_control( 'button_animation_hover', [
						'label'      => esc_html__( 'Animation', 'onnat-extra' ),
						'type'       => KFCM::HOVER_ANIMATION,
						'conditions' => $btn_condition,
					]);
				$this->end_controls_tab();
			$this->end_controls_tabs();
		$this->end_controls_section();
	}

	public function _render_icon( $icon , $echo = true ) {
		$icon = array_filter( $icon );

		if( !empty( $icon ) ) {
			echo '<span class="kfw-elementor-icon-wrap">';
				KFIM::render_icon( $icon, [ 'aria-hidden' => 'true' ] );
			echo '</span>';
		}
	}

	/**
	 * Get attachment image src.
	 */
	public static function _get_image_src( $attachment_id, $image_size_key, array $settings ) {
        return KFGCIS::get_attachment_image_src( $attachment_id, $image_size_key, $settings );
	}

	public function _get_image_title( $attachment ) {
		if ( empty( $attachment['id'] ) ) {
			return '';
		}

		return get_the_title( $attachment['id'] );
	}

	public function _get_image_alt( $instance ) {
		if ( empty( $instance['id'] ) ) {
			// For `Insert From URL` images.
			return isset( $instance['alt'] ) ? trim( strip_tags( $instance['alt'] ) ) : '';
		}

		$attachment_id = $instance['id'];
		if ( ! $attachment_id ) {
			return '';
		}

		$attachment = get_post( $attachment_id );
		if ( ! $attachment ) {
			return '';
		}

		$alt = get_post_meta( $attachment_id, '_wp_attachment_image_alt', true );
		if ( ! $alt ) {
			$alt = $attachment->post_excerpt;
			if ( ! $alt ) {
				$alt = $attachment->post_title;
			}
		}
		return trim( strip_tags( $alt ) );
	}

	public function _button( $settings ) {
		$this->add_render_attribute( 'cta-wrapper', [
			'class' => array_filter([
				'kfw-ele-widget-flip-box-button-wrap',
				!empty( trim( $settings['back_btn_text'] ) ) ? 'kfw-ele-widget-flip-box-button-has-txt' : ''
			])
		]);

		echo '<div '.$this->get_render_attribute_string( 'cta-wrapper' ).'>';

			$this->add_render_attribute( 'cta', [
				'class' => array_filter([
					'kfw-ele-widget-flip-box-button',
					'kfw-ele-widget-flip-box-button-icon-align-' . esc_attr( $settings['back_btn_icon_pos'] ),
					!empty( $settings['button_animation_hover'] ) ? 'elementor-animation-'.esc_attr( $settings['button_animation_hover'] ): '',
				])
			]);

			if ( ! empty( $settings['back_btn_link']['url'] ) ) {
				$this->add_render_attribute( 'cta', 'href', esc_url( $settings['back_btn_link']['url'] ) );

				if ( ! empty( $settings['back_btn_link']['is_external'] ) ) {
					$this->add_render_attribute( 'cta', 'target', '_blank' );
				}

				if ( ! empty( $settings['back_btn_link']['nofollow'] ) ) {
					$this->add_render_attribute( 'cta', 'rel', 'nofollow' );
				}
			}

			echo '<a '.$this->get_render_attribute_string( 'cta' ).'>';

				if( $settings['back_btn_icon_pos'] === 'left' ) {
					$this->_render_icon( $settings['back_btn_icon']);
				}

				echo esc_html( $settings['back_btn_text'] );

				if( $settings['back_btn_icon_pos'] === 'right' ) {
					$this->_render_icon( $settings['back_btn_icon']);
				}

			echo '</a>';

		echo '</div>';
	}

	/**
	 * Render widget output on the frontend.
	 */
    protected function render() {
		$render = kf_extra_plugin_elementor()->get_template("kfw-flip-box-cube");
		include $render;
    }
}