<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

use \Elementor\Widget_Base as WB;
use \Elementor\Controls_Manager as KFCM;
use \Elementor\Icons_Manager as KFIM;

use \Elementor\Group_Control_Background as KFGCBG;
use \Elementor\Group_Control_Typography as KFGCT;
use \Elementor\Group_Control_Border as KFGCBR;
use \Elementor\Group_Control_Box_Shadow as KFGCBS;
use \Elementor\Group_Control_Text_Shadow as KFGCTS;
use \Elementor\Group_Control_Text_Stroke as KFGCTST;

class KFW_Icon_Box extends WB {

	/**
	 * Get element name.
	 *
	 * Retrieve the element name.
	 *
	 * @return string The name.
	 */
	public function get_name() {
        return 'kfw-icon-box';
    }

	/**
	 * Get element title.
	 *
	 * Retrieve the element title.
	 *
	 * @return string Element title.
	 */
	public function get_title() {
		return esc_html__('Icon Box', 'onnat-extra');
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the widget categories.
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ '_kfw_core_widgets' ];
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve the widget icon.
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'kfw-badge eicon-icon-box';
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the widget keywords.
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'kfw', 'kin', 'icon', 'box', 'icon box' ];
	}

	/**
	 * Get widget help url.
	 *
	 * Retrieve the widget help url.
	 *
	 * @return string Widget help url.
	 */
	public function get_custom_help_url() {
        return KFE_CONST_THEME_DOC . 'elementor-custom-widgets/icon-box';
    }

	/**
	 * Get HTML wrapper class.
     *
	 * Retrieve the widget container class.
     *
     * @return string Widget container css classes.
	 */
	public function get_html_wrapper_class() {
        $name = $this->get_name();

		return sprintf( 'kfw-ele-%s elementor-widget-%s', $name, $name );
	}

	/**
	 * Widget base constructor.
	 *
	 * Initializing the widget base class.
	 */
	public function __construct( $data = [], $args = null ) {
        parent::__construct( $data, $args );

		wp_register_style( 'kfw-icon-box',
			KFE_CONST_URL . 'libraries/elementor/assets/css/icon-box' . KFE_CONST_DEBUG_SUFFIX . '.css',
			[],
			KFE_CONST_VERSION
		);
    }

	/**
	 * Get style dependencies.
	 *
	 * Retrieve the list of style dependencies the element requires.
	 *
	 * @return array Element styles dependencies.
	 */
	public function get_style_depends() {
		return [ 'kfw-icon-box' ];
	}

	/**
	 * Get script dependencies.
	 *
	 * Retrieve the list of script dependencies the element requires.
	 *
	 * @return array Element scripts dependencies.
	 */
	public function get_script_depends() {
		return [ ];
	}

	/**
	 * Register controls.
	 *
	 * Used to add new controls to any element type.
	 */
	protected function register_controls() {
		$this->_register_icon_section();
		$this->_register_content_section();
		$this->_register_button_section();

		$this->_register_box_style_section();
		$this->_register_icon_style_section();
		$this->_register_title_style_section();
		$this->_register_sub_title_style_section();
		$this->_register_description_style_section();
		$this->_register_button_style_section();
    }

	public function _register_icon_section() {
		$this->start_controls_section( 'section_icon', [
			'label' => esc_html__( 'Icon', 'onnat-extra' )
		]);
			$this->add_control('icon_type',[
				'label'   => esc_html__( 'Type', 'onnat-extra' ),
				'type'    => KFCM::CHOOSE,
				'options' => [
					'blank' => [
						'title' => esc_html__('Blank', 'onnat-extra'),
						'icon'  => '',
					],
					'icon'  => [
						'title' => esc_html__('Icon', 'onnat-extra'),
						'icon'  => 'eicon-plus-circle-o',
					],
					'image'  => [
						'title' => esc_html__('Image', 'onnat-extra'),
						'icon'  => 'eicon-image',
					],
				],
				'default' => 'icon',
				'toggle'  => false
			]);
			$this->add_control( 'icon', [
				'label'     => esc_html__( 'Icon', 'onnat-extra' ),
				'type'      => KFCM::ICONS,
				'default'   => [
					'value'   => 'fas fa-heart',
					'library' => 'fa-solid',
				],
				'condition' => [ 'icon_type' => 'icon' ],
			]);
			$this->add_control( 'image', [
				'label'     => esc_html__( 'Image', 'onnat-extra' ),
				'type'      => KFCM::MEDIA,
				'dynamic'   => [ 'active' => true ],
				'default'   => [
					'url' => \Elementor\Utils::get_placeholder_image_src()
				],
				'condition' => [ 'icon_type' => 'image' ],
			]);
			$this->add_control('icon_position', [
				'label'          => esc_html__( 'Position', 'onnat-extra' ),
				'type'           => KFCM::CHOOSE,
				'options'        => [
					'left'   => [
						'title' => esc_html__( 'Left', 'onnat-extra' ),
						'icon'  => 'eicon-h-align-left',
					],
					'top'    => [
						'title' => esc_html__( 'Top', 'onnat-extra' ),
						'icon'  => 'eicon-v-align-top',
					],
					'right'  => [
						'title' => esc_html__( 'Right', 'onnat-extra' ),
						'icon'  => 'eicon-h-align-right',
					],
					'bottom' => [
						'title' => esc_html__( 'Bottom', 'onnat-extra' ),
						'icon'  => 'eicon-v-align-bottom',
					],
				],
				'default'        => 'top',
				'toggle'         => false,
				'prefix_class'   => 'kfw-ele-icon-box-icon-position-',
				'conditions'     => [
					'relation' => 'and',
					'terms'    => [
						[
							'name'     => 'icon_type',
							'operator' => '!==',
							'value'    => 'blank'
						],
						[
							'relation' => 'or',
							'terms'    => [
								[
									'name'     => 'icon[value]',
									'operator' => '!=',
									'value'    => '',
								],
								[
									'name'     => 'image[url]',
									'operator' => '!=',
									'value'    => '',
								],
							]
						]
					]
				]
			]);
			$this->add_control('icon_align',[
				'label'        => esc_html__( 'Alignment', 'onnat-extra' ),
				'type'         => KFCM::CHOOSE,
				'options'      => [
					'top'    => [
						'title' => esc_html__( 'Top', 'onnat-extra' ),
						'icon'  => 'eicon-v-align-top',
					],
					'middle' => [
						'title' => esc_html__( 'Middle', 'onnat-extra' ),
						'icon'  => 'eicon-v-align-middle',
					],
					'bottom' => [
						'title' => esc_html__( 'Bottom', 'onnat-extra' ),
						'icon'  => 'eicon-v-align-bottom',
					],
				],
				'prefix_class' => 'kfw-ele-icon-box-icon-vertical-align-',
				'default'      => 'top',
				'toggle'       => false,
				'conditions'   => [
					'relation' => 'and',
					'terms'    => [
						[
							'name'     => 'icon_type',
							'operator' => '!==',
							'value'    => 'blank'
						],
						[
							'name'     => 'icon_position',
							'operator' => 'in',
							'value'    => [ 'left', 'right' ]
						],
					]
				]
			]);
			$this->add_responsive_control('lr_icon_vertical_offset',[
				'label'          => esc_html__( 'Vertical Offset', 'onnat-extra' ),
				'type'           => KFCM::SLIDER,
				'default'        => ['size' => 0,],
				'tablet_default' => ['size' => 0,],
				'mobile_default' => ['size' => 0,],
				'range'          => [
					'px' => [
						'min' => -200,
						'max' => 200,
					],
				],
				'selectors'      => [
					'{{WRAPPER}}' => '--kfw-icon-box-lr-v-offet:{{SIZE}}px;'
				],
				'conditions'     => [
					'relation' => 'and',
					'terms'    => [
						[
							'name'     => 'icon_type',
							'operator' => '!==',
							'value'    => 'blank'
						],
						[
							'name'     => 'icon_position',
							'operator' => 'in',
							'value'    => [ 'left', 'right' ]
						],
						[
							'relation' => 'or',
							'terms'    => [
								[
									'name'     => 'icon[value]',
									'operator' => '!=',
									'value'    => '',
								],
								[
									'name'     => 'image[url]',
									'operator' => '!=',
									'value'    => '',
								],
							]
						]
					]
				],
			]);
			$this->add_responsive_control('lr_icon_horizontal_offset',[
				'label'          => esc_html__( 'Horizontal Offset', 'onnat-extra' ),
				'type'           => KFCM::SLIDER,
				'default'        => ['size' => 0,],
				'tablet_default' => ['size' => 0,],
				'mobile_default' => ['size' => 0,],
				'range'          => [
					'px' => [
						'min' => -200,
						'max' => 200,
					],
				],
				'selectors'      => [
					'{{WRAPPER}}' => '--kfw-icon-box-lr-h-offet:{{SIZE}}px;'
				],
				'conditions'     => [
					'relation' => 'and',
					'terms'    => [
						[
							'name'     => 'icon_type',
							'operator' => '!==',
							'value'    => 'blank'
						],
						[
							'name'     => 'icon_position',
							'operator' => 'in',
							'value'    => [ 'left', 'right' ]
						],
						[
							'relation' => 'or',
							'terms'    => [
								[
									'name'     => 'icon[value]',
									'operator' => '!=',
									'value'    => '',
								],
								[
									'name'     => 'image[url]',
									'operator' => '!=',
									'value'    => '',
								],
							]
						]
					]
				],
			]);
			$this->add_responsive_control('tb_icon_vertical_offset',[
				'label'          => esc_html__( 'Vertical Offset', 'onnat-extra' ),
				'type'           => KFCM::SLIDER,
				'default'        => ['size' => 0,],
				'tablet_default' => ['size' => 0,],
				'mobile_default' => ['size' => 0,],
				'range'          => [
					'px' => [
						'min' => -200,
						'max' => 200,
					],
				],
				'selectors'      => [
					'{{WRAPPER}}' => '--kfw-icon-box-tb-v-offet:{{SIZE}}px;'
				],
				'conditions'     => [
					'relation' => 'and',
					'terms'    => [
						[
							'name'     => 'icon_type',
							'operator' => '!==',
							'value'    => 'blank'
						],
						[
							'name'     => 'icon_position',
							'operator' => 'in',
							'value'    => [ 'top', 'bottom' ]
						],
						[
							'relation' => 'or',
							'terms'    => [
								[
									'name'     => 'icon[value]',
									'operator' => '!=',
									'value'    => '',
								],
								[
									'name'     => 'image[url]',
									'operator' => '!=',
									'value'    => '',
								],
							]
						]
					]
				],
			]);
			$this->add_responsive_control('tb_icon_horizontal_offset',[
				'label'          => esc_html__( 'Horizontal Offset', 'onnat-extra' ),
				'type'           => KFCM::SLIDER,
				'default'        => ['size' => 0,],
				'tablet_default' => ['size' => 0,],
				'mobile_default' => ['size' => 0,],
				'range'          => [
					'px' => [
						'min' => -200,
						'max' => 200,
					],
				],
				'selectors'      => [
					'{{WRAPPER}}' => '--kfw-icon-box-tb-h-offet:{{SIZE}}px;'
				],
				'conditions'     => [
					'relation' => 'and',
					'terms'    => [
						[
							'name'     => 'icon_type',
							'operator' => '!==',
							'value'    => 'blank'
						],
						[
							'name'     => 'icon_position',
							'operator' => 'in',
							'value'    => [ 'top', 'bottom' ]
						],
						[
							'relation' => 'or',
							'terms'    => [
								[
									'name'     => 'icon[value]',
									'operator' => '!=',
									'value'    => '',
								],
								[
									'name'     => 'image[url]',
									'operator' => '!=',
									'value'    => '',
								],
							]
						]
					]
				],
			]);
		$this->end_controls_section();
	}

    public function _register_content_section() {
		$this->start_controls_section( 'section_content', [
			'label' => esc_html__( 'Content', 'onnat-extra' )
		]);
            $this->add_control('title',[
                'label'       => esc_html__( 'Title', 'onnat-extra' ),
                'type'        => KFCM::TEXT,
                'dynamic'     => [ 'active' => true ],
                'default'     => esc_html__( 'Icon Box Title' , 'onnat-extra' ),
                'label_block' => true,
            ]);
            $this->add_control('title_link',[
                'label'     => esc_html__( 'Title Link', 'onnat-extra' ),
                'type'      => KFCM::URL,
                'dynamic'   => [
                    'active' => true
                ],
                'options'   => [
                    'url',
                    'is_external',
                    'nofollow'
                ],
                'default'   => [
                    'url'         => '',
                    'is_external' => true,
                    'nofollow'    => true,
                ],
                'condition' => [
                    'title!' => ''
                ]
            ]);
			$this->add_control('sub_title',[
				'label'       => esc_html__( 'Sub Title', 'onnat-extra' ),
				'type'        => KFCM::TEXT,
				'dynamic'     => [ 'active' => true, ],
				'placeholder' => esc_html__( 'Enter Sub Title', 'onnat-extra' ),
				'label_block' => true,
			]);
			$this->add_control('description',[
				'label'       => esc_html__( 'Description', 'onnat-extra' ),
				'type'        => KFCM::TEXTAREA,
				'dynamic'     => [ 'active' => true, ],
				'placeholder' => esc_html__( 'Enter Description', 'onnat-extra' ),
				'default'     => esc_html__( 'Lorem ipsum dolor sit amet consectetur adipiscing elit dolor', 'onnat-extra' ),
				'label_block' => true,
			]);
		$this->end_controls_section();
    }

    public function _register_button_section() {
		$this->start_controls_section( 'section_button', [
			'label' => esc_html__( 'Button', 'onnat-extra' )
		]);
            $this->add_control('use_button',[
                'label'        => esc_html__( 'Use Button', 'onnat-extra' ),
                'type'         => KFCM::SWITCHER,
                'return_value' => 'yes',
                'default'      => 'yes'
            ]);
			$this->add_control( 'btn_text', [
				'label'     => esc_html__( 'Text', 'onnat-extra' ),
				'type'      => KFCM::TEXT,
				'default'   => esc_html__( 'Button', 'onnat-extra' ),
				'condition' => [
					'use_button' => 'yes'
				]
			]);
			$this->add_control( 'btn_link', [
				'label'       => esc_html__( 'Link', 'onnat-extra' ),
				'type'        => KFCM::URL,
				'placeholder' => 'https://your-link.com',
				'default'     => [ 'url' => '#' ],
				'conditions'  => [
					'relation' => 'and',
					'terms'    => [
						[
							'name'     => 'use_button',
							'operator' => '===',
							'value'    => 'yes'
						],
						[
							'relation' => 'or',
							'terms'    => [
								[
									'name'     => 'btn_text',
									'operator' => '!=',
									'value'    => ''
								],
								[
									'name'     => 'btn_icon[value]',
									'operator' => '!=',
									'value'    => ''
								],
							],
						]
					]
				],
			]);
			$this->add_control( 'btn_icon', [
				'label'       => esc_html__( 'Icon', 'onnat-extra' ),
				'type'        => KFCM::ICONS,
				'skin'        => 'inline',
				'label_block' => false,
				'condition'   => [
					'use_button' => 'yes'
				],
				'default'     => [
					'value'   => 'fa fa-check',
					'library' => 'fa-solid'
				]
			]);
		$this->end_controls_section();
    }

	public function _register_box_style_section() {
		$this->start_controls_section( 'section_box_style', [
			'label' => esc_html__( 'Box', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
			$this->add_responsive_control( 'box_align', [
				'label'     => esc_html__( 'Alignment', 'onnat-extra' ),
				'type'      => KFCM::CHOOSE,
				'toggle'    => false,
				'default'   => 'center',
				'options'   => [
					'left'    => [
						'title' => esc_html__( 'Left', 'onnat-extra' ),
						'icon'  => 'eicon-text-align-left',
					],
					'center'  => [
						'title' => esc_html__( 'Center', 'onnat-extra' ),
						'icon'  => 'eicon-text-align-center',
					],
					'right'   => [
						'title' => esc_html__( 'Right', 'onnat-extra' ),
						'icon'  => 'eicon-text-align-right',
					],
					'justify' => [
						'title' => esc_html__( 'Justified', 'onnat-extra' ),
						'icon'  => 'eicon-text-align-justify',
					],
				],
				'selectors' => [
					'{{WRAPPER}} .kfw-ele-widget-icon-box-main' => 'text-align: {{VALUE}};',
				],
			]);
            $this->add_responsive_control( 'box_margin', [
                'label'      => esc_html__('Margin', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-ele-widget-icon-box-main' => 'margin:{{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->add_responsive_control( 'box_padding', [
                'label'      => esc_html__('Padding', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-ele-widget-icon-box-main' => 'padding:{{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->start_controls_tabs( 'box_customize_tabs');
				$this->start_controls_tab( 'box_normal_state', [
					'label' => esc_html__( 'Normal', 'onnat-extra' ),
				]);
					$this->add_group_control( KFGCBG::get_type(), [
						'name'           => 'box_bg',
						'types'          => ['classic', 'gradient'],
						'fields_options' => [
							'background' => [ 'default' => 'classic' ],
							'color'      => [ 'default' => '' ],
						],
						'selector'       => '{{WRAPPER}} .kfw-ele-widget-icon-box-main',
					]);
                    $this->add_group_control( KFGCBR::get_type(), [
                        'name'      => 'box_border',
                        'selector'  => '{{WRAPPER}} .kfw-ele-widget-icon-box-main',
                    ]);
                    $this->add_responsive_control( 'box_border_radius', [
                        'label'      => esc_html__('Border Radius', 'onnat-extra' ),
                        'type'       => KFCM::DIMENSIONS,
                        'size_units' => [ 'px', '%' ],
                        'selectors'  => [
                            '{{WRAPPER}} .kfw-ele-widget-icon-box-main' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                        ]
                    ]);
                    $this->add_group_control( KFGCBS::get_type(), [
                        'name'      => 'box_box_shadow',
                        'selector'  => '{{WRAPPER}} .kfw-ele-widget-icon-box-main',
                    ]);
                $this->end_controls_tab();
				$this->start_controls_tab( 'box_hover_state', [
					'label' => esc_html__( 'Hover', 'onnat-extra' ),
				]);
					$this->add_group_control( KFGCBG::get_type(), [
						'name'           => 'box_hover_bg',
						'types'          => ['classic', 'gradient'],
						'fields_options' => [
							'background' => [ 'default' => 'classic' ],
							'color'      => [ 'default' => '' ],
						],
						'selector'       => '{{WRAPPER}} .kfw-ele-widget-icon-box-main:before',
					]);
					$this->add_group_control( KFGCBR::get_type(), [
						'name'      => 'box_hover_border',
						'selector'  => '{{WRAPPER}} .kfw-ele-widget-icon-box-main:hover',
					]);
					$this->add_responsive_control( 'box_hover_border_radius', [
						'label'      => esc_html__('Border Radius', 'onnat-extra' ),
						'type'       => KFCM::DIMENSIONS,
						'size_units' => [ 'px', '%' ],
						'selectors'  => [
							'{{WRAPPER}} .kfw-ele-widget-icon-box-main:hover' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
						]
					]);
					$this->add_group_control( KFGCBS::get_type(), [
						'name'      => 'box_hover_box_shadow',
						'selector'  => '{{WRAPPER}} .kfw-ele-widget-icon-box-main:hover',
					]);
                    $this->add_control('box_hover_overlay_color',[
                        'label'     => __( 'Overlay Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'default'   => '',
                        'selectors' => ['{{WRAPPER}} .kfw-ele-widget-icon-box-main:after' => 'background-color: {{VALUE}};' ],
                    ]);
					$this->add_control('box_hover_style', [
						'label'        => esc_html__( 'Hover Style', 'onnat-extra' ),
						'type'         => KFCM::CHOOSE,
						'options'      => [
							'none'    => [
								'title' => esc_html__( 'None', 'onnat-extra' ),
								'icon'  => 'eicon-ban',
							],
							'default' => [
								'title' => esc_html__( 'Default', 'onnat-extra' ),
								'icon'  => 'eicon-globe',
							],
							'circle'  => [
								'title' => esc_html__( 'Circle', 'onnat-extra' ),
								'icon'  => 'eicon-circle',
							],
							'ellipse' => [
								'title' => esc_html__( 'Ellipse', 'onnat-extra' ),
								'icon'  => 'eicon-preview-medium',
							],
							'square'  => [
								'title' => esc_html__( 'Square', 'onnat-extra' ),
								'icon' => 'eicon-square',
							],
						],
						'default'      => 'none',
						'toggle'       => false,
						'prefix_class' => 'kfw-ele-widget-icon-box-hover-style-',
					]);
                $this->end_controls_tab();
			$this->end_controls_tabs();
		$this->end_controls_section();
	}

	public function _register_icon_style_section() {

		$conditions = [
			'relation' => 'and',
			'terms'    => [
				[
					'name'     => 'icon_type',
					'operator' => '!==',
					'value'    => 'blank'
				],
				[
					'relation' => 'or',
					'terms'    => [
						[
							'name'     => 'icon[value]',
							'operator' => '!=',
							'value'    => '',
						],
						[
							'name'     => 'image[url]',
							'operator' => '!=',
							'value'    => '',
						],
					]
				]
			]
		];

		$icon_condition = [
			'icon_type'    => 'icon',
			'icon[value]!' => ''
		];

		$img_condition = [
			'icon_type'   => 'image',
			'image[url]!' => ''
		];

		$this->start_controls_section( 'section_icon_style', [
			'label'      => esc_html__( 'Icon', 'onnat-extra' ),
			'tab'        => KFCM::TAB_STYLE,
			'conditions' => $conditions
		]);
			$this->add_responsive_control( 'icon_size', [
				'label'      => esc_html__('Icon Size', 'onnat-extra' ),
				'type'       => KFCM::SLIDER,
				'size_units' => [ 'px' ],
				'range'      => [
					'px' => [
						'min' => 0,
						'max' => 999,
					],
				],
				'condition'  => $icon_condition,
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-icon-box-icon-wrap.kfw-ele-widget-icon-box-icon-type .kfw-elementor-icon-wrap' => 'font-size: {{SIZE}}{{UNIT}};'
				],
			]);
			$this->add_responsive_control( 'img_size', [
				'label'      => esc_html__('Image Size', 'onnat-extra' ),
				'type'       => KFCM::SLIDER,
				'size_units' => ['%'],
				'range'      => [
					'%' => [
						'min' => 5,
						'max' => 100,
					],
				],
				'condition'  => $img_condition,
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-icon-box-icon-wrap.kfw-ele-widget-icon-box-image-type img' => 'max-width: {{SIZE}}%;'
				],
			]);
			$this->add_responsive_control( 'icon_margin', [
				'label'      => esc_html__('Margin', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'conditions' => $conditions,
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-icon-box-icon-wrap' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
			$this->add_responsive_control( 'icon_padding', [
				'label'      => esc_html__('Padding', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'conditions' => $conditions,
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-icon-box-icon-wrap .kfw-elementor-icon-wrap' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
			$this->start_controls_tabs( 'icon_customize_tabs' );
				$this->start_controls_tab( 'icon_normal_state', [
					'label'     => esc_html__( 'Normal', 'onnat-extra' ),
					'condition' => $icon_condition
				]);
					$this->add_control( 'icon_color', [
						'label'     => esc_html__('Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'condition' => $icon_condition,
						'selectors' => [
							'{{WRAPPER}} .kfw-ele-widget-icon-box-icon-wrap.kfw-ele-widget-icon-box-icon-type .kfw-elementor-icon-wrap' => 'color:{{VALUE}}; fill:{{VALUE}};',
						],
					]);
					$this->add_control('icon_rotate',[
						'label'     => esc_html__( 'Rotate', 'onnat-extra' ),
						'type'      => KFCM::SLIDER,
						'default'   => [
							'size' => 0,
							'unit' => 'deg',
						],
						'range'     => [
							'deg' => [
								'max' => 360,
								'min' => - 360,
							],
						],
						'condition' => $icon_condition,
						'selectors' => [
							'{{WRAPPER}} .kfw-ele-widget-icon-box-icon-wrap.kfw-ele-widget-icon-box-icon-type .kfw-elementor-icon-wrap i'   => 'transform: rotate({{SIZE}}{{UNIT}});',
							'{{WRAPPER}} .kfw-ele-widget-icon-box-icon-wrap.kfw-ele-widget-icon-box-icon-type .kfw-elementor-icon-wrap svg' => 'transform: rotate({{SIZE}}{{UNIT}});',
						],
					]);
					$this->add_group_control( KFGCBG::get_type(), [
						'name'           => 'icon_bg',
						'types'          => ['classic', 'gradient'],
						'exclude'        => [ 'image'],
						'fields_options' => [
							'background' => [ 'default' => 'classic' ],
							'color'      => [ 'default' => '' ],
						],
						'condition'      => $icon_condition,
						'selector'       => '{{WRAPPER}} .kfw-ele-widget-icon-box-icon-wrap.kfw-ele-widget-icon-box-icon-type .kfw-elementor-icon-wrap',
					]);
					$this->add_control('icon_bg_rotate',[
						'label'     => esc_html__( 'Background Rotate', 'onnat-extra' ),
						'type'      => KFCM::SLIDER,
						'default'   => [
							'size' => 0,
							'unit' => 'deg',
						],
						'range'     => [
							'deg' => [
								'max' => 360,
								'min' => -360,
							],
						],
						'condition' => [
							'icon_type'          => 'icon',
							'icon[value]!'       => '',
							'icon_bg_background' => [ 'classic', 'gradient' ],
						],
						'selectors' => [
							'{{WRAPPER}} .kfw-ele-widget-icon-box-icon-wrap.kfw-ele-widget-icon-box-icon-type .kfw-elementor-icon-wrap' => 'transform: rotate({{SIZE}}{{UNIT}});',
						],
					]);
					$this->add_group_control( KFGCBR::get_type(), [
						'name'      => 'icon_border',
						'condition' => $icon_condition,
						'selector'  => '{{WRAPPER}} .kfw-ele-widget-icon-box-icon-wrap.kfw-ele-widget-icon-box-icon-type .kfw-elementor-icon-wrap'
					]);
					$this->add_responsive_control( 'icon_border_radius', [
						'label'      => esc_html__('Border Radius', 'onnat-extra' ),
						'type'       => KFCM::DIMENSIONS,
						'size_units' => [ 'px', '%' ],
						'condition'  => $icon_condition,
						'selectors'  => [
							'{{WRAPPER}} .kfw-ele-widget-icon-box-icon-wrap.kfw-ele-widget-icon-box-icon-type .kfw-elementor-icon-wrap' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
						]
					]);
					$this->add_group_control( KFGCBS::get_type(), [
						'name'      => 'icon_box_shadow',
						'condition' => $icon_condition,
						'selector'  => '{{WRAPPER}} .kfw-ele-widget-icon-box-icon-wrap.kfw-ele-widget-icon-box-icon-type .kfw-elementor-icon-wrap'
					]);
				$this->end_controls_tab();
				$this->start_controls_tab( 'icon_hover_state', [
					'label'     => esc_html__( 'Hover', 'onnat-extra' ),
					'condition' => $icon_condition,
				]);
					$this->add_control( 'icon_hover_color', [
						'label'     => esc_html__('Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'condition' => $icon_condition,
						'selectors' => [
							'{{WRAPPER}}:hover .kfw-ele-widget-icon-box-icon-wrap.kfw-ele-widget-icon-box-icon-type .kfw-elementor-icon-wrap' => 'color:{{VALUE}}; fill:{{VALUE}};',
						],
					]);
					$this->add_control('icon_hover_rotate',[
						'label'     => esc_html__( 'Rotate', 'onnat-extra' ),
						'type'      => KFCM::SLIDER,
						'default'   => [
							'size' => 0,
							'unit' => 'deg',
						],
						'range'     => [
							'deg' => [
								'max' => 360,
								'min' => - 360,
							],
						],
						'condition' => $icon_condition,
						'selectors' => [
							'{{WRAPPER}}:hover .kfw-ele-widget-icon-box-icon-wrap.kfw-ele-widget-icon-box-icon-type .kfw-elementor-icon-wrap i'   => 'transform: rotate({{SIZE}}{{UNIT}});',
							'{{WRAPPER}}:hover .kfw-ele-widget-icon-box-icon-wrap.kfw-ele-widget-icon-box-icon-type .kfw-elementor-icon-wrap svg' => 'transform: rotate({{SIZE}}{{UNIT}});',
						],
					]);
					$this->add_group_control( KFGCBG::get_type(), [
						'name'           => 'icon_hover_bg',
						'types'          => ['classic', 'gradient'],
						'exclude'        => [ 'image'],
						'fields_options' => [
							'background' => [ 'default' => 'classic' ],
							'color'      => [ 'default' => '' ],
						],
						'condition'      => $icon_condition,
						'selector'       => '{{WRAPPER}}:hover .kfw-ele-widget-icon-box-icon-wrap.kfw-ele-widget-icon-box-icon-type .kfw-elementor-icon-wrap',
					]);
					$this->add_control('icon_hover_bg_rotate',[
						'label'     => esc_html__( 'Background Rotate', 'onnat-extra' ),
						'type'      => KFCM::SLIDER,
						'default'   => [
							'size' => 0,
							'unit' => 'deg',
						],
						'range'     => [
							'deg' => [
								'max' => 360,
								'min' => -360,
							],
						],
						'condition' => [
							'icon_type'                => 'icon',
							'icon[value]!'             => '',
							'icon_hover_bg_background' => [ 'classic', 'gradient' ],
						],
						'selectors' => [
							'{{WRAPPER}}:hover .kfw-ele-widget-icon-box-icon-wrap.kfw-ele-widget-icon-box-icon-type .kfw-elementor-icon-wrap' => 'transform: rotate({{SIZE}}{{UNIT}});',
						],
					]);
					$this->add_group_control( KFGCBR::get_type(), [
						'name'      => 'icon_hover_border',
						'condition' => $icon_condition,
						'selector'  => '{{WRAPPER}}:hover .kfw-ele-widget-icon-box-icon-wrap.kfw-ele-widget-icon-box-icon-type .kfw-elementor-icon-wrap'
					]);
					$this->add_responsive_control( 'icon_hover_border_radius', [
						'label'      => esc_html__('Border Radius', 'onnat-extra' ),
						'type'       => KFCM::DIMENSIONS,
						'size_units' => [ 'px', '%' ],
						'condition'  => $icon_condition,
						'selectors'  => [
							'{{WRAPPER}}:hover .kfw-ele-widget-icon-box-icon-wrap.kfw-ele-widget-icon-box-icon-type .kfw-elementor-icon-wrap' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
						]
					]);
					$this->add_group_control( KFGCBS::get_type(), [
						'name'      => 'icon_hover_box_shadow',
						'condition' => $icon_condition,
						'selector'  => '{{WRAPPER}}:hover .kfw-ele-widget-icon-box-icon-wrap.kfw-ele-widget-icon-box-icon-type .kfw-elementor-icon-wrap'
					]);
				$this->end_controls_tab();
			$this->end_controls_tabs();
			$this->start_controls_tabs( 'img_customize_tabs' );
				$this->start_controls_tab( 'img_normal_state', [
					'label'     => esc_html__( 'Normal', 'onnat-extra' ),
					'condition' => $img_condition,
				]);
					$this->add_control('img_rotate',[
						'label'     => esc_html__( 'Rotate', 'onnat-extra' ),
						'type'      => KFCM::SLIDER,
						'default'   => [
							'size' => 0,
							'unit' => 'deg',
						],
						'range'     => [
							'deg' => [
								'max' => 360,
								'min' => - 360,
							],
						],
						'condition' => $img_condition,
						'selectors' => [
							'{{WRAPPER}} .kfw-ele-widget-icon-box-icon-wrap.kfw-ele-widget-icon-box-image-type .kfw-elementor-icon-wrap img' => 'transform: rotate({{SIZE}}{{UNIT}});',
						],
					]);
					$this->add_group_control( KFGCBG::get_type(), [
						'name'           => 'img_bg',
						'types'          => ['classic', 'gradient'],
						'exclude'        => [ 'image'],
						'fields_options' => [
							'background' => [ 'default' => 'classic' ],
							'color'      => [ 'default' => '' ],
						],
						'condition'      => $img_condition,
						'selector'       => '{{WRAPPER}} .kfw-ele-widget-icon-box-icon-wrap.kfw-ele-widget-icon-box-image-type .kfw-elementor-icon-wrap',
					]);
					$this->add_control('img_bg_rotate',[
						'label'     => esc_html__( 'Background Rotate', 'onnat-extra' ),
						'type'      => KFCM::SLIDER,
						'default'   => [
							'size' => 0,
							'unit' => 'deg',
						],
						'range'     => [
							'deg' => [
								'max' => 360,
								'min' => -360,
							],
						],
						'condition' => [
							'icon_type'         => 'image',
							'image[url]!'       => '',
							'img_bg_background' => [ 'classic', 'gradient' ],
						],
						'selectors' => [
							'{{WRAPPER}} .kfw-ele-widget-icon-box-icon-wrap.kfw-ele-widget-icon-box-image-type .kfw-elementor-icon-wrap' => 'transform: rotate({{SIZE}}{{UNIT}});',
						],
					]);
					$this->add_group_control( KFGCBR::get_type(), [
						'name'      => 'img_border',
						'condition' => $img_condition,
						'selector'  => '{{WRAPPER}} .kfw-ele-widget-icon-box-icon-wrap.kfw-ele-widget-icon-box-image-type .kfw-elementor-icon-wrap'
					]);
					$this->add_responsive_control( 'img_border_radius', [
						'label'      => esc_html__('Border Radius', 'onnat-extra' ),
						'type'       => KFCM::DIMENSIONS,
						'size_units' => [ 'px', '%' ],
						'condition'  => $img_condition,
						'selectors'  => [
							'{{WRAPPER}} .kfw-ele-widget-icon-box-icon-wrap.kfw-ele-widget-icon-box-image-type .kfw-elementor-icon-wrap' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
						]
					]);
					$this->add_group_control( KFGCBS::get_type(), [
						'name'      => 'img_box_shadow',
						'condition' => $img_condition,
						'selector'  => '{{WRAPPER}} .kfw-ele-widget-icon-box-icon-wrap.kfw-ele-widget-icon-box-image-type .kfw-elementor-icon-wrap'
					]);
				$this->end_controls_tab();
				$this->start_controls_tab( 'img_hover_state', [
					'label'     => esc_html__( 'Hover', 'onnat-extra' ),
					'condition' => $img_condition,
				]);
					$this->add_control('img_hover_rotate',[
						'label'     => esc_html__( 'Rotate', 'onnat-extra' ),
						'type'      => KFCM::SLIDER,
						'default'   => [
							'size' => 0,
							'unit' => 'deg',
						],
						'range'     => [
							'deg' => [
								'max' => 360,
								'min' => - 360,
							],
						],
						'condition' => $img_condition,
						'selectors' => [
							'{{WRAPPER}}:hover .kfw-ele-widget-icon-box-icon-wrap.kfw-ele-widget-icon-box-image-type .kfw-elementor-icon-wrap img' => 'transform: rotate({{SIZE}}{{UNIT}});',
						],
					]);
					$this->add_group_control( KFGCBG::get_type(), [
						'name'           => 'img_hover_bg',
						'types'          => ['classic', 'gradient'],
						'exclude'        => [ 'image'],
						'fields_options' => [
							'background' => [ 'default' => 'classic' ],
							'color'      => [ 'default' => '' ],
						],
						'condition'      => $img_condition,
						'selector'       => '{{WRAPPER}}:hover .kfw-ele-widget-icon-box-icon-wrap.kfw-ele-widget-icon-box-image-type .kfw-elementor-icon-wrap',
					]);
					$this->add_control('img_hover_bg_rotate',[
						'label'     => esc_html__( 'Background Rotate', 'onnat-extra' ),
						'type'      => KFCM::SLIDER,
						'default'   => [
							'size' => 0,
							'unit' => 'deg',
						],
						'range'     => [
							'deg' => [
								'max' => 360,
								'min' => -360,
							],
						],
						'condition' => [
							'icon_type'         => 'image',
							'image[url]!'       => '',
							'img_bg_background' => [ 'classic', 'gradient' ],
						],
						'selectors' => [
							'{{WRAPPER}}:hover .kfw-ele-widget-icon-box-icon-wrap.kfw-ele-widget-icon-box-image-type .kfw-elementor-icon-wrap' => 'transform: rotate({{SIZE}}{{UNIT}});',
						],
					]);
					$this->add_group_control( KFGCBR::get_type(), [
						'name'      => 'img_hover_border',
						'condition' => $img_condition,
						'selector'  => '{{WRAPPER}}:hover .kfw-ele-widget-icon-box-icon-wrap.kfw-ele-widget-icon-box-image-type .kfw-elementor-icon-wrap'
					]);
					$this->add_responsive_control( 'img_hover_border_radius', [
						'label'      => esc_html__('Border Radius', 'onnat-extra' ),
						'type'       => KFCM::DIMENSIONS,
						'size_units' => [ 'px', '%' ],
						'condition'  => $img_condition,
						'selectors'  => [
							'{{WRAPPER}}:hover .kfw-ele-widget-icon-box-icon-wrap.kfw-ele-widget-icon-box-image-type .kfw-elementor-icon-wrap' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
						]
					]);
					$this->add_group_control( KFGCBS::get_type(), [
						'name'      => 'img_hover_box_shadow',
						'condition' => $img_condition,
						'selector'  => '{{WRAPPER}}:hover .kfw-ele-widget-icon-box-icon-wrap.kfw-ele-widget-icon-box-image-type .kfw-elementor-icon-wrap'
					]);
				$this->end_controls_tab();
			$this->end_controls_tabs();
		$this->end_controls_section();
	}

	public function _register_title_style_section() {
		$this->start_controls_section( 'section_title_style', [
			'label'     => esc_html__( 'Title', 'onnat-extra' ),
			'tab'       => KFCM::TAB_STYLE,
			'condition' => [ 'title!'=> '' ]
		]);
			$this->add_group_control(KFGCT::get_type(), [
				'name'      => 'title_typo',
				'label'     => esc_html__( 'Typography', 'onnat-extra' ),
				'condition' => [ 'title!'=> '' ],
				'selector'  => '{{WRAPPER}} .kfw-ele-widget-icon-box-title',
			]);
			$this->add_responsive_control( 'title_margin', [
				'label'      => esc_html__('Margin', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'condition'  => [ 'title!'=> '' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-icon-box-title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
			$this->add_responsive_control( 'title_padding', [
				'label'      => esc_html__('Padding', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'condition'  => [ 'title!'=> '' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-icon-box-title' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
			$this->start_controls_tabs( 'title_customize_tabs' );
				$this->start_controls_tab( 'title_normal_state', [
					'label'     => esc_html__( 'Normal', 'onnat-extra' ),
					'condition' => [ 'title!'=> '' ],
				]);
					$this->add_control( 'title_color', [
						'label'     => esc_html__('Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'selectors' => ['{{WRAPPER}} .kfw-ele-widget-icon-box-title, {{WRAPPER}} .kfw-ele-widget-icon-box-title a' => 'color: {{VALUE}};' ],
						'condition' => [ 'title!'=> '' ],
					]);
					$this->add_group_control( KFGCTS::get_type(), [
						'name'      => 'title_txt_shadow',
						'condition' => [ 'title!'=> '' ],
						'selector'  => '{{WRAPPER}} .kfw-ele-widget-icon-box-title, {{WRAPPER}} .kfw-ele-widget-icon-box-title a',
					]);
					$this->add_group_control( KFGCTST::get_type(), [
						'name'      => 'title_txt_stroke',
						'condition' => [ 'title!'=> '' ],
						'selector'  => '{{WRAPPER}} .kfw-ele-widget-icon-box-title, {{WRAPPER}} .kfw-ele-widget-icon-box-title a',
					]);
				$this->end_controls_tab();
				$this->start_controls_tab( 'title_hover_state', [
					'label'     => esc_html__( 'Hover', 'onnat-extra' ),
					'condition' => [ 'title!'=> '' ],
				]);
					$this->add_control( 'title_hover_color', [
						'label'     => esc_html__('Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'selectors' => ['{{WRAPPER}}:hover .kfw-ele-widget-icon-box-title, {{WRAPPER}}:hover .kfw-ele-widget-icon-box-title a' => 'color: {{VALUE}};' ],
						'condition' => [ 'title!'=> '' ],
					]);
					$this->add_group_control( KFGCTS::get_type(), [
						'name'      => 'title_hover_txt_shadow',
						'condition' => [ 'title!'=> '' ],
						'selector'  => '{{WRAPPER}}:hover .kfw-ele-widget-icon-box-title, {{WRAPPER}}:hover .kfw-ele-widget-icon-box-title a',
					]);
					$this->add_group_control( KFGCTST::get_type(), [
						'name'      => 'title_hover_txt_stroke',
						'condition' => [ 'title!'=> '' ],
						'selector'  => '{{WRAPPER}}:hover .kfw-ele-widget-icon-box-title, {{WRAPPER}}:hover .kfw-ele-widget-icon-box-title a',
					]);
				$this->end_controls_tab();
			$this->end_controls_tabs();
		$this->end_controls_section();
	}

	public function _register_sub_title_style_section() {
		$this->start_controls_section( 'section_sub_title_style', [
			'label'     => esc_html__( 'Sub Title', 'onnat-extra' ),
			'tab'       => KFCM::TAB_STYLE,
			'condition' => [ 'sub_title!'=> '' ]
		]);
			$this->add_group_control(KFGCT::get_type(), [
				'name'      => 'sub_title_typo',
				'label'     => esc_html__( 'Typography', 'onnat-extra' ),
				'condition' => [ 'sub_title!'=> '' ],
				'selector'  => '{{WRAPPER}} .kfw-ele-widget-icon-box-sub-title',
			]);
			$this->add_responsive_control( 'sub_title_margin', [
				'label'      => esc_html__('Margin', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'condition'  => [ 'sub_title!'=> '' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-icon-box-sub-title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
			$this->add_responsive_control( 'sub_title_padding', [
				'label'      => esc_html__('Padding', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'condition'  => [ 'sub_title!'=> '' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-icon-box-sub-title' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
			$this->start_controls_tabs( 'sub_title_customize_tabs' );
				$this->start_controls_tab( 'sub_title_normal_state', [
					'label'     => esc_html__( 'Normal', 'onnat-extra' ),
					'condition' => [ 'sub_title!'=> '' ],
				]);
					$this->add_control( 'sub_title_color', [
						'label'     => esc_html__('Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'selectors' => ['{{WRAPPER}} .kfw-ele-widget-icon-box-sub-title' => 'color: {{VALUE}};' ],
						'condition' => [ 'sub_title!'=> '' ],
					]);
					$this->add_group_control( KFGCTS::get_type(), [
						'name'      => 'sub_title_txt_shadow',
						'condition' => [ 'sub_title!'=> '' ],
						'selector'  => '{{WRAPPER}} .kfw-ele-widget-icon-box-sub-title',
					]);
					$this->add_group_control( KFGCTST::get_type(), [
						'name'      => 'sub_title_txt_stroke',
						'condition' => [ 'sub_title!'=> '' ],
						'selector'  => '{{WRAPPER}} .kfw-ele-widget-icon-box-sub-title',
					]);
				$this->end_controls_tab();
				$this->start_controls_tab( 'sub_title_hover_state', [
					'label'     => esc_html__( 'Hover', 'onnat-extra' ),
					'condition' => [ 'sub_title!'=> '' ],
				]);
					$this->add_control( 'sub_title_hover_color', [
						'label'     => esc_html__('Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'selectors' => ['{{WRAPPER}}:hover .kfw-ele-widget-icon-box-sub-title' => 'color: {{VALUE}};' ],
						'condition' => [ 'sub_title!'=> '' ],
					]);
					$this->add_group_control( KFGCTS::get_type(), [
						'name'      => 'sub_title_hover_txt_shadow',
						'condition' => [ 'sub_title!'=> '' ],
						'selector'  => '{{WRAPPER}}:hover .kfw-ele-widget-icon-box-sub-title',
					]);
					$this->add_group_control( KFGCTST::get_type(), [
						'name'      => 'sub_title_hover_txt_stroke',
						'condition' => [ 'sub_title!'=> '' ],
						'selector'  => '{{WRAPPER}}:hover .kfw-ele-widget-icon-box-sub-title',
					]);
				$this->end_controls_tab();
			$this->end_controls_tabs();
		$this->end_controls_section();
	}

	public function _register_description_style_section() {
		$this->start_controls_section( 'section_description_style', [
			'label'     => esc_html__( 'Description', 'onnat-extra' ),
			'tab'       => KFCM::TAB_STYLE,
			'condition' => [ 'description!'=> '' ]
		]);
			$this->add_group_control(KFGCT::get_type(), [
				'name'      => 'description_typo',
				'label'     => esc_html__( 'Typography', 'onnat-extra' ),
				'condition' => [ 'description!'=> '' ],
				'selector'  => '{{WRAPPER}} .kfw-ele-widget-icon-box-description',
			]);
			$this->add_responsive_control( 'description_margin', [
				'label'      => esc_html__('Margin', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'condition'  => [ 'description!'=> '' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-icon-box-description' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
			$this->add_responsive_control( 'description_padding', [
				'label'      => esc_html__('Padding', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'condition'  => [ 'description!'=> '' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-icon-box-description' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
			$this->start_controls_tabs( 'description_customize_tabs' );
				$this->start_controls_tab( 'description_normal_state', [
					'label'     => esc_html__( 'Normal', 'onnat-extra' ),
					'condition' => [ 'description!'=> '' ],
				]);
					$this->add_control( 'description_color', [
						'label'     => esc_html__('Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'selectors' => ['{{WRAPPER}} .kfw-ele-widget-icon-box-description' => 'color: {{VALUE}};' ],
						'condition' => [ 'description!'=> '' ],
					]);
					$this->add_group_control( KFGCTS::get_type(), [
						'name'      => 'description_txt_shadow',
						'condition' => [ 'description!'=> '' ],
						'selector'  => '{{WRAPPER}} .kfw-ele-widget-icon-box-description',
					]);
					$this->add_group_control( KFGCTST::get_type(), [
						'name'      => 'description_txt_stroke',
						'condition' => [ 'description!'=> '' ],
						'selector'  => '{{WRAPPER}} .kfw-ele-widget-icon-box-description',
					]);
				$this->end_controls_tab();
				$this->start_controls_tab( 'description_hover_state', [
					'label'     => esc_html__( 'Hover', 'onnat-extra' ),
					'condition' => [ 'description!'=> '' ],
				]);
					$this->add_control( 'description_hover_color', [
						'label'     => esc_html__('Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'selectors' => ['{{WRAPPER}}:hover .kfw-ele-widget-icon-box-description' => 'color: {{VALUE}};' ],
						'condition' => [ 'description!'=> '' ],
					]);
					$this->add_group_control( KFGCTS::get_type(), [
						'name'      => 'description_hover_txt_shadow',
						'condition' => [ 'description!'=> '' ],
						'selector'  => '{{WRAPPER}}:hover .kfw-ele-widget-icon-box-description',
					]);
					$this->add_group_control( KFGCTST::get_type(), [
						'name'      => 'description_hover_txt_stroke',
						'condition' => [ 'description!'=> '' ],
						'selector'  => '{{WRAPPER}}:hover .kfw-ele-widget-icon-box-description',
					]);
				$this->end_controls_tab();
			$this->end_controls_tabs();
		$this->end_controls_section();
	}

	public function _register_button_style_section() {
		$style_condition = [
			'relation' => 'and',
			'terms'    => [
				[
					'name'     => 'use_button',
					'operator' => '===',
					'value'    => 'yes'
				],
				[
					'relation' => 'or',
					'terms'    => [
						[
							'name'     => 'btn_text',
							'operator' => '!=',
							'value'    => ''
						],
						[
							'name'     => 'btn_icon[value]',
							'operator' => '!=',
							'value'    => ''
						],
					],
				]
			]
		];

		$btn_txt_condition = [
			'relation' => 'and',
			'terms'    => [
				[
					'name'     => 'use_button',
					'operator' => '===',
					'value'    => 'yes'
				],
				[
					'name'     => 'btn_text',
					'operator' => '!=',
					'value'    => ''
				],
			]
		];

		$this->start_controls_section( 'section_button_style', [
			'label'      => esc_html__( 'Button', 'onnat-extra' ),
			'tab'        => KFCM::TAB_STYLE,
			'conditions' => $style_condition,
		]);
			$this->add_responsive_control( 'btn_align', [
				'label'     => esc_html__( 'Alignment', 'onnat-extra' ),
				'type'      => KFCM::CHOOSE,
				'toggle'    => false,
				'default'   => '',
				'options'   => [
					'left'    => [
						'title' => esc_html__( 'Left', 'onnat-extra' ),
						'icon'  => 'eicon-text-align-left',
					],
					'center'  => [
						'title' => esc_html__( 'Center', 'onnat-extra' ),
						'icon'  => 'eicon-text-align-center',
					],
					'right'   => [
						'title' => esc_html__( 'Right', 'onnat-extra' ),
						'icon'  => 'eicon-text-align-right',
					],
				],
				'selectors' => [
					'{{WRAPPER}} .kfw-ele-widget-icon-box-button-wrap' => 'text-align: {{VALUE}};',
				],
			]);
			$this->add_responsive_control( 'btn_padding', [
				'label'      => esc_html__( 'Padding', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-icon-box-button' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				],
				'condition' => [
					'use_button!' => ''
				],
			]);
			$this->add_responsive_control( 'btn_margin', [
				'label'      => esc_html__( 'Margin', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'separator'  => 'after',
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-icon-box-button-wrap' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				],
				'condition' => [
					'use_button!' => ''
				]
			]);
			$this->add_control( 'btn_animation_hover', [
				'label'      => esc_html__( 'Hover Animation', 'onnat-extra' ),
				'type'       => KFCM::HOVER_ANIMATION,
				'conditions' => $style_condition,
			]);

			/**
			 * Button Text Style
			 */
				$this->add_control('btn_txt_style_before_hr', [
					'type'       => KFCM::DIVIDER,
					'conditions' => $btn_txt_condition,
				]);
				$this->add_control('btn_txt_style_heading',[
					'label'      => esc_html__( 'BUTTON TEXT', 'onnat-extra' ),
					'type'       => KFCM::HEADING,
					'conditions' => $btn_txt_condition,
				]);
				$this->add_control('btn_txt_style_after_hr', [
					'type'       => KFCM::DIVIDER,
					'conditions' => $btn_txt_condition,
				]);
				$this->add_group_control( KFGCT::get_type(), [
					'name'        => 'btn_typo',
					'description' => esc_html__( 'Typography', 'onnat-extra' ),
					'selector'    => '{{WRAPPER}} .kfw-ele-widget-icon-box-button',
					'conditions'  => $btn_txt_condition,
				]);

				$this->start_controls_tabs( 'btn_customize_tabs' );
					$this->start_controls_tab( 'btn_normal_state', [
						'label'      => esc_html__( 'Normal', 'onnat-extra' ),
						'conditions' => $btn_txt_condition
					]);
						$this->add_control( 'btn_color', [
							'label'      => esc_html__('Text Color', 'onnat-extra' ),
							'type'       => KFCM::COLOR,
							'selectors'  => ['{{WRAPPER}} .kfw-ele-widget-icon-box-button' => 'color: {{VALUE}};' ],
							'conditions' => $btn_txt_condition
						]);
						$this->add_group_control( KFGCBG::get_type(), [
							'name'       => 'btn_bg',
							'label'      => esc_html__('Background', 'onnat-extra' ),
							'types'      => [ 'classic', 'gradient' ],
							'exclude'    => [ 'image' ], # To remove background image option in classic mode
							'selector'   => '{{WRAPPER}} .kfw-ele-widget-icon-box-button',
							'conditions' => $btn_txt_condition
						]);
						$this->add_group_control( KFGCBR::get_type(), [
							'name'           => 'btn_border',
							'fields_options' => [
								'width' => [ 'label' => _x( 'Border Width', 'Border Control', 'onnat-extra' ) ],
								'color' => [ 'label' => _x( 'Border Color', 'Border Control', 'onnat-extra' ) ],
							],
							'selector'       => '{{WRAPPER}} .kfw-ele-widget-icon-box-button',
							'conditions'     => $btn_txt_condition,
						]);
						$this->add_responsive_control( 'btn_border_radius', [
							'label'      => esc_html__('Border Radius', 'onnat-extra' ),
							'type'       => KFCM::DIMENSIONS,
							'size_units' => [ 'px', '%' ],
							'selectors'  => [ '{{WRAPPER}} .kfw-ele-widget-icon-box-button' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};' ],
							'conditions' => $btn_txt_condition,
						]);
						$this->add_group_control( KFGCBS::get_type(), [
							'name'       => 'btn_box_shadow',
							'selector'   => '{{WRAPPER}} .kfw-ele-widget-icon-box-button',
							'conditions' => $btn_txt_condition,
						]);
					$this->end_controls_tab();
					$this->start_controls_tab( 'btn_hover_state', [
						'label'      => esc_html__( 'Hover', 'onnat-extra' ),
						'conditions' => $btn_txt_condition
					]);
						$this->add_control( 'btn_hover_color', [
							'label'      => esc_html__('Text Color', 'onnat-extra' ),
							'type'       => KFCM::COLOR,
							'selectors'  => ['{{WRAPPER}} .kfw-ele-widget-icon-box-button:hover' => 'color: {{VALUE}};' ],
							'conditions' => $btn_txt_condition
						]);
						$this->add_group_control( KFGCBG::get_type(), [
							'name'       => 'btn_hover_bg',
							'label'      => esc_html__('Background', 'onnat-extra' ),
							'types'      => [ 'classic', 'gradient' ],
							'exclude'    => [ 'image' ], # To remove background image option in classic mode
							'selector'   => '{{WRAPPER}} .kfw-ele-widget-icon-box-button:hover',
							'conditions' => $btn_txt_condition
						]);
						$this->add_group_control( KFGCBR::get_type(), [
							'name'           => 'btn_hover_border',
							'fields_options' => [
								'width' => [ 'label' => _x( 'Border Width', 'Border Control', 'onnat-extra' ) ],
								'color' => [ 'label' => _x( 'Border Color', 'Border Control', 'onnat-extra' ) ],
							],
							'selector'       => '{{WRAPPER}} .kfw-ele-widget-icon-box-button:hover',
							'conditions'     => $btn_txt_condition,
						]);
						$this->add_responsive_control( 'btn_hover_border_radius', [
							'label'      => esc_html__('Border Radius', 'onnat-extra' ),
							'type'       => KFCM::DIMENSIONS,
							'size_units' => [ 'px', '%' ],
							'selectors'  => [ '{{WRAPPER}} .kfw-ele-widget-icon-box-button:hover' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};' ],
							'conditions' => $btn_txt_condition,
						]);
						$this->add_group_control( KFGCBS::get_type(), [
							'name'       => 'btn_hover_box_shadow',
							'selector'   => '{{WRAPPER}} .kfw-ele-widget-icon-box-button:hover',
							'conditions' => $btn_txt_condition,
						]);
					$this->end_controls_tab();
				$this->end_controls_tabs();

			/**
			 * Button Icon Style
			 */
				$btn_icon_condition = [
					'relation' => 'and',
					'terms'    => [
						[
							'name'     => 'use_button',
							'operator' => '===',
							'value'    => 'yes'
						],
						[
							'name'     => 'btn_icon[value]',
							'operator' => '!=',
							'value'    => ''
						],
					]
				];
				$btn_icon_animation_condition = [
					'relation' => 'and',
					'terms'    => [
						[
							'name'     => 'use_button',
							'operator' => '===',
							'value'    => 'yes'
						],
						[
							'name'     => 'btn_text',
							'operator' => '!=',
							'value'    => ''
						],
						[
							'name'     => 'btn_icon[value]',
							'operator' => '!=',
							'value'    => ''
						],
					]
				];

				$this->add_control('btn_icon_style_before_hr', [
					'type'       => KFCM::DIVIDER,
					'conditions' => $btn_icon_condition,
				]);
				$this->add_control('btn_icon_style_heading',[
					'label'      => esc_html__( 'BUTTON ICON', 'onnat-extra' ),
					'type'       => KFCM::HEADING,
					'conditions' => $btn_icon_condition,
				]);
				$this->add_control('btn_icon_style_after_hr', [
					'type'       => KFCM::DIVIDER,
					'conditions' => $btn_icon_condition,
				]);
				$this->add_responsive_control( 'btn_icon_padding', [
					'label'      => esc_html__( 'Padding', 'onnat-extra' ),
					'type'       => KFCM::DIMENSIONS,
					'size_units' => [ 'px', 'em', '%' ],
					'selectors'  => [
						'{{WRAPPER}} .kfw-ele-widget-icon-box-button .kfw-elementor-icon-wrap' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
					],
					'conditions' => $btn_icon_condition
				]);
				$this->add_responsive_control( 'btn_icon_margin', [
					'label'      => esc_html__( 'Margin', 'onnat-extra' ),
					'type'       => KFCM::DIMENSIONS,
					'size_units' => [ 'px', 'em', '%' ],
					'separator'  => 'after',
					'selectors'  => [
						'{{WRAPPER}} .kfw-ele-widget-icon-box-button .kfw-elementor-icon-wrap' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
					],
					'conditions' => $btn_icon_condition
				]);
				$this->add_control( 'btn_icon_pos', [
					'label'      => esc_html__( 'Position', 'onnat-extra' ),
					'type'       => KFCM::CHOOSE,
					'options'    => [
						'left'           => [
							'title' => esc_html__('Left', 'onnat-extra'),
							'icon'  => 'eicon-text-align-left',
						],
						'right'          => [
							'title' => esc_html__('Right', 'onnat-extra'),
							'icon'  => 'eicon-text-align-right',
						],
						'animated-left'  => [
							'title' => esc_html__('Animated Left', 'onnat-extra'),
							'icon'  => 'eicon-animation',
						],
						'animated-right' => [
							'title' => esc_html__('Animated Right', 'onnat-extra'),
							'icon'  => 'eicon-animation kfw-ctrl-rotate-180',
						],
						'toggle'         => [
							'title' => esc_html__('Toogle Button Text', 'onnat-extra'),
							'icon'  => 'eicon-share-arrow',
						]
					],
					'default'    => 'left',
					'toggle'     => false,
					'conditions' => $btn_icon_animation_condition
				]);
				$this->add_control( 'alr_btn_icon_animation', [
					'label'      => esc_html__( 'Animation Direction', 'onnat-extra' ),
					'type'       => KFCM::CHOOSE,
					'options'    => [
						'in'  => [
							'title' => esc_html__('Animate In', 'onnat-extra'),
							'icon'  => 'eicon-chevron-left'
						],
						'out' => [
							'title' => esc_html__('Animate Out', 'onnat-extra'),
							'icon'  => 'eicon-chevron-right'
						],
					],
					'default'    => 'in',
					'toggle'     => false,
					'conditions' => [
						'relation' => 'and',
						'terms'    => [
							[
								'name'     => 'use_button',
								'operator' => '===',
								'value'    => 'yes'
							],
							[
								'name'     => 'btn_text',
								'operator' => '!=',
								'value'    => ''
							],
							[
								'name'     => 'btn_icon[value]',
								'operator' => '!=',
								'value'    => ''
							],
							[
								'name'     => 'btn_icon_pos',
								'operator' => 'in',
								'value'    => ['animated-left', 'animated-right' ]
							]
						]
					]
				]);
				$this->add_control( 'toggle_btn_icon_animation', [
					'label'      => esc_html__( 'Animation Direction', 'onnat-extra' ),
					'type'       => KFCM::CHOOSE,
					'options'    => [
						'top'    => [
							'title' => esc_html__('Top', 'onnat-extra'),
							'icon'  => 'eicon-arrow-up',
						],
						'right'  => [
							'title' => esc_html__('Right', 'onnat-extra'),
							'icon'  => 'eicon-arrow-right',
						],
						'bottom' => [
							'title' => esc_html__('Bottom', 'onnat-extra'),
							'icon'  => 'eicon-arrow-down',
						],
						'left'   => [
							'title' => esc_html__('Left', 'onnat-extra'),
							'icon'  => 'eicon-arrow-left',
						],
					],
					'default'    => 'left',
					'toggle'     => false,
					'conditions' => [
						'relation' => 'and',
						'terms'    => [
							[
								'name'     => 'use_button',
								'operator' => '===',
								'value'    => 'yes'
							],
							[
								'name'     => 'btn_text',
								'operator' => '!=',
								'value'    => ''
							],
							[
								'name'     => 'btn_icon[value]',
								'operator' => '!=',
								'value'    => ''
							],
							[
								'name'     => 'btn_icon_pos',
								'operator' => '===',
								'value'    => 'toggle'
							]
						]
					]
				]);
				$this->add_responsive_control( 'btn_icon_size', [
					'label'      => esc_html__('Size', 'onnat-extra' ),
					'type'       => KFCM::SLIDER,
					'size_units' => [ 'px' ],
					'range'      => [
						'px' => [
							'min' => 0,
							'max' => 999,
						],
					],
					'selectors'  => [
						'{{WRAPPER}} .kfw-ele-widget-icon-box-button .kfw-elementor-icon-wrap i'   => 'font-size: {{SIZE}}{{UNIT}};',
						'{{WRAPPER}} .kfw-ele-widget-icon-box-button .kfw-elementor-icon-wrap svg' => 'font-size: {{SIZE}}{{UNIT}};'
					],
					'conditions' => $btn_icon_condition
				]);
				$this->start_controls_tabs( 'btn_icon_customize_tabs' );
					$this->start_controls_tab( 'btn_icon_normal_state', [
						'label'      => esc_html__( 'Normal', 'onnat-extra' ),
						'conditions' => $btn_icon_condition
					]);
						$this->add_control( 'btn_icon_color', [
							'label'      => esc_html__('Icon Color', 'onnat-extra' ),
							'type'       => KFCM::COLOR,
							'selectors'  => [
								'{{WRAPPER}} .kfw-ele-widget-icon-box-button .kfw-elementor-icon-wrap i'   => 'color: {{VALUE}};',
								'{{WRAPPER}} .kfw-ele-widget-icon-box-button .kfw-elementor-icon-wrap svg' => 'fill: {{VALUE}};'
							],
							'conditions' => $btn_icon_condition
						]);
						$this->add_group_control( KFGCBG::get_type(), [
							'name'       => 'btn_icon_bg',
							'label'      => esc_html__('Background', 'onnat-extra' ),
							'types'      => [ 'classic', 'gradient' ],
							'exclude'    => [ 'image' ], # To remove background image option in classic mode
							'selector'   => '{{WRAPPER}} .kfw-ele-widget-icon-box-button .kfw-elementor-icon-wrap',
							'conditions' => $btn_icon_condition
						]);
						$this->add_group_control( KFGCBR::get_type(), [
							'name'           => 'btn_icon_border',
							'fields_options' => [
								'width' => [ 'label' => _x( 'Border Width', 'Border Control', 'onnat-extra' ) ],
								'color' => [ 'label' => _x( 'Border Color', 'Border Control', 'onnat-extra' ) ],
							],
							'selector'       => '{{WRAPPER}} .kfw-ele-widget-icon-box-button .kfw-elementor-icon-wrap',
							'conditions'     => $btn_icon_condition,
						]);
						$this->add_responsive_control( 'btn_icon_border_radius', [
							'label'      => esc_html__('Border Radius', 'onnat-extra' ),
							'type'       => KFCM::DIMENSIONS,
							'size_units' => [ 'px', '%' ],
							'selectors'  => [ '{{WRAPPER}} .kfw-ele-widget-icon-box-button .kfw-elementor-icon-wrap' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};' ],
							'conditions' => $btn_icon_condition,
						]);
						$this->add_group_control( KFGCBS::get_type(), [
							'name'       => 'btn_icon_box_shadow',
							'selector'   => '{{WRAPPER}} .kfw-ele-widget-icon-box-button .kfw-elementor-icon-wrap',
							'conditions' => $btn_icon_condition,
						]);
					$this->end_controls_tab();
					$this->start_controls_tab( 'btn_icon_hover_state', [
						'label'      => esc_html__( 'Hover', 'onnat-extra' ),
						'conditions' => $btn_icon_condition
					]);
						$this->add_control( 'btn_icon_hover_color', [
							'label'      => esc_html__('Color', 'onnat-extra' ),
							'type'       => KFCM::COLOR,
							'selectors'  => [
								'{{WRAPPER}} .kfw-ele-widget-icon-box-button:hover .kfw-elementor-icon-wrap i'   => 'color: {{VALUE}};',
								'{{WRAPPER}} .kfw-ele-widget-icon-box-button:hover .kfw-elementor-icon-wrap svg' => 'fill: {{VALUE}};',
							],
							'conditions' => $btn_icon_condition
						]);
						$this->add_group_control( KFGCBG::get_type(), [
							'name'       => 'btn_icon_hover_bg',
							'label'      => esc_html__('Background', 'onnat-extra' ),
							'types'      => [ 'classic', 'gradient' ],
							'exclude'    => [ 'image' ], # To remove background image option in classic mode
							'selector'   => '{{WRAPPER}} .kfw-ele-widget-icon-box-button:hover .kfw-elementor-icon-wrap',
							'conditions' => $btn_icon_condition
						]);
						$this->add_group_control( KFGCBR::get_type(), [
							'name'           => 'btn_icon_hover_border',
							'fields_options' => [
								'width' => [ 'label' => _x( 'Border Width', 'Border Control', 'onnat-extra' ) ],
								'color' => [ 'label' => _x( 'Border Color', 'Border Control', 'onnat-extra' ) ],
							],
							'selector'       => '{{WRAPPER}} .kfw-ele-widget-icon-box-button:hover .kfw-elementor-icon-wrap',
							'conditions'     => $btn_icon_condition,
						]);
						$this->add_responsive_control( 'btn_icon_hover_border_radius', [
							'label'      => esc_html__('Border Radius', 'onnat-extra' ),
							'type'       => KFCM::DIMENSIONS,
							'size_units' => [ 'px', '%' ],
							'selectors'  => [ '{{WRAPPER}} .kfw-ele-widget-icon-box-button:hover .kfw-elementor-icon-wrap' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};' ],
							'conditions' => $btn_icon_condition,
						]);
						$this->add_group_control( KFGCBS::get_type(), [
							'name'       => 'btn_icon_hover_box_shadow',
							'selector'   => '{{WRAPPER}} .kfw-ele-widget-icon-box-button:hover .kfw-elementor-icon-wrap',
							'conditions' => $btn_icon_condition,
						]);
					$this->end_controls_tab();
				$this->end_controls_tabs();
		$this->end_controls_section();
	}

	public function _title( $settings = [] ) {
		if( !empty( $settings['title'] ) ) {

			$title = $settings['title'];
			$link  = array_filter( $settings['title_link'] );

			if( isset( $link['url'] ) ) {

				$this->add_render_attribute( 'cta', 'href', esc_url( $link['url'] ) );
				if ( !empty( $link['is_external'] ) ) {
					$this->add_render_attribute( 'cta', 'target', '_blank' );
				}


				if ( !empty( $link['nofollow'] ) ) {
					$this->add_render_attribute( 'cta', 'rel', 'nofollow' );
				}

				$title = '';
				$title .= '<a '. $this->get_render_attribute_string( 'cta' ) .'>';
					$title .= esc_html( $settings['title'] );
				$title .= '</a>';
			}

			$this->add_render_attribute( 'title', [
				'class' => array_filter([
					'kfw-ele-widget-icon-box-title'
				])
			]);

			echo '<div '.$this->get_render_attribute_string( 'title' ).'>' . $title . '</div>';
		}
	}

	public function _sub_title( $settings = [] ) {
		if( !empty( $settings['sub_title'] ) ) {

			$this->add_render_attribute( 'sub-title', [
				'class' => array_filter([
					'kfw-ele-widget-icon-box-sub-title'
				])
			]);

			echo '<div '.$this->get_render_attribute_string( 'sub-title' ).'>' . $settings['sub_title'] . '</div>';
		}
	}

	public function _description( $settings = [] ) {
		if( !empty( $settings['description'] ) ) {

			$this->add_render_attribute( 'description', [
				'class' => array_filter([
					'kfw-ele-widget-icon-box-description'
				])
			]);

			echo '<div '.$this->get_render_attribute_string( 'description' ).'>' . esc_html( $settings['description'] ) . '</div>';
		}
	}

	public function _button( $settings = [] ) {
		if( $settings['use_button'] === 'yes' ) {
			$link = array_filter( $settings['btn_link'] );

			$this->add_render_attribute( 'cta-wrapper', [
				'class' => array_filter([
					'kfw-ele-widget-icon-box-button-wrap',
					!empty( trim( $settings['btn_text'] ) ) ? 'kfw-ele-widget-icon-box-button-has-txt' : ''
				])
			]);

			echo '<div '.$this->get_render_attribute_string( 'cta-wrapper' ).'>';

				$this->add_render_attribute( 'cta', [
					'class' => array_filter([
						'kfw-ele-widget-icon-box-button',
						!empty( $settings['btn_animation_hover'] ) ? 'elementor-animation-'.esc_attr( $settings['btn_animation_hover'] ): '',
					])
				]);

				/**
				 * Button Icon related classes
				 */
				if( $settings['btn_icon']['value'] !== '' ) {
					$this->add_render_attribute( 'cta', [
						'class' => array_filter([
							!empty( trim( $settings['btn_text'] ) ) ? 'kfw-ele-widget-icon-box-button-icon-position-'.$settings['btn_icon_pos'] : '',
							( !empty( trim( $settings['btn_text'] ) ) ) && ( $settings['btn_icon_pos'] === 'animated-left' || $settings['btn_icon_pos'] === 'animated-right' ) ? 'kfw-ele-widget-icon-box-button-animated-icon-position-' . $settings['alr_btn_icon_animation'] : '',
							( !empty( trim( $settings['btn_text'] ) ) ) && ( $settings['btn_icon_pos'] === 'toggle' ) ? 'kfw-ele-widget-icon-box-button-toggle-icon-position-' . $settings['toggle_btn_icon_animation'] : '',
						])
					]);
				}

				if ( ! empty( $link['url'] ) ) {
					$this->add_render_attribute( 'cta', 'href', esc_url( $link['url'] ) );

					if ( ! empty( $link['is_external'] ) ) {
						$this->add_render_attribute( 'cta', 'target', '_blank' );
					}

					if ( ! empty( $link['nofollow'] ) ) {
						$this->add_render_attribute( 'cta', 'rel', 'nofollow' );
					}
				}

				echo '<a '.$this->get_render_attribute_string( 'cta' ).'>';

					if( $settings['btn_icon_pos'] === 'left' || $settings['btn_icon_pos'] === 'animated-left' ) {
						$this->_render_icon( $settings['btn_icon'] );
					}

					if( !empty( trim( $settings['btn_text'] ) ) ) {
						printf( '<span class="kfw-elementor-button-text">%s</span>', esc_html( $settings['btn_text'] ) );
					}

					if( $settings['btn_icon_pos'] === 'right'
						|| $settings['btn_icon_pos'] === 'animated-right'
						|| $settings['btn_icon_pos'] === 'toggle'
						|| $settings['btn_icon_pos'] === 'toggle-alt'
					) {
						$this->_render_icon( $settings['btn_icon'] );
					}
				echo '</a>';

			echo '</div>';
		}
	}

	public function _render_icon( $icon , $echo = true ) {
		$icon = array_filter( $icon );

		if( !empty( $icon ) ) {
			echo '<span class="kfw-elementor-icon-wrap">';
				KFIM::render_icon( $icon, [ 'aria-hidden' => 'true' ] );
			echo '</span>';
		}
	}

	public function _render_img( $settings = [] ) {
		if( !empty( $settings['image']['url'] ) ) {
			$image_src = $settings['image']['url'];

			$image_title = $this->_get_image_title( $settings['image'] );
			$image_title = empty( $image_title ) ? $this->get_title() : $image_title;

			$image_alt = $this->_get_image_alt( $settings['image'] );
			$image_alt = empty( $image_alt ) ? $image_title : $image_title;

			$this->add_render_attribute( 'image_attr', [
				'alt'   => esc_attr( $image_alt ),
				'title' => esc_attr( $image_title ),
				'src'   => esc_url( $image_src ),
			] );

			echo '<span class="kfw-elementor-icon-wrap">';
				echo '<img '.$this->get_render_attribute_string( 'image_attr' ).'/>';
			echo '</span>';
		}
	}

	public function _get_image_title( $attachment ) {
		if ( empty( $attachment['id'] ) ) {
			return '';
		}

		return get_the_title( $attachment['id'] );
	}

	public function _get_image_alt( $instance ) {
		if ( empty( $instance['id'] ) ) {
			// For `Insert From URL` images.
			return isset( $instance['alt'] ) ? trim( strip_tags( $instance['alt'] ) ) : '';
		}

		$attachment_id = $instance['id'];
		if ( ! $attachment_id ) {
			return '';
		}

		$attachment = get_post( $attachment_id );
		if ( ! $attachment ) {
			return '';
		}

		$alt = get_post_meta( $attachment_id, '_wp_attachment_image_alt', true );
		if ( ! $alt ) {
			$alt = $attachment->post_excerpt;
			if ( ! $alt ) {
				$alt = $attachment->post_title;
			}
		}
		return trim( strip_tags( $alt ) );
	}

	/**
	 * Render widget output on the frontend.
	 */
    protected function render() {
		$render = kf_extra_plugin_elementor()->get_template("kfw-icon-box");
		include $render;
    }
}