<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

use \Elementor\Widget_Base as WB;
use \Elementor\Controls_Manager as KFCM;
use \Elementor\Icons_Manager as KFIM;

use \Elementor\Group_Control_Background as KFGCBG;
use \Elementor\Group_Control_Typography as KFGCT;
use \Elementor\Group_Control_Border as KFGCBR;
use \Elementor\Group_Control_Box_Shadow as KFGCBS;

class KFW_Mailchimp_Plus extends WB {

	/**
	 * Get element name.
	 *
	 * Retrieve the element name.
	 *
	 * @return string The name.
	 */
	public function get_name() {
        return 'kfw-mailchimp-plus';
    }

	/**
	 * Get element title.
	 *
	 * Retrieve the element title.
	 *
	 * @return string Element title.
	 */
	public function get_title() {
		return esc_html__('Mailchimp Plus', 'onnat-extra');
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the widget categories.
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ '_kfw_core_widgets' ];
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve the widget icon.
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'kfw-badge eicon-mailchimp';
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the widget keywords.
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'kfw', 'kin', 'mailchimp', 'plus', 'subscribe', 'form' ];
	}

	/**
	 * Get widget help url.
	 *
	 * Retrieve the widget help url.
	 *
	 * @return string Widget help url.
	 */
	public function get_custom_help_url() {
        return KFE_CONST_THEME_DOC . 'elementor-custom-widgets/mailchimp-plus';
    }

	/**
	 * Get HTML wrapper class.
     *
	 * Retrieve the widget container class.
     *
     * @return string Widget container css classes.
	 */
	public function get_html_wrapper_class() {
        $name = $this->get_name();

		return sprintf( 'kfw-ele-%s elementor-widget-%s', $name, $name );
	}

	/**
	 * Widget base constructor.
	 *
	 * Initializing the widget base class.
	 */
	public function __construct( $data = [], $args = null ) {
        parent::__construct( $data, $args );

		if ( ! wp_script_is( 'kfw-mailchimp', 'registered' ) ) {
			wp_register_script( 'kfw-mailchimp',
				KFE_CONST_URL . 'libraries/elementor/assets/js/mailchimp' . KFE_CONST_DEBUG_SUFFIX . '.js',
				[ 'jquery', 'elementor-frontend' ],
				KFE_CONST_VERSION,
				true
			);
		}

		wp_register_style( 'kfw-mailchimp-plus',
			KFE_CONST_URL . 'libraries/elementor/assets/css/mailchimp-plus' . KFE_CONST_DEBUG_SUFFIX . '.css',
			[],
			KFE_CONST_VERSION
		);
    }

	/**
	 * Get style dependencies.
	 *
	 * Retrieve the list of style dependencies the element requires.
	 *
	 * @return array Element styles dependencies.
	 */
	public function get_style_depends() {
		return [ 'kfw-mailchimp-plus' ];
	}

	/**
	 * Get script dependencies.
	 *
	 * Retrieve the list of script dependencies the element requires.
	 *
	 * @return array Element scripts dependencies.
	 */
	public function get_script_depends() {
		return [ 'kfw-mailchimp' ];
	}

	/**
	 * Register controls.
	 *
	 * Used to add new controls to any element type.
	 */
	protected function register_controls() {
		$this->_register_app_section();
		$this->_register_email_field_section();
		$this->_register_name_field_section();
		$this->_register_button_section();

		$this->_register_input_field_style_section();
		$this->_register_button_style_section();
	}

    public function _register_app_section() {
		$this->start_controls_section( 'section_app', [
			'label' => esc_html__( 'App Settings', 'onnat-extra' )
		]);
            $this->add_control('api_type',[
                'label'   => esc_html__( 'Source', 'onnat-extra' ),
                'type'    => KFCM::CHOOSE,
                'options' => [
                    'settings' => [
                        'title' => esc_html__('Theme Settings', 'onnat-extra'),
                        'icon'  => 'eicon-cog',
                    ],
                    'widget'   => [
                        'title' => esc_html__('Widget OPtions', 'onnat-extra'),
                        'icon'  => 'eicon-apps',
                    ],
                ],
                'default' => 'settings',
                'toggle'  => false
            ]);
			$this->add_control('api_type_settings_des', [
				'type'            => KFCM::RAW_HTML,
				'content_classes' => 'kfw-elementor-editor-notice-ctrl',
				'raw'             => sprintf( esc_html__('Configure your MailChimp API Key and Audience ID conveniently through the settings panel in the Theme.','onnat-extra') ),
				'condition'       => [
                    'api_type' => 'settings'
                ]
            ]);
			$this->add_control('api_key', [
				'label'       => esc_html__( 'MailChimp API Key', 'onnat-extra' ),
				'label_block' => true,
				'type'        => KFCM::TEXT,
				'description' => sprintf(
                    esc_html__('To obtain MailChimp API Key, you need to follow the %s for detailed instructions.', 'onnat-extra' ),
                    sprintf( '<a href="%s" target="_blank">%s</a>', KFE_CONST_THEME_DOC . 'how-to/get-mailchimp-api-key', esc_html__('documentation', 'onnat-extra') )
                ),
				'condition'   => [
                    'api_type' => 'widget'
                ]
            ]);
			$this->add_control('audience_id', [
				'label'       => esc_html__( 'Audience ID', 'onnat-extra' ),
				'label_block' => true,
				'type'        => KFCM::TEXT,
				'description' => sprintf(
                    esc_html__('Follow the guidelines in the %s for detailed instructions on obtaining the MailChimp Audience ID.', 'onnat-extra' ),
                    sprintf( '<a href="%s" target="_blank">%s</a>', KFE_CONST_THEME_DOC . 'how-to/get-mailchimp-audience-id', esc_html__('documentation', 'onnat-extra') )
                ),
				'condition'   => [
                    'api_type' => 'widget'
                ]
            ]);
            $this->add_control( 'use_redirect', [
                'label'        => esc_html__( 'Use Redirect?', 'onnat-extra' ),
                'type'         => KFCM::SWITCHER,
                'return_value' => 'yes',
                'default'      => 'true'
            ]);
			$this->add_control( 'redirect_url', [
				'label'       => esc_html__( 'Link', 'onnat-extra' ),
				'type'        => KFCM::URL,
				'placeholder' => 'https://your-link.com',
				'options'     => false,
				'default'     => [ 'url' => '/' ],
				'condition'   => [
                    'use_redirect' => 'yes'
                ]
            ]);
			$this->add_control( 'message', [
                'label'       => esc_html__( 'Submission Message?', 'onnat-extra' ),
                'label_block' => true,
                'type'        => KFCM::TEXT,
                'default'     => esc_html__( 'Submission successful! Thank you for your input.', 'onnat-extra' ),
                'condition'   => [
                    'use_redirect!' => 'yes'
                ]
            ]);
			$this->add_control('layout_preset_before_hr', [
				'type' => KFCM::DIVIDER,
			]);
			$this->add_control('layout_preset_style_heading',[
				'label' => esc_html__( 'LAYOUT', 'onnat-extra' ),
				'type'  => KFCM::HEADING,
			]);
			$this->add_control( 'layout', [
				'label'     => esc_html__( 'Layout Preset?', 'onnat-extra' ),
				'type'      => KFCM::SELECT,
				'default'   => 'preset-style-2',
				'options'   => [
                    'preset-style-1' => esc_html__( 'Style 1', 'onnat-extra' ),
                    'preset-style-2' => esc_html__( 'Style 2', 'onnat-extra' ),
				],
			]);
		$this->end_controls_section();
    }

    public function _register_email_field_section() {
		$this->start_controls_section( 'section_email', [
			'label' => esc_html__( 'Email Field', 'onnat-extra' )
		]);
            $this->add_control( 'email_field_placeholder', [
                'label'       => esc_html__( 'Email Field Placeholder', 'onnat-extra' ),
                'type'        => KFCM::TEXT,
                'dynamic'     => [ 'active' => true ],
                'label_block' => true,
                'default'     => esc_html__( 'Email *', 'onnat-extra' ),
                'placeholder' => esc_html__( 'Email *', 'onnat-extra' ),
            ]);
			$this->add_control( 'email_field_icon', [
				'label'       => esc_html__( 'Icon', 'onnat-extra' ),
				'type'        => KFCM::ICONS,
				'skin'        => 'inline',
				'label_block' => false,
				'default'     => [
					'value'   => 'fa fa-check',
					'library' => 'fa-solid'
				]
			]);
		$this->end_controls_section();
    }

    public function _register_name_field_section() {
		$this->start_controls_section( 'section_name', [
			'label' => esc_html__( 'Name Field', 'onnat-extra' )
		]);
            $this->add_control( 'name_field_placeholder', [
                'label'       => esc_html__( 'Name Field Placeholder', 'onnat-extra' ),
                'type'        => KFCM::TEXT,
                'dynamic'     => [ 'active' => true ],
                'label_block' => true,
                'default'     => esc_html__( 'Name *', 'onnat-extra' ),
                'placeholder' => esc_html__( 'Name *', 'onnat-extra' ),
            ]);
			$this->add_control( 'name_field_icon', [
				'label'       => esc_html__( 'Icon', 'onnat-extra' ),
				'type'        => KFCM::ICONS,
				'skin'        => 'inline',
				'label_block' => false,
				'default'     => [
					'value'   => 'fa fa-user',
					'library' => 'fa-solid'
				]
			]);
		$this->end_controls_section();
    }

    public function _register_button_section() {
		$this->start_controls_section( 'section_button', [
			'label' => esc_html__( 'Button', 'onnat-extra' )
		]);
            $this->add_control( 'btn_text', [
                'label'     => esc_html__( 'Text', 'onnat-extra' ),
                'type'      => KFCM::TEXT,
                'default'   => esc_html__( 'Subscribe', 'onnat-extra' ),
            ]);
			$this->add_control( 'btn_icon', [
				'label'       => esc_html__( 'Icon', 'onnat-extra' ),
				'type'        => KFCM::ICONS,
				'skin'        => 'inline',
				'label_block' => false,
				'default'     => [
					'value'   => 'fa fa-check',
					'library' => 'fa-solid'
				]
			]);
		$this->end_controls_section();
    }

	public function _register_input_field_style_section() {
		$this->start_controls_section( 'section_input_field', [
            'label' => esc_html__( 'Input Field', 'onnat-extra' ),
            'tab'   => KFCM::TAB_STYLE,
		]);
			$this->add_responsive_control( 'input_field_padding', [
				'label'      => esc_html__( 'Padding', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-mailchimp-plus-input-field-wrap .kfw-ele-widget-mailchimp-plus-input-field' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				],
			]);
			$this->add_responsive_control( 'input_field_margin', [
				'label'      => esc_html__( 'Margin', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-mailchimp-plus-input-field-wrap .kfw-ele-widget-mailchimp-plus-input-field' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				],
			]);
			$this->add_responsive_control( 'input_field_border_radius', [
				'label'      => esc_html__('Border Radius', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', '%' ],
				'selectors'  => [ '{{WRAPPER}} .kfw-ele-widget-mailchimp-plus-input-field-wrap .kfw-ele-widget-mailchimp-plus-input-field' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};' ],
			]);
			$this->add_control('input_field_typo_before_hr', [
				'type' => KFCM::DIVIDER,
			]);
			$this->add_control('input_field_typo_heading',[
				'label' => esc_html__( 'FIELD TYPO SETTINGS', 'onnat-extra' ),
				'type'  => KFCM::HEADING,
			]);
			$this->add_group_control( KFGCT::get_type(), [
				'name'        => 'input_field_typo',
				'description' => esc_html__( 'Typography', 'onnat-extra' ),
				'selector'    => '{{WRAPPER}} .kfw-ele-widget-mailchimp-plus-input-field-wrap .kfw-ele-widget-mailchimp-plus-input-field',
			]);
			$this->start_controls_tabs( 'input_field_customize_tabs' );
				$this->start_controls_tab( 'input_field_normal_state', [
					'label' => esc_html__( 'Normal', 'onnat-extra' ),
				]);
					$this->add_control( 'input_field_color', [
						'label'     => esc_html__('Text Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'selectors' => ['{{WRAPPER}} .kfw-ele-widget-mailchimp-plus-input-field-wrap .kfw-ele-widget-mailchimp-plus-input-field' => 'color: {{VALUE}};' ],
					]);
					$this->add_control( 'input_field_placeholder_color', [
						'label'     => esc_html__('Placeholder Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'selectors' => ['{{WRAPPER}} .kfw-ele-widget-mailchimp-plus-input-field-wrap .kfw-ele-widget-mailchimp-plus-input-field::placeholder' => 'color: {{VALUE}};' ],
					]);
					$this->add_group_control( KFGCBG::get_type(), [
						'name'     => 'input_field_bg',
						'label'    => esc_html__('Background', 'onnat-extra' ),
						'types'    => [ 'classic', 'gradient' ],
						'exclude'  => [ 'image' ], # To remove background image option in classic mode
						'selector' => '{{WRAPPER}} .kfw-ele-widget-mailchimp-plus-input-field-wrap .kfw-ele-widget-mailchimp-plus-input-field',
					]);
					$this->add_group_control( KFGCBR::get_type(), [
						'name'           => 'input_field_border',
						'fields_options' => [
							'width' => [ 'label' => _x( 'Border Width', 'Border Control', 'onnat-extra' ) ],
							'color' => [ 'label' => _x( 'Border Color', 'Border Control', 'onnat-extra' ) ],
						],
						'selector'       => '{{WRAPPER}} .kfw-ele-widget-mailchimp-plus-input-field-wrap .kfw-ele-widget-mailchimp-plus-input-field',
					]);
					$this->add_group_control( KFGCBS::get_type(), [
						'name'       => 'input_field_box_shadow',
						'selector'   => '{{WRAPPER}} .kfw-ele-widget-mailchimp-plus-input-field-wrap .kfw-ele-widget-mailchimp-plus-input-field',
					]);
				$this->end_controls_tab();

				$this->start_controls_tab( 'input_field_focus_state', [
					'label' => esc_html__( 'Focus', 'onnat-extra' ),
				]);
					$this->add_control( 'input_field_focus_color', [
						'label'     => esc_html__('Text Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'selectors' => ['{{WRAPPER}} .kfw-ele-widget-mailchimp-plus-input-field-wrap .kfw-ele-widget-mailchimp-plus-input-field:focus' => 'color: {{VALUE}};' ],
					]);
					$this->add_control( 'input_field_focus_placeholder_color', [
						'label'     => esc_html__('Placeholder Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'selectors' => ['{{WRAPPER}} .kfw-ele-widget-mailchimp-plus-input-field-wrap .kfw-ele-widget-mailchimp-plus-input-field:focus::placeholder' => 'color: {{VALUE}};' ],
					]);
					$this->add_group_control( KFGCBG::get_type(), [
						'name'     => 'input_field_focus_bg',
						'label'    => esc_html__('Background', 'onnat-extra' ),
						'types'    => [ 'classic', 'gradient' ],
						'exclude'  => [ 'image' ], # To remove background image option in classic mode
						'selector' => '{{WRAPPER}} .kfw-ele-widget-mailchimp-plus-input-field-wrap .kfw-ele-widget-mailchimp-plus-input-field:focus',
					]);
					$this->add_group_control( KFGCBR::get_type(), [
						'name'           => 'input_field_focus_border',
						'fields_options' => [
							'width' => [ 'label' => _x( 'Border Width', 'Border Control', 'onnat-extra' ) ],
							'color' => [ 'label' => _x( 'Border Color', 'Border Control', 'onnat-extra' ) ],
						],
						'selector'       => '{{WRAPPER}} .kfw-ele-widget-mailchimp-plus-input-field-wrap .kfw-ele-widget-mailchimp-plus-input-field:focus',
					]);
					$this->add_group_control( KFGCBS::get_type(), [
						'name'       => 'input_field_focus_box_shadow',
						'selector'   => '{{WRAPPER}} .kfw-ele-widget-mailchimp-plus-input-field-wrap .kfw-ele-widget-mailchimp-plus-input-field:focus',
					]);
				$this->end_controls_tab();

				$this->start_controls_tab( 'input_field_error_state', [
					'label' => esc_html__( 'Error', 'onnat-extra' ),
				]);
					$this->add_control( 'input_field_error_color', [
						'label'     => esc_html__('Text Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'selectors' => ['{{WRAPPER}} .kfw-ele-widget-mailchimp-plus-input-field-wrap .kfw-ele-widget-mailchimp-plus-input-field.kfw-invalid' => 'color: {{VALUE}};' ],
					]);
					$this->add_control( 'input_field_error_placeholder_color', [
						'label'     => esc_html__('Placeholder Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'selectors' => ['{{WRAPPER}} .kfw-ele-widget-mailchimp-plus-input-field-wrap .kfw-ele-widget-mailchimp-plus-input-field.kfw-invalid::placeholder' => 'color: {{VALUE}};' ],
					]);
					$this->add_group_control( KFGCBG::get_type(), [
						'name'     => 'input_field_error_bg',
						'label'    => esc_html__('Background', 'onnat-extra' ),
						'types'    => [ 'classic', 'gradient' ],
						'exclude'  => [ 'image' ], # To remove background image option in classic mode
						'selector' => '{{WRAPPER}} .kfw-ele-widget-mailchimp-plus-input-field-wrap .kfw-ele-widget-mailchimp-plus-input-field.kfw-invalid',
					]);
					$this->add_group_control( KFGCBR::get_type(), [
						'name'           => 'input_field_error_border',
						'fields_options' => [
							'width' => [ 'label' => _x( 'Border Width', 'Border Control', 'onnat-extra' ) ],
							'color' => [ 'label' => _x( 'Border Color', 'Border Control', 'onnat-extra' ) ],
						],
						'selector'       => '{{WRAPPER}} .kfw-ele-widget-mailchimp-plus-input-field-wrap .kfw-ele-widget-mailchimp-plus-input-field.kfw-invalid',
					]);
					$this->add_group_control( KFGCBS::get_type(), [
						'name'       => 'input_field_error_box_shadow',
						'selector'   => '{{WRAPPER}} .kfw-ele-widget-mailchimp-plus-input-field-wrap .kfw-ele-widget-mailchimp-plus-input-field.kfw-invalid',
					]);
				$this->end_controls_tab();
			$this->end_controls_tabs();
			$this->add_control('input_field_icon_before_hr', [
				'type' => KFCM::DIVIDER,
			]);
			$this->add_control('input_field_icon_heading',[
				'label' => esc_html__( 'ICON SETTINGS', 'onnat-extra' ),
				'type'  => KFCM::HEADING,
			]);
			$this->add_responsive_control( 'input_field_icon_margin', [
				'label'      => esc_html__( 'Margin', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [ '{{WRAPPER}} .kfw-ele-widget-mailchimp-plus-input-field-wrap .kfw-elementor-icon-wrap' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'],
			]);
			$this->add_responsive_control( 'input_field_icon_size', [
				'label'      => esc_html__('Size', 'onnat-extra' ),
				'type'       => KFCM::SLIDER,
				'size_units' => [ 'px' ],
				'range'      => [
					'px' => [
						'min' => 0,
						'max' => 999,
					],
				],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-mailchimp-plus-input-field-wrap .kfw-elementor-icon-wrap' => 'font-size: {{SIZE}}{{UNIT}};'
				],
			]);
			$this->add_control( 'input_field_icon_color_settings', [
				'label'        => esc_html__('Colors', 'onnat-extra'),
				'type'         => KFCM::POPOVER_TOGGLE,
				'return_value' => 'yes',
			]);
			$this->start_popover();
				$this->add_control( 'input_field_icon_color', [
					'label'     => esc_html__('Color', 'onnat-extra' ),
					'type'      => KFCM::COLOR,
					'selectors' => ['{{WRAPPER}} .kfw-ele-widget-mailchimp-plus-input-field-wrap .kfw-elementor-icon-wrap' => 'color: {{VALUE}};' ],
				]);
				$this->add_control( 'email_field_icon_focus_color', [
					'label'     => esc_html__('Focus Color', 'onnat-extra' ),
					'type'      => KFCM::COLOR,
					'selectors' => ['{{WRAPPER}} .kfw-ele-widget-mailchimp-plus-input-field-wrap.kfw-field-focus .kfw-elementor-icon-wrap' => 'color: {{VALUE}};' ],
				]);
				$this->add_control( 'email_field_icon_error_color', [
					'label'     => esc_html__('Error Color', 'onnat-extra' ),
					'type'      => KFCM::COLOR,
					'selectors' => ['{{WRAPPER}} .kfw-ele-widget-mailchimp-plus-input-field-wrap.kfw-invalid .kfw-elementor-icon-wrap' => 'color: {{VALUE}};' ],
				]);
			$this->end_popover();
		$this->end_controls_section();
	}

	public function _register_button_style_section() {
		$condition = [
			'relation' => 'or',
			'terms'    => [
				[
					'name'     => 'btn_text',
					'operator' => '!=',
					'value'    => ''
				],
                [
                    'name'     => 'btn_icon[value]',
                    'operator' => '!=',
                    'value'    => ''
                ],
			]
		];

		$this->start_controls_section( 'section_button_style', [
			'label'      => esc_html__( 'Button', 'onnat-extra' ),
			'tab'        => KFCM::TAB_STYLE,
			'conditions' => $condition,
		]);
            $this->add_responsive_control( 'btn_padding', [
                'label'      => esc_html__( 'Padding', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-ele-widget-mailchimp-button' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ],
                'conditions' => $condition
            ]);
			$this->add_responsive_control( 'btn_margin', [
				'label'      => esc_html__( 'Margin', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-mailchimp-button-wrap' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				],
				'conditions' => $condition
			]);
			$this->add_responsive_control( 'btn_border_radius', [
				'label'      => esc_html__('Border Radius', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'separator'  => 'after',
				'size_units' => [ 'px', '%' ],
				'selectors'  => [ '{{WRAPPER}} .kfw-ele-widget-mailchimp-button' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};' ],
				'condition'  => [  'btn_text!' => '' ],
			]);
			$this->add_control( 'btn_animation_hover', [
				'label'      => esc_html__( 'Hover Animation', 'onnat-extra' ),
				'type'       => KFCM::HOVER_ANIMATION,
				'conditions' => $condition,
			]);

			/**
			 * Button Text Style
			 */
                $this->add_control('btn_txt_style_before_hr', [
                    'type'      => KFCM::DIVIDER,
                    'condition' => [
                        'btn_text!' => ''
                    ]
                ]);
				$this->add_control('btn_txt_style_heading',[
					'label'     => esc_html__( 'BUTTON TEXT', 'onnat-extra' ),
					'type'      => KFCM::HEADING,
					'condition' => [  'btn_text!' => '' ]
				]);
				$this->add_control('btn_txt_style_after_hr', [
					'type'      => KFCM::DIVIDER,
					'condition' => [  'btn_text!' => '' ]
				]);
				$this->add_group_control( KFGCT::get_type(), [
					'name'        => 'btn_typo',
					'description' => esc_html__( 'Typography', 'onnat-extra' ),
					'selector'    => '{{WRAPPER}} .kfw-ele-widget-mailchimp-button',
					'condition'   => [  'btn_text!' => '' ]
				]);
				$this->start_controls_tabs( 'btn_customize_tabs' );
                    $this->start_controls_tab( 'btn_normal_state', [
                        'label'     => esc_html__( 'Normal', 'onnat-extra' ),
                        'condition' => [  'btn_text!' => '' ]
                    ]);
                        $this->add_control( 'btn_color', [
                            'label'     => esc_html__('Text Color', 'onnat-extra' ),
                            'type'      => KFCM::COLOR,
                            'selectors' => ['{{WRAPPER}} .kfw-ele-widget-mailchimp-button' => 'color: {{VALUE}};' ],
                            'condition' => [  'btn_text!' => '' ]
                        ]);
                        $this->add_group_control( KFGCBG::get_type(), [
                            'name'      => 'btn_bg',
                            'label'     => esc_html__('Background', 'onnat-extra' ),
                            'types'     => [ 'classic', 'gradient' ],
                            'exclude'   => [ 'image' ],                                    # To remove background image option in classic mode
                            'selector'  => '{{WRAPPER}} .kfw-ele-widget-mailchimp-button',
                            'condition' => [  'btn_text!' => '' ]
                        ]);
                        $this->add_group_control( KFGCBR::get_type(), [
                            'name'           => 'btn_border',
                            'condition'      => [  'btn_text!' => '' ],
                            'fields_options' => [
                                'width' => [ 'label' => _x( 'Border Width', 'Border Control', 'onnat-extra' ) ],
                                'color' => [ 'label' => _x( 'Border Color', 'Border Control', 'onnat-extra' ) ],
                            ],
                            'selector'       => '{{WRAPPER}} .kfw-ele-widget-mailchimp-button',
                            'condition'      => [  'btn_text!' => '' ],
                        ]);
                        $this->add_group_control( KFGCBS::get_type(), [
                            'name'       => 'btn_box_shadow',
                            'selector'   => '{{WRAPPER}} .kfw-ele-widget-mailchimp-button',
                            'condition'  => [  'btn_text!' => '' ],
                        ]);
					$this->end_controls_tab();
					$this->start_controls_tab( 'btn_hover_state', [
                        'label'     => esc_html__( 'Hover', 'onnat-extra' ),
                        'condition' => [  'btn_text!' => '' ]
					]);
                        $this->add_control( 'btn_hover_color', [
                            'label'      => esc_html__('Text Color', 'onnat-extra' ),
                            'type'       => KFCM::COLOR,
                            'selectors'  => ['{{WRAPPER}} .kfw-ele-widget-mailchimp-button:hover' => 'color: {{VALUE}};' ],
                            'condition' => [  'btn_text!' => '' ]
                        ]);
						$this->add_group_control( KFGCBG::get_type(), [
                            'name'      => 'btn_hover_bg',
                            'label'     => esc_html__('Background', 'onnat-extra' ),
                            'types'     => [ 'classic', 'gradient' ],
                            'exclude'   => [ 'image' ],                                         # To remove background image option in classic mode
                            'selector'  => '{{WRAPPER}} .kfw-ele-widget-mailchimp-button:hover',
                            'condition' => [  'btn_text!' => '' ]
						]);
						$this->add_group_control( KFGCBR::get_type(), [
							'name'           => 'btn_hover_border',
							'fields_options' => [
								'width' => [ 'label' => _x( 'Border Width', 'Border Control', 'onnat-extra' ) ],
								'color' => [ 'label' => _x( 'Border Color', 'Border Control', 'onnat-extra' ) ],
							],
							'selector'       => '{{WRAPPER}} .kfw-ele-widget-mailchimp-button:hover',
                            'condition'      => [  'btn_text!' => '' ],
						]);
						$this->add_group_control( KFGCBS::get_type(), [
							'name'       => 'btn_hover_box_shadow',
							'selector'   => '{{WRAPPER}} .kfw-ele-widget-mailchimp-button:hover',
                            'condition'  => [  'btn_text!' => '' ],
						]);
					$this->end_controls_tab();
                $this->end_controls_tabs();

			/**
			 * Button Icon Style
			 */
                $btn_icon_condition = [
                    'btn_icon[value]!' => ''
                ];
				$btn_icon_animation_condition = [
					'relation' => 'and',
					'terms'    => [
						[
							'name'     => 'btn_text',
							'operator' => '!=',
							'value'    => ''
						],
						[
							'name'     => 'btn_icon[value]',
							'operator' => '!=',
							'value'    => ''
						],
					]
                ];

				$this->add_control('btn_icon_style_before_hr', [
					'type'      => KFCM::DIVIDER,
					'condition' => $btn_icon_condition,
				]);
				$this->add_control('btn_icon_style_heading',[
					'label'     => esc_html__( 'BUTTON ICON', 'onnat-extra' ),
					'type'      => KFCM::HEADING,
					'condition' => $btn_icon_condition,
				]);
				$this->add_control('btn_icon_style_after_hr', [
					'type'      => KFCM::DIVIDER,
					'condition' => $btn_icon_condition,
				]);
				$this->add_responsive_control( 'btn_icon_padding', [
					'label'      => esc_html__( 'Padding', 'onnat-extra' ),
					'type'       => KFCM::DIMENSIONS,
					'size_units' => [ 'px', 'em', '%' ],
					'selectors'  => [
                        '{{WRAPPER}} .kfw-ele-widget-mailchimp-button .kfw-elementor-icon-wrap' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                    ],
					'condition'  => $btn_icon_condition
				]);
				$this->add_responsive_control( 'btn_icon_margin', [
					'label'      => esc_html__( 'Margin', 'onnat-extra' ),
					'type'       => KFCM::DIMENSIONS,
					'size_units' => [ 'px', 'em', '%' ],
					'separator'  => 'after',
					'selectors'  => [ '{{WRAPPER}} .kfw-ele-widget-mailchimp-button .kfw-elementor-icon-wrap' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'],
					'condition'  => $btn_icon_condition
				]);
				$this->add_control( 'btn_icon_pos', [
					'label'      => esc_html__( 'Position', 'onnat-extra' ),
					'type'       => KFCM::CHOOSE,
					'options'    => [
						'left'           => [
							'title' => esc_html__('Left', 'onnat-extra'),
							'icon'  => 'eicon-text-align-left',
						],
						'right'          => [
							'title' => esc_html__('Right', 'onnat-extra'),
							'icon'  => 'eicon-text-align-right',
						],
						'animated-left'  => [
							'title' => esc_html__('Animated Left', 'onnat-extra'),
							'icon'  => 'eicon-animation',
						],
						'animated-right' => [
							'title' => esc_html__('Animated Right', 'onnat-extra'),
							'icon'  => 'eicon-animation kfw-ctrl-rotate-180',
						],
						'toggle'         => [
							'title' => esc_html__('Toogle Button Text', 'onnat-extra'),
							'icon'  => 'eicon-share-arrow',
						]
					],
					'default'    => 'left',
					'toggle'     => false,
					'conditions' => $btn_icon_animation_condition
				]);
				$this->add_control( 'alr_btn_icon_animation', [
					'label'      => esc_html__( 'Animation Direction', 'onnat-extra' ),
					'type'       => KFCM::CHOOSE,
					'options'    => [
						'in'  => [
							'title' => esc_html__('Animate In', 'onnat-extra'),
							'icon'  => 'eicon-chevron-left'
						],
						'out' => [
							'title' => esc_html__('Animate Out', 'onnat-extra'),
							'icon'  => 'eicon-chevron-right'
						],
					],
					'default'    => 'in',
					'toggle'     => false,
					'conditions' => [
						'relation' => 'and',
						'terms'    => [
							[
								'name'     => 'btn_text',
								'operator' => '!=',
								'value'    => ''
							],
							[
								'name'     => 'btn_icon[value]',
								'operator' => '!=',
								'value'    => ''
							],
							[
								'name'     => 'btn_icon_pos',
								'operator' => 'in',
								'value'    => ['animated-left', 'animated-right' ]
							]
						]
					]
				]);
				$this->add_control( 'toggle_btn_icon_animation', [
					'label'      => esc_html__( 'Animation Direction', 'onnat-extra' ),
					'type'       => KFCM::CHOOSE,
					'options'    => [
						'top'    => [
							'title' => esc_html__('Top', 'onnat-extra'),
							'icon'  => 'eicon-arrow-up',
						],
						'right'  => [
							'title' => esc_html__('Right', 'onnat-extra'),
							'icon'  => 'eicon-arrow-right',
						],
						'bottom' => [
							'title' => esc_html__('Bottom', 'onnat-extra'),
							'icon'  => 'eicon-arrow-down',
						],
						'left'   => [
							'title' => esc_html__('Left', 'onnat-extra'),
							'icon'  => 'eicon-arrow-left',
						],
					],
					'default'    => 'left',
					'toggle'     => false,
					'conditions' => [
						'relation' => 'and',
						'terms'    => [
							[
								'name'     => 'btn_text',
								'operator' => '!=',
								'value'    => ''
							],
							[
								'name'     => 'btn_icon[value]',
								'operator' => '!=',
								'value'    => ''
							],
							[
								'name'     => 'btn_icon_pos',
								'operator' => '===',
								'value'    => 'toggle'
							]
						]
					]
				]);
				$this->add_responsive_control( 'btn_icon_size', [
					'label'      => esc_html__('Size', 'onnat-extra' ),
					'type'       => KFCM::SLIDER,
					'size_units' => [ 'px' ],
					'range'      => [
						'px' => [
							'min' => 0,
							'max' => 999,
						],
					],
					'selectors'  => [
						'{{WRAPPER}} .kfw-ele-widget-mailchimp-button .kfw-elementor-icon-wrap i'   => 'font-size: {{SIZE}}{{UNIT}};',
						'{{WRAPPER}} .kfw-ele-widget-mailchimp-button .kfw-elementor-icon-wrap svg' => 'font-size: {{SIZE}}{{UNIT}};',
					],
					'condition'  => $btn_icon_condition
				]);
				$this->start_controls_tabs( 'btn_icon_customize_tabs' );
					$this->start_controls_tab( 'btn_icon_normal_state', [
						'label'      => esc_html__( 'Normal', 'onnat-extra' ),
						'condition' => $btn_icon_condition
					]);
						$this->add_control( 'btn_icon_color', [
							'label'      => esc_html__('Icon Color', 'onnat-extra' ),
							'type'       => KFCM::COLOR,
							'selectors'  => [
								'{{WRAPPER}} .kfw-ele-widget-mailchimp-button .kfw-elementor-icon-wrap i'   => 'color: {{VALUE}};',
								'{{WRAPPER}} .kfw-ele-widget-mailchimp-button .kfw-elementor-icon-wrap svg' => 'fill: {{VALUE}};'
							],
							'condition' => $btn_icon_condition
						]);
						$this->add_group_control( KFGCBG::get_type(), [
							'name'       => 'btn_icon_bg',
							'label'      => esc_html__('Background', 'onnat-extra' ),
							'types'      => [ 'classic', 'gradient' ],
							'exclude'    => [ 'image' ], # To remove background image option in classic mode
							'selector'   => '{{WRAPPER}} .kfw-ele-widget-mailchimp-button .kfw-elementor-icon-wrap',
							'condition' => $btn_icon_condition
						]);
						$this->add_group_control( KFGCBR::get_type(), [
							'name'           => 'btn_icon_border',
							'fields_options' => [
								'width' => [ 'label' => _x( 'Border Width', 'Border Control', 'onnat-extra' ) ],
								'color' => [ 'label' => _x( 'Border Color', 'Border Control', 'onnat-extra' ) ],
							],
							'selector'       => '{{WRAPPER}} .kfw-ele-widget-mailchimp-button .kfw-elementor-icon-wrap',
							'condition'     => $btn_icon_condition,
						]);
						$this->add_responsive_control( 'btn_icon_border_radius', [
							'label'      => esc_html__('Border Radius', 'onnat-extra' ),
							'type'       => KFCM::DIMENSIONS,
							'size_units' => [ 'px', '%' ],
							'selectors'  => [ '{{WRAPPER}} .kfw-ele-widget-mailchimp-button .kfw-elementor-icon-wrap' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};' ],
							'condition' => $btn_icon_condition,
						]);
						$this->add_group_control( KFGCBS::get_type(), [
							'name'       => 'btn_icon_box_shadow',
							'selector'   => '{{WRAPPER}} .kfw-ele-widget-mailchimp-button .kfw-elementor-icon-wrap',
							'condition' => $btn_icon_condition,
						]);
					$this->end_controls_tab();
					$this->start_controls_tab( 'btn_icon_hover_state', [
						'label'      => esc_html__( 'Hover', 'onnat-extra' ),
						'condition' => $btn_icon_condition
					]);
						$this->add_control( 'btn_icon_hover_color', [
							'label'      => esc_html__('Color', 'onnat-extra' ),
							'type'       => KFCM::COLOR,
							'selectors'  => [
								'{{WRAPPER}} .kfw-ele-widget-mailchimp-button:hover .kfw-elementor-icon-wrap i'   => 'color: {{VALUE}};',
								'{{WRAPPER}} .kfw-ele-widget-mailchimp-button:hover .kfw-elementor-icon-wrap svg' => 'fill: {{VALUE}};',
							],
							'condition' => $btn_icon_condition
						]);
						$this->add_group_control( KFGCBG::get_type(), [
							'name'       => 'btn_icon_hover_bg',
							'label'      => esc_html__('Background', 'onnat-extra' ),
							'types'      => [ 'classic', 'gradient' ],
							'exclude'    => [ 'image' ], # To remove background image option in classic mode
							'selector'   => '{{WRAPPER}} .kfw-ele-widget-mailchimp-button:hover .kfw-elementor-icon-wrap',
							'condition' => $btn_icon_condition
						]);
						$this->add_group_control( KFGCBR::get_type(), [
							'name'           => 'btn_icon_hover_border',
							'fields_options' => [
								'width' => [ 'label' => _x( 'Border Width', 'Border Control', 'onnat-extra' ) ],
								'color' => [ 'label' => _x( 'Border Color', 'Border Control', 'onnat-extra' ) ],
							],
							'selector'       => '{{WRAPPER}} .kfw-ele-widget-mailchimp-button:hover .kfw-elementor-icon-wrap',
							'condition'     => $btn_icon_condition,
						]);
						$this->add_responsive_control( 'btn_icon_hover_border_radius', [
							'label'      => esc_html__('Border Radius', 'onnat-extra' ),
							'type'       => KFCM::DIMENSIONS,
							'size_units' => [ 'px', '%' ],
							'selectors'  => [ '{{WRAPPER}} .kfw-ele-widget-mailchimp-button:hover .kfw-elementor-icon-wrap' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};' ],
							'condition' => $btn_icon_condition,
						]);
						$this->add_group_control( KFGCBS::get_type(), [
							'name'       => 'btn_icon_hover_box_shadow',
							'selector'   => '{{WRAPPER}} .kfw-ele-widget-mailchimp-button:hover .kfw-elementor-icon-wrap',
							'condition' => $btn_icon_condition,
						]);
					$this->end_controls_tab();
				$this->end_controls_tabs();
		$this->end_controls_section();
    }

	/**
	 * Generate setting json
	 */
	protected function _generate_setting_json( array $settings ) {
		$options = [
			'btn'           => $this->_render_button(),
			'redirect'      => false,
			'message'       => $settings['message'],
			'invalid-email' => esc_html__('Please enter a valid email address.', 'onnat-extra' ),
			'invalid-name'  => esc_html__('Please enter a valid name.', 'onnat-extra' ),
		];

		if( $settings['api_type'] === 'settings' ) {
			$api_key     = kinfw_onnat_theme_options()->kinfw_get_option( 'mailchimp_api_key' );
			$audience_id = kinfw_onnat_theme_options()->kinfw_get_option( 'mailchimp_audience_id' );

			$options['akey' ] = $api_key;
			$options['aid' ]  = $audience_id;

		} else if( $settings['api_type'] === 'widget' ) {
			$api_key     = $settings['api_key'];
			$audience_id = $settings['audience_id'];

			$options['akey' ] = $api_key;
			$options['aid' ]  = $audience_id;
		}

		if( $settings['use_redirect'] === 'yes' ) {
			$options['redirect'] = true;
			$options['redirectURL'] = $settings['redirect_url']['url'];
		}

		return wp_json_encode( $options );
	}

	public function _layout(){
		$layout = $this->get_settings( 'layout' );

		switch( $layout ) {
			case 'preset-style-1':
				$this->_preset_layout( 'kfw-ele-widget-mailchimp-plus-preset-style-1' );
			break;

			case 'preset-style-2':
				$this->_preset_layout( 'kfw-ele-widget-mailchimp-plus-preset-style-2' );
			break;

			default:
				$this->_preset_layout( 'kfw-ele-widget-mailchimp-plus-preset-style-2' );
			break;
		}
	}

	public function _preset_layout( $class = '' ) {

		printf( '<form action="%s" class="kfw-ele-widget-mailchimp-form %s">', esc_url( admin_url('admin-ajax.php') ), $class );
			$this->_email_field();
			$this->_name_field();
			$this->_hidden_fields();
			$this->_button_field();
		print( '</form>' );
	}

	public function _email_field() {
		$settings = $this->get_settings();

		$icon = array_filter( $settings['email_field_icon'] );

		$this->add_render_attribute( 'field-wrapper', [
			'class' => array_filter([
				'kfw-ele-widget-mailchimp-plus-input-field-wrap',
				'kfw-ele-widget-mailchimp-email-field-wrap',
				count( $icon ) ? 'kfw-has-icon' : ''
			])
		]);

		echo '<div '.$this->get_render_attribute_string( 'field-wrapper' ).'>';
			$this->_render_icon( $icon );
			printf(
				'<input type="text" name="email" placeholder="%s" class="kfw-ele-widget-mailchimp-plus-input-field kfw-ele-widget-mailchimp-email-field"/>',
				$settings['email_field_placeholder']
			);
		echo '</div>';
	}

	public function _name_field() {
		$settings = $this->get_settings();

		$icon = array_filter( $settings['name_field_icon'] );

		$this->add_render_attribute( 'name-field-wrapper', [
			'class' => array_filter([
				'kfw-ele-widget-mailchimp-plus-input-field-wrap',
				'kfw-ele-widget-mailchimp-name-field-wrap',
				count( $icon ) ? 'kfw-has-icon' : ''
			])
		]);

		echo '<div '.$this->get_render_attribute_string( 'name-field-wrapper' ).'>';
			$this->_render_icon( $icon );
			printf(
				'<input type="text" name="name" placeholder="%s" class="kfw-ele-widget-mailchimp-plus-input-field kfw-ele-widget-mailchimp-name-field"/>',
				$settings['name_field_placeholder']
			);
		echo '</div>';
	}

	public function _button_field() {
		$settings = $this->get_settings();

		$this->add_render_attribute( 'cta-wrapper', [
			'class' => array_filter([
				'kfw-ele-widget-mailchimp-button-wrap',
				!empty( trim( $settings['btn_text'] ) ) ? 'kfw-ele-widget-mailchimp-button-has-txt' : ''
			])
		]);

		echo '<div '.$this->get_render_attribute_string( 'cta-wrapper' ).'>';

			$this->add_render_attribute( 'cta', [
				'class' => array_filter([
					'kfw-ele-widget-mailchimp-button',
					!empty( $settings['btn_animation_hover'] ) ? 'elementor-animation-'.esc_attr( $settings['btn_animation_hover'] ): '',
				])
			]);

			/**
			 * Button Icon related classes
			 */
			if( $settings['btn_icon']['value'] !== '' ) {
				$this->add_render_attribute( 'cta', [
					'class' => array_filter([
						!empty( trim( $settings['btn_text'] ) ) ? 'kfw-ele-widget-mailchimp-button-icon-position-'.$settings['btn_icon_pos'] : '',
						( !empty( trim( $settings['btn_text'] ) ) ) && ( $settings['btn_icon_pos'] === 'animated-left' || $settings['btn_icon_pos'] === 'animated-right' ) ? 'kfw-ele-widget-mailchimp-button-animated-icon-position-' . $settings['alr_btn_icon_animation'] : '',
						( !empty( trim( $settings['btn_text'] ) ) ) && ( $settings['btn_icon_pos'] === 'toggle' ) ? 'kfw-ele-widget-mailchimp-button-toggle-icon-position-' . $settings['toggle_btn_icon_animation'] : '',								
					])
				]);
			}

			echo '<button type="submit" '.$this->get_render_attribute_string( 'cta' ).'>';

				if( $settings['btn_icon_pos'] === 'left' || $settings['btn_icon_pos'] === 'animated-left' ) {
					$this->_render_icon( $settings['btn_icon'] );
				}

				if( !empty( trim( $settings['btn_text'] ) ) ) {
					printf( '<span class="kfw-elementor-button-text">%s</span>', esc_html( $settings['btn_text'] ) );
				}

				if( $settings['btn_icon_pos'] === 'right' ||
					$settings['btn_icon_pos'] === 'animated-right' ||
					$settings['btn_icon_pos'] === 'toggle' ||
					$settings['btn_icon_pos'] === 'toggle-alt'
				) {
					$this->_render_icon( $settings['btn_icon'] );
				}

			echo '</button>';

		echo '</div>';
	}

	public function _render_button() {
		ob_start();

		$settings = $this->get_settings();

		if( $settings['btn_icon_pos'] === 'left' || $settings['btn_icon_pos'] === 'animated-left' ) {
			$this->_render_icon( $settings['btn_icon'] );
		}

		if( !empty( trim( $settings['btn_text'] ) ) ) {
			printf( '<span class="kfw-elementor-button-text">%s</span>', esc_html( $settings['btn_text'] ) );
		}

		if( $settings['btn_icon_pos'] === 'right' ||
			$settings['btn_icon_pos'] === 'animated-right' ||
			$settings['btn_icon_pos'] === 'toggle' ||
			$settings['btn_icon_pos'] === 'toggle-alt'
		) {
			$this->_render_icon( $settings['btn_icon'] );
		}

		$button = ob_get_contents();
		ob_end_clean();

		return $button;
	}

	public function _hidden_fields() {
		echo '<input type="hidden" name="action" value="kinfw-action/plugins/elementor/widget/kfw-mailchimp-plus/subscribe"/>';
	}

	public function _render_icon( $icon , $echo = true ) {
		$icon = array_filter( $icon );

		if( !empty( $icon ) ) {
			echo '<span class="kfw-elementor-icon-wrap">';
				KFIM::render_icon( $icon, [ 'aria-hidden' => 'true' ] );
			echo '</span>';
		}
	}

	public function _message( $message, $echo = true ) {
		$return = sprintf( '
			<div class="elementor-alert elementor-alert-danger" role="alert">
				<span class="elementor-alert-description">%s</span>
			</div>',
			$message
		);

		if( $echo ) {
			echo $return;
			return;
		}

		return $return;
	}

	/**
	 * Render widget output on the frontend.
	 */
    protected function render() {
		$render = kf_extra_plugin_elementor()->get_template("kfw-mailchimp-plus");
		include $render;
    }
}