<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

use \Elementor\Widget_Base as KFWB;
use \Elementor\Controls_Manager as KFCM;
use \Elementor\Icons_Manager as KFIM;

use \Elementor\Group_Control_Border as KFGCBR;
use \Elementor\Group_Control_Background as KFGCBG;
use \Elementor\Group_Control_Typography as KFGCT;

class KFW_Mobile_Menu extends KFWB {

	/**
	 * Get element name.
	 *
	 * Retrieve the element name.
	 *
	 * @return string The name.
	 */
	public function get_name() {
        return 'kfw-mobile-menu';
    }

	/**
	 * Get element title.
	 *
	 * Retrieve the element title.
	 *
	 * @return string Element title.
	 */
	public function get_title() {
		return esc_html__('Mobile Menu', 'onnat-extra');
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the widget categories.
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ '_kfw_header_footer_widgets' ];
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve the widget icon.
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'kfw-badge eicon-text-align-justify kfw-mobile-menu';
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the widget keywords.
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'kfw', 'kin', 'menu', 'mobile'];
	}

	/**
	 * Get widget help url.
	 *
	 * Retrieve the widget help url.
	 *
	 * @return string Widget help url.
	 */
	public function get_custom_help_url() {
        return KFE_CONST_THEME_DOC . 'elementor-custom-widgets/mobile-menu';
    }

	/**
	 * Get HTML wrapper class.
     *
	 * Retrieve the widget container class.
     *
     * @return string Widget container css classes.
	 */
	public function get_html_wrapper_class() {
        $name = $this->get_name();

		return sprintf( 'kfw-ele-%s elementor-widget-%s', $name, $name );
	}

	/**
	 * Widget base constructor.
	 *
	 * Initializing the widget base class.
	 */
	public function __construct( $data = [], $args = null ) {
        parent::__construct( $data, $args );

		wp_register_script( 'kfw-mobile-menu',
			KFE_CONST_URL . 'libraries/elementor/assets/js/mobile-menu' . KFE_CONST_DEBUG_SUFFIX . '.js',
			[ 'jquery', 'elementor-frontend' ],
			KFE_CONST_VERSION,
			true
		);

		wp_register_style( 'kfw-mobile-menu',
			KFE_CONST_URL . 'libraries/elementor/assets/css/mobile-menu' . KFE_CONST_DEBUG_SUFFIX . '.css',
			[],
			KFE_CONST_VERSION
		);

    }

	/**
	 * Get style dependencies.
	 *
	 * Retrieve the list of style dependencies the element requires.
	 *
	 * @return array Element styles dependencies.
	 */
	public function get_style_depends() {
		return [ 'kfw-mobile-menu' ];
	}

	/**
	 * Get script dependencies.
	 *
	 * Retrieve the list of script dependencies the element requires.
	 *
	 * @return array Element scripts dependencies.
	 */
	public function get_script_depends() {
		return [ 'kfw-mobile-menu' ];
	}

	/**
	 * Register controls.
	 *
	 * Used to add new controls to any element type.
	 */
	protected function register_controls() {
		$this->_register_menu_trigger_section();
		$this->_register_menu_section();

		$this->_register_menu_trigger_style_section();
		$this->_register_menu_style_section();
    }

	public function _register_menu_trigger_section() {
		$this->start_controls_section( 'section_menu_trigger', [
			'label' => esc_html__( 'Menu Trigger', 'onnat-extra' )
		]);
			$this->add_control('menu_trigger_icon_type',[
				'label'   => esc_html__( 'Type', 'onnat-extra' ),
				'type'    => KFCM::CHOOSE,
				'options' => [
					'icon'  => [
						'title' => esc_html__('Custom Icon', 'onnat-extra'),
						'icon'  => 'eicon-plus-circle-o',
					],
					'theme' => [
						'title' => esc_html__('Theme Icon', 'onnat-extra'),
						'icon'  => 'eicon-t-letter-bold',
					],
				],
				'default' => 'theme',
				'toggle'  => false
			]);
			$this->add_control( 'menu_trigger_icon', [
				'label'     => esc_html__( 'Icon', 'onnat-extra' ),
				'type'      => KFCM::ICONS,
				'default'   => [
					'value'   => 'fas fa-heart',
					'library' => 'fa-solid',
				],
				'condition' => [
					'menu_trigger_icon_type' => 'icon'
				],
			]);
		$this->end_controls_section();
	}

	public function _register_menu_section() {
		$this->start_controls_section( 'section_menu', [
			'label' => esc_html__( 'Menu', 'onnat-extra' )
		]);

			$this->add_control( 'menu', [
				'label'       => esc_html__( 'Choose Menu', 'onnat-extra' ),
				'type'        => KFCM::SELECT,
				'options'     => kf_extra_plugin_elementor()->get_available_menus(),
				'default'     => 0,
				'description' => sprintf(
					esc_html__('Go to the %1$s to manage your menus.', 'onnat-extra'),
					sprintf('<a href="%1$s" target="_blank">%2$s</a>', admin_url('nav-menus.php'), esc_html_x( 'Menus screen', 'link text for WordPress Admin Menu Screen', 'onnat-extra' )  )
				),
			] );
		$this->end_controls_section();
	}

	public function _register_menu_trigger_style_section() {
		$this->start_controls_section( 'section_menu_trigger_style', [
			'label' => esc_html__( 'Menu Trigger', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
			$this->add_responsive_control('menu_trigger_align', [
				'label'          => esc_html__( 'Alignment', 'onnat-extra' ),
				'type'           => KFCM::CHOOSE,
				'options'        => [
					'left'   => [
						'title' => esc_html__( 'Left', 'onnat-extra' ),
						'icon'  => 'eicon-text-align-left',
					],
					'center' => [
						'title' => esc_html__( 'Center', 'onnat-extra' ),
						'icon'  => 'eicon-text-align-center',
					],
					'right'  => [
						'title' => esc_html__( 'Right', 'onnat-extra' ),
						'icon'  => 'eicon-text-align-right',
					],
				],
				'default'              => 'center',
				'tablet_default'       => 'center',
				'mobile_default'       => 'center',
				'toggle'               => false,
				'selectors_dictionary' => [
					'left'   => 'text-align:left;',
					'center' => 'text-align:center;',
					'right'  => 'text-align:right;',
				],
				'selectors'            => [
					'{{WRAPPER}} .kfw-ele-widget-mobile-menu-main .kinfw-mobile-navigation-trigger' => '{{VALUE}};'
				],
			]);
			$this->add_responsive_control( 'menu_trigger_margin', [
				'label'      => esc_html__('Margin', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-mobile-menu-main .kinfw-mobile-navigation-trigger .kfw-elementor-icon-wrap' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
			$this->add_responsive_control( 'menu_trigger_padding', [
				'label'      => esc_html__('Padding', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-mobile-menu-main .kinfw-mobile-navigation-trigger .kfw-elementor-icon-wrap' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);

			$this->start_controls_tabs( 'menu_trigger_customize_tabs' );
				$this->start_controls_tab( 'menu_trigger_normal_state', [
					'label'     => esc_html__( 'Normal', 'onnat-extra' ),
				]);
					$this->add_group_control( KFGCBG::get_type(), [
						'name'           => 'menu_trigger_bg',
						'types'          => ['classic', 'gradient'],
						'exclude'        => [ 'image'],
						'fields_options' => [
							'background' => [ 'default' => 'classic' ],
							'color'      => [ 'default' => '' ],
						],
						'selector'       => '{{WRAPPER}} .kfw-ele-widget-mobile-menu-main .kinfw-mobile-navigation-trigger .kfw-elementor-icon-wrap',
					]);
					$this->add_control('menu_trigger_bg_rotate',[
						'label'     => esc_html__( 'Background Rotate', 'onnat-extra' ),
						'type'      => KFCM::SLIDER,
						'default'   => [
							'size' => 0,
							'unit' => 'deg',
						],
						'range'     => [
							'deg' => [
								'max' => 360,
								'min' => -360,
							],
						],
						'condition' => [
							'menu_trigger_bg_background' => [ 'classic', 'gradient' ],
						],
						'selectors' => [
							'{{WRAPPER}} .kfw-ele-widget-mobile-menu-main .kinfw-mobile-navigation-trigger .kfw-elementor-icon-wrap' => 'transform: rotate({{SIZE}}{{UNIT}});',
						],
					]);
					$this->add_group_control( KFGCBR::get_type(), [
						'name'      => 'menu_trigger_border',
						'selector'  => '{{WRAPPER}} .kfw-ele-widget-mobile-menu-main .kinfw-mobile-navigation-trigger .kfw-elementor-icon-wrap'
					]);
					$this->add_responsive_control( 'menu_trigger_border_radius', [
						'label'      => esc_html__('Border Radius', 'onnat-extra' ),
						'type'       => KFCM::DIMENSIONS,
						'size_units' => [ 'px', '%' ],
						'selectors'  => [
							'{{WRAPPER}} .kfw-ele-widget-mobile-menu-main .kinfw-mobile-navigation-trigger .kfw-elementor-icon-wrap' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
						]
					]);
				$this->end_controls_tab();
				$this->start_controls_tab( 'menu_trigger_hover_state', [
					'label'      => esc_html__( 'Hover', 'onnat-extra' ),
				]);
					$this->add_group_control( KFGCBG::get_type(), [
						'name'           => 'menu_trigger_hover_bg',
						'types'          => ['classic', 'gradient'],
						'exclude'        => [ 'image'],
						'fields_options' => [
							'background' => [ 'default' => 'classic' ],
							'color'      => [ 'default' => '' ],
						],
						'selector'       => '{{WRAPPER}} .kfw-ele-widget-mobile-menu-main .kinfw-mobile-navigation-trigger:hover .kfw-elementor-icon-wrap',
					]);
					$this->add_control('menu_trigger_hover_bg_rotate',[
						'label'     => esc_html__( 'Background Rotate', 'onnat-extra' ),
						'type'      => KFCM::SLIDER,
						'default'   => [
							'size' => 0,
							'unit' => 'deg',
						],
						'range'     => [
							'deg' => [
								'max' => 360,
								'min' => -360,
							],
						],
						'condition' => [
							'menu_trigger_bg_background' => [ 'classic', 'gradient' ],
						],
						'selectors' => [
							'{{WRAPPER}} .kfw-ele-widget-mobile-menu-main .kinfw-mobile-navigation-trigger:hover .kfw-elementor-icon-wrap' => 'transform: rotate({{SIZE}}{{UNIT}});',
						],
					]);
					$this->add_group_control( KFGCBR::get_type(), [
						'name'      => 'menu_trigger_hover_border',
						'selector'  => '{{WRAPPER}} .kfw-ele-widget-mobile-menu-main .kinfw-mobile-navigation-trigger:hover .kfw-elementor-icon-wrap'
					]);
					$this->add_responsive_control( 'icon_hover_border_radius', [
						'label'      => esc_html__('Border Radius', 'onnat-extra' ),
						'type'       => KFCM::DIMENSIONS,
						'size_units' => [ 'px', '%' ],
						'selectors'  => [
							'{{WRAPPER}} .kfw-ele-widget-mobile-menu-main .kinfw-mobile-navigation-trigger:hover .kfw-elementor-icon-wrap' => 'border-radius:{{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
						]
					]);
				$this->end_controls_tab();
			$this->end_controls_tabs();

			$this->add_control('menu_trigger_icon_heading_before_hr', [
				'type' => KFCM::DIVIDER,
			]);
			$this->add_control('menu_trigger_icon_heading',[
				'label' => esc_html__( 'ICON', 'onnat-extra' ),
				'type'  => KFCM::HEADING,
			]);
			$this->add_control('menu_trigger_icon_heading_after_hr', [
				'type' => KFCM::DIVIDER,
			]);

			$this->add_responsive_control( 'menu_trigger_icon_size', [
				'label'      => esc_html__('Icon Size', 'onnat-extra' ),
				'type'       => KFCM::SLIDER,
				'size_units' => [ 'px' ],
				'range'      => [
					'px' => [
						'min' => 0,
						'max' => 200,
					],
				],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-mobile-menu-main .kinfw-mobile-navigation-trigger .kinfw-icon' => 'font-size: {{SIZE}}{{UNIT}};'
				],
			]);
			$this->start_controls_tabs( 'menu_trigger_icon_customize_tabs' );
				$this->start_controls_tab( 'menu_trigger_icon_normal_state', [
					'label' => esc_html__( 'Normal', 'onnat-extra' ),
				]);
					$this->add_control( 'menu_trigger_icon_color', [
						'label'     => esc_html__('Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'selectors' => [
							'{{WRAPPER}} .kfw-ele-widget-mobile-menu-main .kinfw-mobile-navigation-trigger .kinfw-icon'     => 'color: {{VALUE}};',
							'{{WRAPPER}} .kfw-ele-widget-mobile-menu-main .kinfw-mobile-navigation-trigger .kinfw-icon svg' => 'fill: {{VALUE}};',
						],
					]);
					$this->add_control('menu_trigger_icon_rotate',[
						'label'     => esc_html__( 'Rotate', 'onnat-extra' ),
						'type'      => KFCM::SLIDER,
						'default'   => [
							'size' => 0,
							'unit' => 'deg',
						],
						'range'     => [
							'deg' => [
								'max' => 360,
								'min' => - 360,
							],
						],
						'selectors' => [
							'{{WRAPPER}} .kfw-ele-widget-mobile-menu-main .kinfw-mobile-navigation-trigger .kinfw-icon'     => 'transform: rotate({{SIZE}}{{UNIT}});',
							'{{WRAPPER}} .kfw-ele-widget-mobile-menu-main .kinfw-mobile-navigation-trigger .kinfw-icon svg' => 'transform: rotate({{SIZE}}{{UNIT}});',
						],
					]);
				$this->end_controls_tab();
				$this->start_controls_tab( 'menu_trigger_icon_hover_state', [
					'label' => esc_html__( 'Hover', 'onnat-extra' ),
				]);
					$this->add_control( 'menu_trigger_icon_h_color', [
						'label'     => esc_html__('Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'selectors' => [
							'{{WRAPPER}} .kfw-ele-widget-mobile-menu-main .kinfw-mobile-navigation-trigger .kinfw-icon:hover'     => 'color: {{VALUE}};',
							'{{WRAPPER}} .kfw-ele-widget-mobile-menu-main .kinfw-mobile-navigation-trigger .kinfw-icon:hover svg' => 'fill: {{VALUE}};',
						],
					]);
					$this->add_control('menu_trigger_icon_h_rotate',[
						'label'     => esc_html__( 'Rotate', 'onnat-extra' ),
						'type'      => KFCM::SLIDER,
						'default'   => [
							'size' => 0,
							'unit' => 'deg',
						],
						'range'     => [
							'deg' => [
								'max' => 360,
								'min' => - 360,
							],
						],
						'selectors' => [
							'{{WRAPPER}} .kfw-ele-widget-mobile-menu-main .kinfw-mobile-navigation-trigger .kinfw-icon:hover'     => 'transform: rotate({{SIZE}}{{UNIT}});',
							'{{WRAPPER}} .kfw-ele-widget-mobile-menu-main .kinfw-mobile-navigation-trigger .kinfw-icon:hover svg' => 'transform: rotate({{SIZE}}{{UNIT}});',
						],
					]);
				$this->end_controls_tab();
			$this->end_controls_tabs();
		$this->end_controls_section();
	}

	public function _register_menu_style_section() {
		$this->start_controls_section( 'section_menu_style', [
			'label' => esc_html__( 'Menu', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
			$this->add_control( 'menu_bg_color', [
				'label'     => esc_html__('Background Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'selectors' => [
					'{{WRAPPER}} .kfw-ele-widget-mobile-menu-main .kinfw-mobile-menu, {{WRAPPER}} .kfw-ele-widget-mobile-menu-main .kinfw-mobile-menu .sub-menu' => 'background-color:{{VALUE}};',
				],
			]);
			$this->add_control( 'menu_overlay_color', [
				'label'     => esc_html__('Overlay Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'selectors' => [
					'{{WRAPPER}} .kfw-ele-widget-mobile-menu-main .kinfw-mobile-menu-overlay' => 'background-color:{{VALUE}};',
				],
			]);

			/**
			 * Menu Link Items
			 */
				$this->add_control('menu_items_heading_before_hr', [
					'type' => KFCM::DIVIDER,
				]);
				$this->add_control('menu_items_heading',[
					'label' => esc_html__( 'MENU ITEMS', 'onnat-extra' ),
					'type'  => KFCM::HEADING,
				]);
				$this->add_control('menu_items_heading_after_hr', [
					'type' => KFCM::DIVIDER,
				]);
				$this->add_group_control(KFGCT::get_type(), [
					'name'     => 'menu_items_typo',
					'label'    => esc_html__( 'Typography', 'onnat-extra' ),
					'selector' => '{{WRAPPER}} .kfw-ele-widget-mobile-menu-main .kinfw-mobile-menu-nav ul li a',
				]);
				$this->add_responsive_control( 'menu_items_margin', [
					'label'      => esc_html__('Margin', 'onnat-extra' ),
					'type'       => KFCM::DIMENSIONS,
					'size_units' => [ 'px', 'em', '%' ],
					'selectors'  => [
						'{{WRAPPER}} .kfw-ele-widget-mobile-menu-main .kinfw-mobile-menu-nav ul li' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
					]
				]);
				$this->add_responsive_control( 'menu_items_padding', [
					'label'      => esc_html__('Padding', 'onnat-extra' ),
					'type'       => KFCM::DIMENSIONS,
					'size_units' => [ 'px', 'em', '%' ],
					'selectors'  => [
						'{{WRAPPER}} .kfw-ele-widget-mobile-menu-main .kinfw-mobile-menu-nav ul li a' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
					]
				]);
				$this->add_group_control( KFGCBR::get_type(), [
					'name'           => 'menu_items_border',
					'fields_options' => [
						'width' => [ 'label' => _x( 'Border Width', 'Border Control', 'onnat-extra' ) ],
						'color' => [ 'label' => _x( 'Border Color', 'Border Control', 'onnat-extra' ) ],
					],
					'selector'       => '{{WRAPPER}} .kfw-ele-widget-mobile-menu-main .kinfw-mobile-menu-nav ul > li',
				]);
				$this->add_control( 'menu_items_color_settings', [
					'label'        => esc_html__('Colors', 'onnat-extra'),
					'type'         => KFCM::POPOVER_TOGGLE,
					'return_value' => 'yes',
				]);
				$this->start_popover();
					$this->add_control( 'menu_items_links_color', [
						'label'     => esc_html__('Normal Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'selectors' => [
							'{{WRAPPER}} .kfw-ele-widget-mobile-menu-main .kinfw-mobile-menu-nav ul li a' => 'color:{{VALUE}};',
						],
					]);
					$this->add_control( 'menu_items_links_h_color', [
						'label'     => esc_html__('Hover Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'selectors' => [
							'{{WRAPPER}} .kfw-ele-widget-mobile-menu-main .kinfw-mobile-menu-nav ul li a:hover' => 'color:{{VALUE}};',
						],
					]);
					$this->add_control( 'menu_items_links_a_color', [
						'label'     => esc_html__('Active Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'selectors' => [
							'{{WRAPPER}} .kfw-ele-widget-mobile-menu-main .kinfw-mobile-menu-nav ul li.current_page_item a' => 'color:{{VALUE}};',
							'{{WRAPPER}} .kfw-ele-widget-mobile-menu-main .kinfw-mobile-menu-nav ul li.current-menu-item a' => 'color:{{VALUE}};',
						],
					]);
				$this->end_popover();

			/**
			 * Close Icon
			 */
				$this->add_control('menu_close_icon_heading_before_hr', [
					'type' => KFCM::DIVIDER,
				]);
				$this->add_control('menu_close_icon_heading',[
					'label' => esc_html__( 'CLOSE ICON', 'onnat-extra' ),
					'type'  => KFCM::HEADING,
				]);
				$this->add_control('menu_close_icon_heading_after_hr', [
					'type' => KFCM::DIVIDER,
				]);

				$this->add_responsive_control( 'menu_close_icon_size', [
					'label'      => esc_html__('Icon Size', 'onnat-extra' ),
					'type'       => KFCM::SLIDER,
					'size_units' => [ 'px' ],
					'range'      => [
						'px' => [
							'min' => 0,
							'max' => 200,
						],
					],
					'selectors'  => [
						'{{WRAPPER}} .kfw-ele-widget-mobile-menu-main .kinfw-mobile-menu-nav ul > li.kinfw-mobile-menu-close a .kinfw-icon' => 'font-size: {{SIZE}}{{UNIT}};'
					],
				]);

				$this->start_controls_tabs( 'menu_close_icon_customize_tabs' );
					$this->start_controls_tab( 'menu_close_icon_normal_state', [
						'label' => esc_html__( 'Normal', 'onnat-extra' ),
					]);
						$this->add_control( 'menu_close_icon_color', [
							'label'     => esc_html__('Color', 'onnat-extra' ),
							'type'      => KFCM::COLOR,
							'selectors' => [
								'{{WRAPPER}} .kfw-ele-widget-mobile-menu-main .kinfw-mobile-menu-nav ul > li.kinfw-mobile-menu-close a .kinfw-icon' => 'color: {{VALUE}};',
							],
						]);
						$this->add_control('menu_close_icon_rotate',[
							'label'     => esc_html__( 'Rotate', 'onnat-extra' ),
							'type'      => KFCM::SLIDER,
							'default'   => [
								'size' => 0,
								'unit' => 'deg',
							],
							'range'     => [
								'deg' => [
									'max' => 360,
									'min' => - 360,
								],
							],
							'selectors' => [
								'{{WRAPPER}} .kfw-ele-widget-mobile-menu-main .kinfw-mobile-menu-nav ul > li.kinfw-mobile-menu-close a .kinfw-icon'   => 'transform: rotate({{SIZE}}{{UNIT}});',
							],
						]);
					$this->end_controls_tab();
					$this->start_controls_tab( 'menu_close_icon_hover_state', [
						'label' => esc_html__( 'Hover', 'onnat-extra' ),
					]);
						$this->add_control( 'menu_close_icon_h_color', [
							'label'     => esc_html__('Color', 'onnat-extra' ),
							'type'      => KFCM::COLOR,
							'selectors' => [
								'{{WRAPPER}} .kfw-ele-widget-mobile-menu-main .kinfw-mobile-menu-nav ul > li.kinfw-mobile-menu-close a:hover .kinfw-icon' => 'color: {{VALUE}};',
							],
						]);
						$this->add_control('menu_close_icon_h_rotate',[
							'label'     => esc_html__( 'Rotate', 'onnat-extra' ),
							'type'      => KFCM::SLIDER,
							'default'   => [
								'size' => 0,
								'unit' => 'deg',
							],
							'range'     => [
								'deg' => [
									'max' => 360,
									'min' => - 360,
								],
							],
							'selectors' => [
								'{{WRAPPER}} .kfw-ele-widget-mobile-menu-main .kinfw-mobile-menu-nav ul > li.kinfw-mobile-menu-close a:hover .kinfw-icon'   => 'transform: rotate({{SIZE}}{{UNIT}});',
							],
						]);
					$this->end_controls_tab();
				$this->end_controls_tabs();

			/**
			 * Sub Menu Arrow Icon
			 */
				$this->add_control('smenu_arrow_icon_heading_before_hr', [
					'type' => KFCM::DIVIDER,
				]);
				$this->add_control('smenu_arrow_icon_heading',[
					'label' => esc_html__( 'ARROW ICON', 'onnat-extra' ),
					'type'  => KFCM::HEADING,
				]);
				$this->add_control('smenu_arrow_icon_heading_after_hr', [
					'type' => KFCM::DIVIDER,
				]);

				$this->add_responsive_control( 'smenu_arrow_icon_size', [
					'label'      => esc_html__('Icon Size', 'onnat-extra' ),
					'type'       => KFCM::SLIDER,
					'size_units' => [ 'px' ],
					'range'      => [
						'px' => [
							'min' => 0,
							'max' => 200,
						],
					],
					'selectors'  => ['
						{{WRAPPER}} .kfw-ele-widget-mobile-menu-main .kinfw-mobile-menu-nav ul li .kinfw-sub-menu-trigger .kinfw-icon,
						{{WRAPPER}} .kfw-ele-widget-mobile-menu-main .kinfw-mobile-menu-nav ul.kinfw-mobile-menu .sub-menu li.kinfw-mobile-menu-back a .kinfw-icon' => 'font-size: {{SIZE}}{{UNIT}};'
					],
				]);

				$this->start_controls_tabs( 'smenu_arrow_icon_customize_tabs' );
					$this->start_controls_tab( 'smenu_arrow_icon_normal_state', [
						'label' => esc_html__( 'Normal', 'onnat-extra' ),
					]);
						$this->add_control( 'smenu_arrow_icon_color', [
							'label'     => esc_html__('Color', 'onnat-extra' ),
							'type'      => KFCM::COLOR,
							'selectors' => [
								'{{WRAPPER}} .kfw-ele-widget-mobile-menu-main .kinfw-mobile-menu-nav ul li .kinfw-sub-menu-trigger .kinfw-icon' => 'color: {{VALUE}};',
								'{{WRAPPER}} .kfw-ele-widget-mobile-menu-main .kinfw-mobile-menu-nav ul.kinfw-mobile-menu .sub-menu li.kinfw-mobile-menu-back a .kinfw-icon' => 'color: {{VALUE}};',
							],
						]);
						$this->add_control('smenu_arrow_icon_rotate',[
							'label'     => esc_html__( 'Rotate', 'onnat-extra' ),
							'type'      => KFCM::SLIDER,
							'default'   => [
								'size' => 0,
								'unit' => 'deg',
							],
							'range'     => [
								'deg' => [
									'max' => 360,
									'min' => - 360,
								],
							],
							'selectors' => [
								'{{WRAPPER}} .kfw-ele-widget-mobile-menu-main .kinfw-mobile-menu-nav ul li .kinfw-sub-menu-trigger .kinfw-icon' => 'transform: rotate({{SIZE}}{{UNIT}});',
							],
						]);
						$this->add_control('smenu_back_arrow_icon_rotate',[
							'label'     => esc_html__( 'Back Arrow Rotate', 'onnat-extra' ),
							'type'      => KFCM::SLIDER,
							'default'   => [
								'size' => 0,
								'unit' => 'deg',
							],
							'range'     => [
								'deg' => [
									'max' => 360,
									'min' => - 360,
								],
							],
							'selectors' => [
								'{{WRAPPER}} .kfw-ele-widget-mobile-menu-main .kinfw-mobile-menu-nav ul.kinfw-mobile-menu .sub-menu li.kinfw-mobile-menu-back a .kinfw-icon' => 'transform: rotate({{SIZE}}{{UNIT}});',
							],
						]);
					$this->end_controls_tab();
					$this->start_controls_tab( 'smenu_arrow_icon_hover_state', [
						'label' => esc_html__( 'Hover', 'onnat-extra' ),
					]);
						$this->add_control( 'smenu_arrow_icon_h_color', [
							'label'     => esc_html__('Color', 'onnat-extra' ),
							'type'      => KFCM::COLOR,
							'selectors' => [
								'{{WRAPPER}} .kfw-ele-widget-mobile-menu-main .kinfw-mobile-menu-nav ul li:hover .kinfw-sub-menu-trigger .kinfw-icon' => 'color: {{VALUE}};',
								'{{WRAPPER}} .kfw-ele-widget-mobile-menu-main .kinfw-mobile-menu-nav ul.kinfw-mobile-menu .sub-menu li.kinfw-mobile-menu-back:hover a .kinfw-icon' => 'color: {{VALUE}};',
							],
						]);
						$this->add_control('smenu_arrow_icon_h_rotate',[
							'label'     => esc_html__( 'Rotate', 'onnat-extra' ),
							'type'      => KFCM::SLIDER,
							'default'   => [
								'size' => 0,
								'unit' => 'deg',
							],
							'range'     => [
								'deg' => [
									'max' => 360,
									'min' => - 360,
								],
							],
							'selectors' => [
								'{{WRAPPER}} .kfw-ele-widget-mobile-menu-main .kinfw-mobile-menu-nav ul li:hover .kinfw-sub-menu-trigger .kinfw-icon' => 'transform: rotate({{SIZE}}{{UNIT}});',
							],
						]);
						$this->add_control('smenu_back_arrow_icon_h_rotate',[
							'label'     => esc_html__( 'Back Arrow Rotate', 'onnat-extra' ),
							'type'      => KFCM::SLIDER,
							'default'   => [
								'size' => 0,
								'unit' => 'deg',
							],
							'range'     => [
								'deg' => [
									'max' => 360,
									'min' => - 360,
								],
							],
							'selectors' => [
								'{{WRAPPER}} .kfw-ele-widget-mobile-menu-main .kinfw-mobile-menu-nav ul.kinfw-mobile-menu .sub-menu li.kinfw-mobile-menu-back:hover a .kinfw-icon' => 'transform: rotate({{SIZE}}{{UNIT}});',
							],
						]);
					$this->end_controls_tab();
				$this->end_controls_tabs();

		$this->end_controls_section();
	}

    public function _render_icons( $icon , $echo = true ) {
		$icon = array_filter( $icon );

		if( !empty( $icon ) ) {
			echo '<span class="kinfw-icon kfw-elementor-icon-wrap">';
				KFIM::render_icon( $icon, [ 'aria-hidden' => 'true' ] );
			echo '</span>';
		}
	}

	/**
	 * Render widget output on the frontend.
	 */
    protected function render() {
		$render = kf_extra_plugin_elementor()->get_template("kfw-mobile-menu");
		include $render;
    }
}