<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

use \Elementor\Widget_Base as KFWB;
use \Elementor\Controls_Manager as KFCM;
use \Elementor\Icons_Manager as KFIM;

use \Elementor\Repeater as KFRTR;
use \Elementor\Group_Control_Typography as KFGCT;
use \Elementor\Group_Control_Border as KFGCBR;
use \Elementor\Group_Control_Background as KFGCBG;
use \Elementor\Group_Control_Box_Shadow as KFGCBS;

class KFW_Pricing_Table extends KFWB {

	/**
	 * Get element name.
	 *
	 * Retrieve the element name.
	 *
	 * @return string The name.
	 */
	public function get_name() {
        return 'kfw-pricing-table';
    }

	/**
	 * Get element title.
	 *
	 * Retrieve the element title.
	 *
	 * @return string Element title.
	 */
	public function get_title() {
		return esc_html__('Pricing Table', 'onnat-extra');
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the widget categories.
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ '_kfw_core_widgets' ];
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve the widget icon.
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'kfw-badge eicon-price-table kfw-pricing-table';
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the widget keywords.
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'kfw', 'kin', 'pricing', 'table',];
	}

	/**
	 * Get widget help url.
	 *
	 * Retrieve the widget help url.
	 *
	 * @return string Widget help url.
	 */
	public function get_custom_help_url() {
		return KFE_CONST_THEME_DOC . 'elementor-custom-widgets/pricing-table';
    }

	/**
	 * Get HTML wrapper class.
     *
	 * Retrieve the widget container class.
     *
     * @return string Widget container css classes.
	 */
	public function get_html_wrapper_class() {
        $name = $this->get_name();

		return sprintf( 'kfw-ele-%s elementor-widget-%s', $name, $name );
	}

	/**
	 * Widget base constructor.
	 *
	 * Initializing the widget base class.
	 */
	public function __construct( $data = [], $args = null ) {
        parent::__construct( $data, $args );

		wp_register_script( 'kfw-pricing-table',
			KFE_CONST_URL . 'libraries/elementor/assets/js/pricing-table' . KFE_CONST_DEBUG_SUFFIX . '.js',
			[ 'jquery', 'elementor-frontend' ],
			KFE_CONST_VERSION,
			true
		);

		wp_register_style( 'kfw-pricing-table',
			KFE_CONST_URL . 'libraries/elementor/assets/css/pricing-table' . KFE_CONST_DEBUG_SUFFIX . '.css',
			[],
			KFE_CONST_VERSION
		);
    }

	/**
	 * Get style dependencies.
	 *
	 * Retrieve the list of style dependencies the element requires.
	 *
	 * @return array Element styles dependencies.
	 */
	public function get_style_depends() {
		return [ 'kfw-pricing-table' ];
	}

	/**
	 * Get script dependencies.
	 *
	 * Retrieve the list of script dependencies the element requires.
	 *
	 * @return array Element scripts dependencies.
	 */
	public function get_script_depends() {
		return [ 'jquery-numerator', 'kfw-pricing-table' ];
	}

	/**
	 * Register controls.
	 *
	 * Used to add new controls to any element type.
	 */
	protected function register_controls() {
		$this->_register_header_section();
		$this->_register_body_section();

		$this->_register_table_style_section();
		$this->_register_header_style_section();
		$this->_register_body_style_section();
    }

    public function _register_header_section() {
        $repeater = new KFRTR();

		$this->start_controls_section( 'section_table_header', [
			'label' => esc_html__( 'Table Header', 'onnat-extra' )
		]);
            $repeater->add_control( 'table_header_type', [
                'label'   => esc_html__( 'Header Type', 'onnat-extra' ),
                'type'    => KFCM::CHOOSE,
                'default' => 'blank',
                'toggle'  => false,
                'options' => [
                    'blank' => [
                        'title' => esc_html__( 'Blank', 'onnat-extra' ),
                        'icon'  => '',
                    ],
                    'data'  => [
                        'title' => esc_html__( 'Info', 'onnat-extra' ),
                        'icon'  => 'eicon-editor-paragraph',                        
                    ],
                ],
            ]);

            $condition = [
                'table_header_type' => 'data'
            ];
            $repeater->start_controls_tabs('table_headers_tabs' );
                $repeater->start_controls_tab('table_headers_title',[
                    'label'     => esc_html__( 'Title', 'onnat-extra' ),
                    'condition' => $condition,
                ]);
                    $repeater->add_control( 'title', [
                        'label'       => esc_html__( 'Title', 'onnat-extra' ),
                        'type'        => KFCM::TEXT,
                        'label_block' => true,
                        'condition'   => $condition,
                        'default'     => esc_html__( 'Title', 'onnat-extra' ),
                    ]);    
                $repeater->end_controls_tab();
                $repeater->start_controls_tab('table_headers_pricing',[
                    'label'     => esc_html__( 'Pricing', 'onnat-extra' ),
                    'condition' => $condition,
                ]);
                    $repeater->add_control( 'currency_heading', [
                        'label'     => esc_html__( 'Currency', 'onnat-extra' ),
                        'type'      => KFCM::HEADING,
                        'condition' => $condition,
                    ]);
                    $repeater->add_control( 'currency_symbol', [
                        'label'     => esc_html__( 'Symbol', 'onnat-extra' ),
                        'type'      => KFCM::SELECT,
                        'condition' => $condition,
                        'options'   => [
                            ''             => esc_html__( 'None', 'onnat-extra' ),
                            'dollar'       => '&#36; ' . _x( 'Dollar', 'Currency Symbol', 'onnat-extra' ),
                            'euro'         => '&#128; ' . _x( 'Euro', 'Currency Symbol', 'onnat-extra' ),
                            'baht'         => '&#3647; ' . _x( 'Baht', 'Currency Symbol', 'onnat-extra' ),
                            'franc'        => '&#8355; ' . _x( 'Franc', 'Currency Symbol', 'onnat-extra' ),
                            'guilder'      => '&fnof; ' . _x( 'Guilder', 'Currency Symbol', 'onnat-extra' ),
                            'krona'        => 'kr ' . _x( 'Krona', 'Currency Symbol', 'onnat-extra' ),
                            'lira'         => '&#8356; ' . _x( 'Lira', 'Currency Symbol', 'onnat-extra' ),
                            'indian_rupee' => '&#8377; ' . _x( 'Rupee (Indian)', 'Currency Symbol', 'onnat-extra' ),
                            'peseta'       => '&#8359 ' . _x( 'Peseta', 'Currency Symbol', 'onnat-extra' ),
                            'peso'         => '&#8369; ' . _x( 'Peso', 'Currency Symbol', 'onnat-extra' ),
                            'pound'        => '&#163; ' . _x( 'Pound Sterling', 'Currency Symbol', 'onnat-extra' ),
                            'real'         => 'R$ ' . _x( 'Real', 'Currency Symbol', 'onnat-extra' ),
                            'ruble'        => '&#8381; ' . _x( 'Ruble', 'Currency Symbol', 'onnat-extra' ),
                            'rupee'        => '&#8360; ' . _x( 'Rupee', 'Currency Symbol', 'onnat-extra' ),
                            'shekel'       => '&#8362; ' . _x( 'Shekel', 'Currency Symbol', 'onnat-extra' ),
                            'yen'          => '&#165; ' . _x( 'Yen/Yuan', 'Currency Symbol', 'onnat-extra' ),
                            'won'          => '&#8361; ' . _x( 'Won', 'Currency Symbol', 'onnat-extra' ),
                            'custom'       => esc_html__( 'Custom', 'onnat-extra' ),
                        ],
                        'default'   => 'dollar',                        
                    ]);
                    $repeater->add_control( 'currency_symbol_custom', [
                        'label'     => esc_html__( 'Symbol', 'onnat-extra' ),
                        'type'      => KFCM::TEXT,
                        'condition' => [
                            'table_header_type' => 'data',
                            'currency_symbol'   => 'custom',
                        ]
                    ]);
                    $repeater->add_control( 'currency_symbol_format', [
                        'label'     => esc_html__( 'Format', 'onnat-extra' ),
                        'type'      => KFCM::CHOOSE,
                        'condition' => $condition,
                        'options'   => [
                            'sub'  => [
                                'title' =>  esc_html__( 'Subscript', 'onnat-extra' ),
                                'icon'  => 'eicon-v-align-bottom'
                            ],
                            'span' => [
                                'title' =>  esc_html__( 'Unset', 'onnat-extra' ),
                                'icon'  => 'eicon-text-align-justify'
                            ],
                            'sup'  => [
                                'title' =>  esc_html__( 'Superscript', 'onnat-extra' ),
                                'icon'  => 'eicon-v-align-top'
                            ],
                        ],
                        'default' => 'sup',
                        'toggle'  => false,
                    ]);
                    $repeater->add_control( 'price', [
                        'label'     => esc_html__( 'Price', 'onnat-extra' ),
                        'type'      => KFCM::TEXT,
                        'default'   => '69.99',
                        'condition' => $condition,
                        'separator' => 'before',
                    ]);
                    $repeater->add_control( 'price_format', [
                        'label'     => esc_html__( 'Fractional Format', 'onnat-extra' ),
                        'type'      => KFCM::CHOOSE,
                        'options'   => [
                            'sub'  => [
                                'title' =>  esc_html__( 'Subscript', 'onnat-extra' ),
                                'icon'  => 'eicon-v-align-bottom'
                            ],
                            'span' => [
                                'title' =>  esc_html__( 'Unset', 'onnat-extra' ),
                                'icon'  => ' eicon-text-align-justify'
                            ],
                            'sup'  => [
                                'title' =>  esc_html__( 'Superscript', 'onnat-extra' ),
                                'icon'  => 'eicon-v-align-top'
                            ],
                        ],
                        'condition' => $condition,
                        'default'   => 'sup',
                        'toggle'    => false,
                    ]);
                    $repeater->add_control( 'price_format_help', [
                        'type'            => KFCM::RAW_HTML,
                        'raw'             => esc_html__( 'This will add a subscript or superscript design to the fractional portion of the price.', 'onnat-extra' ),
                        'content_classes' => 'elementor-control-field-description kfw-elementor-editor-ctrl-remove-margin-top',
                        'condition'       => $condition,
                        'separator'       => 'none',
                    ]);
                    $repeater->add_control( 'period', [
                        'label'     => esc_html__( 'Period', 'onnat-extra' ),
                        'type'      => KFCM::TEXT,
                        'condition' => $condition,
                        'default'   => '/Month',
                    ]);
                $repeater->end_controls_tab();
                $repeater->start_controls_tab('table_headers_button',[
                    'label'     => esc_html__( 'Button', 'onnat-extra' ),
                    'condition' => $condition,
                ]);
                    $repeater->add_control( 'button_text', [
                        'label'     => esc_html__( 'Text', 'onnat-extra' ),
                        'type'      => KFCM::TEXT,
                        'default'   => esc_html__( 'Button', 'onnat-extra' ),
                        'condition' => $condition,
                    ]);
                    $repeater->add_control( 'button_link', [
                        'label'       => esc_html__( 'Link', 'onnat-extra' ),
                        'type'        => KFCM::URL,
                        'placeholder' => 'https://your-link.com',
                        'condition'   => [
                            'table_header_type' => 'data',
                            'button_text!'      => '',
                        ],
                        'default'     => [ 'url' => '#' ],
                    ]);    
                $repeater->end_controls_tab();
                $repeater->start_controls_tab('table_headers_settings',[
                    'label' => esc_html__( 'Settings', 'onnat-extra' ),
                ]);
                    $repeater->add_control( 'settings', [
                        'label' => esc_html__( 'Customise it?', 'onnat-extra' ),
                        'type'  => KFCM::SWITCHER,
                    ]);
                    $repeater->add_control( 'is_active', [
                        'label'     => esc_html__( 'Add Active Class?', 'onnat-extra' ),
                        'type'      => KFCM::SWITCHER,
                        'condition' => [
                            'settings' => 'yes'
                        ],
                    ]);
                    $repeater->add_control( 'custom_width', [
                        'label'     => esc_html__( 'Custom Width?', 'onnat-extra' ),
                        'type'      => KFCM::SWITCHER,
                        'condition' => [
                            'settings' => 'yes'
                        ],
                    ]);
                    $repeater->add_responsive_control( 'width', [
                        'label'      => esc_html__( 'Width', 'onnat-extra' ),
                        'type'       => KFCM::SLIDER,
                        'condition'  => [
                            'settings'     => 'yes',
                            'custom_width' => 'yes',
                        ],
                        'size_units' => [ '%', 'px' ],
                        'selectors'  => [
                            '{{WRAPPER}} table thead tr {{CURRENT_ITEM}}' => 'width: {{SIZE}}{{UNIT}};',
                        ],
                        'range'      => [
                            '%' => [
                                'min' => 0,
                                'max' => 100,
                            ],
                            'px' => [
                                'min' => 1,
                                'max' => 1000,
                            ],
                        ],      
                    ]);
                    $repeater->add_control( 'custom_colspan', [
                        'label'     => esc_html__( 'Custom ColSpan?', 'onnat-extra' ),
                        'type'      => KFCM::SWITCHER,
                        'condition' => [
                            'settings' => 'yes'
                        ],
                    ]);
                    $repeater->add_control( 'colspan', [
                        'label'     => esc_html__( 'ColSpan?', 'onnat-extra' ),
                        'type'      => KFCM::NUMBER,
                        'condition' => [
                            'settings'       => 'yes',
                            'custom_colspan' => 'yes',
                        ],
                    ]);
                $repeater->end_controls_tab();
            $repeater->end_controls_tabs();

            $this->add_control( 'table_headers', [
                'type'        => KFCM::REPEATER,
                'fields'      => $repeater->get_controls(),
                'default'     => [],
                'title_field' => '{{{ table_header_type == "blank" ? "Blank" : title }}}',
            ]);
		$this->end_controls_section();
    }

    public function _register_body_section() {
        $repeater = new KFRTR();

		$this->start_controls_section( 'section_table_body', [
			'label' => esc_html__( 'Table Body', 'onnat-extra' )
		]);
            $repeater->add_control( 'row', [
                'label' => esc_html__( 'Add Row?', 'onnat-extra' ),
                'type'  => KFCM::SWITCHER,
            ]);
            $repeater->add_control( 'table_body_type', [
                'label'   => esc_html__( 'Body Type', 'onnat-extra' ),
                'type'    => KFCM::CHOOSE,
                'default' => 'blank',
                'toggle'  => false,
                'options' => [
                    'blank' => [
                        'title' => esc_html__( 'Blank', 'onnat-extra' ),
                        'icon'  => '',
                    ],
                    'icon'  => [
                        'title' => esc_html__( 'Icon', 'onnat-extra' ),
                        'icon'  => 'eicon-plus-circle-o',                        
                    ],
                    'text'  => [
                        'title' => esc_html__( 'Text', 'onnat-extra' ),
                        'icon'  => 'eicon-editor-paragraph',
                    ],
                ],
            ]);
            $repeater->start_controls_tabs('table_body_tabs');
                $repeater->start_controls_tab('table_body_title',[
                    'label'     => esc_html__( 'Title', 'onnat-extra' ),
                    'condition' => [
                        'table_body_type' => 'text'
                    ],
                ]);
                    $repeater->add_control( 'title', [
                        'label'       => esc_html__( 'Title', 'onnat-extra' ),
                        'type'        => KFCM::TEXT,
                        'label_block' => true,
                        'condition'   => [
                            'table_body_type' => 'text'
                        ],
                    ]);
                    $repeater->add_group_control(KFGCT::get_type(), [
                        'name'      => 'title_typo',
                        'label'     => esc_html__( 'Typography', 'onnat-extra' ),
                        'selector'  => '{{WRAPPER}} table tbody {{CURRENT_ITEM}} .kfw-ele-pricing-table-info',
                        'condition' => [
                            'table_body_type' => 'text',
                        ],
                    ]);
                    $repeater->add_control( 'table_body_title_color_settings', [
                        'label'        => esc_html__('Colors', 'onnat-extra'),
                        'type'         => KFCM::POPOVER_TOGGLE,
                        'return_value' => 'yes',
                        'condition' => [
                            'table_body_type' => 'text',
                        ],
                    ]);
                    $repeater->start_popover();
                        $repeater->add_control( 'title_color', [
                            'label'     => esc_html__( 'Normal', 'onnat-extra' ),
                            'type'      => KFCM::COLOR,
                            'selectors' => [
                                '{{WRAPPER}} table tbody {{CURRENT_ITEM}} .kfw-ele-pricing-table-info' => 'color:{{VALUE}};'
                            ],
                            'condition' => [
                                'table_body_type' => 'text',
                            ],
                        ]);
                        $repeater->add_control( 'title_hover_color', [
                            'label'     => esc_html__( 'Hover', 'onnat-extra' ),
                            'type'      => KFCM::COLOR,
                            'selectors' => [
                                '{{WRAPPER}} table tbody {{CURRENT_ITEM}}:hover .kfw-ele-pricing-table-info' => 'color:{{VALUE}};'
                            ],
                            'condition' => [
                                'table_body_type' => 'text',
                            ],
                        ]);
                    $repeater->end_popover();
                $repeater->end_controls_tab();

                $repeater->start_controls_tab('table_body_icon',[
                    'label'     => esc_html__( 'Icon', 'onnat-extra' ),
                    'condition' => [
                        'table_body_type' => 'icon'
                    ],
                ]);
                    $repeater->add_control( 'icon', [
                        'label'     => esc_html__( 'Icon', 'onnat-extra' ),
                        'type'      => KFCM::ICONS,
                        'default'   => [
                            'value'   => 'fas fa-heart',
                            'library' => 'fa-solid',
                        ],
                        'condition' => [
                            'table_body_type' => 'icon'
                        ],
                    ]);
                    $repeater->add_responsive_control( 'icon_size', [
                        'label'      => esc_html__('Size', 'onnat-extra' ),
                        'type'       => KFCM::SLIDER,
                        'size_units' => [ 'px' ],
                        'range'      => [
                            'px' => [
                                'min' => 0,
                                'max' => 999,
                            ],
                        ],
                        'selectors' => [
                            '{{WRAPPER}} table tbody {{CURRENT_ITEM}} .kfw-elementor-icon-wrapper' => 'font-size: {{SIZE}}{{UNIT}};',
                        ],
                        'condition' => [
                            'table_body_type' => 'icon',
                        ],    
                    ]);
                    $repeater->add_control( 'table_body_icon_color_settings', [
                        'label'        => esc_html__('Colors', 'onnat-extra'),
                        'type'         => KFCM::POPOVER_TOGGLE,
                        'return_value' => 'yes',
                        'condition'    => [
                            'table_body_type' => 'icon',
                        ],
                    ]);
                    $repeater->start_popover();
                        $repeater->add_control( 'icon_color', [
                            'label'     => esc_html__( 'Normal', 'onnat-extra' ),
                            'type'      => KFCM::COLOR,
                            'selectors' => [
                                '{{WRAPPER}} table tbody {{CURRENT_ITEM}} .kfw-elementor-icon-wrapper i'   => 'color:{{VALUE}};',
                                '{{WRAPPER}} table tbody {{CURRENT_ITEM}} .kfw-elementor-icon-wrapper svg' => 'fill:{{VALUE}};',
                            ],
                            'condition' => [
                                'table_body_type' => 'icon',
                            ],
                        ]);
                        $repeater->add_control( 'icon_hover_color', [
                            'label'     => esc_html__( 'Hover', 'onnat-extra' ),
                            'type'      => KFCM::COLOR,
                            'selectors' => [
                                '{{WRAPPER}} table tbody {{CURRENT_ITEM}}:hover .kfw-elementor-icon-wrapper i'   => 'color:{{VALUE}};',
                                '{{WRAPPER}} table tbody {{CURRENT_ITEM}}:hover .kfw-elementor-icon-wrapper svg' => 'fill:{{VALUE}};',
                            ],
                            'condition' => [
                                'table_body_type' => 'icon',
                            ],
                        ]);
                    $repeater->end_popover();                    
                $repeater->end_controls_tab();
                $repeater->start_controls_tab('table_body_settings',[
                    'label' => esc_html__( 'Settings', 'onnat-extra' ),
                ]);
                    $repeater->add_control( 'settings', [
                        'label' => esc_html__( 'Customise it?', 'onnat-extra' ),
                        'type'  => KFCM::SWITCHER,
                    ]);
                    $repeater->add_responsive_control( 'align',[
                        'label'     => esc_html__( 'Alignment', 'onnat-extra' ),
                        'type'      => KFCM::CHOOSE,
                        'toggle'    => false,
                        'options'   => [
                            'left'    => [
                                'title' => esc_html__( 'Left', 'onnat-extra' ),
                                'icon'  => 'eicon-h-align-left',
                            ],
                            'center'  => [
                                'title' => esc_html__( 'Center', 'onnat-extra' ),
                                'icon'  => 'eicon-h-align-center',
                            ],
                            'right'   => [
                                'title' => esc_html__( 'Right', 'onnat-extra' ),
                                'icon'  => 'eicon-h-align-right',
                            ],
                            'justify' => [
                                'title' => esc_html__( 'Justified', 'onnat-extra' ),
                                'icon' => 'eicon-h-align-stretch',
                            ],
                        ],
                        'selectors' => [
                            '{{WRAPPER}} .kfw-ele-pricing-table-wrapper table tbody {{CURRENT_ITEM}}' => 'text-align: {{VALUE}};',
                        ],
                        'condition' => [
                            'settings' => 'yes',
                        ],
                    ]);

                    $repeater->add_responsive_control( 'vertical_align',[
                        'label'     => esc_html__( 'Vertical Alignment', 'onnat-extra' ),
                        'type'      => KFCM::CHOOSE,
                        'toggle'    => false,
                        'options'   => [
                            'top'    => [
                                'title' => esc_html__( 'Top', 'onnat-extra' ),
                                'icon'  => 'eicon-v-align-top',
                            ],
                            'middle' => [
                                'title' => esc_html__( 'Middle', 'onnat-extra' ),
                                'icon'  => 'eicon-v-align-middle',
                            ],
                            'bottom' => [
                                'title' => esc_html__( 'Bottom', 'onnat-extra' ),
                                'icon'  => 'eicon-v-align-bottom',
                            ],
                        ],
                        'selectors' => [
                            '{{WRAPPER}} .kfw-ele-pricing-table-wrapper table tbody {{CURRENT_ITEM}}' => 'vertical-align: {{VALUE}};',
                        ],
                        'condition' => [
                            'settings' => 'yes',
                        ],
                    ]);

                    $repeater->add_control( 'custom_colspan', [
                        'label'     => esc_html__( 'Custom ColSpan?', 'onnat-extra' ),
                        'type'      => KFCM::SWITCHER,
                        'condition' => [
                            'settings' => 'yes'
                        ],
                    ]);
                    $repeater->add_control( 'colspan', [
                        'label'     => esc_html__( 'ColSpan?', 'onnat-extra' ),
                        'type'      => KFCM::NUMBER,
                        'condition' => [
                            'settings'       => 'yes',
                            'custom_colspan' => 'yes',
                        ],
                    ]);
                    $repeater->add_control( 'custom_rowspan', [
                        'label'     => esc_html__( 'Custom RowSpan?', 'onnat-extra' ),
                        'type'      => KFCM::SWITCHER,
                        'condition' => [
                            'settings' => 'yes'
                        ],
                    ]);
                    $repeater->add_control( 'rowspan', [
                        'label'     => esc_html__( 'RowSpan?', 'onnat-extra' ),
                        'type'      => KFCM::NUMBER,
                        'condition' => [
                            'settings'       => 'yes',
                            'custom_rowspan' => 'yes',
                        ],
                    ]);
                    $repeater->add_control( 'table_body_bg_color_settings', [
                        'label'        => esc_html__('Background Colors', 'onnat-extra'),
                        'type'         => KFCM::POPOVER_TOGGLE,
                        'return_value' => 'yes',
                        'condition'    => [
                            'settings' => 'yes',
                        ],
                    ]);
                    $repeater->start_popover();
                        $repeater->add_control( 'bg_color', [
                            'label'     => esc_html__( 'Normal', 'onnat-extra' ),
                            'type'      => KFCM::COLOR,
                            'selectors' => [
                                '{{WRAPPER}} table tbody {{CURRENT_ITEM}}' => 'background-color:{{VALUE}};'
                            ],
                            'condition' => [
                                'settings' => 'yes',
                            ],
                        ]);
                        $repeater->add_control( 'bg_hover_color', [
                            'label'     => esc_html__( 'Hover', 'onnat-extra' ),
                            'type'      => KFCM::COLOR,
                            'selectors' => [
                                '{{WRAPPER}} table tbody {{CURRENT_ITEM}}:hover' => 'background-color:{{VALUE}};'
                            ],
                            'condition' => [
                                'settings' => 'yes',
                            ],
                        ]);
                    $repeater->end_popover();
                $repeater->end_controls_tab();
            $repeater->end_controls_tabs();

            $this->add_control( 'table_body', [
                'type'        => KFCM::REPEATER,
                'fields'      => $repeater->get_controls(),
                'default'     => [],
                'title_field' => '<#
                    var result = "Blank";
                    var type = table_body_type;

                    if( type == "text" ) {
                        result = title
                    }
                    
                    if( type == "icon" ) {
                        result = elementor.helpers.renderIcon( this, icon, {}, "i", "panel" ) || \'<i class="{{ icon }}" aria-hidden="true"></i>\'
                    }
                #>
                {{ row === "yes" ? "ROW:" : ""  }}
                {{{ result }}}',                
            ]);
		$this->end_controls_section();
    }

    public function _register_table_style_section() {
		$this->start_controls_section( 'section_table_style', [
			'label' => esc_html__( 'Table', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
            $this->add_group_control( KFGCBG::get_type(), [
                'name'     => 'table_background',
                'label'    => esc_html__('Background', 'onnat-extra' ),
                'types'    => [ 'classic', 'gradient' ],
                'exclude'  => [ 'image' ], // to remove background image option in classic mode
                'selector' => '{{WRAPPER}} .kfw-ele-pricing-table-wrapper'
            ]);
            $this->add_responsive_control( 'table_padding', [
                'label'      => esc_html__( 'Padding', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-ele-pricing-table-wrapper' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ],
            ]);
            $this->add_responsive_control( 'table_margin', [
                'label'      => esc_html__( 'Margin', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-ele-pricing-table-wrapper' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ],
            ]);
            $this->add_group_control( KFGCBR::get_type(), [
                'name'           => 'table_border',
                'fields_options' => [
                    'width' => [ 'label' => _x( 'Border Width', 'Border Control', 'onnat-extra' ) ],
                    'color' => [ 'label' => _x( 'Border Color', 'Border Control', 'onnat-extra' ) ],
                ],
                'selector'       => '{{WRAPPER}} .kfw-ele-pricing-table-wrapper'
            ]);
            $this->add_responsive_control( 'table_border_radius', [
                'label'      => esc_html__('Border Radius', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-ele-pricing-table-wrapper' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ]
            ]);
            $this->add_group_control( KFGCBS::get_type(), [
                'name'     => 'table_box_shadow',
                'selector' => '{{WRAPPER}} .kfw-ele-pricing-table-wrapper'
            ]);

            $this->add_control('section_table_inner_cell_style_before_hr', [
                'type' => KFCM::DIVIDER,
            ]);
            $this->add_control('section_table_inner_cell_style_heading',[
                'label' => esc_html__( 'CELL', 'onnat-extra' ),
                'type'  => KFCM::HEADING,
            ]);
            $this->add_control('section_table_inner_cell_style_after_hr', [
                'type' => KFCM::DIVIDER,
            ]);
            $this->add_responsive_control( 'cell_padding', [
                'label'      => esc_html__( 'Padding', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-ele-pricing-table-wrapper table td,{{WRAPPER}} .kfw-ele-pricing-table-wrapper table th' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ],
            ]);
            $this->add_group_control( KFGCBR::get_type(), [
                'name'           => 'cell_border',
                'fields_options' => [
                    'width' => [ 'label' => _x( 'Border Width', 'Border Control', 'onnat-extra' ) ],
                    'color' => [ 'label' => _x( 'Border Color', 'Border Control', 'onnat-extra' ) ],
                ],
                'selector'       => '{{WRAPPER}} .kfw-ele-pricing-table-wrapper table td,{{WRAPPER}} .kfw-ele-pricing-table-wrapper table th',
            ]);
		$this->end_controls_section();
    }

    public function _register_header_style_section() {
		$this->start_controls_section( 'section_header_style', [
			'label' => esc_html__( 'Table Header', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
            $this->_register_header_bg_section();
            $this->_register_header_title_section();
            $this->_register_header_price_section();
            $this->_register_header_button_section();
		$this->end_controls_section();
    }

    public function _register_header_bg_section() {
        $this->add_responsive_control( 'header_padding', [
            'label'      => esc_html__( 'Padding', 'onnat-extra' ),
            'type'       => KFCM::DIMENSIONS,
            'size_units' => [ 'px', 'em', '%' ],
            'selectors'  => [
                '{{WRAPPER}} .kfw-ele-pricing-table-wrapper table thead th' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
            ],
        ]);
        $this->add_control('section_header_style_bg_heading',[
            'label' => esc_html__( 'BACKGROUND', 'onnat-extra' ),
            'type'  => KFCM::HEADING,
        ]);
        $this->add_control('section_header_style_bg_after_hr', [
            'type' => KFCM::DIVIDER,
        ]);
        $this->start_controls_tabs( 'section_header_bg_customize_tabs' );
            $this->start_controls_tab( 'section_header_bg_normal_state', [
                'label' => esc_html__( 'Normal', 'onnat-extra' ),
            ]);
                $this->add_group_control( KFGCBG::get_type(), [
                    'name'       => 'table_header_bg',
                    'label'      => esc_html__('Background', 'onnat-extra' ),
                    'types'      => [ 'classic', 'gradient' ],
                    'exclude'    => [ 'image' ], # To remove background image option in classic mode
                    'selector'   => '{{WRAPPER}} .kfw-ele-pricing-table-wrapper table thead th',
                ]);
                $this->add_group_control( KFGCBR::get_type(), [
                    'name'           => 'table_header_border',
                    'fields_options' => [
                        'width' => [ 'label' => _x( 'Border Width', 'Border Control', 'onnat-extra' ) ],
                        'color' => [ 'label' => _x( 'Border Color', 'Border Control', 'onnat-extra' ) ],
                    ],
                    'selector'       => '{{WRAPPER}} .kfw-ele-pricing-table-wrapper table thead th',
                ]);
                $this->add_responsive_control( 'table_header_border_radius', [
                    'label'      => esc_html__('Border Radius', 'onnat-extra' ),
                    'type'       => KFCM::DIMENSIONS,
                    'size_units' => [ 'px', '%' ],
                    'selectors'  => [
                        '{{WRAPPER}} .kfw-ele-pricing-table-wrapper table thead th' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                    ],
                ]);
                $this->add_group_control( KFGCBS::get_type(), [
                    'name'       => 'table_header_box_shadow',
                    'selector'   => '{{WRAPPER}} .kfw-ele-pricing-table-wrapper table thead th',
                ]);
            $this->end_controls_tab();
            $this->start_controls_tab( 'section_header_bg_hover_state', [
                'label' => esc_html__( 'Hover', 'onnat-extra' ),
            ]);
                $this->add_group_control( KFGCBG::get_type(), [
                    'name'       => 'table_header_hover_bg',
                    'label'      => esc_html__('Background', 'onnat-extra' ),
                    'types'      => [ 'classic', 'gradient' ],
                    'exclude'    => [ 'image' ], # To remove background image option in classic mode
                    'selector'   => '{{WRAPPER}} .kfw-ele-pricing-table-wrapper table thead th:hover',
                ]);
                $this->add_group_control( KFGCBR::get_type(), [
                    'name'           => 'table_header_hover_border',
                    'fields_options' => [
                        'width' => [ 'label' => _x( 'Border Width', 'Border Control', 'onnat-extra' ) ],
                        'color' => [ 'label' => _x( 'Border Color', 'Border Control', 'onnat-extra' ) ],
                    ],
                    'selector'       => '{{WRAPPER}} .kfw-ele-pricing-table-wrapper table thead th:hover',
                ]);
                $this->add_responsive_control( 'table_header_hover_border_radius', [
                    'label'      => esc_html__('Border Radius', 'onnat-extra' ),
                    'type'       => KFCM::DIMENSIONS,
                    'size_units' => [ 'px', '%' ],
                    'selectors'  => [
                        '{{WRAPPER}} .kfw-ele-pricing-table-wrapper table thead th:hover' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                    ],
                ]);
                $this->add_group_control( KFGCBS::get_type(), [
                    'name'       => 'table_header_hover_box_shadow',
                    'selector'   => '{{WRAPPER}} .kfw-ele-pricing-table-wrapper table thead th:hover',
                ]);
            $this->end_controls_tab();
            $this->start_controls_tab( 'section_header_bg_active_state', [
                'label' => esc_html__( 'Active', 'onnat-extra' ),
            ]);
                $this->add_group_control( KFGCBG::get_type(), [
                    'name'       => 'table_header_active_bg',
                    'label'      => esc_html__('Background', 'onnat-extra' ),
                    'types'      => [ 'classic', 'gradient' ],
                    'exclude'    => [ 'image' ], # To remove background image option in classic mode
                    'selector'   => '{{WRAPPER}} .kfw-ele-pricing-table-wrapper table thead th.kfw-ele-pricing-table-active',
                ]);
                $this->add_group_control( KFGCBR::get_type(), [
                    'name'           => 'table_header_active_border',
                    'fields_options' => [
                        'width' => [ 'label' => _x( 'Border Width', 'Border Control', 'onnat-extra' ) ],
                        'color' => [ 'label' => _x( 'Border Color', 'Border Control', 'onnat-extra' ) ],
                    ],
                    'selector'       => '{{WRAPPER}} .kfw-ele-pricing-table-wrapper table thead th.kfw-ele-pricing-table-active',
                ]);
                $this->add_responsive_control( 'table_header_active_border_radius', [
                    'label'      => esc_html__('Border Radius', 'onnat-extra' ),
                    'type'       => KFCM::DIMENSIONS,
                    'size_units' => [ 'px', '%' ],
                    'selectors'  => [
                        '{{WRAPPER}} .kfw-ele-pricing-table-wrapper table thead th.kfw-ele-pricing-table-active' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                    ],
                ]);
                $this->add_group_control( KFGCBS::get_type(), [
                    'name'       => 'table_header_active_box_shadow',
                    'selector'   => '{{WRAPPER}} .kfw-ele-pricing-table-wrapper table thead th.kfw-ele-pricing-table-active',
                ]);            
            $this->end_controls_tab();
        $this->end_controls_tabs();        
    }

    public function _register_header_title_section() {
        $this->add_control('section_header_title_style_heading_before_hr', [
            'type' => KFCM::DIVIDER,
        ]);
        $this->add_control('section_header_title_style_heading',[
            'label' => esc_html__( 'TITLE', 'onnat-extra' ),
            'type'  => KFCM::HEADING,
        ]);
        $this->add_control('section_header_title_style_heading_after_hr', [
            'type' => KFCM::DIVIDER,
        ]);
        $this->add_group_control(KFGCT::get_type(), [
            'name'     => 'table_header_title_typo',
            'label'    => esc_html__( 'Typography', 'onnat-extra' ),
            'selector' => '{{WRAPPER}} .kfw-ele-pricing-table-wrapper table thead .kfw-ele-pricing-table-title',
        ]);
        $this->add_control( 'table_header_title_color_settings', [
            'label'        => esc_html__('Colors', 'onnat-extra'),
            'type'         => KFCM::POPOVER_TOGGLE,
            'return_value' => 'yes',
        ]);
        $this->start_popover();
            $this->add_control( 'table_header_title_color', [
                'label'     => esc_html__('Normal Color', 'onnat-extra' ),
                'type'      => KFCM::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .kfw-ele-pricing-table-wrapper table thead .kfw-ele-pricing-table-title' => 'color:{{VALUE}};',
                ],
            ]);
            $this->add_control( 'table_header_title_hover_color', [
                'label'     => esc_html__('Hover Color', 'onnat-extra' ),
                'type'      => KFCM::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .kfw-ele-pricing-table-wrapper table thead th:hover .kfw-ele-pricing-table-title' => 'color:{{VALUE}};',
                ],
            ]);
            $this->add_control( 'table_header_title_active_color', [
                'label'     => esc_html__('Active Color', 'onnat-extra' ),
                'type'      => KFCM::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .kfw-ele-pricing-table-wrapper table thead th.kfw-ele-pricing-table-active .kfw-ele-pricing-table-title' => 'color:{{VALUE}};',
                ],
            ]);
        $this->end_popover();        
    }

    public function _register_header_price_section() {
        $this->add_control('section_header_pricing_style_heading_before_hr', [
            'type' => KFCM::DIVIDER,
        ]);
        $this->add_control('section_header_pricing_style_heading',[
            'label' => esc_html__( 'PRICE', 'onnat-extra' ),
            'type'  => KFCM::HEADING,
        ]);
        $this->add_control('section_header_pricing_style_heading_after_hr', [
            'type' => KFCM::DIVIDER,
        ]);
        $this->add_group_control(KFGCT::get_type(), [
            'name'     => 'table_header_pricing_typo',
            'label'    => esc_html__( 'Typography', 'onnat-extra' ),
            'selector' => '{{WRAPPER}} .kfw-ele-pricing-table-wrapper table thead .kfw-ele-pricing-table-currency',
        ]);
        $this->add_control( 'table_header_pricing_color_settings', [
            'label'        => esc_html__('Colors', 'onnat-extra'),
            'type'         => KFCM::POPOVER_TOGGLE,
            'return_value' => 'yes',
        ]);
        $this->start_popover();
            $this->add_control( 'table_header_pricing_color', [
                'label'     => esc_html__('Normal Color', 'onnat-extra' ),
                'type'      => KFCM::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .kfw-ele-pricing-table-wrapper table thead .kfw-ele-pricing-table-currency' => 'color:{{VALUE}};',
                ],
            ]);
            $this->add_control( 'table_header_pricing_hover_color', [
                'label'     => esc_html__('Hover Color', 'onnat-extra' ),
                'type'      => KFCM::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .kfw-ele-pricing-table-wrapper table thead th:hover .kfw-ele-pricing-table-currency' => 'color:{{VALUE}};',
                ],
            ]);
            $this->add_control( 'table_header_pricing_active_color', [
                'label'     => esc_html__('Active Color', 'onnat-extra' ),
                'type'      => KFCM::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .kfw-ele-pricing-table-wrapper table thead th.kfw-ele-pricing-table-active .kfw-ele-pricing-table-currency' => 'color:{{VALUE}};',
                ],
            ]);
        $this->end_popover();        
    }

    public function _register_header_button_section() {
        $this->add_control('section_header_button_style_heading_before_hr', [
            'type' => KFCM::DIVIDER,
        ]);
        $this->add_control('section_header_button_style_heading',[
            'label' => esc_html__( 'BUTTON', 'onnat-extra' ),
            'type'  => KFCM::HEADING,
        ]);
        $this->add_control('section_header_button_style_heading_after_hr', [
            'type' => KFCM::DIVIDER,
        ]);
        $this->add_group_control( KFGCT::get_type(), [
            'name'        => 'table_header_button_typo',
            'description' => esc_html__( 'Typography', 'onnat-extra' ),
            'selector'    => '{{WRAPPER}} .kfw-ele-pricing-table-wrapper table thead .kfw-ele-pricing-table-button'
        ]);
        $this->add_responsive_control( 'table_header_button_padding', [
            'label'      => esc_html__( 'Padding', 'onnat-extra' ),
            'type'       => KFCM::DIMENSIONS,
            'size_units' => [ 'px', 'em', '%' ],
            'selectors'  => [
                '{{WRAPPER}} .kfw-ele-pricing-table-wrapper table thead .kfw-ele-pricing-table-button' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
            ],
        ]);
        $this->add_responsive_control( 'table_header_button_margin', [
            'label'      => esc_html__( 'Margin', 'onnat-extra' ),
            'type'       => KFCM::DIMENSIONS,
            'size_units' => [ 'px', 'em', '%' ],
            'selectors'  => [
                '{{WRAPPER}} .kfw-ele-pricing-table-wrapper table thead .kfw-ele-pricing-table-button' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
            ],
        ]);
        $this->start_controls_tabs( 'table_header_button_customize_tabs' );
            $this->start_controls_tab( 'table_header_button_normal_state', [
                'label' => esc_html__( 'Normal', 'onnat-extra' ),
            ]);
                $this->add_control( 'table_header_button_color', [
                    'label'     => esc_html__('Button Text Color', 'onnat-extra' ),
                    'type'      => KFCM::COLOR,
                    'selectors' => ['{{WRAPPER}} .kfw-ele-pricing-table-wrapper table thead .kfw-ele-pricing-table-button' => 'color: {{VALUE}};' ],
                ]);
                $this->add_group_control( KFGCBG::get_type(), [
                    'name'     => 'table_header_button_background',
                    'label'    => esc_html__('Background', 'onnat-extra' ),
                    'types'    => [ 'classic', 'gradient' ],
                    'exclude'  => [ 'image' ], // to remove background image option in classic mode
                    'selector' => '{{WRAPPER}} .kfw-ele-pricing-table-wrapper table thead .kfw-ele-pricing-table-button'
                ]);
                $this->add_group_control( KFGCBR::get_type(), [
                    'name'           => 'table_header_button_border',
                    'fields_options' => [
                        'width' => [ 'label' => _x( 'Border Width', 'Border Control', 'onnat-extra' ) ],
                        'color' => [ 'label' => _x( 'Border Color', 'Border Control', 'onnat-extra' ) ],
                    ],
                    'selector'       => '{{WRAPPER}} .kfw-ele-pricing-table-wrapper table thead .kfw-ele-pricing-table-button'
                ]);
                $this->add_responsive_control( 'table_header_button_border_radius', [
                    'label'      => esc_html__('Border Radius', 'onnat-extra' ),
                    'type'       => KFCM::DIMENSIONS,
                    'size_units' => [ 'px', '%' ],
                    'selectors'  => [ '{{WRAPPER}} .kfw-ele-pricing-table-wrapper table thead .kfw-ele-pricing-table-button' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};' ]
                ]);
                $this->add_group_control( KFGCBS::get_type(), [
                    'name'     => 'table_header_button_box_shadow',
                    'selector' => '{{WRAPPER}} .kfw-ele-pricing-table-wrapper table thead .kfw-ele-pricing-table-button'
                ]);
            $this->end_controls_tab();
            $this->start_controls_tab( 'table_header_button_hover_state', [
                'label'     => esc_html__( 'Hover', 'onnat-extra' ),
            ]);
                $this->add_control( 'table_header_button_color_hover', [
                    'label'     => esc_html__('Button Text Color', 'onnat-extra' ),
                    'type'      => KFCM::COLOR,
                    'selectors' => [ '{{WRAPPER}} .kfw-ele-pricing-table-wrapper table thead .kfw-ele-pricing-table-button:hover' => 'color: {{VALUE}};' ],
                ]);
                $this->add_group_control( KFGCBG::get_type(), [
                    'name'     => 'table_header_button_background_hover',
                    'label'    => esc_html__('Background', 'onnat-extra' ),
                    'types'    => [ 'classic', 'gradient' ],
                    'exclude'  => [ 'image' ], // to remove background image option in classic mode
                    'selector' => '{{WRAPPER}} .kfw-ele-pricing-table-wrapper table thead .kfw-ele-pricing-table-button:hover'
                ]);
                $this->add_group_control( KFGCBR::get_type(), [
                    'name'           => 'table_header_button_border_hover',
                    'fields_options' => [
                        'width'  => [ 'label' => _x( 'Border Width', 'Border Control', 'onnat-extra' ) ],
                        'color'  => [ 'label' => _x( 'Border Color', 'Border Control', 'onnat-extra' ) ],
                    ],
                    'selector'       => '{{WRAPPER}} .kfw-ele-pricing-table-wrapper table thead .kfw-ele-pricing-table-button:hover'
                ]);
                $this->add_responsive_control( 'table_header_button_border_radius_hover', [
                    'label'      => esc_html__('Border Radius', 'onnat-extra' ),
                    'type'       => KFCM::DIMENSIONS,
                    'size_units' => [ 'px', '%' ],
                    'selectors'  => [
                        '{{WRAPPER}} .kfw-ele-pricing-table-wrapper table thead .kfw-ele-pricing-table-button:hover' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    ]
                ]);
                $this->add_group_control( KFGCBS::get_type(), [
                    'name'      => 'table_header_button_box_shadow_hover',
                    'selector'  => '{{WRAPPER}} .kfw-ele-pricing-table-wrapper table thead .kfw-ele-pricing-table-button:hover'
                ]);
                $this->add_control( 'table_header_button_animation_hover', [
                    'label' => esc_html__( 'Animation', 'onnat-extra' ),
                    'type'  => KFCM::HOVER_ANIMATION,
                ]);
            $this->end_controls_tab();
            $this->start_controls_tab( 'table_header_button_active_state', [
                'label'     => esc_html__( 'Active', 'onnat-extra' ),
            ]);
                $this->add_control( 'table_header_button_color_active', [
                    'label'     => esc_html__('Button Text Color', 'onnat-extra' ),
                    'type'      => KFCM::COLOR,
                    'selectors' => [ '{{WRAPPER}} .kfw-ele-pricing-table-wrapper table thead th.kfw-ele-pricing-table-active .kfw-ele-pricing-table-button' => 'color: {{VALUE}};' ],
                ]);
                $this->add_group_control( KFGCBG::get_type(), [
                    'name'     => 'table_header_button_background_active',
                    'label'    => esc_html__('Background', 'onnat-extra' ),
                    'types'    => [ 'classic', 'gradient' ],
                    'exclude'  => [ 'image' ], // to remove background image option in classic mode
                    'selector' => '{{WRAPPER}} .kfw-ele-pricing-table-wrapper table thead th.kfw-ele-pricing-table-active .kfw-ele-pricing-table-button'
                ]);
                $this->add_group_control( KFGCBR::get_type(), [
                    'name'           => 'table_header_button_border_active',
                    'fields_options' => [
                        'width'  => [ 'label' => _x( 'Border Width', 'Border Control', 'onnat-extra' ) ],
                        'color'  => [ 'label' => _x( 'Border Color', 'Border Control', 'onnat-extra' ) ],
                    ],
                    'selector'       => '{{WRAPPER}} .kfw-ele-pricing-table-wrapper table thead th.kfw-ele-pricing-table-active .kfw-ele-pricing-table-button'
                ]);
                $this->add_responsive_control( 'table_header_button_border_radius_active', [
                    'label'      => esc_html__('Border Radius', 'onnat-extra' ),
                    'type'       => KFCM::DIMENSIONS,
                    'size_units' => [ 'px', '%' ],
                    'selectors'  => [
                        '{{WRAPPER}} .kfw-ele-pricing-table-wrapper table thead th.kfw-ele-pricing-table-active .kfw-ele-pricing-table-button' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    ]
                ]);
                $this->add_group_control( KFGCBS::get_type(), [
                    'name'      => 'table_header_button_box_shadow_active',
                    'selector'  => '{{WRAPPER}} .kfw-ele-pricing-table-wrapper table thead th.kfw-ele-pricing-table-active .kfw-ele-pricing-table-button'
                ]);
            $this->end_controls_tab();            
        $this->end_controls_tabs();        
    }

    public function _register_body_style_section() {
		$this->start_controls_section( 'section_body_style', [
			'label' => esc_html__( 'Table Body', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
            $this->_register_body_align_section();
            $this->_register_body_bg_section();
            $this->_register_body_title_section();
            $this->_register_body_icon_section();
		$this->end_controls_section();
    }

    public function _register_body_align_section() {
        $this->add_responsive_control( 'body_padding', [
            'label'      => esc_html__( 'Padding', 'onnat-extra' ),
            'type'       => KFCM::DIMENSIONS,
            'size_units' => [ 'px', 'em', '%' ],
            'selectors'  => [
                '{{WRAPPER}} .kfw-ele-pricing-table-wrapper table tbody td' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
            ],
        ]);
        $this->add_responsive_control( 'align',[
            'label'     => esc_html__( 'Alignment', 'onnat-extra' ),
            'type'      => KFCM::CHOOSE,
            'toggle'    => false,
            'options'   => [
                'left'    => [
                    'title' => esc_html__( 'Left', 'onnat-extra' ),
                    'icon'  => 'eicon-h-align-left',
                ],
                'center'  => [
                    'title' => esc_html__( 'Center', 'onnat-extra' ),
                    'icon'  => 'eicon-h-align-center',
                ],
                'right'   => [
                    'title' => esc_html__( 'Right', 'onnat-extra' ),
                    'icon'  => 'eicon-h-align-right',
                ],
                'justify' => [
                    'title' => esc_html__( 'Justified', 'onnat-extra' ),
                    'icon' => 'eicon-h-align-stretch',
                ],
            ],
            'selectors' => [
                '{{WRAPPER}} .kfw-ele-pricing-table-wrapper table tbody td' => 'text-align: {{VALUE}};',
            ],
        ]);
        
        $this->add_responsive_control( 'vertical_align',[
            'label'     => esc_html__( 'Vertical Alignment', 'onnat-extra' ),
            'type'      => KFCM::CHOOSE,
            'toggle'    => false,
            'options'   => [
                'top'    => [
                    'title' => esc_html__( 'Top', 'onnat-extra' ),
                    'icon'  => 'eicon-v-align-top',
                ],
                'middle' => [
                    'title' => esc_html__( 'Middle', 'onnat-extra' ),
                    'icon'  => 'eicon-v-align-middle',
                ],
                'bottom' => [
                    'title' => esc_html__( 'Bottom', 'onnat-extra' ),
                    'icon'  => 'eicon-v-align-bottom',
                ],
            ],
            'selectors' => [
                '{{WRAPPER}} .kfw-ele-pricing-table-wrapper table tbody td' => 'vertical-align: {{VALUE}};',
            ],  
        ]);
    }

    public function _register_body_bg_section() {
        $this->add_control('section_body_style_bg_before_hr', [
            'type' => KFCM::DIVIDER,
        ]);
        $this->add_control('section_body_style_bg_heading',[
            'label' => esc_html__( 'BACKGROUND', 'onnat-extra' ),
            'type'  => KFCM::HEADING,
        ]);
        $this->add_control('section_body_style_bg_after_hr', [
            'type' => KFCM::DIVIDER,
        ]);
        $this->start_controls_tabs( 'section_body_bg_customize_tabs' );
            $this->start_controls_tab( 'section_body_bg_normal_state', [
                'label' => esc_html__( 'Normal', 'onnat-extra' ),
            ]);
                $this->add_control( 'body_bg_color', [
                    'label'     => esc_html__( 'Color', 'onnat-extra' ),
                    'type'      => KFCM::COLOR,
                    'selectors' => [
                        '{{WRAPPER}} .kfw-ele-pricing-table-wrapper table tbody td' => 'background-color:{{VALUE}};'
                    ],
                ]);
                $this->add_control( 'body_bg_color_alt', [
                    'label' => esc_html__( 'Alternate Color', 'onnat-extra' ),
                    'type'  => KFCM::SWITCHER,
                ]);
                $this->add_control( 'body_bg_alt_color', [
                    'label'     => esc_html__( 'Alternate BG Color', 'onnat-extra' ),
                    'type'      => KFCM::COLOR,
                    'condition' => [
                        'body_bg_color_alt' => 'yes',
                    ],
                    'selectors' => [
                        '{{WRAPPER}} .kfw-ele-pricing-table-wrapper table tbody tr:nth-of-type(2n) td' => 'background-color:{{VALUE}};'
                    ],
                ]);
            $this->end_controls_tab();
            $this->start_controls_tab( 'section_body_bg_hover_state', [
                'label' => esc_html__( 'Hover', 'onnat-extra' ),
            ]);
                $this->add_control( 'body_bg_hover_color', [
                    'label'     => esc_html__( 'Color', 'onnat-extra' ),
                    'type'      => KFCM::COLOR,
                    'selectors' => [
                        '{{WRAPPER}} .kfw-ele-pricing-table-wrapper table tbody td:hover' => 'background-color:{{VALUE}};'
                    ],
                ]);
                $this->add_control( 'body_bg_h_alt_color', [
                    'label'     => esc_html__( 'Alternate BG Color', 'onnat-extra' ),
                    'type'      => KFCM::COLOR,
                    'condition' => [
                        'body_bg_color_alt' => 'yes',
                    ],
                    'selectors' => [
                        '{{WRAPPER}} .kfw-ele-pricing-table-wrapper table tbody tr:nth-of-type(2n) td:hover' => 'background-color:{{VALUE}};'
                    ],
                ]);
            $this->end_controls_tab();
        $this->end_controls_tabs();
    }

    public function _register_body_title_section() {
        $this->add_control('section_body_tile_style_before_hr', [
            'type' => KFCM::DIVIDER,
        ]);
        $this->add_control('section_body_title_style_heading',[
            'label' => esc_html__( 'TEXT', 'onnat-extra' ),
            'type'  => KFCM::HEADING,
        ]);
        $this->add_control('section_body_tile_style_after_hr', [
            'type' => KFCM::DIVIDER,
        ]);
        $this->add_group_control(KFGCT::get_type(), [
            'name'      => 'title_typo',
            'label'     => esc_html__( 'Typography', 'onnat-extra' ),
            'selector'  => '{{WRAPPER}} .kfw-ele-pricing-table-wrapper table tbody td .kfw-ele-pricing-table-info',
        ]);
        $this->add_control( 'table_body_title_color_settings', [
            'label'        => esc_html__('Colors', 'onnat-extra'),
            'type'         => KFCM::POPOVER_TOGGLE,
            'return_value' => 'yes',
        ]);
        $this->start_popover();
            $this->add_control( 'title_color', [
                'label'     => esc_html__( 'Normal', 'onnat-extra' ),
                'type'      => KFCM::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .kfw-ele-pricing-table-wrapper table tbody td .kfw-ele-pricing-table-info' => 'color:{{VALUE}};'
                ],
            ]);
            $this->add_control( 'title_hover_color', [
                'label'     => esc_html__( 'Hover', 'onnat-extra' ),
                'type'      => KFCM::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .kfw-ele-pricing-table-wrapper table tbody td:hover .kfw-ele-pricing-table-info' => 'color:{{VALUE}};'
                ],
            ]);
        $this->end_popover();        
    }

    public function _register_body_icon_section() {
        $this->add_control('section_body_icon_style_before_hr', [
            'type' => KFCM::DIVIDER,
        ]);
        $this->add_control('section_body_icon_style_heading',[
            'label' => esc_html__( 'ICON', 'onnat-extra' ),
            'type'  => KFCM::HEADING,
        ]);
        $this->add_control('section_body_icon_style_after_hr', [
            'type' => KFCM::DIVIDER,
        ]);
        $this->add_responsive_control( 'icon_size', [
            'label'      => esc_html__('Size', 'onnat-extra' ),
            'type'       => KFCM::SLIDER,
            'size_units' => [ 'px' ],
            'range'      => [
                'px' => [
                    'min' => 0,
                    'max' => 999,
                ],
            ],
            'selectors' => [
                '{{WRAPPER}} .kfw-ele-pricing-table-wrapper table tbody .kfw-elementor-icon-wrapper' => 'font-size: {{SIZE}}{{UNIT}};',
            ],
        ]);
        $this->add_control( 'table_body_icon_color_settings', [
            'label'        => esc_html__('Colors', 'onnat-extra'),
            'type'         => KFCM::POPOVER_TOGGLE,
            'return_value' => 'yes',
        ]);
        $this->start_popover();
            $this->add_control( 'icon_color', [
                'label'     => esc_html__( 'Normal', 'onnat-extra' ),
                'type'      => KFCM::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .kfw-ele-pricing-table-wrapper table tbody td .kfw-elementor-icon-wrapper i' => 'color:{{VALUE}};',
                    '{{WRAPPER}} .kfw-ele-pricing-table-wrapper table tbody td .kfw-elementor-icon-wrapper svg' => 'color:{{VALUE}};'
                ],
            ]);
            $this->add_control( 'icon_hover_color', [
                'label'     => esc_html__( 'Hover', 'onnat-extra' ),
                'type'      => KFCM::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .kfw-ele-pricing-table-wrapper table tbody td:hover .kfw-elementor-icon-wrapper i' => 'color:{{VALUE}};',
                    '{{WRAPPER}} .kfw-ele-pricing-table-wrapper table tbody td:hover .kfw-elementor-icon-wrapper svg' => 'color:{{VALUE}};'
                ],
            ]);
        $this->end_popover();
    }

    public function _pricing( $settings ) {

        $currency_html = '';
        $symbol        = $settings['currency_symbol'];
        $price_format  = $settings['price_format'];
        $delimiter     = ( $price_format == 'sub' || $price_format == 'sup' ) ? '.' : ' ';
        $price         = explode( $delimiter, $settings['price'] );
        $price1        = $price[0];
        $price2        = '';

        if( count( $price ) === 2 ) {
            $price2 = $price[1];
        }

        if( !empty( $symbol ) ) {

            if( $symbol == 'custom' ) {
                $symbol = $settings['currency_symbol_custom'];
            } else {
                $symbol = kf_extra_plugin_elementor()->get_currency_symbol( $symbol );
            }

            $currency_html .= sprintf( '<%1$s class="kfw-ele-pricing-table-price-currency">%2$s</%1$s>',
                esc_html( $settings['currency_symbol_format'] ),
                esc_html( $symbol )
            );
        }

        if( $price_format ==='sub' || $price_format === 'sup' ) {

            if( !empty( $price1 ) ) {
                $currency_html .= sprintf( '<span class="kfw-ele-pricing-table-price-numeric" data-toval="%1$s" data-round="0"></span>',
                    esc_html( $price1 )
                );
            }

            if( 0 < $price2 ) {
                $currency_html .= sprintf( '<%1$s class="kfw-ele-pricing-table-price-fractional" data-toval="%2$s"></%1$s>',
                    esc_html( $price_format ),
                    esc_html( $price2 )
                );
            }
        } else {
            $price      = explode( ".", $settings['price'] );
            $fractional = 0;

            if( count( $price ) === 2 ) {
                $fractional = strlen( $price[1] );
            }

            $currency_html .= sprintf( '<span class="kfw-ele-pricing-table-price-numeric" data-toval="%1$s" data-round="%2$s"></span>',
                esc_html( $price1 ),
                esc_html( $fractional )
            );
        }

        $period = $settings['period'];
        if( !empty( $settings['price'] ) && !empty( $period) ) {
            $period = sprintf( "<div class='kfw-ele-pricing-table-period'>%s</div>", esc_html( $period )  );
        }
        
        if( !empty( $currency_html ) ) {
            printf( '<div class="kfw-ele-pricing-table-currency"><div class="kfw-ele-pricing-table-price-wrapper">%s</div>%s</div>',
                $currency_html,
                $period
            );
        }
    }

	public function _button( $settings ) {
		$this->add_render_attribute( 'cta-wrapper', [
			'class' => array_filter([
				'kfw-ele-pricing-table-button-wrapper',
			])
		]);

		echo '<div '.$this->get_render_attribute_string( 'cta-wrapper' ).'>';

            $this->add_render_attribute( 'cta', [
				'class' => array_filter([
					'kfw-ele-pricing-table-button',
                ]),
            ]);

			if ( ! empty( $settings['button_link']['url'] ) ) {
				$this->add_render_attribute( 'cta', 'href', esc_url( $settings['button_link']['url'] ) );

				if ( ! empty( $settings['button_link']['is_external'] ) ) {
					$this->add_render_attribute( 'cta', 'target', '_blank' );
				}

				if ( ! empty( $settings['button_link']['nofollow'] ) ) {
					$this->add_render_attribute( 'cta', 'rel', 'nofollow' );
				}
			}

			echo '<a '.$this->get_render_attribute_string( 'cta' ).'>';
				echo esc_html( $settings['button_text'] );
			echo '</a>';
            
		echo '</div>';
    }

    public function _render_icons( $icon , $echo = true ) {
		$icon = array_filter( $icon );

		if( !empty( $icon ) ) {
			echo '<span class="kfw-elementor-icon-wrapper">';
				KFIM::render_icon( $icon, [ 'aria-hidden' => 'true' ] );
			echo '</span>';
		}
	}    

	/**
	 * Render widget output on the frontend.
	 */
    protected function render() {
		$render = kf_extra_plugin_elementor()->get_template("kfw-pricing-table");
		include $render;
    }
}    