<?php

// Do not allow directly accessing this file.
if ( ! defined( 'ABSPATH' ) ) {
	exit( 'Direct script access denied.' );
}

if( !class_exists( 'KinForce_WP_Radio_Taxonomy' ) ) {

    class KinForce_WP_Radio_Taxonomy {

        /**
         * taxonomy name
         */
        public $taxonomy = null;

        /**
         * taxonomy object
         */
        public $tax_obj = null;

        /**
         * whether to filter get_terms() or not
         */
        private $set = true;

        public function __construct( $taxonomy ) {
            $this->taxonomy = $taxonomy;
            $this->tax_obj  = get_taxonomy( $taxonomy );

            // Replace new taxonomy meta box.
            add_action( 'add_meta_boxes', [ $this, 'replace_meta_box' ], 10, 2 );

            // Change checkboxes to radios & trigger get_terms() filter.
            add_filter( 'wp_terms_checklist_args', [ $this, 'filter_terms_checklist_args' ] );

            if( $this->is_wp_version_gte('4.4.0') ) {
                add_action( 'wp_ajax_add-' . $this->taxonomy, [ $this, 'add_non_hierarchical_term' ], 5 );
            }

            // Never save more than 1 term.
            add_action( 'save_post', [ $this, 'save_single_term' ] );
            add_action( 'edit_attachment', [ $this, 'save_single_term' ] );

    		// Hack global taxonomy to switch all radio taxonomies to hierarchical on edit screen.
            add_action( 'load-edit.php', array( $this, 'make_hierarchical' ) );
            add_action( 'wp_ajax_inline-save', array( $this, 'make_hierarchical' ), 0 );
        }

        public function replace_meta_box( $post_type, $post ) {
            if( !is_wp_error( $this->tax_obj ) && isset( $this->tax_obj->object_type ) ) {

                // Get posttypes this taxonomy is connected to.
                $posttypes = $this->tax_obj->object_type;

				if (
					empty( $this->taxonomy ) || empty( $post_type )
					|| ! isset( $posttypes)
					|| ! isset( $post_type )
					|| ! in_array( $post_type,  $posttypes )
				) {
					return;
				}

				// Metaboxes to remove.
				$remove_id = ! is_taxonomy_hierarchical( $this->taxonomy ) ? 'tagsdiv-'. $this->taxonomy : $this->taxonomy .'div';
				remove_meta_box( $remove_id, $post_type, 'side' );

                // Metaboxes to add.
				$add_id = ! is_taxonomy_hierarchical( $this->taxonomy ) ? 'kinfw-radio-tagsdiv-' . $this->taxonomy : 'kinfw-radio-' . $this->taxonomy . 'div';
                add_meta_box( $add_id, $this->tax_obj->labels->name, [ $this, 'meta_box' ], $post_type, 'side', 'core', [ 'taxonomy' => $this->taxonomy ] );
            }
        }

        /**
         * Callback to set up the metabox
         */
        public function meta_box( $post, $box ) {
            $defaults = [ 'taxonomy' => 'category' ];

            if ( ! isset( $box['args'] ) || ! is_array( $box['args'] ) ) {
                $args = [];
            } else {
                $args = $box['args'];
            }

            $r = wp_parse_args( $args, $defaults );
            $tax_name = esc_attr( $r['taxonomy'] );
            $taxonomy = get_taxonomy( $r['taxonomy'] );

            // Get current terms.
            $checked_terms = isset( $post->ID ) ? get_the_terms( $post->ID, $tax_name ) : [];

            // Get first term, a single term.
            $single_term = ! empty( $checked_terms ) && ! is_wp_error( $checked_terms ) ? array_pop( $checked_terms ) : false;
            $single_term_id = $single_term ? (int) $single_term->term_id : 0;
            ?>
            <div id="taxonomy-<?php echo $tax_name; ?>" class="kinfw-radio-buttons-for-taxonomies categorydiv">
                <ul id="<?php echo $tax_name; ?>-tabs" class="category-tabs">
                    <li class="tabs">
                        <a href="#<?php echo $tax_name; ?>-all"><?php echo $taxonomy->labels->all_items; ?></a>
                    </li>
                    <li class="hide-if-no-js">
                        <a href="#<?php echo $tax_name; ?>-pop"><?php echo esc_html( $taxonomy->labels->most_used ); ?></a>
                    </li>
                </ul>
                <div id="<?php echo $tax_name; ?>-pop" class="tabs-panel" style="display: none;">
                    <ul id="<?php echo $tax_name; ?>checklist-pop" class="categorychecklist form-no-clear" >
                        <?php
                            $popular_terms = get_terms( $tax_name, [
                                'orderby'      => 'count',
                                'order'        => 'DESC',
                                'number'       => 10,
                                'hierarchical' => false
                            ] );
                            $popular_ids   = [];

                            foreach( $popular_terms as $term ) {
                                $popular_ids[] = $term->term_id;
                                $value         = is_taxonomy_hierarchical( $tax_name ) ? $term->term_id : $term->slug;
                                $id            = 'popular-' . $tax_name . '-' . $term->term_id;
                                $checked       = checked( $single_term_id, $term->term_id, false );
                                ?>
                                    <li id="<?php echo $id; ?>" class="popular-category">
                                        <label class="selectit">
                                            <input
                                                id="in-<?php echo $id; ?>"
                                                type="radio"
                                                <?php echo $checked; ?>
                                                value="<?php echo (int) $term->term_id; ?>"
                                                <?php disabled( ! current_user_can( $taxonomy->cap->assign_terms ) ); ?>
                                            />
                                            <?php echo esc_html( apply_filters( 'the_category', $term->name, '', '' ) ); ?>
                                        </label>
                                    </li>
                                <?php
                            }
                        ?>
                    </ul>
                </div>

                <div id="<?php echo $tax_name; ?>-all" class="tabs-panel">
                    <ul
                        id="<?php echo $tax_name; ?>checklist"
                        data-wp-lists="list:<?php echo $tax_name; ?>"
                        class="categorychecklist form-no-clear">
                        <?php
                            wp_terms_checklist( $post->ID, [
                                'taxonomy'      => $tax_name,
                                'popular_cats'  => $popular_ids,
                                'selected_cats' => [ $single_term_id ]
                            ] );
                        ?>
                    </ul>
                </div>

                <?php if ( current_user_can( $taxonomy->cap->edit_terms ) ) : ?>
                    <div id="<?php echo $tax_name; ?>-adder" class="wp-hidden-children">
                        <a id="<?php echo $tax_name; ?>-add-toggle" href="#<?php echo $tax_name; ?>-add" class="hide-if-no-js taxonomy-add-new">
                            <?php printf( '+ %s', $taxonomy->labels->add_new_item ); ?>
                        </a>
                        <p id="<?php echo $tax_name; ?>-add" class="category-add wp-hidden-child">
                            <label class="screen-reader-text" for="new<?php echo $tax_name; ?>">
                                <?php echo $taxonomy->labels->add_new_item; ?>
                            </label>
                            <input
                                type="text"
                                name="new<?php echo $tax_name; ?>"
                                id="new<?php echo $tax_name; ?>"
                                class="form-required form-input-tip"
                                value="<?php echo esc_attr( $taxonomy->labels->new_item_name ); ?>"
                                aria-required="true"
                            />
                            <label class="screen-reader-text" for="new<?php echo $tax_name; ?>_parent">
                                <?php echo $taxonomy->labels->parent_item_colon; ?>
                            </label>
                            <?php
                                // Only add parent option for hierarchical taxonomies.
                                if( is_taxonomy_hierarchical( $tax_name ) ) {
                                    $parent_dropdown_args = [
                                        'taxonomy'         => $tax_name,
                                        'hide_empty'       => 0,
                                        'name'             => 'new' . $tax_name . '_parent',
                                        'orderby'          => 'name',
                                        'hierarchical'     => 1,
                                        'show_option_none' => '&mdash; ' . $taxonomy->labels->parent_item . ' &mdash;',
                                    ];

                                    /**
                                     * Filters the arguments for the taxonomy parent dropdown on the Post Edit page.
                                     */
                                    $parent_dropdown_args = apply_filters( 'post_edit_category_parent_dropdown_args', $parent_dropdown_args );
                                    wp_dropdown_categories( $parent_dropdown_args );
                                }
                            ?>
                            <input
                                type="button"
                                id="<?php echo $tax_name; ?>-add-submit"
                                data-wp-lists="add:<?php echo $tax_name; ?>checklist:<?php echo $tax_name; ?>-add"
                                class="button category-add-submit"
                                value="<?php echo esc_attr( $taxonomy->labels->add_new_item ); ?>"
                            />
                            <?php wp_nonce_field( 'add-' . $tax_name, '_ajax_nonce-add-' . $tax_name, false ); ?>
                            <span id="<?php echo $tax_name; ?>-ajax-response"></span>
                        </p>
                    </div>
                <?php endif; ?>
            </div>
            <?php
        }

        /**
         * Tell checklist function to use our new Walker
         */
        public function filter_terms_checklist_args( $args ) {
            if( isset( $args['taxonomy']) && $this->taxonomy == $args['taxonomy'] ) {

                // Add a filter to get_terms() but only for radio lists.
                $this->set_terms_filter( true );
                add_filter( 'get_terms', [ $this, 'get_terms'] , 10, 3 );

                $args['walker'] = new KinForce_Walker_Category_Radio;
            }

            return $args;
        }

        /**
         * Turn on/off the terms filter.
         */
        private function set_terms_filter( $_set = true ) {
            $this->set = (bool) $_set;
        }

        /**
         * Only filter get_terms() in the wp_terms_checklist() function
         */
        private function get_terms_filter() {
            return apply_filters( 'kinfw-filter/plugin/radio-buttons-for-taxonomies/no-term-' . $this->taxonomy, $this->set );
        }

        /**
         * Add new 0 or null term in metabox and quickedit
         * this will allow users to "undo" a term if the taxonomy is not required
         */
        public function get_terms( $terms, $taxonomies, $args ) {
            if( in_array( $this->taxonomy, ( array ) $taxonomies ) && isset( $args['fields'] ) && $args['fields'] == 'all' ) {
                $default_term = intval( get_option( 'default_' . $this->taxonomy, 0 ) );

                if ( ! $default_term && $this->get_terms_filter() ) {
                    // Remove filter after 1st run.
                    remove_filter( current_filter(), __FUNCTION__, 10, 3 );

                    // Turn the switch OFF.
                    $this->set_terms_filter( false );

                    $no_term = sprintf( esc_html__( 'No %s', 'onnat-extra' ), $this->tax_obj->labels->singular_name );
                    $no_term = apply_filters( 'kinfw-filter/plugin/radio-buttons-for-taxonomies/no-term-selected-text', $no_term, $this->tax_obj->labels->singular_name );

                    $uncategorized = (object) [
                        'term_id' => '0',
                        'slug'    => '0',
                        'name'    => $no_term,
                        'parent'  => '0'
                    ];

                    array_push( $terms, $uncategorized );
                }
            }

            return $terms;
        }

        /**
         * Add new term from metabox
         */
        public function add_non_hierarchical_term() {
            $action = $_POST[ 'action' ];
            $tax_name = substr( $action, 4 );

            // If Hierarchical, pass-through to core callback.
            if( is_taxonomy_hierarchical( $tax_name ) ) {
                return false;
            }

            // Non-Hierarchical "Terms".
            $taxonomy = get_taxonomy( $tax_name );
            check_ajax_referer( $action, '_ajax_nonce-add-' . $taxonomy->name );
            if ( ! current_user_can( $taxonomy->cap->edit_terms ) ) {
                wp_die( -1 );
            }

            $names = explode( ',', $_POST['new'.$taxonomy->name] );
            foreach ( $names as $cat_name ) {
                $cat_name = trim($cat_name);
                $category_nicename = sanitize_title($cat_name);

                if ( '' === $category_nicename ) {
                    continue;
                }

                if ( ! $cat_id = term_exists( $cat_name, $taxonomy->name ) ) {
                    $cat_id = wp_insert_term( $cat_name, $taxonomy->name );
                }

                if ( is_wp_error( $cat_id ) ) {
                    continue;
                } else if ( is_array( $cat_id ) ) {
                    $cat_id = $cat_id['term_id'];
                }

                $data = sprintf('
                    <li id="%1$s-%2$s">
                        <label class="selectit">
                            <input id="in-%1$s-%2$s" type="radio" name="radio_tax_input[%1$s][]" value="%2$s" checked="checked">
                            %3$s
                        </label>
                    </li>',
                    esc_attr( $taxonomy->name ),
                    intval( $cat_id ),
                    esc_html( $cat_name )
                );

                $add = [
                    'what'     => $taxonomy->name,
                    'id'       => $cat_id,
                    'data'     => str_replace( array("\n", "\t"), '', $data),
                    'position' => -1
                ];
            }

            $x = new WP_Ajax_Response( $add );
            $x->send();
        }

        /**
         * Only ever save a single term
         */
        public function save_single_term( $post_id ) {

            // Verify if this is an auto save routine. If it is our form has not been submitted, so we dont want to do anything.
            if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
                return $post_id;
            }

            // Prevent weirdness with multisite.
            if( function_exists( 'ms_is_switched' ) && ms_is_switched() ) {
                return $post_id;
            }

            // Make sure we're on a supported post type.
            if ( is_array( $this->tax_obj->object_type ) && isset( $_REQUEST['post_type'] ) && ! in_array ( $_REQUEST['post_type'], $this->tax_obj->object_type ) ) {
                return $post_id;
            }

            // Check capabilities.
            if ( ! current_user_can( $this->tax_obj->cap->assign_terms ) ) {
                return $post_id;
            }

            // If posts are being bulk edited, and no term is selected, do nothing.
            if ( ! empty( $_GET[ 'bulk_edit' ] ) && empty ( $_REQUEST[ 'radio_tax_input' ][ "{$this->taxonomy}" ] ) ) {
                return $post_id;
            }

    		// Verify nonce.
            if ( ! isset( $_REQUEST["_radio_nonce-{$this->taxonomy}"]) || ! wp_verify_nonce( $_REQUEST["_radio_nonce-{$this->taxonomy}"], "radio_nonce-{$this->taxonomy}" ) ) {
                return $post_id;
            }

            // OK, we must be authenticated by now: we need to make sure we're only saving 1 term.
            if ( ! empty ( $_REQUEST["radio_tax_input"]["{$this->taxonomy}"] ) ) {
                $terms       = (array) $_REQUEST["radio_tax_input"]["{$this->taxonomy}"];
                $single_term = intval( array_shift( $terms ) );
            } else {
                // If not saving any terms, set to default.
                $single_term = intval( get_option( 'default_' . $this->taxonomy, 0 ) );
            }

            // Set the single terms.
            wp_set_object_terms( $post_id, $single_term, $this->taxonomy );

            return $post_id;
        }

        public function make_hierarchical() {
            global $wp_taxonomies;
            $wp_taxonomies[$this->taxonomy]->hierarchical = TRUE;
        }

        public function is_wp_version_gte( $version = '4.4.0' ) {
            global $wp_version;
            return version_compare( $wp_version, $version, '>=' );
        }
    }
}